/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.tdb.store;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.HashSet;
import java.util.Set;
import org.apache.jena.atlas.lib.BitsLong;
import org.apache.jena.atlas.lib.Bytes;
import org.apache.jena.atlas.logging.Log;
import org.apache.jena.datatypes.RDFDatatype;
import org.apache.jena.datatypes.xsd.XSDDatatype;
import org.apache.jena.graph.Node;
import org.apache.jena.graph.NodeFactory;
import org.apache.jena.graph.impl.LiteralLabel;
import org.apache.jena.sparql.graph.NodeConst;
import org.apache.jena.sparql.util.NodeUtils;
import org.apache.jena.tdb.TDBException;
import org.apache.jena.tdb.store.DateTimeNode;
import org.apache.jena.tdb.store.DecimalNode;
import org.apache.jena.tdb.store.IntegerNode;
import org.apache.jena.tdb.sys.SystemTDB;

public final class NodeId {
    public static final NodeId NodeDoesNotExist = new NodeId(-8L);
    public static final NodeId NodeIdAny = new NodeId(-9L);
    private static final boolean enableInlineLiterals = SystemTDB.enableInlineLiterals;
    public static final int SIZE = 8;
    final long value;
    public static final int NONE = 0;
    public static final int INTEGER = 1;
    public static final int DECIMAL = 2;
    public static final int DATE = 3;
    public static final int DATETIME = 4;
    public static final int BOOLEAN = 5;
    public static final int SHORT_STRING = 6;
    public static final int SPECIAL = 255;
    private static Set<RDFDatatype> datatypes = new HashSet<RDFDatatype>();

    public static NodeId create(long value) {
        if (value == NodeId.NodeDoesNotExist.value) {
            return NodeDoesNotExist;
        }
        if (value == NodeId.NodeIdAny.value) {
            return NodeIdAny;
        }
        return new NodeId(value);
    }

    public static NodeId create(byte[] b) {
        return NodeId.create(b, 0);
    }

    public static NodeId create(ByteBuffer b) {
        return NodeId.create(b, 0);
    }

    public static NodeId create(byte[] b, int idx) {
        long value = Bytes.getLong(b, idx);
        return NodeId.create(value);
    }

    public static NodeId create(ByteBuffer b, int idx) {
        long value = b.getLong(idx);
        return NodeId.create(value);
    }

    public NodeId(long v) {
        this.value = v;
    }

    public void toByteBuffer(ByteBuffer b, int idx) {
        b.putLong(idx, this.value);
    }

    public void toBytes(byte[] b, int idx) {
        Bytes.setLong(this.value, b, idx);
    }

    public boolean isDirect() {
        return this.type() != 0 && this.type() != 255;
    }

    public int type() {
        return (int)BitsLong.unpack(this.value, 56, 64);
    }

    static long setType(long value, int type) {
        return BitsLong.pack(value, type, 56, 64);
    }

    public long getId() {
        return this.value;
    }

    public int hashCode() {
        return (int)this.value ^ (int)(this.value >> 32);
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (!(other instanceof NodeId)) {
            return false;
        }
        return this.value == ((NodeId)other).value;
    }

    public String toString() {
        if (this == NodeDoesNotExist) {
            return "[DoesNotExist]";
        }
        if (this == NodeIdAny) {
            return "[Any]";
        }
        return String.format("[%016X]", this.value);
    }

    public static NodeId inline(Node node) {
        if (node == null) {
            Log.warn(NodeId.class, "Null node: " + node);
            return null;
        }
        if (!enableInlineLiterals) {
            return null;
        }
        if (!node.isLiteral()) {
            return null;
        }
        if (NodeUtils.isSimpleString(node) || NodeUtils.isLangString(node)) {
            return null;
        }
        try {
            return NodeId.inline$(node);
        }
        catch (Throwable th) {
            Log.warn(NodeId.class, "Failed to process " + node);
            return null;
        }
    }

    public static boolean hasInlineDatatype(Node node) {
        RDFDatatype dtn = node.getLiteralDatatype();
        return datatypes.contains(dtn);
    }

    private static NodeId inline$(Node node) {
        LiteralLabel lit = node.getLiteral();
        if (node.getLiteralDatatype().equals(XSDDatatype.XSDdecimal)) {
            if (!XSDDatatype.XSDdecimal.isValidLiteral(lit)) {
                return null;
            }
            BigDecimal decimal = new BigDecimal(lit.getLexicalForm().trim());
            DecimalNode dn = DecimalNode.valueOf(decimal);
            if (dn != null) {
                return new NodeId(dn.pack());
            }
            return null;
        }
        if (XSDDatatype.XSDinteger.isValidLiteral(lit)) {
            if (lit.getLexicalForm().length() > 19) {
                return null;
            }
            try {
                long v = ((Number)lit.getValue()).longValue();
                v = IntegerNode.pack(v);
                if (v != -1L) {
                    return new NodeId(v);
                }
                return null;
            }
            catch (Throwable ex) {
                return null;
            }
        }
        if (XSDDatatype.XSDdateTime.isValidLiteral(lit)) {
            long v = DateTimeNode.packDateTime(lit.getLexicalForm());
            if (v == -1L) {
                return null;
            }
            v = NodeId.setType(v, 4);
            return new NodeId(v);
        }
        if (XSDDatatype.XSDdate.isValidLiteral(lit)) {
            long v = DateTimeNode.packDate(lit.getLexicalForm());
            if (v == -1L) {
                return null;
            }
            v = NodeId.setType(v, 3);
            return new NodeId(v);
        }
        if (XSDDatatype.XSDboolean.isValidLiteral(lit)) {
            long v = 0L;
            boolean b = (Boolean)lit.getValue();
            v = NodeId.setType(v, 5);
            if (b) {
                v |= 1L;
            }
            return new NodeId(v);
        }
        return null;
    }

    public static boolean isInline(NodeId nodeId) {
        if (nodeId == NodeDoesNotExist) {
            return false;
        }
        long v = nodeId.value;
        int type = nodeId.type();
        switch (type) {
            case 0: {
                return false;
            }
            case 255: {
                return false;
            }
            case 1: 
            case 2: 
            case 3: 
            case 4: 
            case 5: {
                return true;
            }
        }
        throw new TDBException("Unrecognized node id type: " + type);
    }

    public static Node extract(NodeId nodeId) {
        if (nodeId == NodeDoesNotExist) {
            return null;
        }
        long v = nodeId.value;
        int type = nodeId.type();
        switch (type) {
            case 0: {
                return null;
            }
            case 255: {
                return null;
            }
            case 1: {
                long val = IntegerNode.unpack(v);
                Node n = NodeFactory.createLiteral(Long.toString(val), XSDDatatype.XSDinteger);
                return n;
            }
            case 2: {
                BigDecimal d = DecimalNode.unpackAsBigDecimal(v);
                String x = d.toPlainString();
                return NodeFactory.createLiteral(x, XSDDatatype.XSDdecimal);
            }
            case 4: {
                long val = BitsLong.clear(v, 56, 64);
                String lex = DateTimeNode.unpackDateTime(val);
                return NodeFactory.createLiteral(lex, XSDDatatype.XSDdateTime);
            }
            case 3: {
                long val = BitsLong.clear(v, 56, 64);
                String lex = DateTimeNode.unpackDate(val);
                return NodeFactory.createLiteral(lex, XSDDatatype.XSDdate);
            }
            case 5: {
                long val = BitsLong.clear(v, 56, 64);
                if (val == 0L) {
                    return NodeConst.nodeFalse;
                }
                if (val == 1L) {
                    return NodeConst.nodeTrue;
                }
                throw new TDBException("Unrecognized boolean node id : " + val);
            }
        }
        throw new TDBException("Unrecognized node id type: " + type);
    }

    public final boolean isConcrete() {
        return !NodeId.isAny(this) && !NodeId.isDoesNotExist(this);
    }

    public static final boolean isConcrete(NodeId nodeId) {
        if (nodeId == null) {
            return false;
        }
        if (nodeId == NodeIdAny) {
            return false;
        }
        return nodeId != NodeDoesNotExist;
    }

    public static final boolean isAny(NodeId nodeId) {
        return nodeId == NodeIdAny || nodeId == null;
    }

    public static final boolean isDoesNotExist(NodeId nodeId) {
        return nodeId == NodeDoesNotExist;
    }

    static {
        datatypes.add(XSDDatatype.XSDdecimal);
        datatypes.add(XSDDatatype.XSDinteger);
        datatypes.add(XSDDatatype.XSDdate);
        datatypes.add(XSDDatatype.XSDdateTime);
        datatypes.add(XSDDatatype.XSDboolean);
        datatypes.add(XSDDatatype.XSDlong);
        datatypes.add(XSDDatatype.XSDint);
        datatypes.add(XSDDatatype.XSDshort);
        datatypes.add(XSDDatatype.XSDbyte);
        datatypes.add(XSDDatatype.XSDunsignedLong);
        datatypes.add(XSDDatatype.XSDunsignedInt);
        datatypes.add(XSDDatatype.XSDunsignedShort);
        datatypes.add(XSDDatatype.XSDunsignedByte);
        datatypes.add(XSDDatatype.XSDpositiveInteger);
        datatypes.add(XSDDatatype.XSDnegativeInteger);
        datatypes.add(XSDDatatype.XSDnonPositiveInteger);
        datatypes.add(XSDDatatype.XSDnonNegativeInteger);
        datatypes.add(XSDDatatype.XSDdateTimeStamp);
    }
}

