/*
 * $Id: AbstractEndpointURIBuilder.java 14763 2009-05-23 15:17:40Z rossmason $
 * --------------------------------------------------------------------------------------
 * Copyright (c) MuleSource, Inc.  All rights reserved.  http://www.mulesource.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.endpoint;

import org.mule.api.endpoint.EndpointURI;
import org.mule.api.endpoint.EndpointURIBuilder;
import org.mule.api.endpoint.MalformedEndpointException;
import org.mule.api.MuleContext;
import org.mule.util.PropertiesUtils;
import org.mule.RegistryContext;
import org.mule.MuleServer;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLDecoder;
import java.util.Properties;

/**
 * {@link UrlEndpointURIBuilder} is the default endpointUri strategy suitable for
 * most connectors
 */

public abstract class AbstractEndpointURIBuilder implements EndpointURIBuilder
{
    protected String address;
    protected String endpointName;
    protected String connectorName;
    protected String transformers;
    protected String responseTransformers;
    protected String userInfo;
    private URI uri;

    public EndpointURI build(URI uri) throws MalformedEndpointException
    {
        this.uri = uri;
        Properties props = getPropertiesForURI(uri);
        String replaceAddress = null;
        //If the address has been set as a parameter on the URI, then we must ensure that that value is used
        //for the address. We still call the setEndpoint() method so that other information on the URI
        //is still processed
        if (address != null)
        {
            replaceAddress = address;
            setEndpoint(uri, props);
            address = replaceAddress;
        }
        else
        {
            setEndpoint(uri, props);
        }

        EndpointURI ep = new MuleEndpointURI(address, endpointName, connectorName, transformers,
            responseTransformers, props, this.uri, userInfo);
        address = null;
        endpointName = null;
        connectorName = null;
        transformers = null;
        responseTransformers = null;
        uri = null;
        return ep;
    }

    protected void rewriteURI(URI newURI)
    {
        this.uri = newURI;
    }

    protected abstract void setEndpoint(URI uri, Properties props) throws MalformedEndpointException;

    protected Properties getPropertiesForURI(URI uri) throws MalformedEndpointException
    {
        Properties properties = PropertiesUtils.getPropertiesFromQueryString(uri.getQuery());

        String tempEndpointName = (String) properties.get(EndpointURI.PROPERTY_ENDPOINT_NAME);
        if (tempEndpointName != null)
        {
            this.endpointName = tempEndpointName;
        }
        // override the endpointUri if set
        String endpoint = (String) properties.get(EndpointURI.PROPERTY_ENDPOINT_URI);
        if (endpoint != null)
        {
            this.address = endpoint;
            address = decode(address, uri);
        }

        String cnnName = (String) properties.get(EndpointURI.PROPERTY_CONNECTOR_NAME);
        if (cnnName != null)
        {
            this.connectorName = cnnName;
        }

        transformers = (String) properties.get(EndpointURI.PROPERTY_TRANSFORMERS);
        if (transformers != null)
        {
            transformers = transformers.replaceAll(" ", ",");
        }
        responseTransformers = (String) properties.get(EndpointURI.PROPERTY_RESPONSE_TRANSFORMERS);
        if (responseTransformers != null)
        {
            responseTransformers = responseTransformers.replaceAll(" ", ",");
        }
        userInfo = uri.getUserInfo();
        return properties;
    }

    private String decode(String string, URI uri) throws MalformedEndpointException
    {
        try
        {
            MuleContext context = MuleServer.getMuleContext();
            String encoding = "UTF-8";
            if(context!=null)
            {
                encoding = context.getConfiguration().getDefaultEncoding();
            }
            return URLDecoder.decode(string, encoding);
        }
        catch (UnsupportedEncodingException e)
        {
            throw new MalformedEndpointException(uri.toString(), e);
        }
    }
}
