/*
 * $Id: EndpointURIEndpointBuilderTestCase.java 18691 2010-07-31 00:38:46Z mike.schilling $
 * --------------------------------------------------------------------------------------
 * Copyright (c) MuleSource, Inc.  All rights reserved.  http://www.mulesource.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.endpoint;

import org.mule.MessageExchangePattern;
import org.mule.api.MuleException;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.endpoint.InboundEndpoint;
import org.mule.api.endpoint.OutboundEndpoint;
import org.mule.retry.policies.NoRetryPolicyTemplate;
import org.mule.tck.AbstractMuleTestCase;
import org.mule.tck.testmodels.mule.TestConnector;
import org.mule.transaction.MuleTransactionConfig;
import org.mule.util.ObjectNameHelper;

public class EndpointURIEndpointBuilderTestCase extends AbstractMuleTestCase
{
    public void testBuildInboundEndpoint() throws Exception
    {
        String uri = "test://address";
        EndpointBuilder endpointBuilder = new EndpointURIEndpointBuilder(uri, muleContext);
        ImmutableEndpoint ep = endpointBuilder.buildInboundEndpoint();
        assertTrue(ep instanceof InboundEndpoint);
        assertFalse(ep instanceof OutboundEndpoint);
        assertNotNull(ep.getTransformers());
        // We no longer apply default transport transformers as part of endpoint processing
        assertEquals(0, ep.getTransformers().size());
        assertNotNull(ep.getResponseTransformers());
        // We no longer apply default transport transformers as part of endpoint processing
        assertEquals(0, ep.getResponseTransformers().size());
        testDefaultCommonEndpointAttributes(ep);
    }

    public void testBuildOutboundEndpoint() throws MuleException
    {
        String uri = "test://address";
        EndpointBuilder endpointBuilder = new EndpointURIEndpointBuilder(uri, muleContext);
        ImmutableEndpoint ep = endpointBuilder.buildOutboundEndpoint();
        assertFalse(ep instanceof InboundEndpoint);
        assertTrue(ep instanceof OutboundEndpoint);
        // We no longer apply default transport transformers as part of endpoint processing
        assertEquals(0, ep.getTransformers().size());
        assertNotNull(ep.getResponseTransformers());
        // We no longer apply default transport transformers as part of endpoint processing
        assertEquals(0, ep.getResponseTransformers().size());
        testDefaultCommonEndpointAttributes(ep);
    }

    // TODO DF: Test more than defaults with tests using builder to set non-default
    // values

    protected void testDefaultCommonEndpointAttributes(ImmutableEndpoint ep)
    {
        assertEquals(ep.getEndpointURI().getUri().toString(), "test://address");
        assertEquals(muleContext.getConfiguration().getDefaultResponseTimeout(), ep.getResponseTimeout());
        assertTrue("ep.getRetryPolicyTemplate() = " + ep.getRetryPolicyTemplate().getClass(), ep.getRetryPolicyTemplate() instanceof NoRetryPolicyTemplate);
        assertTrue(ep.getTransactionConfig() instanceof MuleTransactionConfig);
        assertTrue(ep.getTransactionConfig() instanceof MuleTransactionConfig);
        assertEquals(null, ep.getSecurityFilter());
        assertTrue(ep.getConnector() instanceof TestConnector);
        assertEquals(new ObjectNameHelper(muleContext).getEndpointName(ep.getEndpointURI()), ep.getName());
        assertFalse(ep.isDeleteUnacceptedMessages());
        assertEquals(muleContext.getConfiguration().getDefaultEncoding(), ep.getEncoding());
        assertEquals(null, ep.getFilter());
        assertEquals(ImmutableEndpoint.INITIAL_STATE_STARTED, ep.getInitialState());
    }
    
    public void testHasSetEncodingMethod() throws Exception
    {
        String uri = "test://address";
        EndpointBuilder endpointBuilder = new EndpointURIEndpointBuilder(uri, muleContext);
        assertNotNull(endpointBuilder.getClass().getMethod("setEncoding", new Class[]{String.class}));
    }
    
    public void testEndpointBuilderFromEndpoint() throws Exception
    {
        String uri = "test://address";
        ImmutableEndpoint endpoint = getTestInboundEndpoint("endpoint.test.address", uri);
        SensingEndpointURIEndpointBuilder builder = new SensingEndpointURIEndpointBuilder(endpoint);
        assertEquals(uri, builder.getEndpointBuilder().getEndpoint().getUri().toString());
        assertEquals(endpoint.getConnector(), builder.getConnector());
        assertEquals(endpoint.getProperties(), builder.getProperties());
        assertEquals(endpoint.getTransactionConfig(), builder.getTransactionConfig());
        assertEquals(endpoint.isDeleteUnacceptedMessages(), builder.getDeleteUnacceptedMessages(builder.getConnector()));
        assertEquals(endpoint.getInitialState(), builder.getInitialState(builder.getConnector()));
        assertEquals(endpoint.getResponseTimeout(), builder.getResponseTimeout(builder.getConnector()));
        assertEquals(endpoint.getSecurityFilter(), builder.getSecurityFilter());
        assertEquals(endpoint.getRetryPolicyTemplate(), builder.getRetryPolicyTemplate(builder.getConnector()));
        assertEquals(MessageExchangePattern.ONE_WAY, builder.getExchangePattern());
    }
    
    private static class SensingEndpointURIEndpointBuilder extends EndpointURIEndpointBuilder
    {
        public SensingEndpointURIEndpointBuilder(ImmutableEndpoint endpoint)
        {
            super(endpoint);
        }
        
        public MessageExchangePattern getExchangePattern()
        {
            return messageExchangePattern;
        }
    }
}
