/*
 * $Id: DefaultMuleSession.java 18952 2010-08-11 16:07:25Z rossmason $
 * -------------------------------------------------------------------------------------
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.session;

import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.MuleMessage;
import org.mule.api.MuleSession;
import org.mule.api.construct.FlowConstruct;
import org.mule.api.security.SecurityContext;
import org.mule.api.transport.SessionHandler;
import org.mule.config.i18n.CoreMessages;
import org.mule.util.CaseInsensitiveHashMap;
import org.mule.util.UUID;
import org.mule.util.store.DeserializationPostInitialisable;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OptionalDataException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.map.CaseInsensitiveMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * <code>DefaultMuleSession</code> manages the interaction and distribution of events for
 * Mule Services.
 */

public final class DefaultMuleSession implements MuleSession, DeserializationPostInitialisable
{
    /**
     * Serial version
     */
    private static final long serialVersionUID = 3380926585676521866L;

    /**
     * logger used by this class
     */
    private static Log logger = LogFactory.getLog(DefaultMuleSession.class);

    /**
     * The Mule service associated with the session
     * <p/>
     * Note: This object uses custom serialization via the writeObject()/readObject() methods.
     */
    private transient FlowConstruct flowConstruct = null;

    /**
     * Determines if the service is valid
     */
    private boolean valid = true;

    private String id;

    /**
     * The security context associated with the session.
     * Note that this context will only be serialized if the SecurityContext object is Serializable.
     */
    private SecurityContext securityContext;

    private Map<String, Object> properties = null;

    /**
     * The Mule context
     * <p/>
     * Note: This object uses custom serialization via the readObject() method.
     */
    private transient MuleContext muleContext;

    private transient Map<String, Object> serializedData = null;

    public DefaultMuleSession(MuleContext muleContext)
    {
        this((FlowConstruct) null, muleContext);
    }

    public DefaultMuleSession(FlowConstruct flowConstruct, MuleContext muleContext)
    {
        this.muleContext = muleContext;
        properties = new CaseInsensitiveHashMap/*<String, Object>*/();
        id = UUID.getUUID();
        this.flowConstruct = flowConstruct;
    }

    /**
     * @deprecated Use DefaultMuleSession(Service service, MuleContext muleContext) instead
     */
    @Deprecated
    public DefaultMuleSession(MuleMessage message,
                              SessionHandler requestSessionHandler,
                              FlowConstruct flowConstruct,
                              MuleContext muleContext) throws MuleException
    {
        this(message, requestSessionHandler, muleContext);
        if (flowConstruct == null)
        {
            throw new IllegalArgumentException(CoreMessages.propertiesNotSet("flowConstruct").toString());
        }
        this.flowConstruct = flowConstruct;
    }

    /**
     * @deprecated Use DefaultMuleSession(MuleContext muleContext) instead
     */
    @Deprecated
    public DefaultMuleSession(MuleMessage message, SessionHandler requestSessionHandler, MuleContext muleContext) throws MuleException
    {
        this(muleContext);

        if (requestSessionHandler == null)
        {
            throw new IllegalArgumentException(
                    CoreMessages.propertiesNotSet("requestSessionHandler").toString());
        }

        if (message == null)
        {
            throw new IllegalArgumentException(
                    CoreMessages.propertiesNotSet("message").toString());
        }

        properties = new CaseInsensitiveMap/*<String, Object>*/();
        requestSessionHandler.retrieveSessionInfoFromMessage(message, this);
        id = getProperty(requestSessionHandler.getSessionIDKey());
        if (id == null)
        {
            id = UUID.getUUID();
            if (logger.isDebugEnabled())
            {
                logger.debug("There is no session id on the request using key: "
                        + requestSessionHandler.getSessionIDKey() + ". Generating new session id: " + id);
            }
        }
        else if (logger.isDebugEnabled())
        {
            logger.debug("Got session with id: " + id);
        }
    }

    public DefaultMuleSession(MuleSession session, MuleContext muleContext)
    {
        this.muleContext = muleContext;
        this.id = session.getId();
        this.securityContext = session.getSecurityContext();
        this.flowConstruct = session.getFlowConstruct();
        this.valid = session.isValid();

        this.properties = new HashMap<String, Object>();
        for (String key : session.getPropertyNamesAsSet())
        {
            this.properties.put(key, session.getProperty(key));
        }
    }

    public String getId()
    {
        return id;
    }

    public boolean isValid()
    {
        return valid;
    }

    public void setValid(boolean value)
    {
        valid = value;
    }

    /**
     * @return Returns the service.
     */
    public FlowConstruct getFlowConstruct()
    {
        return flowConstruct;
    }

    public void setFlowConstruct(FlowConstruct flowConstruct)
    {
        this.flowConstruct = flowConstruct;
    }

    /**
     * The security context for this session. If not null outbound, inbound and/or
     * method invocations will be authenticated using this context
     *
     * @param context the context for this session or null if the request is not
     *                secure.
     */
    public void setSecurityContext(SecurityContext context)
    {
        securityContext = context;
    }

    /**
     * The security context for this session. If not null outbound, inbound and/or
     * method invocations will be authenticated using this context
     *
     * @return the context for this session or null if the request is not secure.
     */
    public SecurityContext getSecurityContext()
    {
        return securityContext;
    }

    /**
     * Will set a session level property. These will either be stored and retrieved
     * using the underlying transport mechanism of stored using a default mechanism
     *
     * @param key   the key for the object data being stored on the session
     * @param value the value of the session data
     */
    public void setProperty(String key, Object value)
    {
        properties.put(key, value);
    }

    /**
     * Will retrieve a session level property.
     *
     * @param key the key for the object data being stored on the session
     * @return the value of the session data or null if the property does not exist
     */
    @SuppressWarnings("unchecked")
    public <T> T getProperty(Object key)
    {
        return (T) properties.get(key);
    }

    /**
     * Will retrieve a session level property and remove it from the session
     *
     * @param key the key for the object data being stored on the session
     * @return the value of the session data or null if the property does not exist
     */
    public Object removeProperty(Object key)
    {
        return properties.remove(key);
    }

    /**
     * Returns an iterater of property keys for the session properties on this
     * session
     *
     * @return an iterater of property keys for the session properties on this
     *         session
     * @deprecated Use getPropertyNamesAsSet() instead
     */
    @Deprecated
    public Iterator<String> getPropertyNames()
    {
        return properties.keySet().iterator();
    }

    public Set<String> getPropertyNamesAsSet()
    {
        return Collections.unmodifiableSet(properties.keySet());
    }

    ////////////////////////////
    // Serialization methods
    ////////////////////////////

    private void writeObject(ObjectOutputStream out) throws IOException
    {
        out.defaultWriteObject();
        //Can be null if service call originates from MuleClient
        if (getFlowConstruct() != null)
        {
            out.writeObject(getFlowConstruct() != null ? getFlowConstruct().getName() : "null");
        }
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException
    {
        in.defaultReadObject();
        serializedData = new HashMap<String, Object>();

        try
        {
            //Optional
            serializedData.put("serviceName", in.readObject());
        }
        catch (OptionalDataException e)
        {
            //ignore
        }
    }

    /**
     * Invoked after deserialization. This is called when the marker interface
     * {@link org.mule.util.store.DeserializationPostInitialisable} is used. This will get invoked
     * after the object has been deserialized passing in the current mulecontext when using either
     * {@link org.mule.transformer.wire.SerializationWireFormat},
     * {@link org.mule.transformer.wire.SerializedMuleMessageWireFormat}, or the
     * {@link org.mule.transformer.simple.ByteArrayToSerializable} transformer.
     *
     * @param muleContext the current muleContext instance
     * @throws MuleException if there is an error initializing
     */
    public void initAfterDeserialisation(MuleContext muleContext) throws MuleException
    {
        String serviceName = (String) serializedData.get("serviceName");
        //Can be null if service call originates from MuleClient
        if (serviceName != null)
        {
            flowConstruct = muleContext.getRegistry().lookupService(serviceName);
        }
        serializedData = null;
    }

}
