/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.core.routing;

import static org.mule.runtime.core.config.i18n.CoreMessages.cannotCopyStreamPayload;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.core.api.DefaultMuleException;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.connector.DispatchException;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.api.routing.RoutingException;
import org.mule.runtime.core.config.i18n.CoreMessages;
import org.mule.runtime.core.api.util.StringMessageUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract routing strategy with utility methods to be reused by routing strategies
 */
public abstract class AbstractRoutingStrategy implements RoutingStrategy {

  /**
   * logger used by this class
   */
  protected static transient Logger logger = LoggerFactory.getLogger(AbstractRoutingStrategy.class);

  private final MuleContext muleContext;

  public AbstractRoutingStrategy(final MuleContext muleContext) {
    this.muleContext = muleContext;
  }


  /**
   * Send message event to destination.
   *
   * Creates a new event that will be used to process the route.
   *
   * @param routedEvent event to route
   * @param message message to route
   * @param route message processor to be executed
   * @param awaitResponse if the
   * @return
   * @throws MuleException
   */
  protected final Event sendRequest(final Event routedEvent, final Message message, final Processor route,
                                    boolean awaitResponse)
      throws MuleException {
    Event result;
    try {
      result = sendRequestEvent(routedEvent, message, route, awaitResponse);
    } catch (MuleException me) {
      throw me;
    } catch (Exception e) {
      throw new RoutingException(null, e);
    }

    if (result != null) {
      Message resultMessage = result.getMessage();
      if (logger.isTraceEnabled()) {
        if (resultMessage != null) {
          try {
            logger.trace("Response payload: \n" + StringMessageUtils
                .truncate(muleContext.getTransformationService().getPayloadForLogging(resultMessage), 100, false));
          } catch (Exception e) {
            logger.trace("Response payload: \n(unable to retrieve payload: " + e.getMessage());
          }
        }
      }
    }
    return result;
  }

  private Event sendRequestEvent(Event routedEvent, Message message, Processor route, boolean awaitResponse)
      throws MuleException {
    if (route == null) {
      throw new DispatchException(CoreMessages.objectIsNull("route"), null);
    }

    return route.process(createEventToRoute(routedEvent, message, route));
  }

  /**
   * Create a new event to be routed to the target MP
   */
  protected Event createEventToRoute(Event routedEvent, Message message, Processor route) {
    return Event.builder(routedEvent).message(message).build();
  }

  protected MuleContext getMuleContext() {
    return muleContext;
  }

  /**
   * Validates that the payload is not consumable so it can be copied.
   *
   * If validation fails then throws a MessagingException
   *
   * @param event
   * @param message
   * @throws MuleException if the payload is consumable
   */
  public static void validateMessageIsNotConsumable(Event event, Message message) throws MuleException {
    if (message.getPayload().getDataType().isStreamType()) {
      throw new DefaultMuleException(cannotCopyStreamPayload(message.getPayload().getDataType().getType().getName()));
    }
  }
}
