/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.runtime.module.embedded.internal;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.toFile;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.Product;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.io.FileUtils;

/**
 * Creates a {@link ClassLoader} with the {@link URL}'s for the Container.
 *
 * @since 4.0
 */
// TODO MULE-11925 Improve MavenContainerClassLoaderFactory so it can work in terms of Dependency instead of URLs
public class MavenContainerClassLoaderFactory {

  private final MavenClient mavenClient;

  public MavenContainerClassLoaderFactory(MavenClient mavenClient) {
    this.mavenClient = mavenClient;
  }

  /**
   * Creates the {@link ClassLoader} Container for a given version.
   *
   * @param version Maven version. Not null.
   * @param product
   * @param containerBaseFolder @return a {@link ClassLoader} Container.
   */
  public ClassLoader create(String version, Product product, ClassLoader parentClassLoader, URL containerBaseFolder) {
    try {
      BundleDescriptor containerBomBundleDescriptor = getContainerBomBundleDescriptor(version, product);
      List<BundleDependency> bundleDependencies =
          mavenClient.resolveBundleDescriptorDependencies(false, containerBomBundleDescriptor);

      List<URL> urls = bundleDependencies.stream()
          .filter(bundleDependency -> !bundleDependency.getDescriptor().getGroupId().equals("org.mule.services"))
          .map(dep -> {
            try {
              return dep.getBundleUri().toURL();
            } catch (MalformedURLException e) {
              throw new RuntimeException(e);
            }
          })
          .collect(Collectors.toList());
      urls = new ArrayList<>(urls);
      File containerFolderFile = toFile(containerBaseFolder);
      // the URL has to be constructed this way since File.toURI().toURL() gets rid of the final slash
      urls.add(new URL(new File(containerFolderFile, "conf").toURI().toString() + "/"));
      return new URLClassLoader(urls.toArray(new URL[urls.size()]), parentClassLoader);
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  public List<URL> getServices(String version, Product product) {
    BundleDescriptor containerBomBundleDescriptor = getContainerBomBundleDescriptor(version, product);
    List<BundleDependency> containerDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, containerBomBundleDescriptor);
    List<URL> urls = containerDependencies.stream()
        .map(dep -> {
          try {
            return dep.getBundleUri().toURL();
          } catch (MalformedURLException e) {
            throw new RuntimeException(e);
          }
        })
        .collect(toList());
    return urls.stream().filter(u -> FileUtils.toFile(u).getPath().toLowerCase().endsWith(".zip")).collect(toList());
  }

  private BundleDescriptor getContainerBomBundleDescriptor(String version, Product product) {
    return new BundleDescriptor.Builder()
        .setGroupId(product.equals(Product.MULE) ? "org.mule.distributions" : "com.mulesoft.mule.distributions")
        .setArtifactId(product.equals(Product.MULE) ? "mule-runtime-impl-bom" : "mule-runtime-impl-bom")
        .setVersion(version)
        .setType("pom").build();
  }


}
