/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.catalog.internal.builder;

import static java.lang.String.format;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.catalog.internal.model.loaders.TypeLoaderFactory;

import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Scanner;

public class TypesResolverBuilder {

  public static final String JAR = "jar";

  private final URI baseUri;
  private String catalogName;
  private String catalogFormat;

  private String schemaFormat;
  private String schemaLocation;
  private String schemaContent;

  private String exampleFormat;
  private String exampleLocation;
  private String exampleContent;

  public TypesResolverBuilder(URI baseUri) {
    this.baseUri = baseUri;
  }

  public void catalogName(String name) {
    this.catalogName = name;
  }

  public String getCatalogName() {
    return catalogName;
  }

  public void catalogFormat(String format) {
    this.catalogFormat = format;
  }

  public void schemaFormat(String schemaFormat) {
    this.schemaFormat = schemaFormat;
  }

  public void schemaLocation(String schemaLocation) {
    this.schemaLocation = schemaLocation;
  }

  public void schemaContent(String schemaContent) {
    this.schemaContent = schemaContent;
  }

  public void exampleFormat(String exampleFormat) {
    this.exampleFormat = exampleFormat;
  }

  public void exampleLocation(String exampleLocation) {
    this.exampleLocation = exampleLocation;
  }

  public void exampleContent(String exampleContent) {
    this.exampleContent = exampleContent;
  }

  TypeLoader build(TypeLoaderRegistry shapesRegistry, TypeLoaderRegistry examplesRegistry) throws Exception {
    TypeLoader typeLoader;

    if (catalogName == null || catalogName.isEmpty()) {
      throw new RuntimeException("Catalog must have a non empty 'name' attribute");
    }

    if (schemaFormat != null) {
      if (schemaLocation != null) {
        typeLoader = createFromUrl(shapesRegistry, catalogFormat, schemaFormat, resolveURI(schemaLocation));
      } else {
        typeLoader = createFromContent(shapesRegistry, catalogFormat, schemaFormat, schemaContent);
      }
    } else if (exampleFormat != null) {
      if (exampleLocation != null) {
        URI exampleURI = resolveURI(exampleLocation);
        typeLoader = createFromUrl(examplesRegistry, catalogFormat, exampleFormat, exampleURI);
      } else {
        typeLoader = createFromContent(examplesRegistry, catalogFormat, exampleFormat, exampleContent);
      }
    } else {
      throw new RuntimeException(format("Catalog '%s' must have either <schema> or <example> to read from.", catalogName));
    }
    return typeLoader;
  }

  private static TypeLoaderFactory getTypeLoaderFactory(TypeLoaderRegistry registry, String typeFormat, String shapeFormat) {
    final TypeLoaderFactory typeLoaderFactory = registry.getFactory(typeFormat, shapeFormat)
        .orElseThrow(() -> new RuntimeException(String.format("Unsupported typeFormat %s shapeFormat %s", typeFormat,
                                                              shapeFormat)));
    return typeLoaderFactory;
  }

  private TypeLoader createFromUrl(TypeLoaderRegistry registry, String typeFormat, String shapeFormat, URI schemaUri)
      throws IOException {
    try (InputStream inputStream = schemaUri.toURL().openStream()) {
      Scanner scanner = new Scanner(inputStream).useDelimiter("\\A");
      String content = scanner.hasNext() ? scanner.next() : "";
      return getTypeLoaderFactory(registry, typeFormat, shapeFormat).createTypeLoader(content);
    }
  }

  private TypeLoader createFromContent(TypeLoaderRegistry registry, String typeFormat, String shapeFormat, String content) {
    return getTypeLoaderFactory(registry, typeFormat, shapeFormat).createTypeLoader(content);
  }

  private URI resolveURI(String location) throws URISyntaxException {
    return baseUri != null ? customResolve(location) : new URI(location);
  }

  /**
   * If the current {@code baseUri} is a JAR file the resolution of the schema (under {@code location}) is trickier as it needs
   * to be recalculated from the entry name (see {@link JarURLConnection#getEntryName()}), forming a new {@link URI}.
   * <p/>
   * As sample, let's say we have:
   * <ol>
   *   <li>{@code baseUri}: "jar:file:/Users/lautaro/.m2/repository/org/mule/extensions/smart/smart-connector-using-custom-types/1.0.0-SNAPSHOT/smart-connector-using-custom-types-1.0.0-SNAPSHOT-mule-plugin.jar!/module-using-custom-types-catalog.xml"</li>
   *   <li>{@code location}: "./get-issues-return-schema.json"</li>
   * </ol>
   * <p/>
   * The expected {@link URI} will have the value of
   * "jar:file:/Users/lautaro/.m2/repository/org/mule/extensions/smart/smart-connector-using-custom-types/1.0.0-SNAPSHOT/smart-connector-using-custom-types-1.0.0-SNAPSHOT-mule-plugin.jar!/get-issues-return-schema.json",
   * making that resource available through the usual {@link URI}/{@link URL} methods.
   *
   * @return an URI targeting the schema for the current {@code location} and {@code baseUri}
   */
  private URI customResolve(String location) {
    final URI result;
    if (baseUri.getScheme().equals(JAR)) {
      try {
        final URI fileCatalogBase = new URI(baseUri.getRawSchemeSpecificPart());
        final URI resolvedScheme = fileCatalogBase.resolve(location);
        result = new URI(JAR + ":" + resolvedScheme);
      } catch (URISyntaxException e) {
        throw new RuntimeException(
                                   format("Cannot generate a relative URI for the resource [%s] given the base URI path [%s]",
                                          location, baseUri.toString()),
                                   e);
      }
    } else {
      result = baseUri.resolve(location);
    }
    return result;
  }

}
