#!/usr/bin/env node

/* eslint-disable max-lines-per-function */
"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = run;

var _yargs = _interopRequireDefault(require("yargs/yargs"));

var _myrmidon = require("myrmidon");

var _ms = _interopRequireDefault(require("ms"));

var _date = _interopRequireDefault(require("../date"));

var _Jira = _interopRequireDefault(require("../Jira"));

var _package = _interopRequireDefault(require("../../package.json"));

var _adfUtils = require("../utils/adfUtils");

var _utils = require("./utils");

var _init = _interopRequireDefault(require("./init"));

var _logger = _interopRequireDefault(require("./logger"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const isMain = !module.parent;
const onYargsFail = (0, _utils.getYargsFail)(isMain, _logger.default);
const cliCommand = (0, _utils.getCLIRunner)({
  isMain,
  profile: 'jira'
});

function prettyDate(date) {
  return (0, _date.default)(date).format('DD-MM-YYYY');
}

async function list(args, profile) {
  const jira = new _Jira.default(profile, _logger.default);
  const stages = [];
  if (args.dev) stages.push('dev');
  const tasks = await jira.list({
    isMine: args.mine,
    search: args.search,
    sprint: args.sprint,
    stages
  });

  for (const t of tasks) {
    _logger.default.info(`%s ${t.summary}`, t.key);
  }
}

async function test(args, profile) {
  const jira = new _Jira.default(profile, _logger.default);

  for (const issueId of args.issueId) {
    await jira.test(issueId);
  }
}

async function show(args, profile) {
  const jira = new _Jira.default(profile, _logger.default);
  const task = await jira.show(args.issueId);

  _logger.default.info(`%s: ${task.summary}\n`, task.key);

  _logger.default.info(`Assignee: ${task.assigneeName} (${task.assignee})`);

  _logger.default.info(`Status: ${task.status} (${task.statusName})`);

  _logger.default.info(`Priority: ${task.priority}`);

  _logger.default.info((0, _adfUtils.adfToText)(task.description));

  if (args.comments && task.comments.length > 0) {
    _logger.default.info('\nComments:');

    for (const com of task.comments) {
      _logger.default.info(`\n%s (${com.author}) ${prettyDate(com.date)}`, com.authorName);

      _logger.default.info((0, _adfUtils.adfToText)(com.text));
    }
  }
}

async function exportLog(args, profile) {
  const jira = new _Jira.default(profile, _logger.default);
  await jira.exportLog([args.start, args.end], args.file);
}

async function clearWorklog(args, profile) {
  const jira = new _Jira.default(profile, _logger.default);
  const cleared = await jira.clearWorklog(args.issueId, {
    period: [args.from, args.to]
  });
  if (cleared.length === 0) _logger.default.warn('No worklogs found');

  for (const i of cleared) {
    _logger.default.info('Removed %s for %s', (0, _ms.default)(i.time), (0, _date.default)(i.start).format('DD MMM YYYY'));
  }
}

async function getStatuses(args, profile) {
  const jira = new _Jira.default(profile, _logger.default);
  const statuses = await jira.loadStatuses();

  for (const stat of statuses) {
    _logger.default.info(`%s: ${stat.name} (${stat.category})`, stat.id);
  }
}

async function logIssues(args, profile) {
  const jira = new _Jira.default(profile, _logger.default);
  await jira.logIssues(args);
}

const FORMATS = ['DD MM', 'DD MMM', 'DD-MMM', 'DD-MM', 'DD-MM-YY', 'DD MM YY', 'DD-MM-YYYY', 'DD MM YYYY'];
const dateSuffix = `\npossible formats: ${FORMATS.join(', ')}`;

function asDate(date) {
  if (!date) return;
  if ((0, _myrmidon.isArray)(date)) return date.map(d => asDate(d));

  for (const format of FORMATS) {
    const dated = (0, _date.default)(date, format, true);
    if (dated.isValid()) return dated;
  }

  throw new Error(`Invalid date ${date}`);
}

async function run(cmd) {
  const Argv = (0, _yargs.default)(cmd).usage('Usage: $0 <command> [options]').command({
    command: 'init',
    desc: 'Add attlasian profile',
    handler: cliCommand(_init.default, {
      noLoadProfile: true
    })
  }).command({
    command: `list [--dev] [--mine] [--search=<search>] [--sprint=<sprint>] ${_utils.commonCommandArgs}`,
    aliases: ['ls'],
    builder: y => (0, _utils.commonYargsOpts)(y).option('dev', {
      alias: ['d', 'development'],
      describe: 'filter only tasks in development',
      type: 'boolean'
    }).option('mine', {
      alias: ['m', 'my'],
      describe: 'filter only mine issues',
      type: 'boolean'
    }).option('search', {
      alias: ['s', 'grep'],
      describe: 'search issues by summary',
      type: 'string'
    }).option('sprint', {
      describe: 'specify sprints for filter',
      choices: ['all', 'open'],
      default: ['open'],
      type: 'array'
    }),
    desc: 'List Tasks',
    handler: cliCommand(list)
  }).command({
    command: `test ${_utils.commonCommandArgs} <issueId...>`,
    desc: 'Send task(s) to testing',
    builder: y => (0, _utils.commonYargsOpts)(y).option('issueId', {
      describe: 'id(s) of task',
      type: 'array'
    }),
    handler: cliCommand(test)
  }).command({
    command: `show ${_utils.commonCommandArgs} [--comments] <issueId>`,
    desc: 'Show task description',
    builder: y => (0, _utils.commonYargsOpts)(y).option('issueId', {
      describe: 'id of task',
      type: 'string'
    }).option('comments', {
      describe: 'Show comments',
      type: 'boolean'
    }),
    handler: cliCommand(show)
  }).command({
    command: `export log ${_utils.commonCommandArgs} <start> <end> [--file=<file>]`,
    desc: 'Export tasks for time tracking',
    builder: y => (0, _utils.commonYargsOpts)(y).option('start', {
      describe: `issues with updatedDate >= start will be included ${dateSuffix}`,
      type: 'date'
    }).option('end', {
      describe: `issues with created <= end will be included ${dateSuffix}`,
      type: 'date'
    }).option('file', {
      describe: 'path to resulting file',
      type: 'string'
    }).coerce('start', asDate).coerce('end', asDate),
    handler: cliCommand(exportLog)
  }).command({
    command: `worklog clear <issueId> [--from=<from>] [--to=<to>] ${_utils.commonCommandArgs}`,
    desc: 'Clear worklog',
    builder: y => (0, _utils.commonYargsOpts)(y).option('from', {
      demandOption: false,
      describe: `start of worklog period ${dateSuffix}`,
      type: 'date'
    }).option('to', {
      demandOption: false,
      describe: `end of worklog period ${dateSuffix}`,
      type: 'date'
    }).positional('<issueId>', {
      describe: 'Id of the issue',
      type: 'string'
    }).coerce('from', asDate).coerce('to', asDate),
    handler: cliCommand(clearWorklog)
  }).command({
    command: `log ${_utils.commonCommandArgs} [--issues=<issues>] [--from=<from>] [--to=<to>] [--include=<include>] [--exclude=<exclude>] [--strategy=<strategy>] [--confirm]`,
    desc: 'Log time in issues',
    builder: y => (0, _utils.commonYargsOpts)(y).option('issues', {
      demandOption: true,
      describe: 'path to file with issues',
      type: 'path'
    }).option('include', {
      describe: 'add day to worklog',
      type: 'array'
    }).option('exclude', {
      describe: 'remove day from worklog',
      type: 'array'
    }).option('from', {
      demandOption: true,
      describe: `start of worklog period ${dateSuffix}`,
      type: 'date'
    }).option('to', {
      demandOption: true,
      describe: `end of worklog period ${dateSuffix}`,
      type: 'date'
    }).option('strategy', {
      describe: 'use time tracker strategy',
      choices: ['id', 'actions'],
      default: ['id']
    }).option('confirm', {
      describe: 'actually log time',
      alias: 'y',
      type: 'boolean'
    }).coerce('from', asDate).coerce('to', asDate).coerce('include', asDate).coerce('exclude', asDate),
    handler: cliCommand(logIssues)
  }).command({
    command: `statuses ${_utils.commonCommandArgs}`,
    desc: 'List jira configuration statuses',
    handler: cliCommand(getStatuses)
  }).help('h').alias('h', 'help').wrap(Math.min(_utils.minTerminalWidth, process.stdout.columns)).version(_package.default.version).demandCommand(1, '').recommendCommands().strict().epilog(`${_package.default.name} v.${_package.default.version}`).fail(onYargsFail);
  await Argv.argv;
}

const firstCmdArgIndex = 2;
if (isMain) run(process.argv.slice(firstCmdArgIndex));