"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _myrmidon = require("myrmidon");

var _dumpUtils = require("../utils/dumpUtils");

var _date = _interopRequireDefault(require("../date"));

var _AtlassianApi = _interopRequireDefault(require("./AtlassianApi"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/* eslint-disable no-param-reassign */
class JiraApi extends _AtlassianApi.default {
  async getStatuses() {
    const res = await this.get('/rest/api/latest/status/');
    return res.map(s => (0, _dumpUtils.dumpStatus)(s));
  }

  async getAllSprints() {
    const boards = await this.get('/rest/agile/1.0/board');
    const sprints = [];
    await Promise.all(boards.values.map(async board => {
      sprints.push(...(await this.getSprints(board)));
    }));
    return sprints;
  }

  async getSprints(board, {
    startAt = 0,
    maxResults = 50,
    ...params
  } = {}) {
    const {
      isLast,
      values
    } = await this.get(`/rest/agile/1.0/board/${board.id}/sprint`, {
      startAt,
      maxResults,
      ...params
    });
    const nextStart = startAt + values.length;

    if (!isLast) {
      const next = await this.getSprints(board, { ...params,
        startAt: nextStart
      });
      return [...values.map(v => (0, _dumpUtils.dumpSprint)(v)), ...next].filter((element, index, array) => (0, _myrmidon.uniqueIdFilter)(element, index, array));
    }

    return values.map(v => (0, _dumpUtils.dumpSprint)(v));
  }

  async getIssues(params, includes = []) {
    const extraParams = {};
    if (includes.length > 0 && includes.includes('changelog')) extraParams.expand = 'changelog';
    const {
      issues,
      startAt,
      total
    } = await this.get('/rest/api/3/search', { ...params,
      ...extraParams
    });
    const nextStart = startAt + issues.length;

    if (includes.length > 0) {
      await Promise.all(issues.map(async issue => {
        const promises = [];

        if (includes.includes('transitions')) {
          promises.push({
            key: '_transitions',
            action: this.getTransitions(issue.id)
          });
        }

        if (includes.includes('comments')) {
          promises.push({
            key: '_comments',
            action: this.getComments(issue.id)
          });
        }

        if (includes.includes('worklogs')) {
          promises.push({
            key: '_worklog',
            action: this.getWorklog(issue.id)
          });
        }

        await Promise.all(promises.map(async p => {
          const res = await p.action;
          issue[p.key] = res;
        }));
      }));
    }

    if (total > nextStart) {
      const next = await this.getIssues({ ...params,
        startAt: nextStart
      }, includes);
      return [...issues.map(i => (0, _dumpUtils.dumpTask)(i)), ...next].filter((element, index, array) => (0, _myrmidon.uniqueIdFilter)(element, index, array));
    }

    return issues.map(t => (0, _dumpUtils.dumpTask)(t));
  }

  async getIssue(id, includes = []) {
    const extraParams = {};
    const promises = [];

    if (includes.length > 0) {
      if (includes.includes('changelog')) extraParams.expand = 'changelog';

      if (includes.includes('transitions')) {
        promises.push({
          key: '_transitions',
          action: this.getTransitions(id)
        });
      }

      if (includes.includes('comments')) {
        promises.push({
          key: '_comments',
          action: this.getComments(id)
        });
      }

      if (includes.includes('worklogs')) {
        promises.push({
          key: '_worklog',
          action: this.getWorklog(id)
        });
      }
    }

    const [issue, ...results] = await Promise.all([this.get(`/rest/api/3/issue/${id}`, extraParams), // TODO
    ...promises.map(p => p.action)]);

    for (const [i, p] of promises.entries()) issue[p.key] = results[i];

    return (0, _dumpUtils.dumpTask)(issue);
  }

  async getTransitions(id) {
    const res = await this.get(`/rest/api/3/issue/${id}/transitions`);
    return res.transitions.map(element => (0, _dumpUtils.dumpTransition)(element));
  }

  async transit(issueId, transitionId) {
    await this.post(`/rest/api/3/issue/${issueId}/transitions`, {
      transition: transitionId
    });
  }

  async getComments(issueId) {
    const res = await this.get(`/rest/api/3/issue/${issueId}/comment`);
    return res.comments.map(element => (0, _dumpUtils.dumpComment)(element));
  }

  async getWorklog(issueId) {
    const res = await this.get(`/rest/api/3/issue/${issueId}/worklog`);
    return res.worklogs.map(element => (0, _dumpUtils.dumpWorklog)(element));
  } // async getAttachments(issueId) {
  //     await this.get(`/rest/api/3/issue/${issueId}/attachment`);
  // }


  async deleteWorklog(issueId, worklogId) {
    await this.delete(`/rest/api/3/issue/${issueId}/worklog/${worklogId}`);
  }

  async logTime(issueID, day, time) {
    const midDay = 12;
    const res = await this.post(`/rest/api/3/issue/${issueID}/worklog`, {
      'timeSpentSeconds': time * 60 * 60,
      'started': (0, _date.default)(day, 'D MMM YYYY').set('hour', midDay).format('YYYY-MM-DD[T]HH:m:s.sssZZ')
    });
    return res.data;
  }

}

exports.default = JiraApi;