"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configPath = exports.commonYargsOpts = exports.commonCommandArgs = void 0;
exports.getCLIRunner = getCLIRunner;
exports.getDefaultProfile = getDefaultProfile;
exports.getYargsFail = getYargsFail;
exports.installLogger = installLogger;
exports.loadConfig = loadConfig;
exports.loadProfile = loadProfile;
exports.minTerminalWidth = void 0;
exports.untilConfirm = untilConfirm;

var _path = _interopRequireDefault(require("path"));

var _os = _interopRequireDefault(require("os"));

var _inquirer = _interopRequireDefault(require("inquirer"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _AtlassianApi = _interopRequireDefault(require("../api/AtlassianApi"));

var _logger = _interopRequireDefault(require("./logger"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const homedir = _os.default.homedir();

const defaultConfigPath = _path.default.join(homedir, '.atlassian');

const configPath = process.env.ATLASSIAN_CONFIG_PATH || defaultConfigPath;
exports.configPath = configPath;

function getDefaultProfile(config, scope) {
  return Object.keys(config).find(key => {
    var _config$key$scope;

    return (_config$key$scope = config[key][scope]) === null || _config$key$scope === void 0 ? void 0 : _config$key$scope.isDefault;
  });
}

async function loadConfig() {
  try {
    return await _fsExtra.default.readJSON(configPath);
  } catch {
    await _fsExtra.default.ensureDir(_path.default.dirname(configPath));
    return {};
  }
}

async function loadProfile(scope, name) {
  const config = await loadConfig();
  let profileName = name;
  if (!profileName) profileName = process.env.ATLASSIAN_PROFILE;
  if (!profileName) profileName = getDefaultProfile(config, scope);
  if (!profileName) throw new Error('no profile selected');
  const profile = config[profileName];
  if (!profile) throw new Error(`no profile ${profileName} found`);
  const api = new _AtlassianApi.default(profile.host, {
    username: profile.email,
    password: profile.token
  });
  const myself = await api.getMyself();
  if (myself.id !== profile.userId) throw new Error(`Profile ${profileName} not matches user ${JSON.stringify(myself)}`);
  return profile;
}

function installLogger(logger, {
  logLevel,
  verbose,
  debug
}) {
  const level = logLevel || verbose && 'verbose' || debug && 'debug'; // eslint-disable-next-line no-param-reassign

  if (level) logger.level = level;
}

async function untilConfirm(q) {
  const {
    confirm,
    ...res
  } = await _inquirer.default.prompt(q);
  if (confirm !== false) return res;
  return untilConfirm(q);
}

function getCLIRunner({
  isMain,
  profile
}) {
  function onError(e) {
    _logger.default.error(e.toString());

    _logger.default.verbose(e.stack);

    if (isMain) process.exit(1);
    throw e;
  }

  return function cliCommand(method, {
    noLoadProfile
  } = {}) {
    return async function (args) {
      try {
        installLogger(_logger.default, args);
        const profileConf = noLoadProfile ? null : await loadProfile(profile, args.profile);
        await method(args, profileConf);
      } catch (error) {
        onError(error);
      }
    };
  };
}

const commonYargsOpts = y => y.option('verbose', {
  describe: 'verbose logs',
  alias: ['v'],
  type: 'boolean'
}).option('debug', {
  describe: 'debug logs',
  type: 'boolean'
}).option('profile', {
  alias: ['p'],
  describe: 'specify profile name',
  type: 'string'
});

exports.commonYargsOpts = commonYargsOpts;
const minTerminalWidth = 95;
exports.minTerminalWidth = minTerminalWidth;
const commonCommandArgs = '[--verbose] [--debug] [--profile=<profile>]';
exports.commonCommandArgs = commonCommandArgs;

function getYargsFail(isMain, logger) {
  return function onYargsFail(message, error, ygs) {
    const failMessage = message || error;
    ygs.showHelp('error');

    if (failMessage) {
      console.log();
      logger.error(failMessage.toString());
      logger.verbose(error === null || error === void 0 ? void 0 : error.stack);
    }

    if (!isMain) throw failMessage;
    process.exit(1);
  };
}