import ReactiveElement from "../core/ReactiveElement.js"; // eslint-disable-line no-unused-vars
import {
  defaultState,
  goDown,
  goEnd,
  goLeft,
  goRight,
  goStart,
  goUp,
  keydown,
  state,
} from "./internal.js";

/**
 * Maps direction keys to direction semantics.
 *
 * This mixin is useful for components that want to map direction keys (Left,
 * Right, etc.) to movement in the indicated direction (go left, go right,
 * etc.).
 *
 * This mixin expects the component to invoke a `keydown` method when a key is
 * pressed. You can use [KeyboardMixin](KeyboardMixin) for that
 * purpose, or wire up your own keyboard handling and call `keydown` yourself.
 *
 * This mixin calls methods such as `goLeft` and `goRight`. You can define
 * what that means by implementing those methods yourself. If you want to use
 * direction keys to navigate a selection, use this mixin with
 * [DirectionCursorMixin](DirectionCursorMixin).
 *
 * If the component defines a property called `orientation`, the value of that
 * property will constrain navigation to the horizontal or vertical axis.
 *
 * @module KeyboardDirectionMixin
 * @param {Constructor<ReactiveElement>} Base
 */
export default function KeyboardDirectionMixin(Base) {
  // The class prototype added by the mixin.
  class KeyboardDirection extends Base {
    // @ts-ignore
    get [defaultState]() {
      return Object.assign(super[defaultState], {
        handleBubblingDirectionKeys: false,
      });
    }

    /**
     * Invoked when the user wants to go/navigate down.
     * The default implementation of this method does nothing.
     */
    [goDown]() {
      if (super[goDown]) {
        return super[goDown]();
      }
    }

    /**
     * Invoked when the user wants to go/navigate to the end (e.g., of a list).
     * The default implementation of this method does nothing.
     */
    [goEnd]() {
      if (super[goEnd]) {
        return super[goEnd]();
      }
    }

    /**
     * Invoked when the user wants to go/navigate left.
     * The default implementation of this method does nothing.
     */
    [goLeft]() {
      if (super[goLeft]) {
        return super[goLeft]();
      }
    }

    /**
     * Invoked when the user wants to go/navigate right.
     * The default implementation of this method does nothing.
     */
    [goRight]() {
      if (super[goRight]) {
        return super[goRight]();
      }
    }

    /**
     * Invoked when the user wants to go/navigate to the start (e.g., of a
     * list). The default implementation of this method does nothing.
     */
    [goStart]() {
      if (super[goStart]) {
        return super[goStart]();
      }
    }

    /**
     * Invoked when the user wants to go/navigate up.
     * The default implementation of this method does nothing.
     */
    [goUp]() {
      if (super[goUp]) {
        return super[goUp]();
      }
    }

    [keydown](/** @type {KeyboardEvent} */ event) {
      let handled = false;

      // Direction keys generally are low-priority keys: if a shadow element
      // like an input has focus, we want to let that focused element handle
      // direction keys. So we only handle the event if we're the target.
      //
      // (We'd really like to be able to provide direction key handling as a
      // default — i.e., if the focused element doesn't handle a key, then we
      // would handle it here. Unfortunately, there doesn't seem to be any
      // general way for us to do that.)
      if (this[state].handleBubblingDirectionKeys || event.target === this) {
        // Respect orientation state if defined, otherwise assume "both".
        const orientation = this[state].orientation || "both";
        const horizontal =
          orientation === "horizontal" || orientation === "both";
        const vertical = orientation === "vertical" || orientation === "both";

        // Ignore Left/Right keys when metaKey or altKey modifier is also pressed,
        // as the user may be trying to navigate back or forward in the browser.
        switch (event.key) {
          case "ArrowDown":
            if (vertical) {
              handled = event.altKey ? this[goEnd]() : this[goDown]();
            }
            break;

          case "ArrowLeft":
            if (horizontal && !event.metaKey && !event.altKey) {
              handled = this[goLeft]();
            }
            break;

          case "ArrowRight":
            if (horizontal && !event.metaKey && !event.altKey) {
              handled = this[goRight]();
            }
            break;

          case "ArrowUp":
            if (vertical) {
              handled = event.altKey ? this[goStart]() : this[goUp]();
            }
            break;

          case "End":
            handled = this[goEnd]();
            break;

          case "Home":
            handled = this[goStart]();
            break;
        }
      }

      // Prefer mixin result if it's defined, otherwise use base result.
      return handled || (super[keydown] && super[keydown](event)) || false;
    }
  }

  return KeyboardDirection;
}
