import { fragmentFrom } from "../core/htmlLiterals.js";
import ReactiveElement from "../core/ReactiveElement.js"; // eslint-disable-line no-unused-vars
import { ids, render, state } from "./internal.js";

const wrap = Symbol("wrap");

/**
 * Adds a page number and total page count to a carousel-like element.
 *
 * This can be applied to components like [Carousel](Carousel) that renders
 * their content as pages.
 *
 * @module PageNumbersMixin
 * @param {Constructor<ReactiveElement>} Base
 * @part {div} page-number - the page number
 */
function PageNumbersMixin(Base) {
  class PageNumbers extends Base {
    [render](/** @type {ChangedFlags} */ changed) {
      if (super[render]) {
        super[render](changed);
      }
      if (changed.currentIndex) {
        const { items, currentIndex } = this[state];
        const textContent =
          currentIndex >= 0 && items
            ? `${currentIndex + 1} / ${items.length}`
            : "";
        this[ids].pageNumber.textContent = textContent;
      }
    }

    /**
     * Destructively wrap a node with elements to show page numbers.
     *
     * @param {Element} target - the element that should be wrapped by page numbers
     */
    [wrap](target) {
      const pageNumbers = fragmentFrom.html`
        <div
          id="pageNumbers"
          role="none"
          style="display: flex; flex: 1; overflow: hidden;"
        >
          <style>
            [part~="page-number"] {
              bottom: 0;
              color: white;
              padding: 0.5em;
              position: absolute;
              right: 0;
            }
          </style>
          <div
            id="pageNumbersContainer"
            role="none"
            style="display: flex; flex: 1; overflow: hidden; position: relative;"
          ></div>
          <div id="pageNumber" part="page-number"></div>
        </div>
      `;

      // Wrap the target with the page numbers.
      const container = pageNumbers.getElementById("pageNumbersContainer");
      if (container) {
        target.replaceWith(pageNumbers);
        container.append(target);
      }
    }
  }

  return PageNumbers;
}

PageNumbersMixin.wrap = wrap;

export default PageNumbersMixin;
