import { indexOfItemContainingTarget } from "../core/dom.js";
import ReactiveElement from "../core/ReactiveElement.js"; // eslint-disable-line no-unused-vars
import {
  firstRender,
  raiseChangeEvents,
  render,
  setState,
  state,
  tap,
} from "./internal.js";

/**
 * A tap/mousedown on a list item makes that item current.
 *
 * This simple mixin is useful in list-like elements like [ListBox](ListBox),
 * where a tap/mousedown on a list item implicitly selects it.
 *
 * The standard use for this mixin is in list-like elements. Native list
 * boxes don't appear to be consistent with regard to whether they select
 * on mousedown or click/mouseup. This mixin assumes the use of mousedown.
 * On touch devices, that event appears to trigger when the touch is *released*.
 *
 * This mixin only listens to mousedown events for the primary mouse button
 * (typically the left button). Right clicks are ignored so that the browser may
 * display a context menu.
 *
 * This mixin expects the component to provide an `state.items` member. It also
 * expects the component to define a `state.currentIndex` member; you can
 * provide that yourself, or use [ItemsCursorMixin](ItemsCursorMixin).
 *
 * If the component receives an event that doesn't correspond to an item (e.g.,
 * the user taps on the element background visible between items), the cursor
 * will be removed. However, if the component sets `state.currentItemRequired` to
 * true, a background tap will *not* remove the cursor.
 *
 * @module TapCursorMixin
 * @param {Constructor<ReactiveElement>} Base
 */
export default function TapCursorMixin(Base) {
  // The class prototype added by the mixin.
  return class TapCursor extends Base {
    constructor() {
      // @ts-ignore
      super();
      this.addEventListener("mousedown", (event) => {
        // Only process events for the main (usually left) button.
        if (event.button !== 0) {
          return;
        }
        this[raiseChangeEvents] = true;
        this[tap](event);
        this[raiseChangeEvents] = false;
      });
    }

    [render](/** @type {ChangedFlags} */ changed) {
      if (super[render]) {
        super[render](changed);
      }
      if (this[firstRender]) {
        Object.assign(this.style, {
          touchAction: "manipulation", // for iOS Safari
          mozUserSelect: "none",
          msUserSelect: "none",
          webkitUserSelect: "none",
          userSelect: "none",
        });
      }
    }

    [tap](/** @type {MouseEvent} */ event) {
      // In some situations, the event target will not be the child which was
      // originally clicked on. E.g., if the item clicked on is a button, the
      // event seems to be raised in phase 2 (AT_TARGET) — but the event target
      // will be the component, not the item that was clicked on. Instead of
      // using the event target, we get the first node in the event's composed
      // path.
      // @ts-ignore
      const target = event.composedPath
        ? event.composedPath()[0]
        : event.target;

      // Find which item was clicked on and, if found, make it current. Ignore
      // clicks on disabled items.
      //
      // For elements which don't require a cursor, a background click will
      // determine the item was null, in which we case we'll remove the cursor.
      const { items, currentItemRequired } = this[state];
      if (items && target instanceof Node) {
        const targetIndex = indexOfItemContainingTarget(items, target);
        const item = targetIndex >= 0 ? items[targetIndex] : null;
        if ((item && !item.disabled) || (!item && !currentItemRequired)) {
          this[setState]({
            currentIndex: targetIndex,
          });
          event.stopPropagation();
        }
      }
    }
  };
}
