import { transformSync } from '@lwc/compiler';
import { resolveModule } from '@lwc/module-resolver';
import { parse as parseModule } from 'es-module-lexer';
import { loadConfig } from '@lwrjs/config';
import { explodeSpecifier } from '@lwrjs/shared-utils';
import { Command } from 'commander';
import chalk from 'chalk';
import path from 'path';
import fs from 'fs';
import treeify from 'object-treeify';
import { parseStringPromise } from 'xml2js';
const CAPABILITY_SSR_ONLY = 'lightning__ServerRenderable'; // SSR-only
const CAPABILITY_SSR_HYDRATION = 'lightning__ServerRenderableWithHydration'; // SSR with hydration
const CAPABILITY_UNKNOWN = 'Unknown'; // unresolvable modules
const CAPABILITY_PRIVATE = 'Private'; // unexposed platform modules (eg: "lwr/routerUtils")
const PLATFORM_NS = ['lwr', 'webruntime'];
const LWC_REGEX = /^[\w-]+\/[\w-]+$/; // LWC component specifier format
const LWC_TRANSFORM_CONFIG = {
    // pass to the LWC compiler
    experimentalDynamicComponent: { strictSpecifier: false },
    enableDynamicComponents: true,
    enableScopedSlots: true,
};
const SFDX_CONFIG_PATH = 'sfdx-project.json';
const CACHE = new Map();
export function createAuditCommand() {
    return new Command('audit')
        .description('Audit the SSR capabilities of components')
        .requiredOption('--components <component...>', '[string] space-separated list of component names to analyze (eg "header footer nav")')
        .option('--namespace <string>', '[string] namespace of the components', 'c')
        .option('--depth <number>', '[number] number of levels to analyze', '1')
        .option('--verbose', '[boolean] show tree of all analyzed components', false)
        .action(async (options, cmd) => {
        // Read options and config
        CACHE.clear(); // use a fresh cache for each audit
        const opts = cmd.optsWithGlobals();
        const { components, config, depth: depthStr, namespace: ns, rootDir, verbose } = opts;
        const lwcConfig = getLwcConfig(rootDir, config);
        // Check that the depth is at least 1
        let depth = parseInt(depthStr);
        if (isNaN(depth) || depth < 1) {
            console.log(chalk.bgYellow('WARN'), `Depth cannot be ${depthStr}, defaulting to 1`);
            depth = 1;
        }
        // Analyze the components and print the results
        const cmpList = components.map((c) => `${ns}/${c}`).join(', ');
        console.log(`✨ Auditing SSR capabilities for ${chalk.bgGray(cmpList)}...`);
        let greatSuccess = true;
        for (const c of components) {
            const info = await analyzeComponentCapabilities(`${ns}/${c}`, rootDir, lwcConfig, depth);
            if (info) {
                const success = print(info, verbose);
                greatSuccess = success && greatSuccess;
            }
            else {
                greatSuccess = false;
                console.log(chalk.bgRed('FAIL'), chalk.bgGray(`${ns}/${c}`), 'cannot be found');
            }
        }
        if (!greatSuccess)
            process.exit(1);
    });
}
/** COMPONENT AUDIT HELPERS **/
/**
 * Analyzes a component tree's SSR capabilities to a given depth and prints out a compatibility audit
 * @param specifier Specifier for the given component
 * @param rootDir Path for the project root directory
 * @param modules LWC modules configuration (ie: alias, dir and npm module records)
 * @param depth The number of levels to check in the component tree
 * @returns Specifier, SSR capability and children for a component, or undefined if the component cannot be resolved
 */
async function analyzeComponentCapabilities(specifier, rootDir, modules, depth) {
    if (CACHE.has(specifier))
        return CACHE.get(specifier);
    const cmpDir = await getComponentDir(specifier, rootDir, modules);
    if (!cmpDir)
        return undefined;
    const info = {
        specifier: specifier,
        ssrCapability: await parseSsrCapability(specifier, cmpDir),
        children: [],
    };
    info.children = await analyzeChildren(info, cmpDir, rootDir, modules, depth);
    info && CACHE.set(specifier, info);
    return info;
}
/**
 * Resolve a component to find its directory path
 * @param specifier Specifier for the given component
 * @param rootDir Path for the project root directory
 * @param modules LWC modules configuration (ie: alias, dir and npm module records)
 * @returns Path to the directory of the component, or undefined if the component cannot be resolved
 */
async function getComponentDir(specifier, rootDir, modules) {
    try {
        // SFDX projects do not have namespace folders, but they have a top-level "lwc" dir
        const s = isSFDX(rootDir) ? specifier.replace(/^c\//, 'lwc/') : specifier;
        const { entry } = resolveModule(s, rootDir, { modules, rootDir });
        return path.dirname(entry);
    }
    catch (e) {
        if (e.code === 'NO_LWC_MODULE_FOUND') {
            // The component is not on the local file system
            return undefined;
        }
        else {
            throw e;
        }
    }
}
/**
 * Parse the SSR capability from a component's js-meta file
 * @param specifier Specifier for the given component
 * @param cmpDir Path to the component directory
 * @returns An SSR capability or undefined (ie: CSR-only)
 */
async function parseSsrCapability(specifier, cmpDir) {
    const xml = readFile(path.join(cmpDir, `${explodeSpecifier(specifier).name}.js-meta.xml`));
    if (!xml) {
        // some components may not have a js-meta file
        const { namespace = '' } = explodeSpecifier(specifier);
        if (!PLATFORM_NS.includes(namespace)) {
            // do not warn for private modules
            console.log(chalk.bgYellow('WARN'), chalk.bgGray(specifier), 'does not have a js-meta.xml file');
            return undefined;
        }
        return CAPABILITY_PRIVATE;
    }
    const metadata = await parseStringPromise(xml);
    const capabilities = metadata.LightningComponentBundle?.capabilities;
    const capability = Array.isArray(capabilities) ? capabilities[0].capability : undefined;
    if (Array.isArray(capability)) {
        return capability.includes(CAPABILITY_SSR_ONLY)
            ? CAPABILITY_SSR_ONLY
            : capability.includes(CAPABILITY_SSR_HYDRATION)
                ? CAPABILITY_SSR_HYDRATION
                : undefined;
    }
    else if (typeof capability === 'string') {
        return capability.startsWith(CAPABILITY_SSR_ONLY) ? capability : undefined;
    }
    return undefined;
}
/**
 * Recursively get the SSR capabilities for the components in a tree to a given depth
 * @param specifier Specifier for the given component
 * @param cmpDir Path to the component directory
 * @param rootDir Path for the project root directory
 * @param modules LWC modules configuration (ie: alias, dir and npm module records)
 * @param maxDepth The number of levels to check in the component tree
 * @param curDepth The depth currently being analyzed
 * @returns An array of component specifiers and SSR capabilities
 */
async function analyzeChildren(info, cmpDir, rootDir, modules, maxDepth = 1, curDepth = 0) {
    const { isLWC, imports } = await parseJsDeps(info.specifier, cmpDir);
    info.isLWC = isLWC; // always get the LWC boolean for the parent
    if (curDepth === maxDepth)
        return [];
    const htmlDeps = await parseHtmlDeps(info.specifier, cmpDir);
    const children = [];
    for (const d of [...htmlDeps, ...imports]) {
        if (CACHE.has(d)) {
            children.push(CACHE.get(d));
            continue;
        }
        const dir = await getComponentDir(d, rootDir, modules);
        const child = { specifier: d, children: [] };
        if (dir) {
            // child exists on the file system
            child.ssrCapability = await parseSsrCapability(d, dir);
            child.children = await analyzeChildren(child, dir, rootDir, modules, maxDepth, curDepth + 1);
        }
        else {
            const { namespace = '' } = explodeSpecifier(d);
            child.ssrCapability = PLATFORM_NS.includes(namespace) ? CAPABILITY_PRIVATE : CAPABILITY_UNKNOWN;
        }
        CACHE.set(d, child);
        children.push(child);
    }
    return children;
}
/**
 * Parse the LWC component dependencies from a component's HTML
 * @param specifier Specifier for the given component
 * @param cmpDir Path to the component directory
 * @returns An array of component specifiers
 */
async function parseHtmlDeps(specifier, cmpDir) {
    const filename = `${explodeSpecifier(specifier).name}.html`;
    const htmlCode = readFile(path.join(cmpDir, filename));
    if (!htmlCode)
        return []; // some LWC modules do not have HTML
    const { namespace, name } = explodeSpecifier(specifier);
    const { code } = transformSync(htmlCode, filename, { ...LWC_TRANSFORM_CONFIG, namespace, name });
    return (await getModuleMetadata(code)).imports;
}
/**
 * Parse the LWC component dependencies from a component's JS
 * @param specifier Specifier for the given component
 * @param cmpDir Path to the component directory
 * @returns An array of component specifiers and a boolean indicating if this is an LWC component
 */
async function parseJsDeps(specifier, cmpDir) {
    let jsCode = readFile(path.join(cmpDir, `${explodeSpecifier(specifier).name}.js`));
    if (!jsCode) {
        // try looking for a ts file if a js file not found
        jsCode = readFile(path.join(cmpDir, `${explodeSpecifier(specifier).name}.ts`));
    }
    return await getModuleMetadata(jsCode);
}
/**
 * Parse a module to get its LWC module imports, and determine if it is an LWC component
 * @param code Code string for an ES module
 * @returns An array of component specifiers and a boolean indicating if this is an LWC component
 */
async function getModuleMetadata(code) {
    if (!code)
        return { imports: [] };
    const [imps, exports] = await parseModule(code); // use await, so we do not need to call init()
    // assume it's an LWC component if it has a named default export
    // ideally, we'd check if it extends from LightningElement (either directly or chained)
    const isLWC = exports.some((e) => e.n === 'default' && e.ln);
    // filter out imports which are not LWC specifiers via a regex
    const imports = imps.map((i) => i.n || code.substring(i.s, i.e)).filter((d) => LWC_REGEX.test(d));
    return { isLWC, imports };
}
/** FILE SYSTEM HELPERS **/
/**
 * Read in the LWC modules configuration for the project
 * @param rootDir Path for the project root directory
 * @param lwrConfigPath Path to the lwr.config.json file
 * @returns LWC modules configuration (ie: alias, dir and npm module records)
 */
function getLwcConfig(rootDir, lwrFile) {
    // LWR project: lwr.config.json
    const absLwrConfigPath = path.isAbsolute(lwrFile) ? lwrFile : path.join(rootDir, lwrFile);
    if (fs.existsSync(absLwrConfigPath)) {
        const { appConfig } = loadConfig({ rootDir, lwrConfigFile: lwrFile });
        return appConfig.lwc.modules;
    }
    // LWC project: lwc.config.json
    const lwcConfigStr = readFile(path.join(rootDir, 'lwc.config.json'));
    if (lwcConfigStr) {
        const lwcConfig = JSON.parse(lwcConfigStr).modules;
        if (lwcConfig)
            return lwcConfig;
    }
    // SFDX project: sfdx-project.json
    const sfdxConfigStr = readFile(path.join(rootDir, SFDX_CONFIG_PATH));
    const modules = [];
    if (sfdxConfigStr) {
        const sfdxConfig = JSON.parse(sfdxConfigStr);
        const pkgDirs = sfdxConfig.packageDirectories || [];
        // configure LWC to resolve modules from each package directory
        pkgDirs.forEach((d) => modules.push({ dir: path.join(d.path, 'main/default') }));
    }
    // LWC project: package.json
    // SFDX projects can have LWC config in a package.json
    const pkgConfigStr = readFile(path.join(rootDir, 'package.json'));
    if (pkgConfigStr) {
        const pkgConfig = JSON.parse(pkgConfigStr).lwc?.modules;
        if (pkgConfig)
            modules.push(...pkgConfig);
    }
    if (!modules.length) {
        // project has no config!
        console.log(chalk.bgRed('FAIL'), 'LWC config not found in', chalk.bgGray(rootDir));
        process.exit(1);
    }
    return modules;
}
/**
 * Read a file if it exists
 * @param filePath File path
 * @returns File contents or undefined if the file is not found
 */
function readFile(filePath) {
    if (fs.existsSync(filePath)) {
        return fs.readFileSync(filePath, 'utf8');
    }
}
/**
 * Determine if this is an SFDX project
 * @param rootDir Path for the project root directory
 * @returns True if this is an SFDX project
 */
function isSFDX(rootDir) {
    return fs.existsSync(path.join(rootDir, SFDX_CONFIG_PATH));
}
/** PRINTING HELPERS **/
const SUCCESS_MSG = 'contains no capability incompatibilities!';
const WARN_MSG = 'depends on one or more components which are not available locally and must be analyzed manually:';
const SSR_ERROR_MSG = 'has the ServerRenderable capability so it cannot depend on components with no SSR capability or the ServerRenderableWithHydration capability:';
const HYDRATE_ERROR_MSG = 'has the ServerRenderableWithHydration capability so it cannot depend on components with no SSR capability:';
const CSR_ONLY = 'No SSR capability';
const LIST_SEP = '\n    - ';
/**
 * Given a root component, print out whether or not SSR capability conflicts exist in its tree
 * @param info Specifier, SSR capability and children for a component
 * @param verbose True if the component tree should be printed out
 * @returns True if there are no incompatibilities
 */
function print(info, verbose = false) {
    let success = true;
    const childCaps = { hydrate: new Set(), csr: new Set(), unknown: new Set() };
    groupCapabilitiesByType(childCaps, info);
    if (verbose) {
        const tree = buildTree([info], info.ssrCapability, true);
        tree && console.log('\n', treeify(tree), '\n');
    }
    if (info.ssrCapability === CAPABILITY_SSR_HYDRATION && childCaps.csr.size) {
        // FAIL: root component is SSRed with hydration, but has CSR-only children
        success = false;
        console.log(chalk.bgRed('FAIL'), chalk.bgGray(info.specifier), HYDRATE_ERROR_MSG, LIST_SEP + [...childCaps.csr].join(LIST_SEP));
    }
    else if (info.ssrCapability === CAPABILITY_SSR_ONLY && (childCaps.csr.size || childCaps.hydrate.size)) {
        // FAIL: root component is SSR-only, but has CSR-only or hydrated children
        success = false;
        console.log(chalk.bgRed('FAIL'), chalk.bgGray(info.specifier), SSR_ERROR_MSG, LIST_SEP + [...childCaps.csr, ...childCaps.hydrate].join(LIST_SEP));
    }
    else {
        // PASS: root component is CSR-only, or there are no conflicts
        console.log(chalk.bgGreen('PASS'), chalk.bgGray(info.specifier), SUCCESS_MSG);
    }
    if (childCaps.unknown.size) {
        // WARN: there are components in the tree with unknown SSR capabilities
        console.log(chalk.bgYellow('WARN'), chalk.bgGray(info.specifier), WARN_MSG, LIST_SEP + [...childCaps.unknown].join(LIST_SEP));
    }
    return success;
}
/**
 * Organize all the capabilities in a component tree by type
 * @param all Arrays of component specifiers by SSR capability type: hydrate, CSR-only and unknown
 * @param info Specifier, SSR capability and children for a component
 * @returns Arrays of component specifiers by SSR capability type
 */
function groupCapabilitiesByType(all, info) {
    for (const c of info.children) {
        if (c.ssrCapability === CAPABILITY_UNKNOWN) {
            all.unknown.add(c.specifier);
        }
        else if (!c.isLWC) {
            continue; // incompatible LWC util modules are just warnings
        }
        else if (!c.ssrCapability) {
            all.csr.add(c.specifier);
        }
        else if (c.ssrCapability === CAPABILITY_SSR_HYDRATION) {
            all.hydrate.add(c.specifier);
        }
        groupCapabilitiesByType(all, c);
    }
    return all;
}
/**
 * Recursively build a tree object which shows the status, specifier and SSR capability of each component
 * @param info Specifier, SSR capability and children for a component
 * @param rootCap Capability of the root component being analyzed
 * @param isRoot True if the root is being processed
 * @returns A tree object which can be printed in verbose mode
 */
function buildTree(info, rootCap, isRoot = false) {
    if (!info.length)
        return null;
    const tree = {};
    for (const i of info) {
        const cap = i.ssrCapability;
        if (cap === CAPABILITY_PRIVATE)
            continue; // exclude private platform deps
        const pass = !rootCap ||
            (rootCap === CAPABILITY_SSR_ONLY && cap === CAPABILITY_SSR_ONLY) ||
            (rootCap === CAPABILITY_SSR_HYDRATION && cap);
        const status = isRoot
            ? ''
            : cap === CAPABILITY_UNKNOWN
                ? chalk.bgYellow('WARN')
                : pass
                    ? chalk.bgGreen('PASS')
                    : !i.isLWC
                        ? chalk.bgYellow('WARN (non-component module)')
                        : chalk.bgRed('FAIL');
        tree[`${status} ${chalk.bgGray(i.specifier)} ${cap || CSR_ONLY}`] = buildTree(i.children, rootCap);
    }
    return tree;
}
//# sourceMappingURL=audit.js.map