/// <reference path="PrimeFaces-module.d.ts" />
/// <reference types="chart.js" />
/// <reference types="cropperjs" />
/// <reference types="downloadjs" />
/// <reference types="google.maps" />
/// <reference types="inputmask" />
/// <reference types="jquery" />
/// <reference types="jqueryui" />
/// <reference types="js-cookie" />
/// <reference types="moment-timezone" />
// Type definitions for PrimeFaces
// Project: PrimeFaces https://github.com/primefaces
// Definitions by: Andre Wachsmuth https://github.com/blutorange/
/**
 * Exposes the Print.js library as a global variable. Normally you do not
 * have to use this yourself, as PrimeFaces calls it for you when you use
 * the printer component. Also note that while PrimeFaces currently uses
 * this library for printing, this is subject to change in future releases.
 */
declare const printJS: typeof import("print-js");
// Additional required type declarations for PrimeFaces that are not picked up from the source code
/**
 * An object that can be used to emulate classes and a class hierarchy in JavaScript. This works even for old
 * browsers that do no support the native `class` syntax yet. Note however, that this should be mostly compatible
 * with the new `class` syntax of JavaScript, so consider creating your own widgets as a class:
 *
 * ```javascript
 * class MyWidget extends PrimeFaces.widget.BaseWidget {
 *   init(cfg){
 *     // ...
 *   }
 * }
 * ```
 *
 * __Note to TypeScript users__: You will need to specify the type parameters explicitly. The best way to do so is by
 * defining the interfaces for the classes separately:
 * ```typescript
 * interface BaseWidgetCfg {
 *  prop1: string;
 * }
 * interface AccordionPanelCfg extends BaseWidgetCfg {
 *   prop2: boolean;
 * }
 * interface BaseWidget {
 *   init(cfg: BaseWidgetCfg): void;
 *   method1(x: number): boolean;
 * }
 * interface Accordion extends BaseWidget {
 *   init(cfg: AccordionPanelCfg): void;
 *   method1(): boolean;
 *   method2(): boolean;
 * }
 * ```
 *
 * Now you can use it normally:
 * ```typescript
 * const BaseWidget = Class.extend<BaseWidget, [BaseWidgetCfg]>({
 *   init(cfg: BaseWidgetCfg) {
 *     // ...
 *   },
 *   method1(x: number): boolean {
 *     return x > 0;
 *   },
 * });
 * const Accordion = BaseWidget.extend<Accordion, [AccordionCfg]>({
 *   init(cfg: AccordionCfg) {
 *     this._super(cfg);
 *   },
 *   method1() {
 *     return !this._super(42);
 *   },
 *   method2() {
 *     return true;
 *   }
 * });
 * const base: BaseWidget = new BaseWidget({prop1: ""});
 * const accordion: Accordion = new Accordion({prop1: "", prop2: true});
 * base.method1(2);
 * accordion.method1();
 * accordion.method2();
 * ```
 */
declare const Class: PrimeFaces.Class;
declare namespace PrimeFaces {
    /**
     * Similar to `Partial<Base>`, but in addition to making the properties optional, also makes the properties
     * nullable.
     * @typeparam Base Type of an object with properties to make partial.
     * @return A new type with all properties of the given type `Base` made optional and nullable.
     */
    export type PartialOrNull<Base> = {
        [P in keyof Base]?: Base[P] | null;
    };
    /**
     * Constructs a new type by making all properties `Key` in `Base` optional.
     * ```typescript
     * type X = {foo: string, bar: string, baz: string};
     * type Y = PartialBy<X, "foo" | "baz">;
     * // type Y = {foo?: string, bar: string, baz?: string};
     * ```
     * @typeparam Base Type for which some properties are made optional.
     * @typeparam Key Type of the keys that are made optional.
     * @return A subtype of `Base` with the all properties `Key` made optional.
     */
    export type PartialBy<Base, Key extends keyof Base> = Omit<Base, Key> & Partial<Pick<Base, Key>>;
    /**
     * Constructs a new type by intersecting the given `Base` with `void`. This is a hack, required because some methods
     * of a parent class explicitly return a value, but some derived classes do not.
     * @typeparam Base Base type to intersect
     * @return A new type that is the intersection of the given `Base` with `void`.
     */
    export type ReturnOrVoid<Base> =
    // eslint-disable-next-line @typescript-eslint/no-invalid-void-type
    Base | void;
    /**
     * Constructs a new type that is the union of all types of each property in T.
     * ```typescript
     * type X = {a: ["bar", RegExp], b: ["foo", number]};
     * type Y = ValueOf<X>;
     * // type Y = ["bar", RegExp] | ["foo", number]
     * ```
     * @typeparam T A type with keys.template
     */
    export type ValueOf<T> = T[keyof T];
    /**
     * Constructs an object type out of a union of two-tuples. The first item in
     * the pair is used as the key, the second item as the type of the
     * property's value.
     * ```typescript
     * type Y = ["bar", RegExp] | ["foo", number];
     * type Z = KeyValueTupleToObject<Y>;
     * // type Z = { bar: RegExp; foo: number; }
     * ```
     * @typeparam T A union of pairs with the key and value for the object's properties.
     */
    export type KeyValueTupleToObject<T extends [
        PropertyKey,
        unknown
    ]> = {
        [K in T[0]]: Extract<T, [
            K,
            unknown
        ]>[1];
    };
    /**
     * Constructs a new type by renaming the properties in `Base` according to the `RenameMap`.
     * ```typescript
     * type Z = { bar: RegExp; foo: number; };
     * type S = RenameKeys<Z, {bar: "b", foo: "f"}>;
     * // type S = { b: RegExp; f: number; }
     * ```
     * @typeparam Base Type with properties to rename
     * @typeparam RenameMap Type with string properties that indicate how to rename the keys of `Base`.
     */
    export type RenameKeys<Base, RenameMap extends Record<string, string>> = KeyValueTupleToObject<ValueOf<{
        [Key in keyof Base]: [
            Key extends keyof RenameMap ? RenameMap[Key] : Key,
            Base[Key]
        ];
    }>>;
    /**
     * Constructs a new type by binding the this context of `Base` to another type `ThisContext`.
     * ```typescript
     * type X = (this: string, x: string) => boolean;
     * type Y = BindThis<X, number>;
     * // type Y = (this: number, x: string) => boolean
     * ```
     * @typeparam Base Type to rebind.
     * @typeparam ThisContext New this context for `Base`.
     * @return If `Base` is a function type, that type with the this context bound to `ThisContext`. Otherwise, returns
     * just `Base`.
     */
    export type BindThis<Base, ThisContext> = Base extends (...args: never[]) => unknown ? (this: ThisContext, ...args: Parameters<Base>) => ReturnType<Base> : Base;
    /**
     * Constructs a new type by binding the this context of `Base` to the `ThisContext`. Additionally, also adds a new
     * property `_super` to the this context of `Base` that is also of type `Base`.
     * ```typescript
     * type X = {foo: string, bar: number};
     * type Y = (this: string, k: string) => boolean;
     * type Z = BindThis<Y, X>;
     * // type Z = (this: {foo: string, bar: number, _super: Y}, k: string) => boolean
     * ```
     * @typeparam Base Type to rebind.
     * @typeparam ThisContext New this context for `Base`.
     * @return If `Base` is a function type, that type with the this context bound to `ThisContext` and with an
     * additional property `_super` of type `Base` added. Otherwise, returns just `Base`.
     */
    export type BindThisAndSuper<Base, ThisContext> = Base extends (...args: never[]) => unknown ? (this: ThisContext & {
        _super: Base;
    }, ...args: Parameters<Base>) => ReturnType<Base> : Base;
    /**
     * Constructs a new type that is the intersection of all property names in `Base` whose type is assignable to
     * `Condition`:
     * ```typescript
     * interface User {
     *   name: string;
     *   mail: string;
     *   active: boolean;
     * }
     *
     * type UserStringKeys = MatchingKeys<User, string>;
     * // type UserStringKeys = "name" | "mail";
     * ```
     * @typeparam Base Type from which to pick some properties.
     * @typeparam Condition Type which the properties in the base type have to match.
     * @return A string intersection type of property names from the base type that match the condition.
     */
    type MatchingKeys<Base, Condition> = {
        [Key in keyof Base]: Base[Key] extends Condition ? Key : never;
    }[keyof Base];
    /**
     * Constructs a new type by picking from `Base` all properties that are assignable to `Condition`.
     * ```typescript
     * interface User {
     *   name: string;
     *   mail: string;
     *   active: boolean;
     * }
     *
     * type UserStringProperties = PickMatching<User, string>;
     * // type UserStringProperties = {name: string, mail: string};
     * ```
     * @typeparam Base Type from which to pick some properties.
     * @typeparam Condition Type which the properties in the base type have to match.
     * @return A subtype of the base type with all properties excluded that do not match the condition.
     */
    export type PickMatching<Base, Condition> = Pick<Base, MatchingKeys<Base, Condition>>;
    /**
     * Given the type of the base class and the sub class, constructs a new type for the argument of `Class.extend(...)`,
     * except that all properties are required.
     * @typeparam TBase Type of the base class.
     * @typeparam TSub Type of the subclass.
     * @return A mapped type with properties P. If the property P is function type F, then the this context is bound to
     * TSub. Additionally, if P is a property only of TBase and not of TSub, a special property `_super` of type F is
     * added to the this context.
     */
    export type ClassExtendProps<TBase, TSub> = {
        [P in keyof TSub]: P extends keyof TBase ? BindThisAndSuper<TBase[P], TSub> : BindThis<TSub[P], TSub>;
    };
    /**
     * The widget configuration of each widget may contain only some of the declared properties. For example, when the
     * value of a property is equal to the default value, it is not transmitted from the server to the client. Only the
     * two properties `id` and `widgetVar` are guaranteed to be always available.
     * @typeparam TCfg Type of a widget configuration. It must have at least the two properties `id` and `widgetVar`.
     * @return A new type with all properties in the given type made optional, exception for `id` and `widgetVar`.
     */
    export type PartialWidgetCfg<TCfg extends {
        id: string | string[];
        widgetVar: string;
    }> = Partial<Omit<TCfg, "id" | "widgetVar">> & Pick<TCfg, "id" | "widgetVar">;
    /**
     * An object that can be used to emulate classes and a class hierarchy in JavaScript. This works even for old
     * browsers that do no support the native `class` syntax yet. Note however, that this should be mostly compatible
     * with the new `class` syntax of JavaScript, so consider creating your own widgets as a class:
     *
     * ```javascript
     * class MyWidget extends PrimeFaces.widget.BaseWidget {
     *   init(cfg){
     *     // ...
     *   }
     * }
     * ```
     *
     * Note for TypeScript users: Normally you should just write a widget as a class that extends from the appropriate
     * base class. If you must use this method you will need to specify the type parameters explicitly. The best way to
     * do so is by defining the interfaces for the classes separately:
     * ```typescript
     * interface BaseWidgetCfg {
     *  prop1: string;
     * }
     * interface AccordionCfg extends BaseWidgetCfg {
     *   prop2: boolean;
     * }
     * interface BaseWidget {
     *   init(cfg: BaseWidgetCfg): void;
     *   method1(x: number): boolean;
     * }
     * interface Accordion extends BaseWidget {
     *   init(cfg: AccordionCfg): void;
     *   method1(): boolean;
     *   method2(): boolean;
     * }
     * ```
     *
     * Now you can use it normally:
     * ```typescript
     * const BaseWidget = Class.extend<BaseWidget, [BaseWidgetCfg]>({
     *   init(cfg: BaseWidgetCfg) {
     *     // ...
     *   },
     *   method1(x: number): boolean {
     *     return x > 0;
     *   },
     * });
     * const Accordion = BaseWidget.extend<Accordion, [AccordionCfg]>({
     *   init(cfg: AccordionCfg) {
     *     this._super(cfg);
     *   },
     *   method1() {
     *     return !this._super(42);
     *   },
     *   method2() {
     *     return true;
     *   }
     * });
     * const base: BaseWidget = new BaseWidget({prop1: ""});
     * const accordion: Accordion = new Accordion({prop1: "", prop2: true});
     * base.method1(2);
     * accordion.method1();
     * accordion.method2();
     * ```
     */
    export interface Class<TBase = Record<string, unknown>> {
        new (): Class<TBase>;
        extend<TSub extends {
            init(...args: TArgs): void;
        } & Omit<TBase, "init">, TArgs extends never[]>(prop: PartialBy<ClassExtendProps<TBase, TSub>, keyof Omit<TBase, "init">>): Class<TSub> & (new (...args: TArgs) => TSub) & {
            prototype: TSub;
        };
    }
    /**
     * Maps the return type of a method of an instance method of a JQueryUI widget instance to the return type of the
     * JQueryUI wrapper:
     * - When an instance method returns `undefined` or the instance itself, the JQuery instance is returned.
     * - Otherwise, the value of the instance method is returned.
     * @typeparam W Type of the widget instance.
     * @typeparam R Type of a value returned by a widget instance method.
     * @typeparam JQ Type of the JQuery instance.
     * @return The type that is returned by the JQueryUI wrapper method.
     */
    export type ToJQueryUIWidgetReturnType<W, R, JQ> =
    // eslint-disable-next-line @typescript-eslint/no-invalid-void-type
    R extends W | undefined | void ? JQ : R extends undefined | void ? R | JQ : R;
    /**
     * An object with all localized strings required on the client side.
     */
    export interface Locale {
        allDayText?: string;
        aria?: Record<string, string>;
        closeText?: string;
        prevText?: string;
        nextText?: string;
        monthNames?: [
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        monthNamesShort?: [
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        dayNames?: [
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        dayNamesShort?: [
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        dayNamesMin?: [
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        weekHeader?: string;
        weekNumberTitle?: string;
        firstDay?: number;
        isRTL?: boolean;
        showMonthAfterYear?: boolean;
        yearSuffix?: string;
        timeOnlyTitle?: string;
        timeText?: string;
        hourText?: string;
        minuteText?: string;
        secondText?: string;
        currentText?: string;
        year?: string;
        ampm?: boolean;
        month?: string;
        week?: string;
        day?: string;
        noEventsText?: string;
        moreLinkText?: string;
        list?: string;
        messages?: Record<string, string>;
        [i18nKey: string]: unknown;
    }
}
declare namespace PrimeFaces {
    /**
     * Defines the possible severity levels of a faces message (a message shown to the user).
     *
     * - fatal: Indicates that the message reports a grave error that needs the immediate attention of the reader.
     * - error: Indicates that the message reports an error that occurred, such as invalid user input or database
     * connection failures etc.
     * - warn: Indicates that the message reports a possible issue, but it does not prevent the normal operation of the
     * system.
     * - info: Indicates that the message provides additional information, if the reader is interested.
     */
    export type FacesMessageSeverity = "fatal" | "error" | "warn" | "info";
    /**
     * A converter for converting string values to the correct data type.
     */
    export interface Converter<T = unknown> {
        /**
         * Converts a string value to the correct data type.
         * @param element Element for which the value was submitted.
         * @param submittedValue The submitted string value
         * @return The converted value.
         */
        convert(element: JQuery, submittedValue: string | null | undefined): T;
    }
    /**
     * A validator for checking whether the value of an element confirms to certain restrictions.
     */
    export interface Validator<T = unknown> {
        /**
         * Validates the given element. If it is not valid, the error message should be thrown.
         * @param element Element to validate
         * @param value Current value of the element
         * @throws The error message as the string when the element with its current value is not valid.
         */
        validate(element: JQuery, value?: T): void;
    }
    /**
     * A 'FacesMessage' with a short summary message and a more detailed message, as well as a severity level that
     * indicates the type of this message. Used by the client-side validation framework and some widgets such as the
     * growl widget.
     */
    export interface FacesMessage {
        /**
         * A short summary of the message.
         */
        summary: string;
        /**
         * In-depth details of the message.
         */
        detail: string;
        /**
         * The severity of this message, i.e. whether it is an information message, a warning message, or an error
         * message.
         */
        severity: FacesMessageSeverity;
        /**
         * The severity in I18N human readable text for ARIA screen readers.
         */
        severityText?: string;
        /**
         * If the message was successfully rendered by a message/growl component.
         */
        rendered: boolean;
    }
    /*
                 * __Note__: Do not parametrize the this context via a type parameter. This would require changing the return type
                 * of BaseWidget#getBehavior to "PrimeFaces.Behavior<this>"". If that were done, however, it would not be longer be
                 * possible to assign, for example, an object of type AccordionPanel to a variable of type BaseWidget - as that
                 * would allow calling "getBehavior" on the AccordionPanel and only passing a BaseWidget as the this context.
                 */
    /**
     * A callback function for a behavior of a widget. A behavior is a way for associating client-side scripts with UI
     * components that opens all sorts of possibilities, including client-side validation, DOM and style manipulation,
     * keyboard handling, and more. When the behavior is triggered, the configured JavaScript gets executed.
     *
     * Behaviors are often, but not necessarily, AJAX behavior. When triggered, it initiates a request the server and
     * processes the response once it is received. This enables several features such as updating or replacing elements
     * dynamically. You can add an AJAX behavior via `<p:ajax event="name" actionListener="#{...}" onstart="..." />`.
     */
    export type Behavior =
    /**
     * @this This callback takes the widget instance as the this context. This must be the widget instance that owns
     * the behavior. The type is only required to be a {@link BaseWidget} as only common widget properties such as
     * its ID are used.
     * @param ext Additional data to be sent with the AJAX request that is made to the server.
     */
    (this: PrimeFaces.widget.BaseWidget, ext?: Partial<PrimeFaces.ajax.ConfigurationExtender>) => void;
    /**
     * The most recent instance of a {@link PrimeFaces.widget.ConfirmDialog} instance that was opened in response to a
     * global confirmation request.
     */
    export let confirmDialog: PrimeFaces.widget.ConfirmDialog | undefined;
    /**
     * The main container element of the source component that issued the confirmation request, used e.g. by the
     * `ConfirmDialog` widget.
     */
    export let confirmSource: JQuery | undefined | null;
    /**
     * The main container element of the source component that issued the popup confirmation request, used e.g. by the
     * `ConfirmPopup` widget.
     */
    export let confirmPopupSource: JQuery | undefined | null;
    /**
     * CSS transition callbacks that can be passed to the methods in {@link CssTransitionHandler}.
     * @since 10.0.0
     */
    export interface CssTransitionCallback {
        /**
         * Called when the entering process is about to start.
         */
        onEnter?: (this: Window) => void;
        /**
         * Called during the entering process.
         * @this The event that occurred. When animations are globally disabled, this callback may still be called, but
         * no event is passed and the this context is the Window.
         */
        onEntering?: (this: JQuery.TriggeredEvent | Window) => void;
        /**
         * Called when the entering process has finished.
         * @this The event that occurred. When animations are globally disabled, this callback may still be called, but
         * no event is passed and the this context is the Window.
         */
        onEntered?: (this: JQuery.TriggeredEvent | Window) => void;
        /**
         * Called when the exiting process is about to start.
         */
        onExit?: () => void;
        /**
         * Called during the exiting process.
         * @this The event that occurred. When animations are globally disabled, this callback may still be called, but
         * no event is passed and the this context is the Window.
         */
        onExiting?: (this: JQuery.TriggeredEvent | Window) => void;
        /**
         * Called when the exiting process has finished.
         * @this The event that occurred. When animations are globally disabled, this callback may still be called, but
         * no event is passed and the this context is the Window.
         */
        onExited?: (this: JQuery.TriggeredEvent | Window) => void;
    }
    /**
     * Methods for a CSS transition that are returned by {@link PrimeFaces.utils.registerCSSTransition}.
     * @since 10.0.0
     */
    export interface CssTransitionHandler {
        /**
         * Should be called when an element gets shown.
         * @param callbacks Optional callbacks that will be invoked at the appropriate time.
         */
        show(callbacks?: CssTransitionCallback): void;
        /**
         * Should be called when an element gets hidden.
         * @param callbacks Optional callbacks that will be invoked at the appropriate time.
         */
        hide(callbacks?: CssTransitionCallback): void;
    }
    /**
     * Handler returned by methods that add an event handler that can be used to dispose of bound handler. Removes the
     * event handler that was added.
     */
    export interface UnbindCallback {
        /**
         * Removes the event handler or handlers that were added when this callback was created.
         */
        unbind(): void;
    }
}
declare namespace PrimeFaces.ajax {
    /**
     * An entry on the {@link JQuery.jqXHR} request object with additional values added by PrimeFaces. For example, when
     * you call `PrimeFaces.current().ajax().addCallbackParam(...)` on the server in a bean method, the added parameters
     * are available in this object. This is also how you can access pass values from the server to the client after
     * calling a remote command. See {@link PrimeFaces.ajax.pfXHR} and {@link PrimeFaces.ab}.
     */
    export type PrimeFacesArgs = Record<string, unknown>;
    /**
     * Additional settings on a {@link JQuery.jqXHR} request, such as portlet forms and nonces.
     */
    export type PrimeFacesSettings = Record<string, unknown>;
    /**
     * Callback for an AJAX request that is always called after the request completes, irrespective of whether it
     * succeeded or failed.
     *
     * This is the type of function that you can set as a client side callback for the `oncomplete` attribute of a
     * component or an AJX behavior.
     */
    export type CallbackOncomplete =
    /**
     * @this The current AJAX settings as they were passed to JQuery when the request was made.
     * @param xhrOrErrorThrown Either the XHR request that was made (in case of success), or the error that was
     * thrown (in case of an error).
     * @param status The type of error or success.
     * @param pfArgs Additional arguments returned by PrimeFaces, such as AJAX callback params from beans.
     * @param dataOrXhr Either the XMLDocument (in case of success), or the XHR request (in case of an error).
     */
    (this: JQuery.AjaxSettings, xhrOrErrorThrown: unknown, status: JQuery.Ajax.TextStatus, pfArgs: PrimeFacesArgs, dataOrXhr: XMLDocument | pfXHR) => void;
    /**
     * Callback for an AJAX request that is called in case any error occurred during the request, such as a a network
     * error. Note that this is not called for errors in the application logic, such as when bean validation fails.
     *
     * This is the type of function that you can set as a client side callback for the `onerror` attribute of a
     * component or an AJX behavior.
     */
    export type CallbackOnerror =
    /**
     * @this The current AJAX settings as they were passed to JQuery when the request was made.
     * @param xhr The XHR request that failed.
     * @param status The type of error that occurred.
     * @param errorThrown The error with details on why the request failed.
     */
    (this: JQuery.AjaxSettings, xhr: pfXHR, status: JQuery.Ajax.ErrorTextStatus, errorThrown: string) => void;
    /**
     * Callback for an AJAX request that is called before the request is sent. Return `false` to cancel the request.
     *
     * This is the type of function that you can set as a client side callback for the `onstart` attribute of a
     * component or an AJX behavior.
     */
    export type CallbackOnstart =
    /**
     * @this The {@link PrimeFaces.ajax.Request} singleton instance responsible for handling the request.
     * @param cfg The current AJAX configuration.
     * @return {boolean | undefined} `false` to abort and not send the request, `true` or `undefined` otherwise.
     */
    (this: PrimeFaces.ajax.Request, cfg: Configuration) => boolean;
    /**
     * Callback for an AJAX request that is called when the request succeeds.
     *
     * This is the type of function that you can set as a client side callback for the `onsuccess` attribute of a
     * component or an AJX behavior.
     */
    export type CallbackOnsuccess =
    /**
     * @this The current AJAX settings as they were passed to JQuery when the request was made.
     * @param data The XML document representing the partial response returned the JSF application in response
     * to the faces request. It usually looks like this: `<changes>...</changes>`
     * @param status The type of success, usually `success`.
     * @param xhr The XHR request that succeeded.
     * @return `true` if this handler already handle and/or parsed the response, `false` or `undefined` otherwise.
     */
    (this: JQuery.AjaxSettings, data: XMLDocument, status: JQuery.Ajax.SuccessTextStatus, xhr: pfXHR) => boolean | undefined;
    /**
     * The XHR request object used by PrimeFaces. It extends the `jqXHR` object as used by JQuery, but adds additional
     * properties specific to PrimeFaces.
     * @typeparam P Data made available by the server via {@link pfXHR.pfArgs}.
     */
    export interface pfXHR<P extends PrimeFacesArgs = PrimeFacesArgs> extends JQuery.jqXHR {
        /**
         * An object with additional values added by PrimeFaces. For example, when you call
         * `PrimeFaces.current().ajax().addCallbackParam(...)` on the server in a bean method, the added parameters are
         * available in this object. This is also how you can access pass values from the server to the client after
         * calling a remote command.  See {@link PrimeFaces.ajax.pfXHR} and {@link PrimeFaces.ab}.
         */
        pfArgs?: P;
        /**
         * Additional settings, such as portlet forms and nonces.
         */
        pfSettings?: PrimeFacesSettings;
    }
    /**
     * Represents the data of a PrimeFaces AJAX request. This is the value that is returned by {@link PrimeFaces.ab} and
     * {@link PrimeFaces.ajax.Request.handle}.
     * @typeparam P Record type of the data made available in the property {@link PrimeFaces.ajax.pfXHR.pfArgs} by the
     * server.
     */
    export interface ResponseData<P extends PrimeFacesArgs = PrimeFacesArgs> {
        /**
         * The XML document that was returned by the server. This may include several elements such as `update` for DOM
         * updates that need to be performed, `executeScript` for running JavaScript code. A typical response might look
         * as follows:
         *
         * ```xml
         * <?xml version="1.0" encoding="UTF-8"?>
         * <partial-response>
         *    <changes>
         *       <update id="content:msgs"><![CDATA[
         *           <span id="content:msgs" class="ui-growl-pl">...</span>
         *           <script id="content:msgs_s" type="text/javascript">...</script>
         *       ]]></update>
         *       <update id="content:javax.faces.ViewState:0"><![CDATA[3644438980748093603:2519460806875181703]]></update>
         *    </changes>
         * </partial-response>
         * ```
         */
        document: XMLDocument;
        /**
         * The jQuery XHR request object that was used for the request.
         *
         * __Note__: This object has a `pfArgs` entry that contains the values added to the response by the server. See
         * {@link PrimeFaces.ajax.pfXHR.pfArgs} and {@link PrimeFaces.ajax.RemoteCommand}.
         */
        jqXHR: PrimeFaces.ajax.pfXHR<P>;
        /**
         * A string describing the type of success. Usually the HTTP status text.
         */
        textStatus: JQuery.Ajax.SuccessTextStatus;
    }
    /**
     * Represents the data passed to the exception handler of the promise returned by {@link PrimeFaces.ab} and
     * {@link PrimeFaces.ajax.Request.handle}.
     */
    export interface FailedRequestData {
        /**
         * An optional exception message, if an error occurred.
         */
        errorThrown: string;
        /**
         * The jQuery XHR request object that was used for the request. May not be available when no HTTP request was
         * sent, such as when validation failed.
         */
        jqXHR?: PrimeFaces.ajax.pfXHR;
        /**
         * A string describing the type of error that occurred.
         */
        textStatus: JQuery.Ajax.SuccessTextStatus;
    }
    /**
     * Describes a server callback parameter for an AJAX call. For example, when you call a
     * `<p:remoteCommand name="myCommand" />` from the client, you may pass additional parameters to the backing
     * bean like this:
     *
     * ```javascript
     * myCommand([
     *   {
     *     name: "MyParam",
     *     value: "MyValue",
     *   }
     * ]);
     * ```
     *
     * In the backing bean, you can access this parameter like this:
     *
     * ```java
     * final String myParam = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("myParam");
     * ```
     * @typeparam K Name of this parameter.
     * @typeparam V Type of the value of the callback parameter. Please note that it will be converted to string
     * before it is passed to the server.
     */
    export interface RequestParameter<K extends string = string, V = unknown> {
        /**
         * The name of the parameter to pass to the server.
         */
        name: K;
        /**
         * The value of the parameter to pass to the server.
         */
        value: V;
    }
    /**
     * The response of an AJAX request may contain one or more actions such as executing scripts or updating DOM nodes.
     * This interface represents a handler for an `update` action.
     * @typeparam TWidget Type of the widget which
     * triggered the AJAX request.
     */
    export interface UpdateHandler<TWidget extends PrimeFaces.widget.BaseWidget = PrimeFaces.widget.BaseWidget> {
        /**
         * The widget which triggered the AJAX request.
         */
        widget: TWidget;
        /**
         * The handle function which is given the HTML string of the update
         * @param content The new HTML content from the update.
         */
        handle(this: TWidget, content: string): void;
    }
    /**
     * Represents the selection of an INPUT or TEXTAREA element.
     */
    export interface ActiveElementSelection {
        /**
         * Start of the selection, that is, the index of the first selected character.
         */
        start: number;
        /**
         * End of the selection, that is, one plus the index of the last selected character.
         */
        end: number;
        /**
         * The number of selected characters.
         */
        length: number;
        /**
         * The selected text
         */
        text: string;
    }
    /**
     * The options that can be passed to AJAX calls made by PrimeFaces. Note that you do not have to provide a value
     * for all these property. Most methods methods such as `PrimeFaces.ab` have got sensible defaults in case you
     * do not.
     */
    export interface Configuration {
        /**
         * If `true`, the the request is sent immediately. When set to `false`, the AJAX request is added to a
         * global queue to ensure that only one request is active at a time, and that each response is processed
         * in order. Defaults to `false`.
         */
        async: boolean;
        /**
         * Delay in milliseconds. If less than this delay elapses between AJAX requests, only the most recent one is
         * sent and all other requests are discarded. If this option is not specified, no delay is used.
         */
        delay: number;
        /**
         * A PrimeFaces client-side search expression (such as `@widgetVar` or `@(.my-class)` for locating the form
         * to with the input elements that are serialized. If not given, defaults to the enclosing form.
         */
        formId: string;
        /**
         * The AJAX behavior event that triggered the AJAX request.
         */
        event: string;
        /**
         * Additional options that can be passed when sending an AJAX request to override the current options.
         */
        ext: Partial<ConfigurationExtender>;
        /**
         * Additional search expression that is added to the `process` option.
         */
        fragmentId: string;
        /**
         * Whether this AJAX request is global, ie whether it should trigger the global `<p:ajaxStatus />`. Defaults
         * to `true`.
         */
        global: boolean;
        /**
         * `true` if components with `<p:autoUpdate/`> should be ignored and updated only if specified explicitly
         * in the `update` option; or `false` otherwise. Defaults to `false`.
         */
        ignoreAutoUpdate: boolean;
        /**
         * Callback that is always called after the request completes, irrespective of whether it succeeded or
         * failed.
         */
        oncomplete: CallbackOncomplete;
        /**
         * Callback that is called in case any error occurred during the request, such as a a network error. Note
         * that this is not called for errors in the application logic, such as when bean validation fails.
         */
        onerror: CallbackOnerror;
        /**
         * Callback that is called before the request is sent. Return `false` to cancel the request.
         */
        onstart: CallbackOnstart;
        /**
         * Callback that is called when the request succeeds.
         */
        onsuccess: CallbackOnsuccess;
        /**
         * Additional parameters that are passed to the server. These can be accessed as follows:
         *
         * ```java
         * final String myParam = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("myParam");
         * ```
         */
        params: RequestParameter[];
        /**
         * `true` to perform a partial submit and not send the entire form data, but only the processed components;
         * or `false` to send the entire form data. Defaults to `false`.
         */
        partialSubmit: boolean;
        /**
         * A CSS selector for finding the input elements of partially processed components. Defaults to `:input`.
         */
        partialSubmitFilter: string;
        /**
         * A (client-side) PrimeFaces search expression for the components to process in the AJAX request.
         */
        process: string;
        /**
         * A promise object that is resolved when the AJAX request is complete. You can use this option to register
         * a custom callback. Please note that usually you do not have to set this option explicitly, you can use the
         * return value of {@link PrimeFaces.ab} or {@link PrimeFaces.ajax.Request.handle}. It will create a new promise
         * object when none was provided, and return that.
         */
        promise: Promise<PrimeFaces.ajax.ResponseData>;
        /**
         * `true` if the AJAX request is a reset request that resets the value of all form elements to their
         * initial values, or `false` otherwise. Defaults to `false`.
         */
        resetValues: boolean;
        /**
         * `true` if child components should be skipped for the AJAX request, `false` otherwise. Used only by a few
         * specific components.
         */
        skipChildren: boolean;
        /**
         * The source that triggered the AJAX request.
         */
        source: string | JQuery | HTMLElement;
        /**
         * Set a timeout (in milliseconds) for the request. A value of 0 means there will be no timeout.
         */
        timeout: number;
        /**
         * A (client-side) PrimeFaces search expression for the components to update in the AJAX request.
         */
        update: string;
    }
    /**
     * Additional options that can be passed when sending an AJAX request to override the current options.
     */
    export type ConfigurationExtender = Pick<Configuration, "update" | "process" | "onstart" | "params" | "partialSubmit" | "onerror" | "onsuccess" | "oncomplete"> & {
        /**
         * If given, this function is called once for each component. It is passed that serialized values for the
         * component and should return the filtered values that are to be sent to the server. If not given, no
         * values are filtered, and all values are send to the server.
         * @param componentPostParams The serialized values of a component.
         * @return The filtered values that are to be sent to the server.
         */
        partialSubmitParameterFilter(this: Request, componentPostParams: RequestParameter[]): RequestParameter[];
    };
    /**
     * Options passed to AJAX calls made by PrimeFaces. This is the same as `Configuration`, but with shorter
     * option names and is used mainly by the method `PrimeFaces.ab`. See `Configuration` for a detailed description
     * of these options.
     */
    export type ShorthandConfiguration = RenameKeys<Configuration, {
        source: "s";
        formId: "f";
        process: "p";
        update: "u";
        event: "e";
        async: "a";
        global: "g";
        delay: "d";
        timeout: "t";
        skipChildren: "sc";
        ignoreAutoUpdate: "iau";
        partialSubmit: "ps";
        partialSubmitFilter: "psf";
        resetValues: "rv";
        fragmentId: "fi";
        params: "pa";
        onstart: "onst";
        onerror: "oner";
        onsuccess: "onsu";
        oncomplete: "onco";
    }>;
    /**
     * Helper type for the parameters of the remote command. You can specify an object type with the allowed parameter
     * names and their expected value types. This helps to increase type safety for remote commands. For example, when
     * this remote command with an appropriate bean implementation is defined:
     *
     * ```xml
     * <p:remoteCommand name="RemoteCommands.checkMaturity" ... />
     * ```
     *
     * Then you can declare (or generate automatically from the bean method!) this remote command in TypeScript like
     * this:
     *
     * ```typescript
     * declare const RemoteCommands {
     *   const checkMaturity: RemoteCommand<
     *     {name: string, age: number},
     *     {success: boolean, message: string}
     *   >;
     * }
     *
     * RemoteCommand.checkMaturity( [ { name: "name", value: "John Doe" } ] ) // works
     * RemoteCommand.checkMaturity( [ { name: "age", value: 12 } ] ) // works
     *
     * RemoteCommand.checkMaturity( [ { name: "username", value: "John Doe" } ] ) // error
     * RemoteCommand.checkMaturity( [ { name: "age", value: "12" } ] ) // error
     *
     * const response = await RemoteCommand.checkMaturity( [ { name: "name", value: "John Doe" } ];
     *
     * const success: boolean = response.jqXHR.pfArgs.success; // works
     * const message: string = response.jqXHR.pfArgs.message; // works
     * const message: string = response.jqXHR.pfArgs.errormessage; // error
     * ```
     * @typeparam T Record type with the param names and the corresponding param values.
     * @return An array type of {@link PrimeFaces.ajax.RequestParameter | request parameters} where the `name` can be
     * one of the keys of `T` and the `value` is the corresponding value from `T`. Array values are mapped to the item
     * type, so that `RemoteCommandParams<{names: string[]}>` is the same as `RemoteCommandParams<{names: string}>`.
     * This is done because multiple values for the same name should be send by including multiple items in the request
     * callback parameter array.
     */
    export type RemoteCommandParams<T extends Record<string, unknown> = Record<string, unknown>> = {
        [P in keyof T]: P extends string ? PrimeFaces.ajax.RequestParameter<P, T[P] extends (infer R)[] ? R : T[P]> : never;
    }[keyof T][];
    /**
     * Type for the JavaScript remote command function that is created via
     *
     * ```xml
     * <p:remoteCommand name="myCommand" listener="#{myBean.action}" />
     * ```
     *
     * This creates a variable `window.myCommand` that is of this type. On the client-side, you can pass parameters to
     * the remote command via
     *
     * ```javascript
     * window.myCommand([ { name: "myParamName", value: 9 } ]);
     * ```
     *
     * On the server-side, you can access them as follows:
     *
     * ```java
     * String myParamValue = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("myParamName")
     * ```
     *
     * To send data back to the client, use
     *
     * ```java
     * PrimeFaces.current().ajax().addCallbackParam("returnParamName", true);
     * ```
     *
     * Finally, to access the returned value on the client, do
     *
     * ```javascript
     * try {
     *   const response = await window.myCommand([ { name: "myParamName", value: 9 } ]);
     *   // Success, do something with the data
     *   const value = response.jqXHR.pfArgs.returnParamName;
     * }
     * catch (e) {
     *   // Handle error
     *   console.error("Could not invoke remote command", e);
     * }
     * ```
     *
     * Please note that you should not use async-await if you need to target old browsers, use `then`/`catch` on the
     * returned promise instead. See {@link RemoteCommandParams} for more details on how to use this TypeScript type.
     * @typeparam T Object type with the param names and the corresponding param values.
     * @typeparam R Object type of the data returned by the remote command.
     */
    export type RemoteCommand<T extends Record<string, unknown> = Record<string, unknown>, R extends PrimeFacesArgs = PrimeFacesArgs> =
    /**
     * @param params Optional parameters that are passed to the remote command.
     * @return A promise that is settled when the remote command it complete. It is resolved with the data received
     * from the server, and rejected when a network or server error occurred.
     */
    (params?: RemoteCommandParams<T>) => Promise<ResponseData<R>>;
}
declare namespace PrimeFaces.validation {
    /**
     * The validation result.
     */
    export interface ValidationResult {
        /**
         * A map between the client ID of an element and a list of faces message for that element.
         * @type {Record<string, PrimeFaces.FacesMessage[]>}
         */
        messages: Record<string, PrimeFaces.FacesMessage[]>;
        /**
         * If the result is valid / if it has any validation errors.
         */
        valid: boolean;
        /**
         * If the result has any unrendered message.
         */
        hasUnrenderedMessage: boolean;
    }
    /**
     * When an element is invalid due to a validation error, the user needs to be informed. A highlight handler is
     * responsible for changing the visual state of an element so that the user notices the invalid element. A highlight
     * handler is usually registered for a particular type of element or widget.
     */
    export interface Highlighter {
        /**
         * When an element is invalid due to a validation error, the user needs to be informed. This method must
         * highlight the given element in a way that makes the user notice that the element is invalid.
         * @param element An element to highlight.
         */
        highlight(element: JQuery): void;
        /**
         * When an element is invalid due to a validation error, the user needs to be informed. This method must
         * remove the highlighting of the given element that was added by `highlight`.
         */
        unhighlight(element: JQuery): void;
    }
    /**
     * The options that can be passed to the Validation method. Note that you do not have to provide a value
     * for all these property. Most methods such as `PrimeFaces.vb` have got sensible defaults in case you
     * do not.
     */
    export interface Configuration {
        /**
         * The source that triggered the validationt.
         */
        source: string | JQuery | HTMLElement;
        /**
         * `true` if the validation is triggered by AJAXified compoment. Defaults to `false`.
         */
        ajax: boolean;
        /**
         * A (client-side) PrimeFaces search expression for the components to process in the validation.
         */
        process: string;
        /**
         * A (client-side) PrimeFaces search expression for the components to update in the validation.
         */
        update: string;
        /**
         * `true` if invalid elements should be highlighted as invalid. Default is `true`.
         */
        highlight: boolean;
        /**
         * `true` if the first invalid element should be focussed. Default is `true`.
         */
        focus: boolean;
        /**
         * `true` if messages should be rendered. Default is `true`.
         */
        renderMessages: boolean;
        /**
         * `true` if invisible elements should be validated. Default is `false`.
         */
        validateInvisibleElements: boolean;
    }
    /**
     * Options passed to `PrimeFaces.vb` as shortcut. This is the same as `Configuration`, but with shorter
     * option names and is used mainly by the method `PrimeFaces.vb`. See `Configuration` for a detailed description
     * of these options.
     */
    export type ShorthandConfiguration = RenameKeys<Configuration, {
        source: "s";
        ajax: "a";
        process: "p";
        update: "u";
        highlight: "h";
        focus: "f";
        renderMessages: "r";
        validateInvisibleElements: "v";
    }>;
}
// JQuery extensions
/**
 * Here you can find additional methods on JQuery instances defined by various JQuery plugins. These methods are
 * usually defined by certain widgets and may not be available unless the widget and its JavaScript dependencies were
 * loaded.
 */
interface JQuery {
}
/**
 * Here you can find additional properties and methods defined on the global JQuery object, such as `$.browser`. These
 * properties and methods are usually defined by certain widgets and may not be available unless the widget and its
 * JavaScript dependencies were loaded.
 */
interface JQueryStatic {
}
declare namespace JQuery {
    /**
     * This interface contains all known types of events triggered by various jQuery extensions. It maps the name of the
     * event to the type the event that is triggered. Please note that this interface does not define the custom
     * additional arguments that may be passed when triggering event. These are deprecated in favor of using
     * `CustomEvent`.
     */
    interface TypeToTriggeredEventMap<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        /**
         * Triggered on the document before an AJAX request made by {@link PrimeFaces.ajax} starts.
         *
         * Usually the following arguments are passed to the callback:
         * - {@link PrimeFaces.ajax.pfXHR}: The AJAX request that is about to be sent.
         * - {@link JQuery.AjaxSettings}: The settings of the AJAX request.
         */
        pfAjaxSend: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered on the document when an AJAX request made by {@link PrimeFaces.ajax} starts.
         *
         * Usually no arguments are passed to the callback.
         */
        pfAjaxStart: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered on the document when an AJAX request made by {@link PrimeFaces.ajax} fails.
         *
         * Usually the following arguments are passed to the callback:
         * - {@link PrimeFaces.ajax.pfXHR}: The AJAX request that failed.
         * - {@link JQuery.AjaxSettings}: The settings of the AJAX request.
         * - A string: The error that occurred.
         */
        pfAjaxError: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered on the document when an AJAX request made by {@link PrimeFaces.ajax} succeeds.
         *
         * Usually the following arguments are passed to the callback:
         * - {@link PrimeFaces.ajax.pfXHR}: The AJAX request that was successful.
         * - {@link JQuery.AjaxSettings}: The settings of the AJAX request.
         */
        pfAjaxSuccess: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered on the document when an AJAX request completes (both success and failure). Only when `global` is set to `true`.
         *
         * Usually the following arguments are passed to the callback:
         * - {@link PrimeFaces.ajax.pfXHR}: The AJAX request that completed
         * - {@link JQuery.AjaxSettings}: The settings of the AJAX request.
         */
        pfAjaxComplete: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
    }
}
// Type declarations for the JSF core, see for example
// https://docs.oracle.com/javaee/7/javaserver-faces-2-2/jsdocs/symbols/jsf.html
/**
 * The top level global namespace for JavaServer Faces functionality.
 *
 * Please note that this functionality may only be available when you use certain tags such as `<f:ajax .../>` tags.
 */
declare namespace jsf {
    /**
     * An integer specifying the implementation version that this file implements. It's a monotonically increasing
     * number, reset with every increment of `jsf.specversion` This number is implementation dependent.
     */
    export const implversion: number;
    /**
     * The result of calling `UINamingContainer.getNamingContainerSeparatorChar()`: Used for separating IDs, such as a
     * colon (`:`).
     */
    export const separatorchar: string;
    /**
     * An integer specifying the specification version that this file implements. It's format is: rightmost two digits,
     * bug release number, next two digits, minor release number, leftmost digits, major release number. This number may
     * only be incremented by a new release of the specification.
     */
    export const specversion: number;
    /**
     * Return the `windowId` of the window in which the argument form is rendered.
     * @param node Determine the nature of the argument. If not present, search for the `windowId` within
     * `document.forms`. If present and the value is a string, assume the string is a DOM id and get the element with
     * that id and start the search from there. If present and the value is a DOM element, start the search from there.
     * @return The `windowId` of the current window, or `null` if the `windowId` cannot be determined.
     * @throws An error if more than one unique `WindowId` is found.
     */
    export function getClientWindow(node?: HTMLElement | string): string | null;
    /**
     * Return the value of `Application.getProjectStage()` for the currently running application instance. Calling this
     * method must not cause any network transaction to happen to the server.
     *
     * __Usage:__
     *
     * ```javascript
     * var stage = jsf.getProjectStage();
     * if (stage === ProjectStage.Development) {
     *     // ...
     * } else if stage === ProjectStage.Production) {
     *    // ...
     * }
     * ```
     * @return A string representing the current state of the running application in a typical product development
     * life cycle. Refer to `javax.faces.application.Application.getProjectStage` and
     * `javax.faces.application.ProjectStage`.
     */
    export function getProjectStage(): string;
    /**
     * Collect and encode state for input controls associated with the specified form element. This will include all
     * input controls of type hidden.
     *
     * __Usage:__
     *
     * ```javascript
     * var state = jsf.getViewState(form);
     * ```
     * @param form The form element whose contained input controls will be collected and encoded. Only successful
     * controls will be collected and encoded in accordance with: `Section 17.13.2` of the HTML Specification.
     * @return The encoded state for the specified form's input controls.
     */
    export function getViewState(form: HTMLFormElement): string;
}
/**
 * The namespace for Ajax functionality provided by JSF.
 *
 * Please note that this functionality may only be available when you use certain tags such as `<f:ajax .../>` tags.
 */
declare namespace jsf.ajax {
    /**
     * Possible status codes when an AJAX request fails.
     */
    export type ErrorStatus = "httpError" | "emptyResponse" | "serverError" | "malformedXML";
    /**
     * Possible status codes when an AJAX request succeeds.
     */
    export type EventStatus = "complete" | "success" | "begin";
    /**
     * Possible status codes when an AJAX request succeeds or fails.
     */
    export type Status = ErrorStatus | EventStatus;
    /**
     * A reference to a function to call on an error, see {@link addOnError}.
     */
    export type OnErrorCallback =
    /**
     * @param data Data with details about the error and the received response.
     */
    (data: OnErrorCallbackData) => void;
    /**
     * A reference to a function to call on an event, see {@link addOnEvent}.
     */
    export type OnEventCallback =
    /**
     * @param data Data with details about the received response.
     */
    (data: OnEventCallbackData) => void;
    /**
     * Base data for an event callback, see {@link addOnError} and {@link addOnEvent}.
     * @typeparam T Type of the `type` property.
     * @typeparam S Type of the `status` property.
     */
    export interface CallbackData<T extends string, S extends string> {
        /**
         * Type of the request result.
         */
        type: T;
        /**
         * Status describing the type of success or error.
         */
        status: S;
        /**
         * ID of the source component that triggered the AJAX request.
         */
        source: string;
        /**
         * HTTP response code of the request.
         */
        responseCode?: number;
        /**
         * XML of the response.
         */
        responseXML?: XMLDocument;
        /**
         * Raw text of the response.
         */
        responseText?: string;
    }
    /**
     * Data for the callback when an AJAX request fails, see {@link addOnError}.
     */
    export interface OnErrorCallbackData extends CallbackData<"error", ErrorStatus> {
        /**
         * Name of the error, if {@link status} is set to `serverError`.
         */
        errorName?: string;
        /**
         * Message of the error, if {@link status} is set to `serverError`.
         */
        errorMessage?: string;
        /**
         * Human readable description of the error.
         */
        description: string;
    }
    /**
     * Data for the callback when an AJAX request succeeds, see {@link addOnEvent}.
     */
    export interface OnEventCallbackData extends CallbackData<"event", EventStatus> {
    }
    /**
     * The request context that will be used during error/event handling. {@link jsf.ajax.response}.
     */
    export interface RequestContext {
        /**
         * The source DOM element for the request.
         */
        source: HTMLElement;
        /**
         * The error handler for the request.
         */
        onerror: OnErrorCallback;
        /**
         * The event handler for the request.
         */
        onevent: OnEventCallback;
    }
    /**
     * The set of available options that can be sent as request parameters to control client and/or server side request
     * processing. Used by {@link jsf.ajax.request}.
     */
    export interface RequestOptions {
        /**
         * Space separated list of client identifiers
         */
        execute: string;
        /**
         * Space separated list of client identifiers.
         */
        render: string;
        /**
         * Function to callback for event.
         */
        onevent: OnEventCallback;
        /**
         * Function to callback for error.
         */
        onerror: OnErrorCallback;
        /**
         * An object containing parameters to include in the request.
         */
        params: Record<string, unknown>;
        /**
         * If less than delay milliseconds elapses between calls to request() only the most recent one is sent and all
         * other requests are discarded. If the value of delay is the literal string 'none' without the quotes, or no
         * delay is specified, no delay is used.
         */
        delay: number;
        /**
         * If true, ensure a post data argument with the name `javax.faces.partial.resetValues` and the value true is
         * sent in addition to the other post data arguments. This will cause `UIViewRoot.resetValues()` to be called,
         * passing the value of the "render" attribute. Note: do not use any of the `@` keywords such as `@form` or
         * `@this` with this option because `UIViewRoot.resetValues()` does not descend into the children of the listed
         * components.
         */
        resetValues: boolean;
    }
    /**
     * Register a callback for error handling.
     *
     * __Usage:__
     *
     * ```javascript
     * jsf.ajax.addOnError(handleError);
     * // ...
     * var handleError = function handleError(data) {
     *   //...
     * }
     * ```
     *
     * __Implementation Requirements:__
     *
     * This function must accept a reference to an existing JavaScript function. The JavaScript function reference must
     * be added to a list of callbacks, making it possible to register more than one callback by invoking
     * {@link jsf.ajax.addOnError} more than once. This function must throw an error if the callback argument is not a
     * function.
     * @param callback A reference to a function to call on an error.
     */
    export function addOnError(callback: OnErrorCallback): void;
    /**
     * Register a callback for event handling.
     *
     * __Usage:__
     *
     * ```javascript
     * jsf.ajax.addOnEvent(statusUpdate);
     *   // ...
     * var statusUpdate = function statusUpdate(data) {
     *   // ...
     * }
     * ```
     *
     * __Implementation Requirements:__
     *
     * This function must accept a reference to an existing JavaScript function. The JavaScript function reference must
     * be added to a list of callbacks, making it possible to register more than one callback by invoking
     * {@link jsf.ajax.addOnEvent} more than once. This function must throw an error if the callback argument is not a
     * function.
     * @param callback A reference to a function to call on an event.
     */
    export function addOnEvent(callback: OnEventCallback): void;
    /**
     * Send an asynchronous Ajax request to the server.
     *
     * __Usage:__
     *
     * Example showing all optional arguments:
     *
     * ```xml
     * <commandButton id="button1" value="submit"
     *     onclick="jsf.ajax.request(this,event,
     *       {execute:'button1',render:'status',onevent: handleEvent,onerror: handleError});return false;"/>
     * </commandButton/>
     * ```
     *
     * __Implementation Requirements:__
     *
     * This function must:
     * - Be used within the context of a form.
     * - Capture the element that triggered this Ajax request (from the source argument, also known as the source
     *   element.
     * - If the source element is `null` or `undefined` throw an error.
     * - If the source argument is not a string or DOM element object, throw an error.
     * - If the source argument is a string, find the DOM element for that string identifier.
     * - If the DOM element could not be determined, throw an error.
     * - If the `onerror` and `onevent` arguments are set, they must be functions, or throw an error.
     * - Determine the source element's form element.
     * - Get the form view state by calling {@link jsf.getViewState} passing the form element as the argument.
     * - Collect post data arguments for the Ajax request.
     *     - The following name/value pairs are required post data arguments:
     *         - `javax.faces.ViewState` - Contents of `javax.faces.ViewState` hidden field. This is included when
     *           {@link jsf.getViewState} is used.
     *         - `javax.faces.partial.ajax` - `true`
     *         - `javax.faces.source` - The identifier of the element that triggered this request.
     *         - `javax.faces.ClientWindow` - Call {@link jsf.getClientWindow}, passing the current form. If the return
     *           is non-null, it must be set as the value of this name/value pair, otherwise, a name/value pair for
     *           client window must not be sent.
     * - Collect optional post data arguments for the Ajax request.
     *     - Determine additional arguments (if any) from the `options` argument. If `options.execute` exists:
     *         - If the keyword `@none` is present, do not create and send the post data argument
     *           `javax.faces.partial.execute`.
     *         - If the keyword `@all` is present, create the post data argument with the name
     *           `javax.faces.partial.execute` and the value `@all`.
     *         - Otherwise, there are specific identifiers that need to be sent. Create the post data argument with the
     *           name `javax.faces.partial.execute` and the value as a space delimited string of client identifiers.
     *     - If `options.execute` does not exist, create the post data argument with the name
     *       `javax.faces.partial.execute` and the value as the identifier of the element that caused this request.
     *     - If `options.render` exists:
     *         - If the keyword `@none` is present, do not create and send the post data argument
     *           `javax.faces.partial.render`.
     *         - If the keyword `@all` is present, create the post data argument with the name
     *           `javax.faces.partial.render` and the value `@all`.
     *         - Otherwise, there are specific identifiers that need to be sent. Create the post data argument with the
     *           name `javax.faces.partial.render` and the value as a space delimited string of client identifiers.
     *     - If `options.render` does not exist do not create and send the post data argument
     *       `javax.faces.partial.render`.
     *     - If `options.delay` exists let it be the value delay, for this discussion. If options.delay does not exist,
     *       or is the literal string 'none', without the quotes, no delay is used. If less than delay milliseconds
     *       elapses between calls to `request()` only the most recent one is sent and all other requests are discarded.
     *     - If `options.resetValues` exists and its value is true, ensure a post data argument with the name
     *       `javax.faces.partial.resetValues` and the value true is sent in addition to the other post data arguments.
     *       This will cause `UIViewRoot.resetValues()` to be called, passing the value of the `render` attribute. Note:
     *       do not use any of the `@` keywords such as `@form` or `@this` with this option because
     *       `UIViewRoot.resetValues()` does not descend into the children of the listed components.
     *     - Determine additional arguments (if any) from the event argument. The following name/value pairs may be used
     *       from the event object:
     *         - `target` - the `ID` of the element that triggered the event.
     *         - `captured` - the `ID` of the element that captured the event.
     *         - `type` - the type of event (ex: `onkeypress`)
     *         - `alt` - true if `ALT` key was pressed.
     *         - `ctrl` - true if `CTRL` key was pressed.
     *         - `shift` - true if `SHIFT` key was pressed.
     *         - `meta` - true if `META` key was pressed.
     *         - `right` - true if right mouse button was pressed.
     *         - `left` - true if left mouse button was pressed.
     *         - `keycode` - the key code.
     * - Encode the set of post data arguments.
     * - Join the encoded view state with the encoded set of post data arguments to form the query string that will be
     *   sent to the server.
     * - Create a request context object that will be used during error/event handling. The function must set the
     *   properties:
     *     - `source` (the source DOM element for this request)
     *     - `onerror` (the error handler for this request)
     *     - `onevent` (the event handler for this request)
     * - Send a begin event following the procedure as outlined in the Chapter 13 `Sending Events` section of the Java
     *   Server Faces spec.
     * - Set the request header with the name: `Faces-Request` and the value: `partial/ajax`.
     * - Determine the `posting URL` as follows: If the hidden field `javax.faces.encodedURL` is present in the
     *   submitting form, use its value as the posting URL. Otherwise, use the action property of the form element as
     *   the URL.
     * - Determine whether or not the submitting form is using `multipart/form-data` as its enctype attribute. If not,
     *   send the request as an asynchronous POST using the posting URL that was determined in the previous step.
     *   Otherwise, send the request using a multi-part capable transport layer, such as a hidden inline frame. Note
     *   that using a hidden inline frame does not use `XMLHttpRequest`, but the request must be sent with all the
     *   parameters that a JSF `XMLHttpRequest` would have been sent with. In this way, the server side processing of
     *   the request will be identical whether or the request is multipart or not.
     *
     *   The begin, complete, and success events must be emulated when using the multipart transport. This allows any
     * listeners to behave uniformly regardless of the multipart or XMLHttpRequest nature of the transport.
     *
     * Form serialization should occur just before the request is sent to minimize the amount of time between the
     * creation of the serialized form data and the sending of the serialized form data (in the case of long requests in
     * the queue). Before the request is sent it must be put into a queue to ensure requests are sent in the same order
     * as when they were initiated. The request callback function must examine the queue and determine the next request
     * to be sent. The behavior of the request callback function must be as follows:
     *
     * - If the request completed successfully invoke {@link jsf.ajax.response} passing the request object.
     * - If the request did not complete successfully, notify the client.
     * - Regardless of the outcome of the request (success or error) every request in the queue must be handled. Examine
     *   the status of each request in the queue starting from the request that has been in the queue the longest. If
     *   the status of the request is complete (readyState 4), dequeue the request (remove it from the queue). If the
     *   request has not been sent (readyState 0), send the request. Requests that are taken off the queue and sent
     *   should not be put back on the queue.
     * @param source The DOM element that triggered this AJAX request, or an ID string of the element to use as the
     * triggering element.
     * @param event The DOM event that triggered this Ajax request. The event argument is optional.
     * @param options The set of available options that can be sent as request parameters to control client and/or
     * server side request processing.
     * @throws Error if first required argument element is not specified, or if one or more of the components in the
     * `options.execute` list is a file upload component, but the form's enctype is not set to `multipart/form-data`.
     */
    export function request(source: HTMLElement | string, event?: Event, options?: Partial<RequestOptions>): void;
    /**
     * Receive an Ajax response from the server.
     *
     * __Usage:__
     *
     * ```javascript
     * jsf.ajax.response(request, context);
     * ```
     *
     * __Implementation Requirements:__
     *
     * This function must evaluate the markup returned in the `request.responseXML` object and perform the following
     * action:
     *
     * - If there is no XML response returned, signal an emptyResponse error. If the XML response does not follow the
     *   format as outlined in Appendix A of the Java Server Faces spec prose document signal a `malformedError` error.
     *   Refer to section `Signaling Errors` in Chapter 13 of the Java Server Faces spec prose document.
     * - If the response was successfully processed, send a success event as outlined in Chapter 13 `Sending Events`
     *   section of the Java Server Faces spec prose document.
     *
     * _Update Element Processing_
     *
     * The `update` element is used to update a single DOM element. The `id` attribute of the update element refers to
     * the DOM element that will be updated. The contents of the `CDATA` section is the data that will be used when
     * updating the contents of the DOM element as specified by the `<update>` element identifier.
     *
     * - If an `<update>` element is found in the response with the identifier `javax.faces.ViewRoot`:
     *
     *   ```xml
     *   <update id="javax.faces.ViewRoot">
     *       <![CDATA[...]]>
     *   </update>
     *   ```
     *
     *   Update the entire DOM replacing the appropriate `head` and/or `body` sections with the content from the
     *   response.
     *
     * - If an `<update>` element is found in the response with an identifier containing `javax.faces.ViewState`:
     *
     *   ```xml
     *   <update id="<VIEW_ROOT_CONTAINER_CLIENT_ID><SEP>javax.faces.ViewState<SEP><UNIQUE_PER_VIEW_NUMBER>">
     *       <![CDATA[...]]>
     *   </update>
     *   ```
     *
     *   locate and update the submitting form's `javax.faces.ViewState` value with the `CDATA` contents from the
     *   response. `<SEP>`: is the currently configured `UINamingContainer.getSeparatorChar()`.
     *   `<VIEW_ROOT_CONTAINER_CLIENT_ID>` is the return from `UIViewRoot.getContainerClientId()` on the view from
     *   whence this state originated. `<UNIQUE_PER_VIEW_NUMBER>` is a number that must be unique within this view, but
     *   must not be included in the view state. This requirement is simply to satisfy XML correctness in parity with
     *   what is done in the corresponding non-partial JSF view. Locate and update the `javax.faces.ViewState` value for
     *   all forms specified in the render target list.
     *
     * - If an update element is found in the response with an identifier containing `javax.faces.ClientWindow`:
     *
     *   ```xml
     *   <update id="<VIEW_ROOT_CONTAINER_CLIENT_ID><SEP>javax.faces.ClientWindow<SEP><UNIQUE_PER_VIEW_NUMBER>">
     *       <![CDATA[...]]>
     *   </update>
     *   ```
     *
     *   locate and update the submitting form's `javax.faces.ClientWindow` value with the `CDATA` contents from the
     *   response. `<SEP>`: is the currently configured `UINamingContainer.getSeparatorChar()`.
     *   `<VIEW_ROOT_CONTAINER_CLIENT_ID>` is the return from `UIViewRoot.getContainerClientId()` on the view from
     *   whence this state originated. `<UNIQUE_PER_VIEW_NUMBER>` is a number that must be unique within this view, but
     *   must not be included in the view state. This requirement is simply to satisfy XML correctness in parity with
     *   what is done in the corresponding non-partial JSF view. Locate and update the `javax.faces.ClientWindow` value
     *   for all forms specified in the render target list.
     *
     * - If an update element is found in the response with the identifier `javax.faces.ViewHead`:
     *
     *   ```xml
     *   <update id="javax.faces.ViewHead">
     *       <![CDATA[...]]>
     *   </update>
     *   ```
     *
     *   update the document's `head` section with the `CDATA` contents from the response.
     *
     * - If an update element is found in the response with the identifier `javax.faces.ViewBody`:
     *
     *   ```xml
     *   <update id="javax.faces.ViewBody">
     *       <![CDATA[...]]>
     *   </update>
     *   ```
     *
     *   update the document's `body` section with the `CDATA` contents from the response.
     *
     * - For any other `<update>` element:
     *
     *   ```xml
     *   <update id="update id">
     *       <![CDATA[...]]>
     *   </update>
     *   ```
     *
     *   Find the DOM element with the identifier that matches the `<update>` element identifier, and replace its
     *   contents with the `<update>` element's `CDATA` contents.
     *
     * _Insert Element Processing_
     *
     * - If an `<insert>` element is found in the response with a nested `<before>` element:
     *
     *   ```xml
     *   <insert>
     *       <before id="before id">
     *           <![CDATA[...]]>
     *       </before>
     *   </insert>
     *   ```
     *
     *     - Extract this `<before>` element's `CDATA` contents from the response.
     *     - Find the DOM element whose identifier matches before id and insert the `<before>` element's `CDATA` content
     *       before the DOM element in the document.
     *
     * - If an <insert> element is found in the response with a nested <after> element:
     *
     *   ```xml
     *   <insert>
     *       <after id="after id">
     *           <![CDATA[...]]>
     *       </after>
     *   </insert>
     *   ```
     *
     *     - Extract this `<after>` element's `CDATA` contents from the response.
     *     - Find the DOM element whose identifier matches after id and insert the `<after>` element's `CDATA` content
     *       after the DOM element in the document.
     *
     * _Delete Element Processing_
     *
     * - If a `<delete>` element is found in the response:
     *
     *   ```xml
     *   <delete id="delete id"/>
     *   ```
     *
     *   Find the DOM element whose identifier matches `delete id` and remove it from the DOM.
     *
     * _Element Attribute Update Processing_
     *
     * - If an `<attributes>` element is found in the response:
     *
     *   ```xml
     *   <attributes id="id of element with attribute">
     *       <attribute name="attribute name" value="attribute value">
     *        <!-- ... -->
     *   </attributes>
     *   ```
     *
     *     - Find the DOM element that matches the `<attributes>` identifier.
     *     - For each nested `<attribute>` element in `<attribute>`, update the DOM element attribute value (whose name
     *       matches attribute name), with attribute value.
     *
     * _JavaScript Processing_
     *
     * - If an `<eval>` element is found in the response:
     *
     *   ```xml
     *   <eval>
     *       <![CDATA[...JavaScript...]]>
     *   </eval>
     *   ```
     *
     *     - Extract this `<eval>` element's `CDATA` contents from the response and execute it as if it were JavaScript
     *       code.
     *
     * _Redirect Processing_
     *
     * - If a `<redirect>` element is found in the response:
     *
     *   ```xml
     *   <redirect url="redirect url"/>
     *   ```
     *
     *   Cause a redirect to the `url redirect url`.
     *
     * _Error Processing_
     *
     * - If an `<error>` element is found in the response:
     *
     *   ```xml
     *   <error>
     *       <error-name>..fully qualified class name string...<error-name>
     *       <error-message><![CDATA[...]]><error-message>
     *   </error>
     *   ```
     *
     *   Extract this `<error>` element's error-name contents and the error-message contents. Signal a `serverError`
     *   passing the `errorName` and `errorMessage`. Refer to section `Signaling Errors` in Chapter 13 of the Java
     *   Server Faces spec prose document.
     *
     * _Extensions_
     *
     * - The `<extensions>` element provides a way for framework implementations to provide their own information.
     * - The implementation must check if `<script>` elements in the response can be automatically run, as some browsers
     *   support this feature and some do not. If they can not be run, then scripts should be extracted from the
     *   response and run separately.
     * @param request The `XMLHttpRequest` instance that contains the status code and response message from the server.
     * @param context An object containing the request context, including the following properties: the source element,
     * per call onerror callback function, and per call onevent callback function.
     * @throws Error if request contains no data.
     */
    export function response(request: XMLHttpRequest, context: RequestContext): void;
}
/**
 * The namespace for JavaServer Faces JavaScript utilities.
 *
 * Please note that this functionality may only be available when you use certain tags such as `<f:ajax .../>` tags.
 */
declare namespace jsf.util {
    /**
     * A varargs function that invokes an arbitrary number of scripts. If any script in the chain returns false, the
     * chain is short-circuited and subsequent scripts are not invoked. Any number of scripts may specified after the
     * `event` argument.
     * @param source The DOM element that triggered this Ajax request, or an id string of the element to use as the
     * triggering element.
     * @param event The DOM event that triggered this Ajax request. The `event` argument is optional.
     * @param scripts List of scripts to execute in a function scope. Receives the source as the this context and one
     * parameter `event`, set to the value passed to this function.
     * @return `false` if any scripts in the chain return `false`, otherwise returns `true`.
     */
    export function chain(source: HTMLElement | string, event?: Event | undefined, ...scripts: string[]): boolean;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces AccordionPanel Widget__
     *
     * The AccordionPanel is a container component that displays content in a stacked format.
     * @typeparam TCfg Defaults to `AccordionPanelCfg`. Type of the configuration object for this widget.
     */
    export class AccordionPanel<TCfg extends AccordionPanelCfg = AccordionPanelCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM elements for the header of each tab.
         */
        headers: JQuery;
        /**
         * The DOM elements for the content of each tab panel.
         */
        panels: JQuery;
        /**
         * The DOM elements for the hidden input storing which panels are expanded and collapsed.
         */
        stateHolder: JQuery;
        /**
         * Adds the given panel node to the list of currently selected nodes.
         * @param nodeId ID of a panel node.
         */
        private addToSelection(nodeId: string): void;
        /**
         * Binds all event listeners required by this accordion panel.
         */
        private bindEvents(): void;
        /**
         * Sets up all event listeners for keyboard interactions.
         */
        private bindKeyEvents(): void;
        /**
         * Handles the event listeners and behaviors when switching to a different tab.
         * @param panel The tab which is now active.
         */
        private fireTabChangeEvent(panel: JQueryStatic): void;
        /**
         * Handles the event listeners and behaviors when a tab was closed.
         * @param index 0-based index of the closed tab.
         */
        private fireTabCloseEvent(index: number): void;
        /**
         * Hides one of the panels of this accordion.
         * @param index 0-based index of the panel to hide.
         */
        private hide(index: number): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Called when this accordion panel is initialized. Reads the selected panels from the saved state, see also
         * `saveState`.
         */
        private initActive(): void;
        /**
         * The content of a tab panel may be loaded dynamically on demand via AJAX. This method checks whether the content
         * of a tab panel is currently loaded.
         * @param panel A tab panel to check.
         * @return `true` if the content of the tab panel is loaded, `false` otherwise.
         */
        isLoaded(panel: JQuery): boolean;
        /**
         * The content of a tab panel may be loaded dynamically on demand via AJAX. This method loads the content of the
         * given tab. Make sure to check first that this widget has got a dynamic tab panel (see
         * {@link AccordionPanelCfg.dynamic}) and that the given tab panel is not loaded already (see {@link isLoaded}).
         * @param panel A tab panel to load.
         */
        loadDynamicTab(panel: JQuery): void;
        /**
         * When loading tab content dynamically, marks the content as loaded.
         * @param panel A panel of this accordion that was loaded.
         */
        private markAsLoaded(panel: JQuery): void;
        /**
         * Marks the currently active panels as loaded; their content does not need to be retrieved from the server anymore.
         */
        private markLoadedPanels(): void;
        /**
         * Handles event listeners and behaviors when switching to a different tab.
         * @param newPanel The new tab the is shown.
         */
        private postTabShow(newPanel: JQuery): void;
        /**
         * Removes the given panel node from the list of currently selected nodes.
         * @param nodeId ID of a panel node.
         */
        private removeFromSelection(nodeId: string): void;
        /**
         * Saves the current state of this widget, used for example to preserve the state during AJAX updates.
         */
        private saveState(): void;
        /**
         * Activates (opens) the tab with given index. This may fail by returning `false`, such
         * as when a callback is registered that prevent the tab from being opened.
         * @param index 0-based index of the tab to open. Must not be out of range.
         * @return `true` when the given panel is now active, `false` otherwise.
         */
        select(index: number): boolean;
        /**
         * Activates (opens) all the tabs if multiple mode is enabled and the first tab in single mode.
         */
        selectAll(): void;
        /**
         * Hides other panels and makes the given panel visible, such as by adding or removing the appropriate CSS classes.
         * @param panel A tab panel to show.
         */
        private show(panel: JQuery): void;
        /**
         * Deactivates (closes) the tab with given index.
         * @param index 0-based index of the tab to close. Must not be out of range.
         */
        unselect(index: number): void;
        /**
         * Deactivates (closes) all the tabs.
         */
        unselectAll(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the
     * {@link  AccordionPanel| AccordionPanel widget}. You can access this configuration via
     * {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this configuration is usually meant to be
     * read-only and should not be modified.
     */
    export interface AccordionPanelCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * List of tabs that are currently active (open). Each item is a 0-based index of a tab.
         */
        active: number[];
        /**
         * `true` if activating a dynamic tab should not load the contents from server again and use
         * the cached contents; or `false` if the caching is disabled.
         */
        cache: boolean;
        /**
         * The icon class name for the collapsed icon.
         */
        collapsedIcon: string;
        /**
         * `true` if a tab controller was specified for this widget; or `false` otherwise. A tab
         * controller is a server side listener that decides whether a tab change or tab close should be allowed.
         */
        controlled: boolean;
        /**
         * `true` if the contents of each panel are loaded on-demand via AJAX; `false` otherwise.
         */
        dynamic: boolean;
        /**
         * The icon class name for the expanded icon.
         */
        expandedIcon: string;
        /**
         * Whether to keep AccordionPanel state across views.
         */
        multiViewState: boolean;
        /**
         * `true` if multiple tabs may be open at the same time; or `false` if opening one tab
         * closes all other tabs.
         */
        multiple: boolean;
        /**
         * `true` if the current text direction `rtl` (right-to-left); or `false` otherwise.
         */
        rtl: boolean;
        /**
         * Should the tab scroll into view. One of start, center, end, nearest, or NULL if disabled.
         */
        scrollIntoView: number;
        /**
         * Speed of toggling in milliseconds.
         */
        toggleSpeed: number;
    }
}
declare namespace PrimeFaces.widget.AjaxStatus {
    /**
     * Available
     * types of AJAX related events to which you can listen.
     */
    export type AjaxStatusEventType = "start" | "success" | "error" | "complete";
}
declare namespace PrimeFaces.widget.AjaxStatus {
    /**
     * Callback for when an AJAX request completes, either
     * successfully or with an error. Usually set via `<p:ajaxStatus oncomplete="..."/>`. This callback applies when
     * `<p:ajax />` is used.
     */
    export type PfAjaxCompleteCallback =
    /**
     * @param xhr The request that succeeded.
     * @param settings The settings of the jQuery
     * AJAX request.
     */
    (this: Document, xhr: JQuery.jqXHR, settings: JQuery.AjaxSettings) => void;
}
declare namespace PrimeFaces.widget.AjaxStatus {
    /**
     * Callback for when an AJAX request fails. Usually set via
     * `<p:ajaxStatus onerror="..."/>`. This callback applies when `<p:ajax />` is used.
     */
    export type PfAjaxErrorCallback =
    /**
     * @param xhr The request that failed.
     * @param settings The settings of the jQuery
     * AJAX request.
     * @param errorThrown The error that cause the request to
     * fail.
     */
    (this: Document, xhr: JQuery.jqXHR, settings: JQuery.AjaxSettings, errorThrown: string) => void;
}
declare namespace PrimeFaces.widget.AjaxStatus {
    /**
     * Callback for when an AJAX request starts. Usually set via
     * `<p:ajaxStatus onstart="..."/>`. This callback applies when `<p:ajax />` is used.
     */
    export type PfAjaxStartCallback = (this: Document) => void;
}
declare namespace PrimeFaces.widget.AjaxStatus {
    /**
     * Callback for when an AJAX request succeeds. Usually set
     * via `<p:ajaxStatus onsuccess="..."/>`. This callback applies when `<p:ajax />` is used.
     */
    export type PfAjaxSuccessCallback =
    /**
     * @param xhr The request that succeeded.
     * @param settings The settings of the jQuery
     * AJAX request.
     */
    (this: Document, xhr: JQuery.jqXHR, settings: JQuery.AjaxSettings) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces AjaxStatus Widget__
     *
     * AjaxStatus is a global notifier for AJAX requests.
     *
     * For the callbacks that can be set via the `onstart`, `onsuccess`, `onerror` and `oncomplete` attributes, see
     * {@link PfAjaxStartCallback}, {@link PfAjaxSuccessCallback}, {@link PfAjaxErrorCallback}, and
     * {@link PfAjaxCompleteCallback}.
     * @typeparam TCfg Defaults to `AjaxStatusCfg`. Type of the configuration object for this widget.
     */
    export class AjaxStatus<TCfg extends AjaxStatusCfg = AjaxStatusCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The set-timeout timer ID for the timer of the delay before the AJAX status is
         * triggered.
         */
        timeout: number | null;
        /**
         * Listen to the relevant events on the document element.
         */
        private bind(): void;
        /**
         * Clears the ste-timeout timer for the delay.
         */
        private deleteTimeout(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Finds the facet ID of the given event.
         * @param event One of the supported event
         * @return The ID of the facet element for the given event
         */
        private toFacetId(event: PrimeFaces.widget.AjaxStatus.AjaxStatusEventType): string;
        /**
         * Triggers the given event by invoking the event handler, usually defined on the `<p:ajaxStatus/>` tag.
         * @typeparam K A name of one of the supported events that should
         * be triggered.
         * @param event A name of one of the supported events that should
         * be triggered.
         * @param args Arguments that are passed to the
         * event handler.
         */
        trigger<K extends PrimeFaces.widget.AjaxStatus.AjaxStatusEventType>(event: K, args: Parameters<PrimeFaces.widget.AjaxStatus.EventToCallbackMap[K]>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  AjaxStatus| AjaxStatus widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface AjaxStatusCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Client-side callback for when the AJAX behavior completes, i.e. when the request finishes, irrespective of whether it
         * succeeded or failed.
         */
        complete: PrimeFaces.widget.AjaxStatus.PfAjaxCompleteCallback | jsf.ajax.OnEventCallback | jsf.ajax.OnErrorCallback;
        /**
         * Delay in milliseconds before displaying the AJAX status. Default is `0`, meaning immediate.
         */
        delay: number;
        /**
         * Client-side callback
         * for when the AJAX behavior fails, i.e. when the request fails.
         */
        error: PrimeFaces.widget.AjaxStatus.PfAjaxErrorCallback | jsf.ajax.OnErrorCallback;
        /**
         * Client-side callback
         * for when the AJAX behavior starts, i.e. the request is about to be sent.
         */
        start: PrimeFaces.widget.AjaxStatus.PfAjaxStartCallback | jsf.ajax.OnEventCallback;
        /**
         * Client-side
         * callback for when the AJAX  behavior completes successfully, i.e. when the request succeeds.
         */
        success: PrimeFaces.widget.AjaxStatus.PfAjaxSuccessCallback | jsf.ajax.OnEventCallback;
    }
}
declare namespace PrimeFaces.widget.AjaxStatus {
    /**
     * Maps between the
     * {@link AjaxStatusEventType} and the corresponding event handlers. Used by the {@link AjaxStatus} component.
     */
    export interface EventToCallbackMap {
        /**
         * Callback for when an AJAX request completes, either successfully or with an error. Usually set via
         * `<p:ajaxStatus oncomplete="..."/>`.
         */
        complete: PrimeFaces.widget.AjaxStatus.PfAjaxCompleteCallback | jsf.ajax.OnEventCallback | jsf.ajax.OnErrorCallback;
        /**
         * Callback
         * for when an AJAX request fails. Usually set via `<p:ajaxStatus onerror="..."/>`.
         */
        error: PrimeFaces.widget.AjaxStatus.PfAjaxErrorCallback | jsf.ajax.OnErrorCallback;
        /**
         * Callback
         * for when an AJAX request starts. Usually set via `<p:ajaxStatus onstart="..."/>`.
         */
        start: PrimeFaces.widget.AjaxStatus.PfAjaxStartCallback | jsf.ajax.OnEventCallback;
        /**
         * Callback for when an AJAX request succeeds. Usually set via `<p:ajaxStatus onsuccess="..."/>`.
         */
        success: PrimeFaces.widget.AjaxStatus.PfAjaxSuccessCallback | jsf.ajax.OnEventCallback;
    }
}
declare namespace PrimeFaces.widget.AutoComplete {
    /**
     * Specifies the behavior of the dropdown
     * button.
     * - `blank`: Sends an empty string.
     * - `current`: Send the input value.
     */
    export type DropdownMode = "blank" | "current";
}
declare namespace PrimeFaces.widget.AutoComplete {
    /**
     * Client side callback to invoke when value changes.
     */
    export type OnChangeCallback =
    /**
     * @param input (Input) element on which the change occurred.
     */
    (input: JQuery) => void;
}
declare namespace PrimeFaces.widget.AutoComplete {
    /**
     * Event to initiate the autocomplete search.
     * - `enter`: Starts the search for suggestion items when the enter key is pressed.
     * - `keyup`: Starts the search for suggestion items as soon as a key is released.
     */
    export type QueryEvent = "keyup" | "enter";
}
declare namespace PrimeFaces.widget.AutoComplete {
    /**
     * Specifies whether filter requests
     * are evaluated by the client's browser or whether they are sent to the server.
     */
    export type QueryMode = "server" | "client" | "hybrid";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces AutoComplete Widget__
     *
     * AutoComplete provides live suggestions while the user is entering text
     * @typeparam TCfg Defaults to `AutoCompleteCfg`. Type of the configuration object for this widget.
     */
    export class AutoComplete<TCfg extends AutoCompleteCfg = AutoCompleteCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Whether the autocomplete is active.
         */
        active: boolean;
        /**
         * The cache for the results of an autocomplete search.
         */
        cache?: Record<string, string>;
        /**
         * The set-interval timer ID for the cache timeout.
         */
        cacheTimeout?: number;
        /**
         * Whether the click event is fired on the selected items when a `blur` occurs.
         */
        checkMatchedItem?: boolean;
        /**
         * Column span count for the options in the overlay panel with the available completion items.
         */
        colspan?: number;
        /**
         * Current option group when creating the options in the overlay  with the available
         * completion items.
         */
        currentGroup?: string;
        /**
         * Current value in the input field where the user can search for completion items.
         */
        currentInputValue: string;
        /**
         * Currently selected items, when `forceSelection` is enabled.
         */
        currentItems?: string[];
        /**
         * Text currently entered in the input field.
         */
        currentText?: string;
        /**
         * The DOM element for the container with the dropdown suggestions.
         */
        dropdown: JQuery;
        /**
         * Text to display when there is no data to display.
         */
        emptyMessage: string;
        /**
         * Unbind callback for the hide overlay handler.
         */
        hideOverlayHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM element for the hidden input with the selected value.
         */
        hinput: JQuery;
        /**
         * The DOM element for the input element.
         */
        input: JQuery;
        /**
         * When multiple mode is enabled that allows multiple items to be selected: The DOM
         * element of the container with the input element used to enter text and search for an item to select.
         */
        inputContainer: JQuery;
        /**
         * If dynamic loading is enabled, whether the content was loaded already.
         */
        isDynamicLoaded: boolean;
        /**
         * Whether to use a drop down menu when searching for completion options.
         */
        isSearchWithDropdown?: boolean;
        /**
         * Whether the tab key is currently pressed.
         */
        isTabPressed: boolean;
        /**
         * Whether an item was clicked.
         */
        itemClick?: boolean;
        /**
         * The DOM element for the container with the suggestion items.
         */
        itemContainer?: JQuery;
        /**
         * Whether an item was selected via the enter key.
         */
        itemSelectedWithEnter?: boolean;
        /**
         * The DOM elements for the suggestion items.
         */
        items?: JQuery;
        /**
         * The DOM element for the tooltip of a suggestion item.
         */
        itemtip: JQuery;
        /**
         * The DOM element for the container with multiple selection items.
         */
        multiItemContainer?: JQuery;
        /**
         * The DOM element for the overlay panel with the suggestion items.
         */
        panel: JQuery;
        /**
         * The client ID of the overlay panel with the suggestion items.
         */
        panelId: string;
        /**
         * Placeholder shown in the input field when no text is entered.
         */
        placeholder: string;
        /**
         * Whether to suppress the change event when the input's value changes.
         */
        preventInputChangeEvent?: boolean;
        /**
         * Text previously entered in the input field.
         */
        previousText?: string;
        /**
         * Whether an AJAX request for the autocompletion items is currently in progress.
         */
        querying?: boolean;
        /**
         * Tracking number to make sure search requests match up in query mode
         */
        requestId: number;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Hint text for screen readers to provide information about the search results.
         */
        resultsMessage: string;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM element for the autocomplete status ARIA element.
         */
        status: JQuery;
        /**
         * Whether key input events should be ignored currently.
         */
        suppressInput: boolean;
        /**
         * Timeout ID for the timer used to clear the autocompletion cache in regular
         * intervals.
         */
        timeout?: number;
        /**
         * Whether a touch is made on the dropdown button.
         */
        touchToDropdownButton: boolean;
        /**
         * Handler for CSS transitions used by this widget.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * The finishing HTML with the wrapper element of the suggestions box.
         */
        wrapperEndTag: string;
        /**
         * The starting HTML with the wrapper element of the suggestions box.
         */
        wrapperStartTag: string;
        /**
         * Activates search behavior.
         */
        activate(): void;
        /**
         * Add the given suggestion item.
         * @param item Suggestion item to add.
         */
        addItem(item: JQuery | string): void;
        /**
         * Aligns (positions) the overlay panel that shows the found suggestions.
         */
        alignPanel(): void;
        /**
         * Appends the overlay panel to the DOM.
         */
        private appendPanel(): void;
        /**
         * Sets up all event listeners for the dropdown menu.
         */
        private bindDropdownEvents(): void;
        /**
         * Sets up all event listeners for mouse and click events.
         */
        private bindDynamicEvents(): void;
        /**
         * Sets up all keyboard related event listeners.
         */
        private bindKeyEvents(): void;
        /**
         * Sets up all panel event listeners
         */
        private bindPanelEvents(): void;
        /**
         * Sets up all global event listeners for the overlay.
         */
        private bindStaticEvents(): void;
        /**
         * Adjusts the value of the aria attributes for the given selectable option.
         * @param item An option for which to set the aria attributes.
         */
        private changeAriaValue(item: Element): void;
        /**
         * Clears the input field.
         */
        clear(): void;
        /**
         * Clears the cache with the results of an autocomplete search.
         */
        private clearCache(): void;
        /**
         * Hides suggested items menu.
         */
        close(): void;
        /**
         * Localizes the ARIA accessibility labels for the autocomplete.
         */
        private configureLocale(): void;
        /**
         * Deactivates search behavior.
         */
        deactivate(): void;
        /**
         * Clears the set-timeout timer for the autocomplete search.
         */
        private deleteTimeout(): void;
        /**
         * Disables the component.
         */
        disable(): void;
        /**
         * Disables the dropdown menu.
         */
        private disableDropdown(): void;
        /**
         * Adds the given text to the ARIA status label element.
         * @param text Label text to display.
         */
        private displayAriaStatus(text: string): void;
        /**
         * Enables the component.
         */
        enable(): void;
        /**
         * Enables the dropdown menu.
         */
        private enableDropdown(): void;
        /**
         * Fetches the suggestion items for the current query from the server.
         */
        private fetchItems(): void;
        /**
         * Finds and sets the wrapper HTML snippets on this instance.
         * @param wrapper Wrapper element with the suggestions fetched from the server.
         */
        private findWrapperTag(wrapper: JQuery): void;
        /**
         * Triggers the behavior for when the input was cleared.
         */
        private fireClearEvent(): void;
        /**
         * Creates the grouped suggestion item for the given parameters.
         * @param group A group where to look for the item.
         * @param container Container element of the group.
         * @param tooltip Optional tooltip for the group item.
         * @return The newly created group item.
         */
        private getGroupItem(group: string, container: JQuery, tooltip: string): JQuery;
        /**
         * Takes the available suggestion items and groups them.
         */
        private groupItems(): void;
        /**
         * Fired when the browser viewport is resized or scrolled.  In Mobile environment we don't want to hider the overlay
         * we want to re-align it.  This is because on some mobile browser the popup may force the browser to trigger a
         * resize immediately and close the overlay. See GitHub #7075.
         */
        private handleViewportChange(): void;
        /**
         * Hides the panel with the suggestions.
         */
        private hide(): void;
        /**
         * Adjusts the highlighting and aria attributes for the given selectable option.
         * @param item An option for which to set the aria attributes.
         * @param highlight Flag to indicate to highlight or not
         */
        private highlightItem(item: Element, highlight: boolean): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Initializes the cache that stores the retrieved suggestions for a search term.
         */
        private initCache(): void;
        /**
         * Invokes the appropriate behavior for when empty message was selected.
         */
        private invokeEmptyMessageBehavior(): void;
        /**
         * Invokes the appropriate behavior for when a suggestion item was selected.
         * @param itemValue Value of the selected item.
         */
        private invokeItemSelectBehavior(itemValue: string): void;
        /**
         * Invokes the appropriate behavior when a suggestion item was unselected.
         * @param itemValue Value of the unselected item.
         */
        private invokeItemUnselectBehavior(itemValue: string): void;
        /**
         * Invokes the appropriate behavior for when more text was selected.
         */
        private invokeMoreTextBehavior(): void;
        /**
         * Checks whether the given value is part of the available suggestion items.
         * @param value A value to check.
         * @param shouldFireClearEvent `true` if clear event should be fired.
         * @return Whether the given value matches a value in the list of available suggestion items;
         * or `undefined` if {@link AutoCompleteCfg.forceSelection} is set to `false`.
         */
        isValid(value: string, shouldFireClearEvent?: boolean): boolean | undefined;
        /**
         * Callback for when a key event occurred.
         * @param e Key event that occurred.
         */
        private processKeyEvent(e: JQuery.TriggeredEvent): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Removes all items if in multiple mode.
         */
        removeAllItems(): void;
        /**
         * Removes the given suggestion item.
         * @param item Suggestion item to remove.
         */
        removeItem(item: JQuery | string): void;
        /**
         * Initiates a search with given value, that is, look for matching options and present the options that were found
         * to the user.
         * @param query Keyword for the search.
         */
        search(query: string): void;
        /**
         * Performs a search the same ways as if the user had opened the dropdown menu. Depending on the configured
         * `dropdownMode`, performs the search either with an empty string or with the current value.
         */
        searchWithDropdown(): void;
        /**
         * Adds the suggestions items in the given wrapper to the local cache of suggestion items.
         * @param wrapper Wrapper element with the suggestions fetched from the server.
         */
        private setCache(wrapper: JQuery): void;
        /**
         * Sets the querying state.
         * @param state Querying state to set.
         */
        private setQuerying(state: boolean): void;
        /**
         * Sets up the event listener for the blur event to force a selection, when that feature is enabled.
         */
        private setupForceSelection(): void;
        /**
         * Binds events for multiple selection mode.
         */
        private setupMultipleMode(): void;
        /**
         * Shows the panel with the suggestions.
         */
        private show(): void;
        /**
         * Shows the tooltip for the given suggestion item.
         * @param item Item with a tooltip.
         */
        private showItemtip(item: JQuery): void;
        /**
         * Performs the search for the available suggestion items.
         * @param query Keyword for the search.
         */
        private showSuggestions(query: string): void;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  AutoComplete| AutoComplete widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface AutoCompleteCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether autocompletion search is initially active.
         */
        active: boolean;
        /**
         * ID of the container to which the suggestion box is appended.
         */
        appendTo: string;
        /**
         * Defines which position on the target element to align the positioned element against.
         */
        atPos: string;
        /**
         * Highlights the first suggested item automatically.
         */
        autoHighlight: boolean;
        /**
         * Defines if auto selection of items that are equal to the typed input is enabled. If
         * `true`, an item that is equal to the typed input is selected.
         */
        autoSelection: boolean;
        /**
         * When enabled autocomplete caches the searched result list.
         */
        cache: boolean;
        /**
         * Timeout in milliseconds value for cached results.
         */
        cacheTimeout: number;
        /**
         * REST endpoint for fetching autocomplete suggestions. Takes precedence over the
         * bean command specified via `completeMethod` on the component.
         */
        completeEndpoint: string;
        /**
         * The delay in milliseconds before an autocomplete search is triggered.
         */
        delay: number;
        /**
         * Specifies the behavior of the dropdown button.
         */
        dropdownMode: PrimeFaces.widget.AutoComplete.DropdownMode;
        /**
         * Defines if dynamic loading is enabled for the element's panel. If the value is `true`,
         * the overlay is not rendered on page load to improve performance.
         */
        dynamic: boolean;
        /**
         * Whether the text of the suggestion items is escaped for HTML.
         */
        escape: boolean;
        /**
         * Whether one of the available suggestion items is forced to be preselected.
         */
        forceSelection: boolean;
        /**
         * Whether suggestion items are grouped.
         */
        grouping: boolean;
        /**
         * Whether a footer facet is present.
         */
        hasFooter: boolean;
        /**
         * Whether a tooltip is shown for the suggestion items.
         */
        itemtip: boolean;
        /**
         * Position of item corner relative to item tip.
         */
        itemtipAtPosition: string;
        /**
         * Position of itemtip corner relative to item.
         */
        itemtipMyPosition: string;
        /**
         * Minimum length before an autocomplete search is triggered.
         */
        minLength: number;
        /**
         * The text shown in the panel when the number of suggestions is greater than `maxResults`.
         */
        moreText: string;
        /**
         * When `true`, enables multiple selection.
         */
        multiple: boolean;
        /**
         * Defines which position on the element being positioned to align with the target element.
         */
        myPos: string;
        /**
         * Client side callback to invoke when value
         * changes.
         */
        onChange: PrimeFaces.widget.AutoComplete.OnChangeCallback;
        /**
         * Event to initiate the the autocomplete search.
         */
        queryEvent: PrimeFaces.widget.AutoComplete.QueryEvent;
        /**
         * Specifies query mode, whether autocomplete contacts
         * the server.
         */
        queryMode: PrimeFaces.widget.AutoComplete.QueryMode;
        /**
         * Height of the container with the suggestion items.
         */
        scrollHeight: number;
        /**
         * Limits the number of simultaneously selected items. Default is unlimited.
         */
        selectLimit: number;
        /**
         * Ensures uniqueness of the selected items.
         */
        unique: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces BlockUI Widget__
     *
     * BlockUI is used to block interactivity of JSF components with optional AJAX integration.
     * @typeparam TCfg Defaults to `BlockUICfg`. Type of the configuration object for this widget.
     */
    export class BlockUI<TCfg extends BlockUICfg = BlockUICfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the content of the blocking overlay.
         */
        blocker: JQuery;
        /**
         * The DOM element for the content of the blocker.
         */
        content: JQuery;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM element for the overlay that blocks the UI.
         */
        target: JQuery;
        /**
         * The set-timeout timer ID for the timer of the delay before the AJAX status is
         * triggered.
         */
        timeout: number | null;
        /**
         * Clean up this widget and remove elements from DOM.
         */
        private _cleanup(): void;
        /**
         * Align the overlay so it covers its target component.
         */
        private alignOverlay(): void;
        /**
         * Sets up the global resize listener on the document.
         */
        private bindResizer(): void;
        /**
         * Sets up the global event listeners on the document.
         */
        private bindTriggers(): void;
        /**
         * Clears the ste-timeout timer for the delay.
         */
        private deleteTimeout(): void;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         * @override
         */
        override destroy(): void;
        /**
         * Checks whether the blocking overlay contains any content items.
         * @return `true` if this blocking overlay has got any content, `false` otherwise.
         */
        private hasContent(): boolean;
        /**
         * Checks whether this blocker has more than 1 target.
         * @return `true` if this blocker has more than 1 target, `false` otherwise.
         */
        private hasMultipleTargets(): boolean;
        /**
         * Hide the component with optional duration animation.
         * @param duration Durations are given in milliseconds; higher values indicate slower animations, not
         * faster ones. The strings `fast` and `slow` can be supplied to indicate durations of 200 and 600 milliseconds,
         * respectively.
         */
        hide(duration?: number): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Checks whether this blockUI is currently blocking.
         * @return `true` if this blockUI is blocking, or `false` otherwise.
         */
        isBlocking(): boolean;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Renders the client-side parts of this widget.
         */
        private render(): void;
        /**
         * Show the component with optional duration animation.
         * @param duration Durations are given in milliseconds; higher values indicate slower
         * animations, not faster ones. The strings `fast` and `slow` can be supplied to indicate durations of 200 and 600
         * milliseconds, respectively.
         */
        show(duration?: number | string): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  BlockUI| BlockUI widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface BlockUICfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * When disabled, displays block without animation effect.
         */
        animate: boolean;
        /**
         * Search expression for block targets.
         */
        block: string;
        /**
         * Blocks the UI by default when enabled.
         */
        blocked: boolean;
        /**
         * Delay in milliseconds before displaying the block. Default is `0`, meaning immediate.
         */
        delay: number;
        /**
         * Style class of the component.
         */
        styleClass: string;
        /**
         * Search expression of the components to bind.
         */
        triggers: string;
    }
}
// Global pollution ...
/**
 * Additional properties that will be set on the global `Date` object when the `Calendar` widget is loaded.
 */
interface Date {
    /**
     * Gets the microseconds.
     *
     * Defined globally by the Calendar widget. __Do not use this.__
     * @deprecated
     * @return The microseconds field of this date.
     */
    getMicroseconds(): number;
    /**
     * Set the microseconds.
     *
     * Defined globally by the Calendar widget. __Do not use this.__
     * @deprecated
     * @param microseconds The microseconds to set.
     * @return this for chaining.
     */
    setMicroseconds(microseconds: number): this;
}
/**
 * Namespace for the timepicker JQueryUI plugin, available as `JQuery.fn.timepicker` and `JQuery.fn.datetimepicker`.
 * Contains some additional types and interfaces required for the typings.
 */
declare namespace JQueryUITimepickerAddon {
    /**
     * Time units for selecting a time in the calendar widget.
     */
    export type TimeUnit = "hour" | "minute" | "second" | "millisec" | "microsec";
    /**
     * Whether to use sliders, select elements or a custom control type for selecting a time (hour / minute / second) in
     * the time picker.
     */
    export type ControlType = "slider" | "select";
    /**
     * An offset of a timezone, in minutes relative to UTC. For example, `UTC-4` is represented as `-240`.
     */
    export type TimezoneOffset = number;
    /**
     * How dates are parsed by the Timepicker.
     *
     * - `loose`: Uses the JavaScript method `new Date(timeString)` to guess the time
     * - `strict`: A date text must match the timeFormat exactly.
     */
    export type TimeParseType = "loose" | "strict";
    /**
     * A custom function for parsing a time string.
     */
    export type TimeParseFunction =
    /**
     * @param timeFormat Format according to which to parse the time.
     * @param timeString Time string to parse.
     * @param optins Current options of the time picker.
     * @return The parsed time, or `undefined` if the time string could not be parsed.
     */
    (timeFormat: string, timeString: string, options: Partial<DatetimepickerOptions>) => TimeParseResult | undefined;
    /**
     * Represents the available methods on a JQuery instance for the date and / or time picker.
     */
    export type PickerMethod = "datepicker" | "timepicker" | "datetimepicker";
    /**
     * Represents a timezone of the world.
     */
    export interface Timezone {
        /**
         * Name of the timezone.
         */
        label: string;
        /**
         * Offset of the timezone.
         */
        value: TimezoneOffset;
    }
    /**
     * The timepicker for working with times, such as formatting and parsing times.
     */
    export interface Timepicker {
        /**
         * A map with a locale name (`fr`, `de`, etc.) as the key and the locale as the value.
         */
        regional: Record<string, Locale>;
        /**
         * Current version of the DateTimePicker JQueryUI add-on.
         */
        version: string;
        /**
         * Override the default settings for all instances of the time picker.
         * @param settings The new settings to use as defaults.
         * @return this for chaining.
         */
        setDefaults(settings: Partial<DatetimepickerOptions>): this;
        /**
         * Calls `datetimepicker` on the `startTime` and `endTime` elements, and configures them to enforce the date /
         * time range limits.
         * @param startTime DOM element of the date/time picker with the start date/time.
         * @param endTime DOM element of the date/time picker with the end date/time
         * @param options Options for the `$.fn.datetimepicker` call.
         */
        datetimeRange(startTime: JQuery, endTime: JQuery, options: Partial<RangeOptions>): void;
        /**
         * Calls `timepicker` on the `startTime` and `endTime` elements, and configures them to enforce the time range
         * limits.
         * @param startTime DOM element of the date/time picker with the start date/time.
         * @param endTime DOM element of the date/time picker with the end date/time
         * @param options Options for the `$.fn.timepicker` call.
         */
        timeRange(startTime: JQuery, endTime: JQuery, options: Partial<RangeOptions>): void;
        /**
         * Calls `datepicker` on the `startTime` and `endTime` elements, and configures them to enforce the date
         * range limits.
         * @param startTime DOM element of the date/time picker with the start date/time.
         * @param endTime DOM element of the date/time picker with the end date/time
         * @param options Options for the `$.fn.datepicker` call.
         */
        dateRange(startTime: JQuery, endTime: JQuery, options: Partial<RangeOptions>): void;
        /**
         * Calls the given method on the `startTime` and `endTime` elements, and configures them to enforce the date /
         * time range limits.
         * @param method Whether to call the `datepicker`, `timepicker`, or `datetimepicker` method on the elements.
         * @param startTime DOM element of the date/time picker with the start date/time.
         * @param endTime DOM element of the date/time picker with the end date/time
         * @param options Options for the `$.fn.datepicker` call.
         * @return A JQuery instance containing the given `startTime` and `endTime` elements.
         */
        handleRange(method: PickerMethod, startTime: JQuery, endTime: JQuery, options: Partial<RangeOptions>): JQuery;
        /**
         * Get the timezone offset as string from a timezone offset.
         * @param tzMinutes If not a number, less than `-720` (`UTC-12`), or greater than `840` (`UTC+14`),
         * this value is returned as-is
         * @param iso8601 If `true` formats in accordance to `iso8601` (sucha as `+12:45`).
         * @return The timezone offset as a string, such as `+0530` for `UTC+5.5`.
         */
        timezoneOffsetString(tzMinutes: TimezoneOffset | string, iso8601: boolean): string;
        /**
         * Get the number in minutes that represents a timezone string.
         * @param tzString A formatted time zone string, such as `+0500`, `-1245`, or `Z`.
         * @return The offset in minutes, or the given `tzString` when it does not represent a valid timezone.
         */
        timezoneOffsetNumber(tzString: string): TimezoneOffset | string;
        /**
         * JavaScript `Date`s have not support for timezones, so we must adjust the minutes to compensate.
         * @param date Date to adjust.
         * @param fromTimezone Timezone of the given date.
         * @param toTimezone Timezone to adjust the date to, relative to the `fromTimezone`.
         * @return The given date, adjusted from the `fromTimezone` to the `toTimezone`.
         */
        timezoneAdjust(date: Date, fromTimezone: string, toTimezone: string): Date;
        /**
         * Log error or data to the console during error or debugging.
         * @param args Data to log.
         */
        log(...args: readonly unknown[]): void;
    }
    /**
     * Represents localized messages for a certain locale that are displayed by the datetimepicker.
     */
    export interface Locale {
        /**
         * Default: `["AM", "A"]`, A Localization Setting - Array of strings to try and parse against to determine AM.
         */
        amNames: string[];
        /**
         * Default: `["PM", "P"]`, A Localization Setting - Array of strings to try and parse against to determine PM.
         */
        pmNames: string[];
        /**
         * Default: `HH:mm`, A Localization Setting - String of format tokens to be replaced with the time.
         */
        timeFormat: string;
        /**
         * Default: Empty string, A Localization Setting - String to place after the formatted time.
         */
        timeSuffix: string;
        /**
         * Default: `Choose Time`, A Localization Setting - Title of the wigit when using only timepicker.
         */
        timeOnlyTitle: string;
        /**
         * Default: `Time`, A Localization Setting - Label used within timepicker for the formatted time.
         */
        timeText: string;
        /**
         * Default: `Hour`, A Localization Setting - Label used to identify the hour slider.
         */
        hourText: string;
        /**
         * Default: `Minute`, A Localization Setting - Label used to identify the minute slider.
         */
        minuteText: string;
        /**
         * Default: `Second`, A Localization Setting - Label used to identify the second slider.
         */
        secondText: string;
        /**
         * Default: `Millisecond`, A Localization Setting - Label used to identify the millisecond slider.
         */
        millisecText: string;
        /**
         * Default: `Microsecond`, A Localization Setting - Label used to identify the microsecond slider.
         */
        microsecText: string;
        /**
         * Default: `Timezone`, A Localization Setting - Label used to identify the timezone slider.
         */
        timezoneText: string;
    }
    /**
     * Represents the result of parsing a time string.
     */
    export interface TimeParseResult {
        /**
         * Hour of the time, starting at `0`.
         */
        hour: number;
        /**
         * Minute of the time, starting at `0`.
         */
        minute: number;
        /**
         * Seconds of the time, starting at `0`.
         */
        seconds: number;
        /**
         * Milliseconds of the time, starting at `0`.
         */
        millisec: number;
        /**
         * Microseconds of the time, starting at `0`.
         */
        microsec: number;
        /**
         * Timezone of the time.
         */
        timezone?: TimezoneOffset;
    }
    /**
     * Options for the date time picker that lets the user select a time.
     */
    export interface DatetimepickerOptions extends JQueryUI.DatepickerOptions, Locale {
        /**
         * Default: `true` - When `altField` is used from datepicker, `altField` will only receive the formatted time
         * and the original field only receives date.
         */
        altFieldTimeOnly: boolean;
        /**
         * Default: (separator option) - String placed between formatted date and formatted time in the altField.
         */
        altSeparator: string;
        /**
         * Default: (timeSuffix option) - String always placed after the formatted time in the altField.
         */
        altTimeSuffix: string;
        /**
         * Default: (timeFormat option) - The time format to use with the altField.
         */
        altTimeFormat: string;
        /**
         * Default: true - Whether to immediately focus the main field whenever the altField receives focus. Effective
         * at construction time only, changing it later has no effect.
         */
        altRedirectFocus: boolean;
        /**
         * Default: [generated timezones] - An array of timezones used to populate the timezone select.
         */
        timezoneList: Timezone[] | Record<string, TimezoneOffset>;
        /**
         * Default: `slider` - How to select a time (hour / minute / second). If `slider` is unavailable through
         * jQueryUI, `select` will be used. For advanced usage you may set this to a custom control to use controls
         * other than sliders or selects.
         */
        controlType: ControlType | CustomControl;
        /**
         * Default: `null` - Whether to show the hour control.  The default of `null` will use detection from timeFormat.
         */
        showHour: boolean | null;
        /**
         * Default: `null` - Whether to show the minute control.  The default of `null` will use detection from
         * timeFormat.
         */
        showMinute: boolean | null;
        /**
         * Default: `null` - Whether to show the second control.  The default of `null` will use detection from
         * timeFormat.
         */
        showSecond: boolean | null;
        /**
         * Default: `null` - Whether to show the millisecond control.  The default of `null` will use detection from
         * timeFormat.
         */
        showMillisec: boolean | null;
        /**
         * Default: `null` - Whether to show the microsecond control.  The default of `null` will use detection from
         * timeFormat.
         */
        showMicrosec: boolean | null;
        /**
         * Default: `null` - Whether to show the timezone select.
         */
        showTimezone: boolean | null;
        /**
         * Default: true - Whether to show the time selected within the datetimepicker.
         */
        showTime: boolean;
        /**
         * Default: `1` - Hours per step the slider makes.
         */
        stepHour: number;
        /**
         * Default: `1` - Minutes per step the slider makes.
         */
        stepMinute: number;
        /**
         * Default: `1` - Seconds per step the slider makes.
         */
        stepSecond: number;
        /**
         * Default: `1` - Milliseconds per step the slider makes.
         */
        stepMillisec: number;
        /**
         * Default: `1` - Microseconds per step the slider makes.
         */
        stepMicrosec: number;
        /**
         * Default: `0` - Initial hour set.
         */
        hour: number;
        /**
         * Default: `0` - Initial minute set.
         */
        minute: number;
        /**
         * Default: `0` - Initial second set.
         */
        second: number;
        /**
         * Default: `0` - Initial millisecond set.
         */
        millisec: number;
        /**
         * Default: `0` - Initial microsecond set.  Note: Javascript's native `Date` object does not natively support
         * microseconds.  Timepicker extends the Date object with `Date.prototype.setMicroseconds(m)` and
         * `Date.prototype.getMicroseconds()`. Date comparisons will not acknowledge microseconds. Use this only for
         * display purposes.
         */
        microsec: number;
        /**
         * Default: `null` - Initial timezone set.  If `null`, the browser's local timezone will be used.
         */
        timezone: TimezoneOffset | null;
        /**
         * Default: `0` - The minimum hour allowed for all dates.
         */
        hourMin: number;
        /**
         * Default: `0` - The minimum minute allowed for all dates.
         */
        minuteMin: number;
        /**
         * Default: `0` - The minimum second allowed for all dates.
         */
        secondMin: number;
        /**
         * Default: `0` - The minimum millisecond allowed for all dates.
         */
        millisecMin: number;
        /**
         * Default: `0` - The minimum microsecond allowed for all dates.
         */
        microsecMin: number;
        /**
         * Default: `23` - The maximum hour allowed for all dates.
         */
        hourMax: number;
        /**
         * Default: `59` - The maximum minute allowed for all dates.
         */
        minuteMax: number;
        /**
         * Default: `59` - The maximum second allowed for all dates.
         */
        secondMax: number;
        /**
         * Default: `999` - The maximum millisecond allowed for all dates.
         */
        millisecMax: number;
        /**
         * Default: `999` - The maximum microsecond allowed for all dates.
         */
        microsecMax: number;
        /**
         * Default: `0` - When greater than `0` a label grid will be generated under the slider.  This number represents
         * the units (in hours) between labels.
         */
        hourGrid: number;
        /**
         * Default: `0` - When greater than `0` a label grid will be generated under the slider. This number represents
         * the units (in minutes) between labels.
         */
        minuteGrid: number;
        /**
         * Default: `0` - When greater than `0` a label grid will be genereated under the slider. This number represents
         * the units (in seconds) between labels.
         */
        secondGrid: number;
        /**
         * Default: `0` - When greater than `0` a label grid will be genereated under the slider. This number represents
         * the units (in milliseconds) between labels.
         */
        millisecGrid: number;
        /**
         * Default: `0` - When greater than `0` a label grid will be genereated under the slider. This number represents
         * the units (in microseconds) between labels.
         */
        microsecGrid: number;
        /**
         * Default: `true` - Whether to show the button panel at the bottom. This is generally needed.
         */
        showButtonPanel: boolean;
        /**
         * Default: `false` - Allows direct input in time field
         */
        timeInput: boolean;
        /**
         * Default: `false` - Hide the datepicker and only provide a time interface.
         */
        timeOnly: boolean;
        /**
         * Default: `false` - Show the date and time in the input, but only allow the timepicker.
         */
        timeOnlyShowDate: boolean;
        /**
         * Default: unset - Function to be called when the timepicker or selection control is injected or re-rendered.
         */
        afterInject(this: Timepicker): void;
        /**
         * Default: unset - Function to be called when a date is chosen or time has changed.
         * @param datetimeText Currently selected date as text.
         * @param timepicker The current timepicker instance.
         */
        onSelect(this: HTMLElement | null, datetimeText: string, timepicker: Timepicker): void;
        /**
         * Default: `true` - Always have a time set internally, even before user has chosen one.
         */
        alwaysSetTime: boolean;
        /**
         * Default: space (` `) - When formatting the time this string is placed between the formatted date and
         * formatted time.
         */
        separator: string;
        /**
         * Default: (timeFormat option) - How to format the time displayed within the timepicker.
         */
        pickerTimeFormat: string;
        /**
         * Default: (timeSuffix option) - String to place after the formatted time within the timepicker.
         */
        pickerTimeSuffix: string;
        /**
         * Default: `true` - Whether to show the timepicker within the datepicker.
         */
        showTimepicker: boolean;
        /**
         * Default: `false` - Try to show the time dropdowns all on one line. This should be used with `controlType`
         * `select` and as few units as possible.
         */
        oneLine: boolean;
        /**
         * Default: `null` - String of the default time value placed in the input on focus when the input is empty.
         */
        defaultValue: string | null;
        /**
         * Default: `null` - Date object of the minimum datetime allowed.  Also available as minDate.
         */
        minDateTime: Date | null;
        /**
         * Default: `null` - Date object of the maximum datetime allowed. Also Available as maxDate.
         */
        maxDateTime: Date | null;
        /**
         * Default: `null` - String of the minimum time allowed. '8:00 am' will restrict to times after 8am
         */
        minTime: string | null;
        /**
         * Default: `null` - String of the maximum time allowed. '8:00 pm' will restrict to times before 8pm
         */
        maxTime: string | null;
        /**
         * Default: `strict` - How to parse the time string. You may also set this to a function to handle the parsing
         * yourself.
         */
        parse: TimeParseType | TimeParseFunction;
    }
    /**
     * Optionts for the various methods of the `Timepicker` for working time date / time ranges.
     */
    export interface RangeOptions extends DatetimepickerOptions {
        /**
         * Min allowed interval in milliseconds
         */
        minInterval: number;
        /**
         * Max allowed interval in milliseconds
         */
        maxInterval: number;
        /**
         * Options that are applied only to the date / time picker for the start date / time.
         */
        start: Partial<DatetimepickerOptions>;
        /**
         * Options that are applied only to the date / time picker for the end date / time.
         */
        end: Partial<DatetimepickerOptions>;
    }
    /**
     * Options for a custom control for selecting an hour, minute, or seconds. The control should behave in such a way
     * that the user may select a number in the set `{ min, min+step, min+2*step, ..., max }`.
     */
    export interface ControlOptions {
        /**
         * Maximum allowed value for the time unit the user may select.
         */
        max: number;
        /**
         * Minumum allowed value for the time unit the user may select.
         */
        min: number;
        /**
         * Desired step size for selecting a value.
         */
        step: number;
    }
    /**
     * For advanced usage of the Calendar, you may pass an object of this type to use controls other than sliders and
     * selects for selecting an hour, minute, or second.
     */
    export interface CustomControl {
        /**
         * Creates the control for the given time unit and appends it to the given `container` element.
         * @param instance The current date time picker instance.
         * @param container The container element to which the created control must be appended.
         * @param unit The type of control for which to set the value.
         * @param val Initial value for the control
         * @param min Minumum allowed value for the time unit the user may select.
         * @param max Maximum allowed value for the time unit the user may select.
         * @param step Desired step size for selecting a value.
         * @return The `container` element as passed to this method.
         */
        create(instance: Timepicker, container: JQuery, unit: TimeUnit, val: number, min: number, max: number, step: number): JQuery;
        /**
         * Sets the given ooptions on the control for the given time unit.
         * @param instance The current date time picker instance.
         * @param container The container element of the control, as passed to `create`.
         * @param unit The type of control for which to apply the options.
         * @param opts Options to apply on the control
         * @return The `container` element as passed to this method.
         */
        options(instance: Timepicker, container: JQuery, unit: TimeUnit, opts: Partial<ControlOptions>): JQuery;
        /**
         * Sets the value of control for the given time uit.
         * @param instance The current date time picker instance.
         * @param container The container element of the control, as passed to `create`.
         * @param unit The type of control for which to set the value.
         * @param val Value to set on this control.
         * @return The `container` element as passed to this method.
         */
        value(instance: Timepicker, container: JQuery, unit: TimeUnit, val: number): JQuery;
        /**
         * Gets the current value of the control for the given time unit.
         * @param instance The current date time picker instance.
         * @param container The container element of the control, as passed to `create`.
         * @param unit The type of control for which to get the value.
         * @return The current value of the control.
         */
        value(instance: Timepicker, container: JQuery, unit: TimeUnit): number;
    }
}
interface JQuery {
    /**
     * Initializes the datetimepicker on this element. It lets the user select both a date and a time (hour and
     * minute).
     * @param cfg Options for the datetimepicker.
     * @return this for chaining.
     */
    datetimepicker(cfg?: Partial<JQueryUITimepickerAddon.DatetimepickerOptions>): this;
    /**
     * Sets and selects the given date.
     * @param methodName Name of the method to invoke.
     * @param date The new date to select. When not given, unselects the date.
     * @return this for chaining.
     */
    datetimepicker(methodName: "setDate", date?: Date): this;
    /**
     * Finds the currently selected date of the datetimepicker.
     * @param methodName Name of the method to invoke.
     * @return The currently selected date, or `null` if no date is selected.
     */
    datetimepicker(methodName: "getDate"): Date | null;
    /**
     * Enables the datetimepicker so that the user can now select a date.
     * @param methodName Name of the method to invoke.
     * @return this for chaining.
     */
    datetimepicker(methodName: "enable"): this;
    /**
     * Disables the datetimepicker so that the user cannot select a date anymore.
     * @param methodName Name of the method to invoke.
     * @return this for chaining.
     */
    datetimepicker(methodName: "disable"): this;
    /**
     * Sets the minimum allowed date the user may select.
     * @param methodName Name of the method to invoke.
     * @param optionName Name of the option to set.
     * @param date New value for the option.
     * @return this for chaining.
     */
    datetimepicker(methodName: "option", optionName: "minDate", date: Date): this;
    /**
     * Sets the maximum allowed date the user may select.
     * @param methodName Name of the method to invoke.
     * @param optionName Name of the option to set.
     * @param date New value for the option.
     * @return this for chaining.
     */
    datetimepicker(methodName: "option", optionName: "maxDate", date: Date): this;
    /**
     * Initializes the timepicker on this element. It lets the user select a time (hour and minute).
     * @param cfg Options for the datetimepicker.
     * @return this for chaining.
     */
    timepicker(cfg?: Partial<JQueryUITimepickerAddon.DatetimepickerOptions>): this;
}
interface JQueryStatic {
    /**
     * The global instance of the timepicker utility class for working with times.
     */
    timepicker: JQueryUITimepickerAddon.Timepicker;
}
declare namespace PrimeFaces.widget.Calendar {
    /**
     * Callback invoked before the calendar is opened.
     */
    export type PreShowCallback =
    /**
     * @param input Input element for the date.
     * @param instance Current time picker
     * instance controlling the calendar. `false` to prevent the time picker from being shown.
     * @return A new
     * set of options for the time picker.
     */
    (this: PrimeFaces.widget.Calendar, input: JQuery, instance: JQueryUITimepickerAddon.Timepicker) => Partial<JQueryUI.DatepickerOptions> | boolean | undefined;
}
declare namespace PrimeFaces.widget.Calendar {
    /**
     * Callback invoked before a day is shown.
     */
    export type PreShowDayCallback =
    /**
     * @param date The current date of the calendar.
     * @return Two to three
     * values indicating:
     * 1. true/false indicating whether or not this date is selectable
     * 1. a CSS class name to add to the date's cell or "" for the default presentation
     * 1. an optional popup tooltip for this date
     */
    (this: Window, date: Date) => [
        boolean,
        string
    ] | [
        boolean,
        string,
        string
    ];
}
declare namespace PrimeFaces.widget.Calendar {
    /**
     * Client-side event to display the
     * calendar. `focus` is when the input field receives focus. `popup` is when the popup button is clicked. `both` is
     * both `focus` and `popup`.
     */
    export type ShowOnType = "focus" | "button" | "both";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Calendar Widget__
     *
     * __Deprecated__: Use the {@link DatePicker|p:datePicker} component instead.
     *
     * Calendar is an input component used to select a date featuring display modes, paging, localization, ajax selection
     * and more.
     *
     * To interact with the calendar, use the `timepicker` or `datetimepicker` JQuery plugin, for example:
     *
     * ```javascript
     * PF("calendarWidget").jqEl.datetimepicker("getDate");
     * PF("calendarWidget").jqEl.datetimepicker("setDate", new Date());
     * ```
     * @typeparam TCfg Defaults to `CalendarCfg`. Type of the configuration object for this widget.
     */
    export class Calendar<TCfg extends CalendarCfg = CalendarCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * DOM element of the plain-text input field for the date and/or time.
         */
        input: JQuery;
        /**
         * The DOM element on which the JQuery plugin `datepicker` or `datetimepicker` was initialized. You
         * can use this element to interact with the date picker.
         */
        jqEl: JQuery;
        /**
         * Whether the calendar is read-only and cannot be edited.
         */
        readonly?: boolean;
        /**
         * Whether the input needs to be refocused.
         */
        refocusInput: boolean;
        /**
         * Aligns the overlay panel with the date picker according to the current configuration. It is usually positioned
         * next to or below the input field to which it is attached.
         */
        alignPanel(): void;
        /**
         * Initializes the mask on the input if using a mask and not an inline picker.
         */
        private applyMask(): void;
        /**
         * Sets up the event listeners for when this calendar is closed.
         */
        private bindCloseListener(): void;
        /**
         * Sets up the event listeners for when the user selects a particular date.
         */
        private bindDateSelectListener(): void;
        /**
         * Sets up the event listeners for when the user switches to a different month or year.
         */
        private bindViewChangeListener(): void;
        /**
         * Sets up the locale so that this calendar is displayed in the configured langauge.
         */
        private configureLocale(): void;
        /**
         * Creates and initializes the confiugration options for the time picker.
         */
        private configureTimePicker(): void;
        /**
         * Disables the calendar, so that the user can no longer select any date..
         */
        disable(): void;
        /**
         * Enables the calendar, so that the user can select a date.
         */
        enable(): void;
        /**
         * Triggers the `close` event when this calendar is closed.
         */
        private fireCloseEvent(): void;
        /**
         * Triggers the behaviors and event listener for when the user has selected a certain date.
         */
        private fireDateSelectEvent(): void;
        /**
         * Triggers the behaviors and event listener for when the user has switched to a different month or year.
         * @param year New year for which a calendar is shown.
         * @param month New month for which a calendar is shown (0=January).
         */
        private fireViewChangeEvent(year: number, month: number): void;
        /**
         * Finds the currently selected date.
         * @return The selected date of the calendar, or `null` when no date is selected.
         */
        getDate(): Date | null;
        /**
         * Fired when the browser viewport is resized or scrolled.  In Mobile environment we don't want to hider the overlay
         * we want to re-align it.  This is because on some mobile browser the popup may force the browser to trigger a
         * resize immediately and close the overlay. See GitHub #7075.
         */
        private handleViewportChange(): void;
        /**
         * Checks whether this calendar lets the user specify a clock time (and not just a date).
         * @return `true` when this calendar includes a clock time picker, `false` otherwise.
         */
        hasTimePicker(): boolean;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Sets the currently selected date of the datepicker.
         * @param date Date to display, or `null` or `undefined` to clear the date.
         */
        setDate(date: Date | null | undefined): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Calendar| Calendar widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface CalendarCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Position of the button in the tabbing order.
         */
        buttonTabindex: string;
        /**
         * How the user selects a time (hour / minute /
         * second). When set to `custom`, the `timeControlObject` must be set.
         */
        controlType: JQueryUITimepickerAddon.ControlType | "custom";
        /**
         * Date format pattern for localization
         */
        dateFormat: string;
        /**
         * Disables the calendar when set to true.
         */
        disabled: boolean;
        /**
         * Disables weekend columns.
         */
        disabledWeekends: boolean;
        /**
         * Duration of the effect.
         */
        duration: string;
        /**
         * If enabled, the input is focused again after selecting a date. Default is false.
         */
        focusOnSelect: boolean;
        /**
         * Default for hour selection, if no date is given. Default is 0.
         */
        hour: number;
        /**
         * Maximum boundary for hour selection.
         */
        hourMax: number;
        /**
         * Minimum boundary for hour selection.
         */
        hourMin: number;
        /**
         * Locale to be used for labels and conversion.
         */
        locale: string;
        /**
         * Applies a mask using the pattern.
         */
        mask: string;
        /**
         * Clears the field on blur when incomplete input is entered
         */
        maskAutoClear: boolean;
        /**
         * Placeholder in mask template.
         */
        maskSlotChar: string;
        /**
         * Sets calendar's maximum visible date; Also used for validation on the server-side.
         */
        maxDate: string;
        /**
         * Default for millisecond selection, if no date is given. Default is 0.
         */
        millisec: number;
        /**
         * Sets calendar's minimum visible date; Also used for validation on the server-side.
         */
        minDate: string;
        /**
         * Default for minute selection, if no date is given. Default is 0.
         */
        minute: number;
        /**
         * Maximum boundary for hour selection.
         */
        minuteMax: number;
        /**
         * Minimum boundary for minute selection.
         */
        minuteMin: number;
        /**
         * Enables multiple page rendering.
         */
        numberOfMonths: number;
        /**
         * Try to show the time dropdowns all on one line. This should be used with the
         * `controlType` set to `select`.
         */
        oneLine: boolean;
        /**
         * `true` if `mode` is set to `popup`.
         */
        popup: boolean;
        /**
         * Callback invoked before the calendar is opened.
         */
        preShow: PrimeFaces.widget.Calendar.PreShowCallback;
        /**
         * Callback invoked before a day is shown.
         */
        preShowDay: PrimeFaces.widget.Calendar.PreShowDayCallback;
        /**
         * Makes the calendar readonly when set to true.
         */
        readonly: boolean;
        /**
         * Default for second selection, if no date is given. Default is 0.
         */
        second: number;
        /**
         * Maximum boundary for second selection.
         */
        secondMax: number;
        /**
         * Minimum boundary for second selection.
         */
        secondMin: number;
        /**
         * Enables selection of days belonging to other months.
         */
        selectOtherMonths: boolean;
        /**
         * The cutoff year for determining the century for a date. Default is `+10`.
         */
        shortYearCutoff: string;
        /**
         * Effect to use when displaying and showing the popup calendar.
         */
        showAnim: string;
        /**
         * Visibility of button panel containing today and done buttons.
         */
        showButtonPanel: boolean;
        /**
         * Whether to show the hour control.
         */
        showHour: string;
        /**
         * Whether to show the millisec control
         */
        showMillisec: string;
        /**
         * Whether to show the minute control.
         */
        showMinute: string;
        /**
         * Client side event that displays the popup calendar.
         */
        showOn: PrimeFaces.widget.Calendar.ShowOnType;
        /**
         * Displays days belonging to other months.
         */
        showOtherMonths: boolean;
        /**
         * Whether to show the second control.
         */
        showSecond: string;
        /**
         * Whether to show the `Current Date` button if `showButtonPanel` is rendered.
         */
        showTodayButton: boolean;
        /**
         * Displays the week number next to each week.
         */
        showWeek: boolean;
        /**
         * Hour steps.
         */
        stepHour: string;
        /**
         * Minute steps.
         */
        stepMinute: number;
        /**
         * Second steps.
         */
        stepSecond: number;
        /**
         * When `controlType` is set to `custom`, an
         * object for creating and handling custom controls for the hour / minute / second inputs.
         */
        timeControlObject: JQueryUITimepickerAddon.CustomControl;
        /**
         * Allows direct input in time field.
         */
        timeInput: boolean;
        /**
         * Shows only timepicker without date.
         */
        timeOnly: boolean;
        /**
         * Year range for the navigator, default is `c-10:c+10`.
         */
        yearRange: string;
    }
}
declare namespace PrimeFaces.widget.Captcha {
    /**
     * Captcha features light and dark modes for theme.
     */
    export type Theme = "auto" | "light" | "dark";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Captcha Widget__
     *
     * Captcha is a form validation component based on Recaptcha API V2.
     * @typeparam TCfg Defaults to `CaptchaCfg`. Type of the configuration object for this widget.
     */
    export class Captcha<TCfg extends CaptchaCfg = CaptchaCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Appends the script to the document body.
         */
        private appendScript(): void;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         * @override
         */
        override destroy(): void;
        /**
         * Finds the name to use for the callback function set globally on the window.
         * @return Name for the global callback.
         */
        private getInitCallbackName(): string;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: TCfg): void;
        /**
         * Renders the client-side parts of this widget.
         */
        private render(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Captcha| Captcha widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface CaptchaCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Name of a function in the global scope. Callback to be executed when the user submits a
         * successful CAPTCHA response. The user's response, goggle recaptcha-response, will be the input for your callback
         * function.
         */
        callback: string;
        /**
         * Name of a function in the global scope. The callback executed when the captcha response
         * expires and the user needs to solve a new captcha.
         */
        expired: string;
        /**
         * Language in which information about this captcha is shown.
         */
        language: string;
        /**
         * Public recaptcha key for a specific domain (deprecated).
         */
        sitekey: string;
        /**
         * Size of the recaptcha.
         */
        size: string;
        /**
         * URL for the ReCaptcha JavaScript file. Some countries do not have access to Google.
         */
        sourceUrl: string;
        /**
         * Position of the input element in the tabbing order.
         */
        tabindex: number;
        /**
         * Theme of the captcha.
         */
        theme: PrimeFaces.widget.Captcha.Theme;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Carousel Widget__
     * Carousel is a content slider featuring various customization options.
     * @typeparam TCfg Defaults to `CarouselCfg`. Type of the configuration object for this widget.
     */
    export class Carousel<TCfg extends CarouselCfg = CarouselCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * Whether autoplay is allowed or not.
         */
        allowAutoplay: boolean;
        /**
         * Style element with the custom CSS for the carousel.
         */
        carouselStyle: HTMLStyleElement;
        /**
         * Whether the viewport is circular or not.
         */
        circular: boolean;
        /**
         * The DOM element for the container of the carousel that contains items container and buttons.
         */
        container: JQuery;
        /**
         * The DOM element for the content of the carousel that shows the carousel.
         */
        content: JQuery;
        /**
         * Callback used to listen to resize events and
         * adjust the carousel accordingly.
         */
        documentResizeListener: ((event: UIEvent) => void) | undefined;
        /**
         * DOM elements of the `LI` indicator of the carousel.
         */
        indicators?: JQuery;
        /**
         * The DOM element for the indicators container of the carousel.
         */
        indicatorsContainer: JQuery;
        /**
         * Timeout ID of the timer used for autoplay.
         */
        interval?: number;
        /**
         * Whether autoplay is allowed or not.
         */
        isAutoplay: boolean;
        /**
         * Whether the circular mode is on or not.
         */
        isCircular: boolean;
        /**
         * Whether the remaining items have been added or not.
         */
        isRemainingItemsAdded: boolean;
        /**
         * Whether the viewport is vertical or not.
         */
        isVertical: boolean;
        /**
         * The DOM elements for the carousel items.
         */
        items: JQuery;
        /**
         * The DOM element for the item container of the carousel.
         */
        itemsContainer: JQuery;
        /**
         * The DOM element for the item container of the carousel.
         */
        itemsContent: JQuery;
        /**
         * The number of simultaneously visible items.
         */
        itemsCount: number;
        /**
         * The DOM element for the button to switch to the previous carousel item.
         */
        nextNav: JQuery;
        /**
         * Instant number of how many items will scroll when scrolled.
         */
        numScroll: number;
        /**
         * Instant number of items visible on the carousel viewport.
         */
        numVisible: number;
        /**
         * Old number of items visible on the carousel viewport.
         */
        oldNumScroll: number;
        /**
         * Old number of how many items will scroll when scrolled.
         */
        oldNumVisible: number;
        /**
         * The currently displayed page of carousel items.
         */
        page: number;
        /**
         * The DOM element for the button to switch to the next carousel item.
         */
        prevNav: JQuery;
        /**
         * How many items remaining for the show.
         */
        remainingItems: number;
        /**
         * The number of indicators currently in the viewport.
         */
        totalIndicators: number;
        /**
         * The number of how many items shifted.
         */
        totalShiftedItems: number;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Retrieves whether the backward button is disabled.
         * @return backward button is disabled.
         */
        private backwardIsDisabled(): boolean;
        /**
         * Adds the resize event listener to the window.
         */
        private bindDocumentListeners(): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Calculates position and visible items and the number of how many items will be scrolled when screen aspect ratio
         * changes then updates current page of the current Carousel widget.
         */
        private calculatePosition(): void;
        /**
         * Scrolls the item container based on the total number of shifted items
         * @param totalShiftedItems total number of shifted items.
         */
        private changePosition(totalShiftedItems: number): void;
        /**
         * Clones items if the carousel widget is circular
         */
        private cloneItems(): void;
        /**
         * Creates responsive styles of the carousel container.
         */
        private createStyle(): void;
        /**
         * Retrieves the first index of visible items.
         * @return first index of the visible items.
         */
        private firstIndex(): number;
        /**
         * Retrieves whether the forward button is disabled.
         * @return forward button is disabled.
         */
        private forwardIsDisabled(): boolean;
        /**
         * Retrieves the total number of the indicators floor to 0 so it can't be negative.
         * @return total number of the indicators.
         */
        private getTotalIndicators(): number;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Initialize current page and variables.
         */
        private initPageState(): void;
        /**
         * Retrieves the last index of visible items.
         * @return last index of the visible items.
         */
        private lastIndex(): number;
        /**
         * Moves this carousel to the previous page. If autoplay is active, it will stop.
         * @param event Event that occurred.
         * @param index 0-based index of the page to display.
         */
        navBackward(event: Event, index: number): void;
        /**
         * Moves this carousel to the next page. If autoplay is active, it will stop.
         * @param event Event that occurred.
         * @param index 0-based index of the page to display.
         */
        navForward(event: Event, index: number): void;
        /**
         * It moves the current Carousel to the index of the clicked indicator on that Carousel viewport.
         * @param event Event that occurred.
         * @param index 0-based index of the indicator.
         */
        private onIndicatorClick(event: Event, index: number): void;
        /**
         * Changes current page according to the state of the page when the transition ends.
         */
        private onTransitionEnd(): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Retrieves the indicators html of the carousel.
         * @return html of the indicators container.
         */
        private renderIndicators(): string;
        /**
         * Enables autoplay and starts the slideshow.
         */
        startAutoplay(): void;
        /**
         * Moves this carousel to the given page.
         * @param dir direction of the move and takes a value of -1 or 1.
         * @param page 0-based index of the page to display.
         */
        step(dir: number, page: number): void;
        /**
         * Disables autoplay and stops the slideshow.
         */
        stopAutoplay(): void;
        /**
         * Styles visible items
         */
        private styleActiveItems(): void;
        /**
         * Applies styles to the clones
         * @param element cloned dom element of the item
         * @param index index of the element
         * @param length length of the clones
         */
        private styleClone(element: JQuery, index: number, length: number): void;
        /**
         * Render the indicators based on the current page state.
         */
        private updateIndicators(): void;
        /**
         * Update styles of the navigator buttons.
         */
        private updateNavigators(): void;
        /**
         * Updates the current page of the carousel.
         */
        private updatePage(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Carousel| Carousel widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface CarouselCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * Sets the time in milliseconds to have Carousel start scrolling automatically
         * after being initialized.
         */
        autoplayInterval: number;
        /**
         * Sets continuous scrolling
         */
        circular: boolean;
        /**
         * Number of items to scroll
         */
        numScroll: number;
        /**
         * Number of visible items per page
         */
        numVisible: number;
        /**
         * Specifies the layout of the component, valid layouts are horizontal or vertical
         */
        orientation: string;
        /**
         * Index of the first item.
         */
        page: number;
        /**
         * Whether to display the paginator or not.
         */
        paginator: boolean;
        /**
         * An array of options for responsive design
         */
        responsiveOptions: {
            breakpoint: string;
            numVisible: number;
            numScroll: number;
        }[];
    }
}
declare namespace PrimeFaces.widget.Chart {
    /**
     * The type of the chart extender. It is invoked when the chart is created and lets you modify the chart by using the
     * [chart.js](https://www.chartjs.org/docs/latest/) API. The current chart widget is passed as the this context. To
     * modify the chart configuration, mutate the `this.cfg.config` object.
     */
    export type ChartExtender = <TWidget extends PrimeFaces.widget.Chart = PrimeFaces.widget.Chart>(this: TWidget) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Chart Widget__
     *
     * Chart.js based components are a modern replacement for the older `<p:chart>` component. Each chart component has its
     * own model api that defines the data and the options to customize the graph.
     *
     * You can also define an extender function. The extender function allows access to the underlying
     * [chart.js](https://www.chartjs.org/docs/latest/) API using the `setExtender` method of the model. You need to define
     * a global function and set it on the model, see the user guide for more details. The required typing of that function
     * is given by `PrimeFaces.widget.Chart.ChartExtender`.
     * @typeparam TCfg Defaults to `ChartCfg`. Type of the configuration object for this widget.
     */
    export class Chart<TCfg extends ChartCfg = ChartCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * The canvas on which this chart is drawn.
         */
        canvas: JQuery<HTMLCanvasElement>;
        /**
         * The chart.js instance creates for this chart widget.
         */
        chart: import("chart.js");
        /**
         * The 2D rendering context of the canvas used by this chart.
         */
        ctx: CanvasRenderingContext2D;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Setups the event listeners required by this widget when an item (data point) in the chart is selected.
         */
        private bindItemSelect(): void;
        /**
         * Cleans up deferred render tasks. When you extend this class and override this method, make sure to call
         * `super`.
         * @override
         */
        override destroy(): void;
        /**
         * Return this chart as an image with a data source URL (`<img src="data:url" />`)
         * @return The content of this chart as an HTML IMAGE.
         */
        exportAsImage(): HTMLImageElement;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Send this chart to the printer.
         */
        print(): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Chart|Base chart widget}. You
     * can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ChartCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * The configuration for the
         * [chart.js](https://www.chartjs.org/docs/latest/) chart. It can be modified within the extender function set for this
         * chart widget.
         */
        config: import("chart.js").ChartConfiguration;
        /**
         * Extender function allows access to the underlying
         * [chart.js](https://www.chartjs.org/docs/latest/) API.
         */
        extender: PrimeFaces.widget.Chart.ChartExtender;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces LineChart Widget__
     *
     * A line chart is a way of plotting data points on a line. Often, it is used to show trend data, or the comparison of
     * two data sets.
     * @typeparam TCfg Defaults to `LineChartCfg`. Type of the configuration object for this widget.
     */
    export class LineChart<TCfg extends LineChartCfg = LineChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  LineChart| LineChart widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface LineChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces BarChart Widget__
     *
     * A bar chart provides a way of showing data values represented as vertical
     * bars. It is sometimes used to show trend data, and the comparison of multiple
     * data sets side by side.
     * @typeparam TCfg Defaults to `BarChartCfg`. Type of the configuration object for this widget.
     */
    export class BarChart<TCfg extends BarChartCfg = BarChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  BarChart| BarChart widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface BarChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces PieChart Widget__
     *
     * Pie chart is divided into segments, the arc of each segment shows the proportional value of each piece of data.
     * @typeparam TCfg Defaults to `PieChartCfg`. Type of the configuration object for this widget.
     */
    export class PieChart<TCfg extends PieChartCfg = PieChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  PieChart| PieChart widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface PieChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DonutChart Widget__
     *
     * A Donut Chart is a variation of a Pie Chart but with a space in the center.
     * @typeparam TCfg Defaults to `DonutChartCfg`. Type of the configuration object for this widget.
     */
    export class DonutChart<TCfg extends DonutChartCfg = DonutChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DonutChart| DonutChart widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DonutChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces PolarAreaChart Widget__
     *
     * Polar area charts are similar to pie charts, but each segment has the same angle - the radius of the segment differs
     * depending on the value.
     * @typeparam TCfg Defaults to `PolarAreaChartCfg`. Type of the configuration object for this widget.
     */
    export class PolarAreaChart<TCfg extends PolarAreaChartCfg = PolarAreaChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the
     * {@link  PolarAreaChart| PolarAreaChart widget}. You can access this configuration via
     * {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this configuration is usually meant to be
     * read-only and should not be modified.
     */
    export interface PolarAreaChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces RadarChart Widget__
     *
     * A radar chart is a way of showing multiple data points and the variation between them.
     * @typeparam TCfg Defaults to `RadarChartCfg`. Type of the configuration object for this widget.
     */
    export class RadarChart<TCfg extends RadarChartCfg = RadarChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  RadarChart| RadarChart widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface RadarChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces BubbleChart Widget__
     *
     * A bubble chart is used to display three dimensions of data at the same time. The location of the bubble is determined
     * by the first two dimensions and the corresponding horizontal and vertical axes. The third dimension is represented by
     * the size of the individual bubbles.
     * @typeparam TCfg Defaults to `BubbleChartCfg`. Type of the configuration object for this widget.
     */
    export class BubbleChart<TCfg extends BubbleChartCfg = BubbleChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  BubbleChart| BubbleChart widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface BubbleChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces ScatterChart Widget__
     * @typeparam TCfg Defaults to `ScatterChartCfg`. Type of the configuration object for this widget.
     */
    export class ScatterChart<TCfg extends ScatterChartCfg = ScatterChartCfg> extends PrimeFaces.widget.Chart<TCfg> {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the
     * {@link  ScatterChart| ScatterChart widget}. You can access this configuration via
     * {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this configuration is usually meant to be
     * read-only and should not be modified.
     */
    export interface ScatterChartCfg extends PrimeFaces.widget.ChartCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Chip Widget__
     *
     * Chip represents entities using icons, labels and images.
     * @typeparam TCfg Defaults to `ChipCfg`. Type of the configuration object for this widget.
     */
    export class Chip<TCfg extends ChipCfg = ChipCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * DOM element of the icon for closing this chip, when this chip is closable (an `x` by
         * default).
         */
        removeIcon: JQuery;
        /**
         * Sets up all event listeners required for this widget.
         */
        private bindEvents(): void;
        /**
         * Closes the chip.
         */
        private close(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Chip| Chip widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ChipCfg extends PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Chips Widget__
     *
     * Chips is used to enter multiple values on an inputfield.
     * @typeparam TCfg Defaults to `ChipsCfg`. Type of the configuration object for this widget.
     */
    export class Chips<TCfg extends ChipsCfg = ChipsCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Is this component wrapped in a float label.
         */
        hasFloatLabel: boolean;
        /**
         * DOM element of the hidden INPUT field with the current value.
         */
        hinput: JQuery;
        /**
         * DOM element of the visible INPUT field.
         */
        input: JQuery;
        /**
         * DOM element of the container for the visible INPUT.
         */
        inputContainer: JQuery;
        /**
         * DOM element of the container of the items (chips).
         */
        itemContainer: JQuery;
        /**
         * Placeholder for the input field.
         */
        placeholder: string;
        /**
         * Adds a new item (chip) to the list of currently displayed items.
         * @param value Value of the chip to add.
         * @param refocus `true` to put focus back on the INPUT again after the chip was added, or `false`
         * otherwise.
         */
        addItem(value: string, refocus?: boolean): void;
        /**
         * Sets up all event listeners required for this widget.
         */
        private bindEvents(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Triggers the behaviors and event listeners for when an item (chip) was selected.
         * @param itemValue Value of the selected item.
         */
        private invokeItemSelectBehavior(itemValue: string): void;
        /**
         * Triggers the behaviors and event listeners for when an item (chip) was unselected.
         * @param itemValue Value of the unselected item.
         */
        private invokeItemUnselectBehavior(itemValue: string): void;
        /**
         * Deletes the currently editing input value and refocus the input box if necessary.
         * @param refocus `true` to put focus back on the INPUT again after the chip was added, or `false`
         * otherwise.
         */
        private refocus(refocus?: boolean): void;
        /**
         * Removes an item (chip) from the list of currently displayed items.
         * @param item An item (LI element) that should be removed.
         * @param silent Flag indicating whether to animate the removal and fire the AJAX behavior.
         */
        removeItem(item: JQuery, silent?: boolean): void;
        /**
         * Converts the current list into a separator delimited list for mass editing while keeping original
         * order of the items or closes the editor turning the values back into chips.
         */
        toggleEditor(): void;
        /**
         * Handles floating label CSS if wrapped in a floating label.
         */
        private updateFloatLabel(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Chips| Chips widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ChipsCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether to add an item when the input loses focus.
         */
        addOnBlur: boolean;
        /**
         * Maximum number of entries allowed.
         */
        max: number;
        /**
         * Separator character to allow multiple values such if a list is pasted into the input.
         * Default is `,`.
         */
        separator: string;
        /**
         * Prevent duplicate entries from being added.
         */
        unique: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SimpleDateFormat widget__
     *
     * Code ported from Tim Down's http://www.timdown.co.uk/code/simpledateformat.php
     *
     * Helper widget for working with `Date`s and date formats.
     * @typeparam TCfg Defaults to `PrimeFaces.widget.SimpleDateFormatCfg`. Type of the configuration
     * object for this widget.
     */
    export class SimpleDateFormat<TCfg extends PrimeFaces.widget.SimpleDateFormatCfg = PrimeFaces.widget.SimpleDateFormatCfg> {
        /**
         * The configuration of this widget
         * instance. Please note that no property is guaranteed to be present, you should always check for `undefined` before
         * accessing a property. This is partly because the value of a property is not transmitted from the server to the client
         * when it equals the default.
         */
        cfg: PrimeFaces.PartialWidgetCfg<PrimeFaces.widget.SimpleDateFormatCfg>;
        /**
         * Format sthe given given according to the pattern of the current widget configuration.
         * @param date A date to format
         * @return The given date as a formatted string.
         */
        format(date: Date): string;
        /**
         * Computes the ordinal index of the given day in the given year.
         * @param date A day to check.
         * @return The ordinal index of the given day relative to the beginning of the year, starting at `1`.
         */
        getDayInYear(date: Date): number;
        /**
         * Computes the difference between the two given dates.
         * @param date1 First input date
         * @param date2 Second input date
         * @return Time in milliseconds between the two dates (`date1-date2`).
         */
        getDifference(date1: Date, date2: Date): number;
        /**
         * Finds the currently configured value of how many days a week must have at least to be considered a "full" week.
         * Weeks with less that that number of days are disregarded in `getWeekInMonth` and `getWeekInYear`.
         * @param days Unused.
         * @return The minimal number of days a week is allowed to have to be considered a "full" week.
         */
        getMinimalDaysInFirstWeek(days: unknown): number;
        /**
         * Finds closest Sunday preceding the given date. If the date is already a Sunday, that day is returned.
         * @param date Input date.
         * @return The date at midnight of the first Sunday before the given date. If the given date is already a
         * Sunday, that day is returned.
         */
        getPreviousSunday(date: Date): Date;
        /**
         * Finds the difference in milliseconds between the two given date (`date1-date2`).
         * @param date1 First input date
         * @param date2 Second input date
         * @return The numer of milliseconds between the two given date (`date1-date2`).
         */
        getTimeSince(date1: Date, date2: Date): number;
        /**
         * Converts the given date to UTC time, that is, the number of milliseconds between midnight, January 1, 1970
         * Universal Coordinated Time (UTC) (or GMT) and the given date.
         * @param date Date to convert to UTC.
         * @return The given date, converted to UTC time.
         */
        getUTCTime(date: Date): number;
        /**
         * Computes the ordinal index of the week of the month of the given date.
         * @param date Date with a month to check.
         * @param minimalDaysInFirstWeek Minimal number of days the first week of the month is allowed to have. If
         * the first week contains less days, the returned output is decremented by one (if you do not want to count, say,
         * 2 days, as week).
         * @return The week of the month of the given date, starting at `0`.
         */
        getWeekInMonth(date: Date, minimalDaysInFirstWeek: number): number;
        /**
         * Computes the ordinal index of the week of the year of the given date.
         * @param date Date to check.
         * @param minimalDaysInFirstWeek Minimal number of days the first week of the year is allowed to have. If
         * the first week contains less days, the returned output is decremented by one (if you do not want to count, say,
         * 2 days, as week).
         * @return The week of the year of the given date, starting at `0`.
         */
        getWeekInYear(date: Date, minimalDaysInFirstWeek: number): number;
        /**
         * A widget class should not have an explicit constructor. Instead, this initialize method is called after the widget
         * was created. You can use this method to perform any initialization that is required. For widgets that need to create
         * custom HTML on the client-side this is also the place where you should call your render method.
         * @param cfg The widget configuration to be used for this widget instance. This widget
         * configuration is usually created on the server by the `javax.faces.render.Renderer` for this component.
         */
        init(cfg: Partial<TCfg>): void;
        /**
         * Checks whether the first given date lies before the second given date.
         * @param date1 First input date
         * @param date2 Second input date
         * @return `true` if `date1` lies before `date2`, or `false` otherwise.
         */
        isBefore(date1: Date, date2: Date): boolean;
        /**
         * Creates a new date object that represents midnighht of the given year, month, and day.
         * @param year A year to set. `0` repesents the year `1900`, `100` the year `2000`.
         * @param month A month (of the year) to set. `0` is January, `11` is `December`.
         * @param day A day (of the month) to set, in the range `1...31`.
         * @return A date for the given year, month, and day at at midnight.
         */
        newDateAtMidnight(year: number, month: number, day: number): Date;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the
     * {@link  SimpleDateFormat| SimpleDateFormat widget}. You can access this configuration via
     * {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this configuration is usually meant to be
     * read-only and should not be modified.
     */
    export interface SimpleDateFormatCfg {
        /**
         * Localized day names (`Monday`, `Tuesday` etc.)
         */
        dayNames: string[];
        /**
         * The client-side ID of this widget, with all parent naming containers, such as
         * `myForm:myWidget`. This is also the ID of the container HTML element for this widget. In case the widget needs
         * multiple container elements (such as {@link Paginator}), this may also be an array if IDs.
         */
        id: string | string[];
        /**
         * The locale for formatting dates.
         */
        locale: string;
        /**
         * Minimal number of days a week is allowed to have to be considered a "full"
         * week, used by `getWeekInMonth` in `getWeekInYear`.
         */
        minimalDaysInFirstWeek: number;
        /**
         * Localized month names (`January`, `February` etc.)
         */
        monthNames: string[];
        /**
         * A regex for splitting a date format into its components.
         */
        regex: RegExp;
        /**
         * Object with the different keywords used by the date format.
         */
        types: Record<string, number>;
        /**
         * The name of the widget variables of this widget. The widget variable can be used to
         * access a widget instance by calling `PF('myWidgetVar')`.
         */
        widgetVar: string;
    }
}
declare namespace PrimeFaces.widget.Clock {
    /**
     * Display mode for the clock widget. `analog`
     * displays an analog clock, `digital` a digitial clock.
     */
    export type DisplayMode = "analog" | "digital";
}
declare namespace PrimeFaces.widget.Clock {
    /**
     * Indicates which time the clock widget uses. `client`
     * uses the time from the client (browser), `server` uses the time from the server.
     */
    export type TimeMode = "client" | "server";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Clock Widget__
     *
     * Clock displays server or client datetime live.
     * @typeparam TCfg Defaults to `ClockCfg`. Type of the configuration object for this widget.
     */
    export class Clock<TCfg extends ClockCfg = ClockCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The canvas for the analog clock.
         */
        canvas: import("raphael").RaphaelPaper;
        /**
         * The drawn element for the clock outline.
         */
        clock: import("raphael").RaphaelElement;
        /**
         * The currently displayed time.
         */
        current: Date;
        /**
         * Calculated sizes for the analog clock elements.
         */
        dimensions: PrimeFaces.widget.Clock.Dimensions;
        /**
         * The drawn element for the hour hand.
         */
        hour_hand: import("raphael").RaphaelElement;
        /**
         * The drawn elements for the hour signs (1-12).
         */
        hour_sign: import("raphael").RaphaelElement[];
        /**
         * The set-interval timer ID for the ticking of the clock.
         */
        interval: number;
        /**
         * The drawn element for the minute hand.
         */
        minute_hand: import("raphael").RaphaelElement;
        /**
         * The drawn element for the pin at the center of the clock.
         */
        pin: import("raphael").RaphaelElement;
        /**
         * The drawn element for the second hand.
         */
        second_hand: import("raphael").RaphaelElement;
        /**
         * Draws this clock according the the current widget configuation.
         */
        private draw(): void;
        /**
         * Draws the clock hands for the analog clock.
         */
        private draw_hands(): void;
        /**
         * Draws the hour marks for the analog clock.
         */
        private draw_hour_signs(): void;
        /**
         * Computes the width of the individual elements of the analog clock for the given target width.
         * @param size Target width of the clock in pixels
         * @return Calculated sizes for the analog clock elements.
         */
        private getDimensions(size: number): PrimeFaces.widget.Clock.Dimensions;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Checks whether this clock is displayed as an analog or digital clock.
         * @return `true` if this clock is displayed as an analog clock, or `false` if it is displayed in an
         * INPUT field.
         */
        isAnalogClock(): boolean;
        /**
         * Checks whether the time of the client is used for this clock.
         * @return `true` if the time of the client is used, or `false` if the time of the server is used.
         */
        isClient(): boolean;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Starts this clock if it is not already running.
         */
        start(): void;
        /**
         * Stops this clock it is currently running.
         */
        stop(): void;
        /**
         * Synchronizes this clock so that it shows the current time. This will trigger an AJAX update of this component.
         */
        sync(): void;
        /**
         * Called each click of the clock, animates the clock hands.
         */
        private update(): void;
        /**
         * Called after a tick of the clock, updates the visual display of this clock.
         */
        private updateOutput(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Clock| Clock widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ClockCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * When `mode` is set to `server`: `true` to automatically sync the time with the server
         * according to the specified `syncInterval`, or `false` otherwise.
         */
        autoSync: boolean;
        /**
         * Whether the clock is displayed as an analog or digital
         * clock.
         */
        displayMode: PrimeFaces.widget.Clock.DisplayMode;
        /**
         * Locale for the clock, determines the time format.
         */
        locale: string;
        /**
         * Whether the clock uses the time of the browser or the time from the
         * server.
         */
        mode: PrimeFaces.widget.Clock.TimeMode;
        /**
         * Datetime format.
         */
        pattern: string;
        /**
         * Defines the sync in ms interval in when `autoSync` is set to `true`.
         */
        syncInterval: number;
        /**
         * The initial time value for the clock to display.
         */
        value: string;
    }
}
declare namespace PrimeFaces.widget.Clock {
    /**
     * Computed dimensions for the individual parts of the analog
     * clock, all in pixels.
     */
    export interface Dimensions {
        /**
         * Width of the clock face in pixels.
         */
        clock_width: number;
        /**
         * Half width of the clock element in pixels.
         */
        half: number;
        /**
         * Radial distance in pixels from the circumference of the circle
         * where the hour hand starts.
         */
        hour_hand_start_position: number;
        /**
         * Stroke width in pixels of the hour hand.
         */
        hour_hand_stroke_width: number;
        /**
         * Distance in pixels from the center of the circle where the hour mark
         * ends.
         */
        hour_sign_max_size: number;
        /**
         * Distance in pixels from the center of the circle where the hour mark
         * starts.
         */
        hour_sign_min_size: number;
        /**
         * Radial distance in pixels from the circumference of the circle
         * where the minute hand starts.
         */
        minute_hand_start_position: number;
        /**
         * Stroke width in pixels of the minute hand.
         */
        minute_hand_stroke_width: number;
        /**
         * Radius in pixels of the pin at the center of the clock face.
         */
        pin_width: number;
        /**
         * Radial distance in pixels from the circumference of the circle
         * where the seconds hand starts.
         */
        second_hand_start_position: number;
        /**
         * Stroke width in pixels of the seconds hand.
         */
        second_hand_stroke_width: number;
        /**
         * Width of the clock element in pixels.
         */
        size: number;
    }
}
declare namespace PrimeFaces.widget.ColorPicker {
    /**
     * Display mode of a color picker. `inline`
     * renders the color picker within the normal content flow, `popup` creates an overlay that is displayed when the user
     * clicks on the color.
     */
    export type DisplayMode = "inline" | "popup";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Color Picker Widget__
     *
     * ColorPicker is an input component with a color palette.
     *
     * This uses Coloris written in vanilla ES6. To interact with the color picker, you can use the following code.
     *
     * ```javascript
     * // Assuming the widget variable of the color picker was set to "myColorPicker"
     * const colorPicker = PF("myColorPicker");
     *
     * // Brings up the color picker (if "mode" was set to "popup")
     * colorPicker.show();
     *
     * // Hides up the color picker (if "mode" was set to "popup")
     * colorPicker.hide();
     *
     * // Sets the currently selected color to "green"
     * colorPicker.setColor("00FF00");
     * ```
     * @typeparam TCfg Defaults to `ColorPickerCfg`. Type of the configuration object for this widget.
     */
    export class ColorPicker<TCfg extends ColorPickerCfg = ColorPickerCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Is this component wrapped in a float label.
         */
        hasFloatLabel: boolean;
        /**
         * DOM element of the INPUT element
         */
        input: JQuery;
        /**
         * True if popup mode, else inline mode
         */
        popup: boolean;
        /**
         * Clean up this widget and remove events from the DOM.
         */
        private _cleanup(): void;
        /**
         * Sets up the event listeners required by this widget for inline mode.
         */
        private bindInlineCallbacks(): void;
        /**
         * Sets up the event listeners required by this widget.
         */
        private bindInputCallbacks(): void;
        /**
         * Configures a single ARIA label from PF locale to Coloris a11y.
         * @param label the PF label to lookup in locale.js
         * @param a11y the a11y JSON object for Coloris
         * @param property the JSON property to set in a11y
         */
        private configureAriaLabel(label: string, a11y: {
            key: string;
        }, property: string): void;
        /**
         * Localizes the ARIA accessibility labels for the color picker.
         */
        private configureLocale(): void;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         * @override
         */
        override destroy(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Gets the current color
         * @return the current color
         */
        getColor(): string;
        /**
         * Close the dialog and revert the color to its original value.
         * @param revert true to revert the color to its original value
         */
        hide(revert: boolean | undefined): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Sets the current color
         * @param color the color to set
         */
        setColor(color: string): void;
        /**
         * Sets up support for using the overlay color picker within an overlay dialog.
         */
        private setupDialogSupport(): void;
        /**
         * Only one instance of Coloris is allowed so ensure it only loads defaults once.
         */
        private setupGlobalDefaults(): void;
        /**
         * Configure the color picker for popup mode.
         */
        private setupPopup(): void;
        /**
         * Shows the popup panel.
         */
        show(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Coloris}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ColorPickerCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * The instance of for configuring in popup mode
         */
        instance: string;
        /**
         * Whether the color picker is displayed inline or as a popup.
         */
        mode: PrimeFaces.widget.ColorPicker.DisplayMode;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces ColumnToggler Widget__
     *
     * ColumnToggler is a helper component for the data table to toggle visibility of columns.
     * @typeparam TCfg Defaults to `ColumnTogglerCfg`. Type of the configuration object for this widget.
     */
    export class ColumnToggler<TCfg extends ColumnTogglerCfg = ColumnTogglerCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * DOM element of the close button for closing the overlay with the available columns.
         */
        closer?: JQuery;
        /**
         * DOM elements for the `TH` columns of the data table.
         */
        columns?: JQuery;
        /**
         * The number of frozen column of table to which this column toggle is
         * attached.
         */
        frozenColumnCount?: number;
        /**
         * Whether the table to which this column toggle is attached has got any frozen columns.
         */
        hasFrozenColumn: boolean;
        /**
         * Whether any prioritized columns exist. Used for responsive mode.
         */
        hasPriorityColumns?: boolean;
        /**
         * Whether the table to which this column toggle is attached has got a sticky header.
         */
        hasStickyHeader: boolean;
        /**
         * DOM elements for the `UL` items in the overlay with the available columns.
         */
        itemContainer?: JQuery;
        /**
         * Overlay column toggler panel with the available columns.
         */
        panel?: JQuery;
        /**
         * DOM element for the select all checkbox.
         */
        selectAllCheckbox?: JQuery;
        /**
         * Table to which this column toggle is attached.
         */
        table: JQuery;
        /**
         * ID of the table to which this column toggle is attached.
         */
        tableId: string;
        /**
         * The DOM element for the table body of the table to which this column toggle is attached.
         */
        tbody: JQuery;
        /**
         * The DOM element for the table foot of the table to which this column toggle is attached.
         */
        tfoot: JQuery;
        /**
         * The DOM element for the table head of the table to which this column toggle is attached.
         */
        thead: JQuery;
        /**
         * IDs of the columns that are selected.
         */
        togglerState?: string[];
        /**
         * DOM element of the hidden input that contains the columns that are
         * selected. Used to preserve that state between AJAX updates.
         */
        togglerStateHolder?: JQuery;
        /**
         * Button that toggles this column toggler.
         */
        trigger: JQuery;
        /**
         * Whether this column toggler is currently displayed.
         */
        visible: boolean;
        /**
         * Whether the width of the overlay panel with the available columns was
         * aligned with the width of the toggler.
         */
        widthAligned?: boolean;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Aligns the overlay panel of this column toggler according to the current widget configuration.
         */
        alignPanel(): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Sets up the event listners for keyboard interaction.
         */
        private bindKeyEvents(): void;
        /**
         * Computes the required `colspan` for the rows.
         * @return The calculated `colspan` for the rows.
         */
        private calculateColspan(): number;
        /**
         * Selects or unselect a column of this column toggler. Also shows or hides the corresponding colum of the table
         * to which this column toggler is attached.
         * @param column A column element (`LI`) of this column toggler.
         * @param isHidden `true` to unselect the column and hide the corresponding table column, or `true`
         * otherwise.
         */
        private changeTogglerState(column: JQuery, isHidden: boolean): void;
        /**
         * Checks the given checkbox for a column, so that the column is now selected. Also display the column of the table
         * to which this column toggler is attached.
         * @param chkbox Checkbox (`.ui-chkbox-box`) of a column of this column toggler.
         */
        check(chkbox: JQuery): void;
        /**
         * Checks all columns to enable all.
         */
        checkAll(): void;
        /**
         * Triggers the events listeners and behaviors when the popup is closed.
         */
        private fireCloseEvent(): void;
        /**
         * Triggers the events listeners and behaviors when a column was selected or unselected.
         * @param visible `true` if the column was selected, `false` otherwise.
         * @param index Index of the toggled column.
         */
        private fireToggleEvent(visible: boolean, index: number): void;
        /**
         * Hides this column toggler.
         */
        hide(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * This render method to check whether the widget container is visible. Do not override this method, or the
         * deferred widget functionality may not work properly anymore.
         * @override
         * @return `true` if the widget container is visible, `false` or
         * `undefined` otherwise.
         */
        override render(): void;
        /**
         * Brings up this column toggler so that the user can which column to hide or show.
         */
        show(): void;
        /**
         * Checks or unchecks the given checkbox for a column, depending on whether it is currently selected. Also shows or
         * hides  the column of the table to which this column toggler is attached.
         * @param chkbox Checkbox (`.ui-chkbox-box`) of a column of this column toggler.
         */
        toggle(chkbox: JQuery): void;
        /**
         * Toggles selecting or deselecting all columns.
         */
        toggleAll(): void;
        /**
         * Unchecks the given checkbox for a column, so that the column is now not selected. Also hides the column of the
         * table to which this column toggler is attached.
         * @param chkbox Checkbox (`.ui-chkbox-box`) of a column of this column toggler.
         */
        uncheck(chkbox: JQuery): void;
        /**
         * Unchecks all columns to disable all.
         */
        uncheckAll(): void;
        /**
         * Updates the colspan attributes of the target table of this column toggler. Called after a column was selected or
         * unselected, which resulted in a column of the data table to be shown or hidden.
         */
        private updateColspan(): void;
        /**
         * Updates the `colspan` attribute fo the columns of the given row.
         * @param row A row to update.
         * @param colspanValue New value for the `colspan` attribute.
         */
        private updateRowColspan(row: JQuery, colspanValue: string): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the
     * {@link  ColumnToggler| ColumnToggler widget}. You can access this configuration via
     * {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this configuration is usually meant to be
     * read-only and should not be modified.
     */
    export interface ColumnTogglerCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * ID of the component (table) to which this column toggler is attached.
         */
        datasource: string;
        /**
         * ID of the button that toggles this column toggler.
         */
        trigger: string;
    }
}
declare namespace PrimeFaces.widget.ConfirmPopup {
    /**
     * Callback invoked after the popup is hidden.
     */
    export type HideCallback = (this: Window) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces ConfirmPopup Widget__
     *
     * ConfirmPopup displays a confirmation overlay displayed relatively to its target.
     * @typeparam TCfg Defaults to `ConfirmPopupCfg`. Type of the configuration object for this widget.
     */
    export class ConfirmPopup<TCfg extends ConfirmPopupCfg = ConfirmPopupCfg> extends PrimeFaces.widget.DynamicOverlayWidget<TCfg> {
        /**
         * The DOM element for the content of the confirm popup.
         */
        content: JQuery;
        /**
         * Element that was focused before the dialog was opened.
         */
        focusedElementBeforeDialogOpened: HTMLElement;
        /**
         * Unbind callback for the hide overlay handler.
         */
        hideOverlayHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM element for the message icon.
         */
        icon: JQuery;
        /**
         * DOM element of the confirmation message displayed in this confirm popup.
         */
        message: JQuery;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * Handler for CSS transitions used by this widget.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * Aligns the popup so that it is shown at the correct position.
         * @param target Jquery selector that is the target of this popup
         */
        private align(target?: JQuery): void;
        /**
         * Applies focus to the first focusable element of the content in the popup.
         */
        applyFocus(): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        protected bindEvents(): void;
        /**
         * Sets up all panel event listeners
         * @param target Selector or DOM element of the target component that triggers this popup.
         */
        private bindPanelEvents(target?: string | JQuery): void;
        /**
         * Hides the popup.
         * @param callback Callback that is invoked after this popup was closed.
         */
        hide(callback: PrimeFaces.widget.ConfirmPopup.HideCallback): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg the widget configuraton
         *
         * (from super type BaseWidget) The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Checks whether this popup is opened and visible.
         * @return `true` if this popup is currently being shown, `false` otherwise.
         */
        isVisible(): boolean;
        /**
         * Puts focus on the element that opened this dialog.
         * @param delay how long to delay before focusing
         */
        protected returnFocus(delay?: number | undefined): void;
        /**
         * Makes the popup visible.
         * @param target Selector or DOM element of the target component that triggers this popup.
         */
        show(target?: string | JQuery): void;
        /**
         * Shows the given message in this confirmation popup.
         * @param msg Message to show.
         */
        showMessage(msg: Partial<PrimeFaces.widget.ConfirmPopup.ConfirmPopupMessage>): void;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  ConfirmPopup| ConfirmPopup widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ConfirmPopupCfg extends PrimeFaces.widget.DynamicOverlayWidgetCfg {
        /**
         * The search expression for the element to which the overlay panel should be
         * appended.
         */
        appendTo: string | null;
        /**
         * When set `true`, clicking outside of the popup hides the overlay.
         */
        dismissable: boolean;
        /**
         * When enabled, confirmPopup becomes a shared for other components that require confirmation.
         */
        global: boolean;
        /**
         * Event on target to hide the popup.
         */
        hideEvent: string;
        /**
         * Event on target to show the popup.
         */
        showEvent: string;
    }
}
declare namespace PrimeFaces.widget.ConfirmPopup {
    /**
     * Interface for the message that
     * is shown in the confirm popup.
     */
    export interface ConfirmPopupMessage {
        /**
         * If `true`, the message is escaped for HTML. If `false`, the message is
         * interpreted as an HTML string.
         */
        escape: boolean;
        /**
         * Main content of the popup message.
         */
        message: string;
        /**
         * A JavaScript code snippet that is be evaluated before the message is
         * shown.
         */
        onShow: string;
    }
}
declare namespace PrimeFaces {
    /**
     * A shortcut for `PrimeFaces.ajax.Request.handle(cfg, ext)`, with shorter option names. Sends an AJAX request to
     * the server and processes the response. You can use this method if you need more fine-grained control over which
     * components you want to update or process, or if you need to change some other AJAX options.
     * @param cfg Configuration for the AJAX request, with shorthand
     * options. The individual options are documented in `PrimeFaces.ajax.Configuration`.
     * @param ext Optional extender with additional options that
     * overwrite the options given in `cfg`.
     * @return A promise that resolves once the AJAX requests is done. Use this
     * to run custom JavaScript logic. When the AJAX request succeeds, the promise is fulfilled. Otherwise, when the
     * AJAX request fails, the promise is rejected. If the promise is rejected, the rejection handler receives an object
     * of type {@link PrimeFaces.ajax.FailedRequestData}.
     */
    export function ab(cfg: Partial<PrimeFaces.ajax.ShorthandConfiguration>, ext?: Partial<PrimeFaces.ajax.ConfigurationExtender>): Promise<PrimeFaces.ajax.ResponseData>;
}
/**
 * The object with functionality related to sending and receiving AJAX requests that are made by PrimeFaces. Each
 * request receives an XML response, which consists of one or multiple actions that are to be performed. This
 * includes creating new DOM elements, deleting or updating existing elements, or executing some JavaScript.
 */
declare namespace PrimeFaces.ajax {
    /**
     * Parameter shortcut mapping for the method `PrimeFaces.ab`.
     */
    export let CFG_SHORTCUTS: Record<string, string>;
    /**
     * This object contains functionality related to queuing AJAX requests to ensure that they are (a) sent in the
     * proper order and (b) that each response is processed in the same order as the requests were sent.
     */
    export const Queue: PrimeFaces.ajax.Queue;
    /**
     * Name for the ID of a resource entry, used in AJAX requests.
     */
    export const RESOURCE: string;
    /**
     * The interface for the object containing low-level functionality related to sending AJAX requests.
     */
    export const Request: PrimeFaces.ajax.Request;
    /**
     * The interface for the object containing low-level functionality related to handling AJAX responses. Note that
     * the different types of AJAX actions are handles by the `PrimeFaces.ResponseProcessor`.
     */
    export const Response: PrimeFaces.ajax.Response;
    /**
     * The interface for the object containing low-level functionality related to processing the different types
     * of actions from AJAX responses.
     */
    export const ResponseProcessor: PrimeFaces.ajax.ResponseProcessor;
    /**
     * This object contains utility methods for AJAX requests, primarily used internally.
     */
    export const Utils: PrimeFaces.ajax.Utils;
    /**
     * Name for the ID of the BODY element, used in AJAX requests.
     */
    export const VIEW_BODY: string;
    /**
     * Name for the ID of the HEAD element, used in AJAX requests.
     */
    export const VIEW_HEAD: string;
    /**
     * Minimum number of milliseconds to show inline Ajax load animations.
     */
    export let minLoadAnimation: number;
    /**
     * Only available for backward compatibility, do not use in new code.
     * @param cfg Configuration for the AJAX request to send, such as
     * the HTTP method, the URL, and the content of the request.
     * @param ext Optional extender with additional options
     * that overwrite the options given in `cfg`.
     * @return Always returns `undefined`.
     * @deprecated Use `PrimeFaces.ajax.Request.handle` instead.
     */
    export function AjaxRequest(cfg: Partial<PrimeFaces.ajax.Configuration>, ext?: Partial<PrimeFaces.ajax.ConfigurationExtender>): undefined;
}
declare namespace PrimeFaces.ajax {
    /**
     * The interface for the object containing functionality related to queuing
     * AJAX requests. The queue ensures that requests are (a) sent in the order as they were issued, and (b) that
     * each response is processed in the same order as the requests were sent.
     */
    export interface Queue {
        /**
         * A map between the source ID and  the timeout IDs (as returned by `setTimeout`). Used for AJAX requests
         * with a specified delay (such as remote commands that have a delay set).
         */
        delays: Record<string, number>;
        /**
         * A list of requests that are waiting to be sent.
         */
        requests: Partial<PrimeFaces.ajax.Configuration>[];
        /**
         * A list of sent AJAX requests, i.e. HTTP requests that were already started. This is used, for example, to
         * abort requests that were sent already when that becomes necessary.
         */
        xhrs: PrimeFaces.ajax.pfXHR[];
        /**
         * Aborts all requests that were already sent, but have not yet received an answer from the server. Also
         * removes all requests that are waiting in the queue and have not been sent yet.
         */
        abortAll(): void;
        /**
         * Adds a newly sent XHR request to the list of sent requests (`PrimeFaces.ajax.xhrs`).
         * @param xhr XHR request to add.
         */
        addXHR(xhr: PrimeFaces.ajax.pfXHR): void;
        /**
         * Checks whether this queue contains any scheduled AJAX requests.
         * @return `true` if this queue contains no scheduled requests, `false` otherwise.
         */
        isEmpty(): boolean;
        /**
         * Offers an AJAX request to this queue. The request is sent once all other requests in this queue have
         * been sent. If a delay is set on the request configuration, the request is not sent before the specified
         * delay has elapsed.
         * @param request The request to send.
         */
        offer(request: Partial<PrimeFaces.ajax.Configuration>): void;
        /**
         * Returns the request that is scheduled to be sent next, but does not modify the queue in any way.
         * @return The topmost request in this queue that is to be sent next,
         * or `null` when this queue is empty.
         */
        peek(): Partial<PrimeFaces.ajax.Configuration> | null;
        /**
         * Removes the topmost request (the requests that was just sent) from this queue; and starts the second
         * topmost request.
         * @return The topmost request in this queue, or `null` if this queue
         * is empty.
         */
        poll(): Partial<PrimeFaces.ajax.Configuration> | null;
        /**
         * Removes an XHR request from the list of sent requests (`PrimeFaces.ajax.xhrs`). Usually called once the
         * AJAX request is done, having resulted in either a success or an error.
         * @param xhr XHR request to remove.
         */
        removeXHR(xhr: PrimeFaces.ajax.pfXHR): void;
    }
}
declare namespace PrimeFaces.ajax {
    /**
     * The interface for the object containing functionality related to
     * sending AJAX requests.
     */
    export interface Request {
        /**
         * Appends a request parameter to the given list of parameters.
         * Optionally add a prefix to the name, this is used for portlet namespacing.
         * @param formData the form data to add to the form.
         * @param name Name of the new parameter to add.
         * @param value Value of the parameter to add.
         * @param parameterPrefix Optional prefix that is added in front of the name.
         */
        addFormData(formData: FormData, name: string, value: string | Blob, parameterPrefix?: string): void;
        /**
         * Adds a new request parameter to the given FormData. The value of the parameter is taken from the input
         * element of the given form. The input element must have the same name as the name of the parameter to add.
         * Optionally add a prefix to the name, which used for portlet namespacing.
         * @param formData The FormData.
         * @param name Name of the new parameter to add
         * @param form An HTML FORM element that contains an INPUT element with the given name.
         * @param parameterPrefix Optional prefix that is added in front of the name.
         */
        addFormDataFromInput(formData: FormData, name: string, form: JQuery, parameterPrefix?: string): void;
        /**
         * Appends a request parameter to the given list of parameters.
         * Optionally add a prefix to the name, this is used for portlet namespacing.
         * @typeparam TValue Defaults to `unknown`. Type of the parameter value.
         * @param params List of parameters to which a new
         * parameter is added.
         * @param name Name of the new parameter to add.
         * @param value Value of the parameter to add.
         * @param parameterPrefix Optional prefix that is added in front of the name.
         */
        addParam<TValue = unknown>(params: PrimeFaces.ajax.RequestParameter<string, TValue>[], name: string, value: TValue, parameterPrefix?: string): void;
        /**
         * Adds a new request parameter to the given list. The value of the parameter is taken from the input
         * element of the given form. The input element must have the same name as the name of the parameter to add.
         * Optionally add a prefix to the name, which used for portlet namespacing.
         * @param params List of request parameters to the new
         * parameter is added.
         * @param name Name of the new parameter to add
         * @param form An HTML FORM element that contains an INPUT element with the given name.
         * @param parameterPrefix Optional prefix that is added in front of the name.
         */
        addParamFromInput(params: PrimeFaces.ajax.RequestParameter[], name: string, form: JQuery, parameterPrefix?: string): void;
        /**
         * Adds a list of callback parameters to the given list. Optionally prepends a prefix to the name of each
         * added parameter.
         * @typeparam TValue Defaults to `unknown`. Type of the parameter values.
         * @param params List of callback parameters to which
         * parameters are added.
         * @param paramsToAdd List of callback parameters to
         * add.
         * @param parameterPrefix Optional prefix that is added in front of the name of the added
         * callback parameters.
         */
        addParams<TValue = unknown>(params: PrimeFaces.ajax.RequestParameter<string, TValue>[], paramsToAdd: PrimeFaces.ajax.RequestParameter<string, TValue>[], parameterPrefix?: string): void;
        /**
         * Creates a new array with all parameters from the second array that are not in the first array. That is,
         * removes all parameters from the second array whose name is equal to one of the parameters in the first
         * array. The given input array are not modified.
         * @typeparam TValue Defaults to `unknown`. Type of the parameter values.
         * @param arr1 A list of parameters for comparison.
         * @param arr2 A list of additional parameters.
         * @return An list of parameters that are in the second
         * array, but not in the first.
         */
        arrayCompare<TValue = unknown>(arr1: PrimeFaces.ajax.RequestParameter<string, TValue>[], arr2: PrimeFaces.ajax.RequestParameter<string, TValue>[]): PrimeFaces.ajax.RequestParameter<string, TValue>[];
        /**
         * Performs the early collection of post parameters (form element values) if the request is configured that
         * way. See: https://github.com/primefaces/primefaces/issues/109
         * @param cfg Configuration for the AJAX request to send, such as
         * the HTTP method, the URL, and the content of the request.
         * @return The collected form element values to be sent with the request.
         */
        collectEarlyPostParams(cfg: Partial<PrimeFaces.ajax.Configuration>): PrimeFaces.ajax.RequestParameter[];
        /**
         * Creates a FormData which can be used for a Faces AJAX request on the current view.
         * It already contains all required parameters like ViewState or ClientWindow.
         * @param form The closest form of the request source.
         * @param parameterPrefix The Portlet parameter namespace.
         * @param source The id of the request source.
         * @param process A comma separated list of components which should be processed.
         * @param update A comma separated list of components which should be updated.
         * @return The newly created form data.
         */
        createFacesAjaxFormData(form: JQuery, parameterPrefix: string, source: string, process?: string, update?: string): FormData;
        /**
         * Finds the namespace (prefix) for the parameters of the given form.
         * This is required for Porlets as a Portlet contains multiple JSF views and we must only process and update the forms/inputs of the current view / application.
         * Later the namespace is used for all post params.
         * @param form An HTML FORM element.
         * @return The namespace for the parameters of the given form, or `null` when the form does
         * not specifiy a namespace.
         */
        extractParameterNamespace(form: JQuery): string | null;
        /**
         * Handles the given AJAX request, either by sending it immediately (if `async` is set to `true`), or by
         * adding it to the AJAX queue otherwise. The AJAX queue ensures that requests are sent and handled in the
         * order they were started. See also {@link jsf.ajax.request}.
         * @param cfg Configuration for the AJAX request to send, such as
         * the HTTP method, the URL, and the content of the request.
         * @param ext Optional extender with additional options
         * that overwrite the options given in `cfg`.
         * @return A promise that resolves once the AJAX requests is done.
         * Use this to run custom JavaScript logic. When the AJAX request succeeds, the promise is fulfilled.
         * Otherwise, when the AJAX request fails, the promise is rejected. If the promise is rejected, the
         * rejection handler receives an object of type {@link PrimeFaces.ajax.FailedRequestData}.
         */
        handle(cfg: Partial<PrimeFaces.ajax.Configuration>, ext?: Partial<PrimeFaces.ajax.ConfigurationExtender>): Promise<PrimeFaces.ajax.ResponseData>;
        /**
         * Given an AJAX call configuration, resolves the components for the `process` or `update` search
         * expressions given by the configurations. Resolves the search expressions to the actual components and
         * returns a list of their IDs.
         * @param source the source element.
         * @param cfg An AJAX call configuration.
         * @param type Whether to resolve the `process` or `update` expressions.
         * @return A list of IDs with the components to which the process or update expressions refer.
         */
        resolveComponentsForAjaxCall(source: JQuery, cfg: Partial<PrimeFaces.ajax.Configuration>, type: "process" | "update"): string[];
        /**
         * Collects all `process` or `update` search expressions from the given AJAX call configuration and returns
         * them as one search expression.
         * @param cfg An AJAX call configuration.
         * @param type Whether to resolve the `process` or `update` expressions.
         * @return All process or update search expression from the given configuration.
         */
        resolveExpressionsForAjaxCall(cfg: Partial<PrimeFaces.ajax.Configuration>, type: "process" | "update"): string;
        /**
         * Starts the given AJAX request immediately by sending the data to the server. Contrast with
         * {@link handle}, which may queue AJAX requests, depending on how they are configured.
         * @param cfg Configuration for the AJAX request to send, such as
         * the HTTP method, the URL, and the content of the request.
         * @return `false` if the AJAX request is to be canceled, `true` or `undefined`
         * otherwise.
         */
        send(cfg: Partial<PrimeFaces.ajax.Configuration>): boolean | undefined;
    }
}
declare namespace PrimeFaces.ajax {
    /**
     * The interface for the object containing functionality related to
     * handling AJAX responses
     */
    export interface Response {
        /**
         * Destroys all widgets that are not part of the DOM anymore, usually because they were removed by an AJAX
         * update. Calls the `destroy` method on the widget and removes the widget from the global widget registry.
         */
        destroyDetachedWidgets(): void;
        /**
         * Handles the response of an AJAX request. The response consists of one or more actions such as executing a
         * script or updating a DOM element. See also {@link jsf.ajax.response}.
         *
         * Also updates the specified components if any and synchronizes the client side JSF state. DOM updates are
         * implemented using jQuery which uses a very algorithm.
         * @typeparam TWidget Defaults to `PrimeFaces.widget.BaseWidget`. Type of the widget which
         * triggered the AJAX request.
         * @param xml The XML that was returned by the AJAX request.
         * @param status Text status of the request.
         * @param xhr The XHR request to which a response was received.
         * @param updateHandler Optional handler for `update` actions.
         */
        handle<TWidget extends PrimeFaces.widget.BaseWidget = PrimeFaces.widget.BaseWidget>(xml: XMLDocument, status: JQuery.Ajax.SuccessTextStatus, xhr: PrimeFaces.ajax.pfXHR, updateHandler?: PrimeFaces.ajax.UpdateHandler<TWidget>): void;
        /**
         * Puts focus on the given element if necessary.
         * @param activeElementId ID of the active to refocus.
         * @param activeElementSelection The range to select, for INPUT
         * and TEXTAREA elements.
         */
        handleReFocus(activeElementId: string, activeElementSelection?: PrimeFaces.ajax.ActiveElementSelection): void;
    }
}
declare namespace PrimeFaces.ajax {
    /**
     * The interface for the object containing functionality related to
     * processing the different types of actions from AJAX responses.
     */
    export interface ResponseProcessor {
        /**
         * Handles an `attributes` AJAX action by setting the attributes on the DOM element.
         * @param node The XML node of the `attributes` action.
         * @return `false` if the AJAX action could not be performed, `true` or `undefined`
         * otherwise.
         */
        doAttributes(node: Node): boolean | undefined;
        /**
         * Handles a `delete` AJAX action by remove the DOM element.
         * @param node The XML node of the `delete` action.
         */
        doDelete(node: Node): void;
        /**
         * Handles an `error` AJAX action by doing nothing currently.
         * @param node The XML node of the `error` action.
         * @param xhr The XHR request to which a response was received.
         */
        doError(node: Node, xhr: PrimeFaces.ajax.pfXHR): void;
        /**
         * Handles an `eval` AJAX action by evaluating the returned JavaScript.
         * @param node The XML node of the `eval` action.
         * @param xhr The XHR request to which a response was received.
         */
        doEval(node: Node, xhr: PrimeFaces.ajax.pfXHR): void;
        /**
         * Handles an `extension` AJAX action by extending the `pfArgs` property on the jQuery XHR object.
         * @param node The XML node of the `extension` action.
         * @param xhr The XHR request to which a response was received.
         */
        doExtension(node: Node, xhr: PrimeFaces.ajax.pfXHR): void;
        /**
         * Handles an `insert` AJAX action by inserting a newly creating DOM element.
         * @param node The XML node of the `insert` action.
         * @return `false` if the AJAX action could not be performed, `true` or `undefined`
         * otherwise.
         */
        doInsert(node: Node): boolean | undefined;
        /**
         * Handles a `redirect` AJAX action by performing a redirect to the target URL.
         * @param node The XML node of the `redirect` action.
         */
        doRedirect(node: Node): void;
        /**
         * Handles an `update` AJAX action by calling the given update handler. When no update handler is given,
         * replaces the HTML content of the element with the new content.
         * @typeparam TWidget Defaults to `PrimeFaces.widget.BaseWidget`. Type of the widget which
         * triggered the AJAX request.
         * @param node The XML node of the `update` action.
         * @param xhr The XHR request to which a response was received.
         * @param updateHandler Optional handler for the update.
         */
        doUpdate<TWidget extends PrimeFaces.widget.BaseWidget = PrimeFaces.widget.BaseWidget>(node: Node, xhr: PrimeFaces.ajax.pfXHR, updateHandler?: PrimeFaces.ajax.UpdateHandler<TWidget>): void;
    }
}
declare namespace PrimeFaces.ajax {
    /**
     * The class for the object with the AJAX utility methods, used for
     * handling and working with AJAX requests and updates.
     */
    export interface Utils {
        /**
         * Iterates over all immediate children of the given node and returns the concatenated content (`node value`)
         * of each such child node. For the document itself, the node value is `null`.
         * For text, comment, and CDATA nodes, the `node value` is the (text) content of the node.
         * For attribute nodes, the value of the attribute is used.
         * @param node An HTML node for which to retrieve the content.
         * @return The content of all immediate child nodes, concatenated together.
         */
        getContent(node: HTMLElement): string;
        /**
         * Gets a selector to resolve all forms which needs to be updated with a new ViewState.
         * This is required in portlets as the DOM contains forms of multiple JSF views / applications.
         * @param form The closest form of the request source.
         * @param parameterPrefix The portlet parameter prefix.
         * @return The selector for the forms, or `null` when no forms need to be updated.
         */
        getPorletForms(form: JQuery, parameterPrefix: string): string | null;
        /**
         * Resolves the URL which should be used for the POST request.
         * For portlets, a different URL is used.
         * @param form The closest form of the request source.
         * @return The POST url.
         */
        getPostUrl(form: JQuery): string;
        /**
         * Get source ID from settings.
         * @param settings containing source ID.
         * @return The source ID from settings or `null` if settings does not contain a source.
         */
        getSourceId(settings: JQuery.AjaxSettings): string;
        /**
         * Is this script an AJAX request?
         * @param script the JS script to check
         * @return `true` if this script contains an AJAX request
         */
        isAjaxRequest(script: string): boolean;
        /**
         * Checks whether the component ID from the provided widget equals the source ID from the provided
         * settings.
         * @param widget of the component to check for being the source.
         * @param settings containing source ID.
         * @return `true` if the component ID from the provided widget equals the source ID from the
         * provided settings.
         */
        isXhrSource(widget: PrimeFaces.widget.BaseWidget, settings: JQuery.AjaxSettings): boolean;
        /**
         * Checks whether one of component's triggers equals the source ID from the provided settings.
         * @param widget of the component to check for being the source.
         * @param settings containing source ID.
         * @param triggerMustExist flag to check if the trigger must exist
         * @return `true` if if one of component's triggers equals the source ID from the provided settings.
         */
        isXhrSourceATrigger(widget: PrimeFaces.widget.BaseWidget, settings: JQuery.AjaxSettings, triggerMustExist: boolean): boolean;
        /**
         * Updates the HTML `body` element of the current document with the content received from an AJAX request.
         * @param content The content of the changeset that was returned by an AJAX request.
         */
        updateBody(content: string): void;
        /**
         * Updates an element with the given ID by applying a change set that was returned by an AJAX request. This
         * involves replacing the HTML content of the element with the new content.
         * @param id ID of the element that is to be updated.
         * @param content The new content of the changeset as returned by an AJAX request.
         * @param xhr Optional XHR request with `pfSettings` or `pfArgs` with further
         * data, such as which forms should be updated.
         */
        updateElement(id: string, content: string, xhr?: PrimeFaces.ajax.pfXHR): void;
        /**
         * Updates the main hidden input element for each form.
         * @param name Name of the hidden form input element, usually the same as the form.
         * @param value Value to set on the hidden input element.
         * @param xhr Optional XHR request with `pfSettings` or `pfArgs` with further
         * data, such as which forms should be updated.
         */
        updateFormStateInput(name: string, value: string, xhr?: PrimeFaces.ajax.pfXHR): void;
        /**
         * Updates the HTML `head` element of the current document with the content received from an AJAX request.
         * @param content The content of the changeset that was returned by an AJAX request.
         */
        updateHead(content: string): void;
    }
}
/**
 * The object with functionality related to multiple window support in PrimeFaces applications.
 */
declare namespace PrimeFaces.clientwindow {
    /**
     * The key for the session storage entry holding the client window ID.
     */
    export const CLIENT_WINDOW_SESSION_STORAGE: string;
    /**
     * The name of the URL parameter holding the client window ID.
     */
    export const CLIENT_WINDOW_URL_PARAM: string;
    /**
     * The number of characters of the client window ID. Each client window ID must be of this length, or it is
     * invalid.
     */
    export const LENGTH_CLIENT_WINDOW_ID: number;
    /**
     * The value of the temporary client window ID, used for requesting a new ID, see
     * {@link requestNewClientWindowId}.
     */
    export const TEMP_CLIENT_WINDOW_ID: string;
    /**
     * The current window ID, as received from the server. May be `null` when to ID was provided.
     */
    export let clientWindowId: null | string;
    /**
     * Whether the currently loaded page is from the first redirect.
     */
    export let initialRedirect: boolean;
    /**
     * Whether the {@link init} function was called already.
     */
    export let initialized: boolean;
    /**
     * Checks whether the client window ID is valid. If not, requests a new client window ID from the server via
     * reloading the current page.
     */
    export function assertClientWindowId(): void;
    /**
     * Makes sure the temporary cookie for the client window ID is expired.
     */
    export function cleanupCookies(): void;
    /**
     * Expires the cookie with the given name by setting a cookie with the appropriate `max-age` and `expires`
     * settings.
     * @param cookieName Name of the cookie to expire.
     */
    export function expireCookie(cookieName: string): void;
    /**
     * Returns the value of the URL parameter with the given name. When the URL contains multiple URL parameters
     * with the same name, the value of the first URL parameter is returned.
     * @param uri An URL from which to extract an URL parameter.
     * @param name Name of the URL parameter to retrieve.
     * @return The value of the given URL parameter. Returns the empty string when the URL parameter
     * is present, but has no value. Returns `null` when no URL parameter with the given name exists.
     */
    export function getUrlParameter(uri: string, name: string): string | null;
    /**
     * Initializes the client window feature. Usually invoked on page load. This method should only be called once
     * per page.
     * @param clientWindowId The current client window ID.
     * @param initialRedirect Whether the currently loaded page is from the first redirect.
     */
    export function init(clientWindowId: string, initialRedirect: boolean): void;
    /**
     * Given a URL, removes all URL parameters with the given name, adds a new URL parameter with the given value,
     * and returns the new URL with the replaced parameter. If the URL contains multiple URL parameters with the
     * same name, they are all removed.
     * @param uri The URL for which to change an URL parameter.
     * @param parameterName Name of the URL parameter to change.
     * @param parameterValue New value for the URL parameter. If `null` or not given, the empty
     * string is used.
     * @return The given URL, but with value of the given URL parameter changed to the new value.
     */
    export function replaceUrlParam(uri: string, parameterName: string, parameterValue?: string | null): string;
    /**
     * Expires the current client window ID by replacing it with a temporary, invalid client window ID. Then reloads
     * the current page to request a new ID from the server.
     */
    export function requestNewClientWindowId(): void;
}
/**
 * The object with functionality related to handling the `script-src` directive of the HTTP Content-Security-Policy
 * (CSP) policy. This makes use of a nonce (number used once). The server must generate a unique nonce value each
 * time it transmits a policy.
 */
declare namespace PrimeFaces.csp {
    /**
     * Map of currently registered CSP events on this page.
     */
    export let EVENT_REGISTRY: Map<string, Map<string, boolean>>;
    /**
     * Name of the POST parameter for transmitting the nonce.
     */
    export const NONCE_INPUT: string;
    /**
     * The value of the nonce to be used.
     */
    export let NONCE_VALUE: string;
    /**
     * GitHub #5790: When using jQuery to trigger a click event on a button while using CSP
     * we must set preventDefault or else it will trigger a non-ajax button click.
     * @param target The target of this click event.
     * @return the JQuery click event
     */
    export function clickEvent(target: JQuery): JQuery.TriggeredEvent;
    /**
     * Perform a CSP safe `eval()`.
     * @param js The JavaScript code to evaluate.
     * @param nonceValue Nonce value. Leave out if not using CSP.
     * @param windowContext Optional Window context to call eval from.
     */
    export function eval(js: string, nonceValue?: string, windowContext?: string): void;
    /**
     * Perform a CSP safe `eval()` with a return result value.
     * @param js The JavaScript code to evaluate.
     * @param nonceValue Nonce value. Leave out if not using CSP.
     * @param windowContext Optional Window context to call eval from.
     * @return The result of the evaluated JavaScript code.
     * @see https://stackoverflow.com/a/33945236/502366
     */
    export function evalResult(js: string, nonceValue?: string, windowContext?: string): unknown;
    /**
     * CSP won't allow string-to-JavaScript methods like `eval()` and `new Function()`.
     * This method uses JQuery `globalEval` to safely evaluate the function if CSP is enabled.
     * @param id The element executing the function (aka `this`).
     * @param js The JavaScript code to evaluate. Two variables will be in scope for the code: (a) the
     * `this` context, which is set to the given `id`, and (b) the `event` variable, which is set to the given `e`.
     * @param e The event from the caller to pass through.
     */
    export function executeEvent(id: HTMLElement, js: string, e: JQuery.TriggeredEvent): void;
    /**
     * Does this component have a registered AJAX event.
     * @param id ID of an element
     * @param event Event to listen to, with the `on` prefix, such as `onclick` or `onblur`.
     * @return true if component has this AJAX event
     */
    export function hasRegisteredAjaxifiedEvent(id: string, event?: string): boolean | undefined;
    /**
     * Sets the given nonce to all forms on the current page.
     * @param nonce Nonce to set. This value is usually supplied by the server.
     */
    export function init(nonce: string): void;
    /**
     * Checks if the given form is a Faces form.
     * @param form The form to check.
     * @return true if the form is a Faces form.
     */
    export function isFacesForm(form?: HTMLInputElement): boolean;
    /**
     * Registers an event listener for the given element.
     * @param id ID of an element
     * @param event Event to listen to, with the `on` prefix, such as `onclick` or `onblur`.
     * @param js Callback that may return `false` to prevent the default behavior of the event.
     */
    export function register(id: string, event?: string, js?: () => boolean): void;
}
/**
 * The object with functionality related to working with dialogs and the dialog framework.
 */
declare namespace PrimeFaces.dialog {
}
declare namespace PrimeFaces.dialog {
    /**
     * An extended confirmation
     * message with an additional `source` attribute for specifying the source component or form.
     */
    export interface ExtendedConfirmDialogMessage extends PrimeFaces.widget.ConfirmDialog.ConfirmDialogMessage {
        /**
         * The source component (command button,
         * AJAX callback etc) that triggered the confirmation. When a string, it is interpreted as the client ID of the
         * component. Otherwise, it must be the main DOM element of the source component.
         */
        source: string | HTMLElement | JQuery;
    }
}
declare namespace PrimeFaces.dialog {
    /**
     * Interface of the dialog
     * configuration object for a dialog of the dialog framework. Used by `PrimeFaces.dialog.DialogHandlerCfg`. This is
     * mainly just the `PrimeFaces.widget.DialogCfg`, but adds a few more properties.
     */
    export interface DialogHandlerCfgOptions extends PrimeFaces.widget.DialogCfg {
        /**
         * Height of the IFRAME in pixels.
         */
        contentHeight: number;
        /**
         * Width of the IFRAME in pixels.
         */
        contentWidth: number;
        /**
         * ID of the header element of the dialog.
         */
        headerElement: string;
    }
}
declare namespace PrimeFaces.dialog {
    /**
     * Interface of the configuration object for a
     * dialog of the dialog framework. Used by `PrimeFaces.dialog.DialogHandler.openDialog`.
     */
    export interface DialogHandlerCfg {
        /**
         * The options for the dialog.
         */
        options: Partial<PrimeFaces.dialog.DialogHandlerCfgOptions>;
        /**
         * PrimeFaces dialog client ID.
         */
        pfdlgcid: string;
        /**
         * ID of the dialog.
         */
        sourceComponentId: string;
        /**
         * Widget variable of the dialog.
         */
        sourceWidgetVar: string;
        /**
         * Source URL for the IFRAME element with the dialog.
         */
        url: string;
    }
}
declare namespace PrimeFaces.dialog {
    /**
     * The interface of the object with all methods for working with dialogs and the dialog framework.
     */
    export interface DialogHandlerObject {
        /**
         * . Dialog used for showing messages, such as confirmation messages.
         */
        messageDialog: PrimeFaces.widget.Dialog;
        /**
         * Closes the dialog as specified by the given configuration.
         * @param cfg Configuration of the dialog.
         */
        closeDialog(cfg: PrimeFaces.dialog.DialogHandlerCfg): void;
        /**
         * Asks the user to confirm an action. Shows a confirmation dialog with the given message. Requires a global
         * `<p:confirmDialog>` to be available on the current page.
         * @param msg Message to show in the confirmation dialog.
         */
        confirm(msg: PrimeFaces.dialog.ExtendedConfirmDialogMessage): void;
        /**
         * Returns the current window instance. When inside an iframe, returns the window instance of the topmost
         * document.
         * @return The root window instance.
         */
        findRootWindow(): Window;
        /**
         * Opens the dialog as specified by the given configuration. When the dialog is dynamic, loads the content from
         * the server.
         * @param cfg Configuration of the dialog.
         */
        openDialog(cfg: PrimeFaces.dialog.DialogHandlerCfg): void;
        /**
         * Displays a message in the messages dialog.
         * @param msg Details of the message to show.
         */
        showMessageInDialog(msg: PrimeFaces.widget.ConfirmDialog.ConfirmDialogMessage): void;
    }
}
declare namespace PrimeFaces.dialog {
    /**
     * The object with all methods for dialogs and the dialog framework.
     */
    export const DialogHandler: PrimeFaces.dialog.DialogHandlerObject;
}
/**
 * The object with functionality related to the browser environment, such as information about the current browser.
 */
declare namespace PrimeFaces.env {
    /**
     * The current browser type.
     */
    export let browser: string;
    /**
     * `true` if the current browser is an IOS browser, `false` otherwise.
     */
    export let ios: boolean;
    /**
     * `true` if the current browser is a mobile browser, `false` otherwise.
     */
    export let mobile: boolean;
    /**
     * `true` if the user's current OS setting prefers dark mode, `false` otherwise.
     */
    export let preferredColorSchemeDark: boolean;
    /**
     * `true` if the user's current OS setting prefers light mode, `false` otherwise.
     */
    export let preferredColorSchemeLight: boolean;
    /**
     * `true` if the user's current OS setting prefers reduced motion or animations, `false` otherwise.
     */
    export let prefersReducedMotion: boolean;
    /**
     * `true` if the current browser supports touch, `false` otherwise.
     */
    export let touch: boolean;
    /**
     * Evaluate a media query and return true/false if its a match.
     * @param mediaquery the media query to evaluate
     * @return true if it matches the query false if not
     */
    export function evaluateMediaQuery(mediaquery: string): boolean;
    /**
     * Gets the user's preferred color scheme set in their operating system.
     * @return either 'dark' or 'light'
     */
    export function getOSPreferredColorScheme(): string;
    /**
     * Gets the currently loaded PrimeFaces theme.
     * @return The current theme, such as `omega` or `luna-amber`. Empty string when no theme is loaded.
     */
    export function getTheme(): string;
    /**
     * Based on the current PrimeFaces theme determine if light or dark contrast is being applied.
     * @return either 'dark' or 'light'
     */
    export function getThemeContrast(): string;
    /**
     * Initializes the environment by reading the browser environment.
     */
    export function init(): void;
    /**
     * Media query to determine if screen size is above pixel count.
     * @param pixels the number of pixels to check
     * @return true if screen is greater than number of pixels
     */
    export function isScreenSizeGreaterThan(pixels: number): boolean;
    /**
     * Media query to determine if screen size is below pixel count.
     * @param pixels the number of pixels to check
     * @return true if screen is less than number of pixels
     */
    export function isScreenSizeLessThan(pixels: number): boolean;
    /**
     * A widget is touch enabled if the browser supports touch AND the widget has the touchable property enabled.
     * The default will be true if it widget status can't be determined.
     * @param cfg the widget configuration
     * @return true if touch is enabled, false if disabled
     */
    export function isTouchable(cfg: PrimeFaces.widget.BaseWidgetCfg): boolean;
}
/**
 * The object with functionality related to working with search expressions.
 */
declare namespace PrimeFaces.expressions {
}
declare namespace PrimeFaces.expressions {
    /**
     * The interface of the object with all methods for working with search expressions.
     */
    export interface SearchExpressionFacadeObject {
        /**
         * Takes a search expression that may contain multiple components, separated by commas or whitespaces. Resolves
         * each search expression to the component it refers to and returns a list of IDs of the resolved components.
         * @param source the source element where to start the search (e.g. required for @form).
         * @param expressions A search expression with one or multiple components to resolve.
         * @return A list of IDs with the resolved components.
         */
        resolveComponents(source: JQuery, expressions: string): string[];
        /**
         * Takes a search expression that may contain multiple components, separated by commas or whitespaces. Resolves
         * each search expression to the component it refers to and returns a JQuery object with the DOM elements of
         * the resolved components.
         * @param source the source element where to start the search (e.g. required for @form).
         * @param expressions A search expression with one or multiple components to resolve.
         * @return A list with the resolved components.
         */
        resolveComponentsAsSelector(source: JQuery, expressions: string | HTMLElement | JQuery): JQuery;
        /**
         * Splits the given search expression into its components. The components of a search expression are separated
         * by either a comman or a whitespace.
         * ```javascript
         * splitExpressions("") // => [""]
         * splitExpressions("form") // => ["form"]
         * splitExpressions("form,input") // => ["form", "input"]
         * splitExpressions("form input") // => ["form", "input"]
         * splitExpressions("form,@child(1,2)") // => ["form", "child(1,2)"]
         * ```
         * @param expression A search expression to split.
         * @return The individual components of the given search expression.
         */
        splitExpressions(expression: string): string[];
    }
}
declare namespace PrimeFaces.expressions {
    /**
     * The object with all methods for working with search
     * expressions.
     */
    export const SearchExpressionFacade: PrimeFaces.expressions.SearchExpressionFacadeObject;
}
declare namespace PrimeFaces {
    /**
     * A registry of all instantiated widgets that are available on the current page.
     */
    export let widgets: Record<string, PrimeFaces.widget.BaseWidget>;
}
declare namespace PrimeFaces {
    /**
     * A map with language specific translations. This is a map between the language keys and another map with the i18n
     * keys mapped to the translation.
     */
    export let locales: Record<string, PrimeFaces.Locale>;
}
declare namespace PrimeFaces {
    /**
     * A map between some HTML entities and their HTML-escaped equivalent.
     */
    export let entityMap: Record<string, string>;
}
/**
 * Finds and returns a widget
 *
 * Note to typescript users: You should define a method that takes a widget variables and widget constructor, and
 * check whether the widget is of the given type. If so, you can return the widget and cast it to the desired type:
 * ```typescript
 * function getWidget<T extends PrimeFaces.widget.BaseWidget>(widgetVar, widgetClass: new() => T): T | undefined {
 *   const widget = PrimeFaces.widget[widgetVar];
 *   return widget !== undefined && widget instanceof constructor ? widgetClass : undefined;
 * }
 * ```
 * @param widgetVar The widget variable of a widget.
 * @return The widget instance, or `undefined` if no such widget exists
 * currently.
 */
declare function PF(widgetVar: string): PrimeFaces.widget.BaseWidget | undefined;
/**
 * This is the main global object for accessing the client-side API of PrimeFaces. Broadly speaking, it consists
 * of the following entries:
 *
 * - {@link PrimeFaces.ajax} The AJAX module with functionality for sending AJAX requests
 * - {@link PrimeFaces.clientwindow} The client window module for multiple window support in PrimeFaces applications.
 * - {@link PrimeFaces.csp} The  CSP module for the HTTP Content-Security-Policy (CSP) policy `script-src` directive.
 * - {@link PrimeFaces.dialog} The dialog module with functionality related to the dialog framework
 * - {@link PrimeFaces.env} The environment module with information about the current browser
 * - {@link PrimeFaces.expressions} The search expressions module with functionality for working with search expression
 * - {@link PrimeFaces.resources} The resources module with functionality for creating resource links
 * - {@link PrimeFaces.utils} The utility module with functionality that does not fit anywhere else
 * - {@link PrimeFaces.widget} The registry with all available widget classes
 * - {@link PrimeFaces.widgets} The registry with all currently instantiated widgets
 * - Several other utility methods defined directly on the `PrimeFaces` object, such as
 * {@link PrimeFaces.monitorDownload}, {@link PrimeFaces.getWidgetById}, or {@link PrimeFaces.escapeHTML}.
 */
declare namespace PrimeFaces {
    /**
     * Name of the POST parameter that contains the name of the current behavior event.
     */
    export const BEHAVIOR_EVENT_PARAM: string;
    /**
     * Name of the POST parameter with the current client ID
     */
    export const CLIENT_ID_DATA: string;
    /**
     * Name of the POST parameter with the current client window.
     */
    export const CLIENT_WINDOW: string;
    /**
     * Name of the POST parameter that indicates whether `<p:autoUpdate>` tags should be ignored.
     */
    export const IGNORE_AUTO_UPDATE_PARAM: string;
    /**
     * Name of the POST parameter that contains the name of the current partial behavior event.
     */
    export const PARTIAL_EVENT_PARAM: string;
    /**
     * Name of the POST parameter that contains the list of components to process.
     */
    export const PARTIAL_PROCESS_PARAM: string;
    /**
     * Name of the POST parameter that indicates whether the request is an AJAX request.
     */
    export const PARTIAL_REQUEST_PARAM: string;
    /**
     * Name of the POST parameter that indicates which element or component triggered the AJAX request.
     */
    export const PARTIAL_SOURCE_PARAM: string;
    /**
     * Name of the POST parameter that contains the list of components to be updated.
     */
    export const PARTIAL_UPDATE_PARAM: string;
    /**
     * Name of the POST parameter that indicates whether forms should have their values reset.
     */
    export const RESET_VALUES_PARAM: string;
    /**
     * Name of the faces resource servlet, eg. `javax.faces.resource`.
     */
    export const RESOURCE_IDENTIFIER: string;
    /**
     * Name of the POST parameter that indicates whether children should be skipped.
     */
    export const SKIP_CHILDREN_PARAM: string;
    /**
     * The current version of PrimeFaces.
     */
    export const VERSION: string;
    /**
     * Name of the POST parameter that contains the view root.
     */
    export const VIEW_ROOT: string;
    /**
     * Name of the POST parameter that contains the current view state.
     */
    export const VIEW_STATE: string;
    /**
     * Flag for detecting whether animation is currently running. Similar to jQuery.active flag and is useful
     * for scripts or automation tests to determine if the animation is currently running.
     */
    export let animationActive: boolean;
    /**
     * Global flag for enabling or disabling both jQuery and CSS animations.
     */
    export let animationEnabled: boolean;
    /**
     * Used to store whether a custom focus has been rendered. This avoids having to retain the last focused element
     * after AJAX update.
     */
    export let customFocus: boolean;
    /**
     * Some widgets need to compute their dimensions based on their parent element(s). This requires that such
     * widgets are not rendered until they have become visible. A widget may not be visible, for example, when it
     * is inside a tab that is not shown when the page is rendered. PrimeFaces provides a global mechanism for
     * widgets to render once they are visible. This is done by keeping a list of widgets that need to be rendered,
     * and checking on every change (AJAX request, tab change etc.) whether any of those have become visible. A
     * widgets should extend `PrimeFaces.widget.DeferredWidget` to make use of this functionality.
     *
     * This is the list of renders for widgets that are currently waiting to become visible.
     */
    export let deferredRenders: PrimeFaces.DeferredRender[];
    /**
     * A list of widgets that were once instantiated, but are not removed from the DOM, such as due to the result
     * of an AJAX update request.
     */
    export const detachedWidgets: PrimeFaces.widget.BaseWidget[];
    /**
     * PrimeFaces per defaults hides all overlays on scrolling/resizing to avoid positioning problems.
     * This is really hard to overcome in selenium tests and we can disable this behavior with this setting.
     */
    export let hideOverlaysOnViewportChange: boolean;
    /**
     * A tracker for the current z-index, used for example when creating multiple modal dialogs.
     */
    export let zindex: number;
    /**
     * Aborts all pending AJAX requests. This includes both requests that were already sent but did not receive a
     * response yet, as well as requests that are waiting in the queue and have not been sent yet.
     */
    export function abortXHRs(): void;
    /**
     * Some widgets need to compute their dimensions based on their parent element(s). This requires that such
     * widgets are not rendered until they have become visible. A widget may not be visible, for example, when it
     * is inside a tab that is not shown when the page is rendered. PrimeFaces provides a global mechanism for
     * widgets to render once they are visible. This is done by keeping a list of widgets that need to be rendered,
     * and checking on every change (AJAX request, tab change etc.) whether any of those have become visible. A
     * widgets should extend `PrimeFaces.widget.DeferredWidget` to make use of this functionality.
     *
     * Adds a deferred render to the global list.  If this widdget has already been added only the last instance
     * will be added to the stack.
     * @param widgetId The ID of a deferred widget.
     * @param containerId ID of the container that should be visible before the widget can be rendered.
     * @param fn Callback that is invoked when the widget _may_ possibly have become visible. Should
     * return `true` when the widget was rendered, or `false` when the widget still needs to be rendered later.
     */
    export function addDeferredRender(widgetId: string, containerId: string, fn: () => boolean): void;
    /**
     * Adds hidden input elements to the given form. For each key-value pair, a new hidden input element is created
     * with the given value and the key used as the name.
     * @param parent The ID of a FORM element.
     * @param params An object with key-value pairs.
     * @return This object for chaining.
     */
    export function addSubmitParam(parent: string, params: Record<string, string>): typeof PrimeFaces;
    /**
     * Attaches the given behaviors to the element. For each behavior, an event listener is registered on the
     * element. Then, when the event is triggered, the behavior callback is invoked.
     * @param element The element for which to attach the behaviors.
     * @param behaviors An object with an event name
     * as the key and event handlers for that event as the value. Each event handler is called with the given
     * element as the this context and the event that occurred as the first argument.
     */
    export function attachBehaviors(element: JQuery, behaviors: Record<string, (this: JQuery, event: JQuery.TriggeredEvent) => void>): void;
    /**
     * A function that is used as the handler function for HTML event tags (`onclick`, `onkeyup` etc.). When a
     * component has got an `onclick` etc attribute, the JavaScript for that attribute is called by this method.
     * @param element Element on which the event occurred.
     * @param event Event that occurred.
     * @param functions A list of callback
     * functions. If any returns `false`, the default action of the event is prevented.
     */
    export function bcn(element: HTMLElement, event: Event, functions: ((this: HTMLElement, event: Event) => boolean | undefined)[]): void;
    /**
     * A function that is used as the handler function for AJAX behaviors. When a component has got an AJAX
     * behavior, the JavaScript that implements behavior's client-side logic is called by this method.
     * @param ext Additional options to override the current
     * options.
     * @param event Event that occurred.
     * @param fns A list of callback functions. If any returns `false`, the other callbacks are not invoked.
     */
    export function bcnu(ext: Partial<PrimeFaces.ajax.ConfigurationExtender>, event: Event, fns: ((this: typeof PrimeFaces, ext: Partial<PrimeFaces.ajax.ConfigurationExtender>, event: Event) => boolean | undefined)[]): void;
    /**
     * Applies the inline AJAX status (ui-state-loading) to the given widget / button.
     * @param widget the widget.
     * @param button The button DOM element.
     * @param isXhrSource Callback that checks if the widget is the source of the current AJAX request.
     */
    export function bindButtonInlineAjaxStatus(widget?: PrimeFaces.widget.BaseWidget, button?: JQuery, isXhrSource?: (widget: PrimeFaces.widget.BaseWidget, settings: JQuery.AjaxSettings) => boolean): void;
    /**
     * Ends the AJAX disabled state.
     * @param widget the widget.
     * @param button The button DOM element.
     */
    export function buttonEndAjaxDisabled(widget?: PrimeFaces.widget.BaseWidget, button?: JQuery): void;
    /**
     * Finds the width of the scrollbar that is used by the current browser, as scrollbar widths are different for
     * across different browsers.
     * @return The width of the scrollbars of the current browser.
     */
    export function calculateScrollbarWidth(): number;
    /**
     * Changes the current theme to the given theme (by exchanging CSS files). Requires that the theme was
     * installed and is available.
     * @param newTheme The new theme, eg. `luna-amber`, `nova-dark`, or `omega`.
     */
    export function changeTheme(newTheme: string): void;
    /**
     * Clears the text selected by the user on the current page.
     */
    export function clearSelection(): void;
    /**
     * Deprecated, use `PrimeFaces.dialog.DialogHandler.closeDialog` instead.
     * @param cfg Configuration of the dialog.
     * @deprecated
     */
    export function closeDialog(cfg: PrimeFaces.dialog.DialogHandlerCfg): void;
    /**
     * Displays dialog or popup according to the type of confirm component.
     * @param msg Message to show with the confirm dialog or popup.
     * @deprecated Deprecated, use {@link PrimeFaces.dialog.DialogHandler.confirm} instead.
     */
    export function confirm(msg: PrimeFaces.dialog.ExtendedConfirmDialogMessage): void;
    /**
     * Checks whether cookies are enabled in the current browser.
     * @return `true` if cookies are enabled and can be used, `false` otherwise.
     */
    export function cookiesEnabled(): boolean;
    /**
     * Generates a unique key for using in HTML5 local storage by combining the context, view, id, and key.
     * @param id ID of the component
     * @param key a unique key name such as the component name
     * @param global if global then do not include the view id
     * @return the generated key comprising of context + view + id + key
     */
    export function createStorageKey(id: string, key: string, global: boolean): string;
    /**
     * Creates a new widget of the given type and with the given configuration. Registers that widget in the widgets
     * registry {@link PrimeFaces.widgets}. If this method is called in response to an AJAX request and the method
     * exists already, it is refreshed.
     * @param widgetName Name of the widget class, as registered in `PrimeFaces.widget`
     * @param widgetVar Widget variable of the widget
     * @param cfg Configuration for the widget
     */
    export function createWidget(widgetName: string, widgetVar: string, cfg: PrimeFaces.widget.BaseWidgetCfg): void;
    /**
     * A shortcut for {@link createWidget}.
     * @param widgetName Name of the widget class, as registered in {@link PrimeFaces.widget}.
     * @param widgetVar Widget variable of the widget
     * @param cfg Configuration for the widget
     */
    export function cw(widgetName: string, widgetVar: string, cfg: PrimeFaces.widget.BaseWidgetCfg): void;
    /**
     * Logs the given message at the `debug` level.
     * @param log Message to log
     */
    export function debug(log: string): void;
    /**
     * Deletes the given cookie.
     * @param name Name of the cookie to delete
     * @param cfg The cookie configuration used to set the cookie.
     */
    export function deleteCookie(name: string, cfg?: Partial<Cookies.CookieAttributes>): void;
    /**
     * Logs the given message at the `error` level.
     * @param log Message to log
     */
    export function error(log: string): void;
    /**
     * Creates an ID to a CSS ID selector that matches elements with that ID. For example:
     * ```
     * PrimeFaces.escapeClientId("form:input"); // => "#form\:input"
     * PrimeFaces.escapeClientId("form#input"); // => "#form#input"
     * ```
     *
     * __Please note that this method does not escape all characters that need to be escaped and will not work with arbitrary IDs__
     * @param id ID to convert.
     * @return A CSS ID selector for the given ID.
     */
    export function escapeClientId(id: string): string;
    /**
     * Escapes the given value to be used as the content of an HTML element or attribute.
     * @param value A string to be escaped
     * @param preventDoubleEscaping if true will not include ampersand to prevent double escaping
     * @return The given value, escaped to be used as a text-literal within an HTML document.
     */
    export function escapeHTML(value: string, preventDoubleEscaping: boolean | undefined): string;
    /**
     * Creates a regexp that matches the given text literal, and HTML-escapes that result.
     * @param text The literal text to escape.
     * @return A regexp that matches the given text, escaped to be used as a text-literal within an HTML
     * document.
     */
    export function escapeRegExp(text: string): string;
    /**
     * Attempts to put focus an element:
     *
     * - When `id` is given, puts focus on the element with that `id`
     * - Otherwise, when `context` is given, puts focus on the first focusable element within that context
     * (container)
     * - Otherwise, puts focus on the first focusable element in the page.
     * @param id ID of an element to focus.
     * @param context The ID of a container with an element to focus
     */
    export function focus(id?: string, context?: string): void;
    /**
     * Puts focus on the given element.
     * @param el Element to focus
     */
    export function focusElement(el: JQuery): void;
    /**
     * Some ARIA attributes have a value that depends on the current locale. This returns the localized version for
     * the given aria key.
     * @param key An aria key
     * @param defaultValue Optional default if key is not found
     * @return The translation for the given aria key
     */
    export function getAriaLabel(key: string, defaultValue: string): string;
    /**
     * Gets the form by id or the closest form if the id is not a form itself.
     * In AJAX we also have a fallback for the first form in DOM, this should not be used here.
     * @param id ID of the component to get the closest form or if its a form itself
     * @return the form or NULL if no form found
     */
    export function getClosestForm(id: string): JQuery;
    /**
     * Fetches the value of a cookie by its name
     * @param name Name of a cookie
     * @return The value of the given cookie, or `undefined` if no such cookie exists
     */
    export function getCookie(name: string): string | undefined;
    /**
     * Deprecated, use {@link PrimeFaces.resources.getFacesResource} instead.
     * @param name Name of the resource
     * @param library Library of the resource
     * @param version Version of the resource
     * @return The URL for accessing the given resource.
     * @deprecated
     */
    export function getFacesResource(name: string, library: string, version: string): string;
    /**
     * Attempt to look up the locale key by current locale and fall back to US English if not found.
     * @param key The locale key
     * @return The translation for the given key
     */
    export function getLocaleLabel(key: string): string;
    /**
     * Finds the current locale with the i18n keys and the associated translations. Uses the current language key
     * as specified by `PrimeFaces.settings.locale`. When no locale was found for the given locale, falls back to
     * the default English locale.
     * @param cfgLocale optional configuration locale from the widget
     * @return The current locale with the key-value pairs.
     */
    export function getLocaleSettings(cfgLocale?: string): PrimeFaces.Locale;
    /**
     * Finds the text currently selected by the user on the current page.
     * @return The text currently selected by the user on the current page.
     */
    export function getSelection(): string | Selection;
    /**
     * Gets the currently loaded PrimeFaces theme.
     * @return The current theme, such as `omega` or `luna-amber`. Empty string when no theme is loaded.
     */
    export function getTheme(): string;
    /**
     * Gets the currently loaded PrimeFaces theme CSS link.
     * @return The full URL to the theme CSS
     */
    export function getThemeLink(): string;
    /**
     * Finds a widget in the current page with the given ID.
     * @param id ID of the widget to retrieve.
     * @return The widget with the given ID, of `null` if no such widget was
     * found.
     */
    export function getWidgetById(id: string): PrimeFaces.widget.BaseWidget | null;
    /**
     * Finds all widgets in the current page that are of the given type.
     * @typeparam TWidget Type of the widgets of interest, e.g.
     * `PrimeFaces.widget.DataTable`.
     * @param type The (proto)type of the widgets of interest, e.g., `PrimeFaces.widget.DataTable`.
     * @return An array of widgets that are of the requested type. If no suitable widgets
     * are found on the current page, an empty array will be returned.
     */
    export function getWidgetsByType<TWidget extends new (...args: never[]) => unknown>(type: TWidget): InstanceType<TWidget>[];
    /**
     * Checks whether any text on the current page is selected by the user.
     * @return `true` if text is selected, `false` otherwise.
     */
    export function hasSelection(): boolean;
    /**
     * Checks whether an items is contained in the given array. The items is compared against the array entries
     * via the `===` operator.
     * @typeparam T Defaults to `unknown`. Type of the array items
     * @param arr An array with items
     * @param item An item to check
     * @return `true` if the given item is in the given array, `false` otherwise.
     */
    export function inArray<T = unknown>(arr: T[], item: T): boolean;
    /**
     * Logs the given message at the `info` level.
     * @param log Message to log
     */
    export function info(log: string): void;
    /**
     * Some widgets need to compute their dimensions based on their parent element(s). This requires that such
     * widgets are not rendered until they have become visible. A widget may not be visible, for example, when it
     * is inside a tab that is not shown when the page is rendered. PrimeFaces provides a global mechanism for
     * widgets to render once they are visible. This is done by keeping a list of widgets that need to be rendered,
     * and checking on every change (AJAX request, tab change etc.) whether any of those have become visible. A
     * widgets should extend `PrimeFaces.widget.DeferredWidget` to make use of this functionality.
     *
     * Invokes all deferred renders. This is usually called when an action was performed that _may_ have resulted
     * in a container now being visible. This includes actions such as an AJAX request request was made or a tab
     * change.
     * @param containerId ID of the container that _may_ have become visible.
     */
    export function invokeDeferredRenders(containerId: string): void;
    /**
     * Checks whether the current application is running in a development environment or a production environment.
     * @return `true` if this is a development environment, `false` otherwise.
     */
    export function isDevelopmentProjectStage(): boolean;
    /**
     * Checks whether a value is of type `number` and is neither `Infinity` nor `NaN`.
     * @param value A value to check
     * @return `true` if the given value is a finite number (neither `NaN` nor +/- `Infinity`),
     * `false` otherwise.
     */
    export function isNumber(value: unknown): boolean;
    /**
     * Checks whether the current application is running in a production environment.
     * @return `true` if this is a production environment, `false` otherwise.
     */
    export function isProductionProjectStage(): boolean;
    /**
     * As a `<p:fileDownload>` process is implemented as a norma, non-AJAX request, `<p:ajaxStatus>` will not work.
     * Still, PrimeFaces provides a feature to monitor file downloads via this client-side function. This is done
     * by sending a cookie with the HTTP response of the file download request. On the client-side, polling is used
     * to check when the cookie is set.
     *
     * The example below displays a modal dialog when a download begins and hides it when the download is complete:
     *
     * Client-side callbacks:
     *
     * ```javascript
     * function showStatus() {
     *   PF('statusDialog').show();
     * }
     * function hideStatus() {
     *   PF('statusDialog').hide();
     * }
     * ```
     *
     * Server-side XHTML view:
     *
     * ```xml
     * <p:commandButton value="Download" ajax="false" onclick="PrimeFaces.monitorDownload(showStatus, hideStatus)">
     *   <p:fileDownload value="#{fileDownloadController.file}"/>
     * </p:commandButton>
     * ```
     * @param start Callback that is invoked when the download starts.
     * @param complete Callback that is invoked when the download ends.
     * @param monitorKey Name of the cookie for monitoring the download. The cookie name defaults to
     * `primefaces.download` + the current viewId. When a monitor key is given, the name of the cookie will consist of a prefix and the
     * given monitor key.
     */
    export function monitorDownload(start: () => void, complete: () => void, monitorKey?: string): void;
    /**
     * Increment and return the next `z-index` for CSS as a string.
     * Note that jQuery will no longer accept numeric values in {@link JQuery.css | $.fn.css} as of version 4.0.
     * @return the next `z-index` as a string.
     */
    export function nextZindex(): string;
    /**
     * Registeres a listener that will be called as soon as the given element was loaded completely. Please note the
     * listener may be called synchronously (immediately) or asynchronously, depending on whether the element is
     * already loaded.
     * @param element Element to wait for
     * @param listener Listener to call once the element is loaded
     */
    export function onElementLoad(element: JQuery, listener: () => void): void;
    /**
     * Deprecated, use `PrimeFaces.dialog.DialogHandler.openDialog` instead.
     * @param cfg Configuration of the dialog.
     * @deprecated
     */
    export function openDialog(cfg: PrimeFaces.dialog.DialogHandlerCfg): void;
    /**
     * Queue a microtask if delay is 0 or less and setTimeout if > 0.
     * @param fn the function to call after the delay
     * @param delay the optional delay in milliseconds
     * @return the id associated to the timeout or undefined if no timeout used
     */
    export function queueTask(fn: () => void, delay: number | undefined): number | undefined;
    /**
     * Some widgets need to compute their dimensions based on their parent element(s). This requires that such
     * widgets are not rendered until they have become visible. A widget may not be visible, for example, when it
     * is inside a tab that is not shown when the page is rendered. PrimeFaces provides a global mechanism for
     * widgets to render once they are visible. This is done by keeping a list of widgets that need to be rendered,
     * and checking on every change (AJAX request, tab change etc.) whether any of those have become visible. A
     * widgets should extend `PrimeFaces.widget.DeferredWidget` to make use of this functionality.
     *
     * Removes a deferred render from the global list.
     * @param widgetId The ID of a deferred widget.
     */
    export function removeDeferredRenders(widgetId: string): void;
    /**
     * Reset any state variables on update="@all".
     */
    export function resetState(): void;
    /**
     * Aligns container scrollbar to keep item in container viewport, algorithm copied from JQueryUI menu widget.
     * @param container The container with a scrollbar that contains the item.
     * @param item The item to scroll into view.
     */
    export function scrollInView(container: JQuery, item: JQuery): void;
    /**
     * Scrolls to a component with given client id
     * @param id The ID of an element to scroll to.
     * @param duration string or number determining how long the animation will run. Default to 400
     */
    export function scrollTo(id: string, duration: string | number | undefined): void;
    /**
     * Takes an input or textarea element and sets the caret (text cursor) position to the end of the the text.
     * @param element An input or textarea element.
     */
    export function setCaretToEnd(element: JQuery): void;
    /**
     * Sets the value of a given cookie.
     * It will set secure=true, if using HTTPS and session-config/cookie-config/secure is set to true in web.xml.
     * It will set sameSite, if secure=true, with the value of the primefaces.COOKIES_SAME_SITE parameter.
     * @param name Name of the cookie to set
     * @param value Value to set
     * @param cfg Configuration for this cookie: when it expires, its
     * paths and domain and whether it is secure cookie.
     */
    export function setCookie(name: string, value: string, cfg?: Partial<Cookies.CookieAttributes>): void;
    /**
     * Deprecated, use {@link PrimeFaces.dialog.DialogHandler.showMessageInDialog} instead.
     * @param msg Message to show in a dialog.
     * @deprecated
     */
    export function showMessageInDialog(msg: PrimeFaces.widget.ConfirmDialog.ConfirmDialogMessage): void;
    /**
     * BUTTON elements may have different states, such as `hovering` or `focused`. For each state, there is a
     * corresponding style class that is added to the button when it is in that state, such as `ui-state-hover` or
     * `ui-state-focus`. These classes are used by CSS rules for styling. This method sets up a button element so
     * that the classes are added correctly (by adding event listeners).
     * @param button BUTTON element to skin
     * @return this for chaining
     */
    export function skinButton(button: JQuery): typeof PrimeFaces;
    /**
     * There are many Close buttons in PF that should get aria-label="close" and role="button".
     * @param element BUTTON or LINK element
     * @return this for chaining
     */
    export function skinCloseAction(element: JQuery): JQuery;
    /**
     * INPUT elements may have different states, such as `hovering` or `focused`. For each state, there is a
     * corresponding style class that is added to the input when it is in that state, such as `ui-state-hover` or
     * `ui-state-focus`. These classes are used by CSS rules for styling. This method sets up an input element so
     * that the classes are added correctly (by adding event listeners).
     * @param input INPUT element to skin
     * @return this for chaining
     */
    export function skinInput(input: JQuery): typeof PrimeFaces;
    /**
     * SELECT elements may have different states, such as `hovering` or `focused`. For each state, there is a
     * corresponding style class that is added to the select when it is in that state, such as `ui-state-hover` or
     * `ui-state-focus`. These classes are used by CSS rules for styling. This method sets up a select element so
     * that the classes are added correctly (by adding event listeners).
     * @param select SELECT element to skin
     * @return this for chaining
     */
    export function skinSelect(select: JQuery): typeof PrimeFaces;
    /**
     * Submits the given form, and clears all `ui-submit-param`s after that to prevent dom caching issues.
     *
     * If a target is given, it is set on the form temporarily before it is submitted. Afterwards, the original
     * target attribute of the form is restored.
     * @param formId ID of the FORM element.
     * @param target The target attribute to use on the form during the submit process.
     */
    export function submit(formId: string, target?: string): void;
    /**
     * Converts a date into an ISO-8601 date without using the browser timezone offset.
     *
     * See https://stackoverflow.com/questions/10830357/javascript-toisostring-ignores-timezone-offset
     * @param date the date to convert
     * @return ISO-8601 version of the date
     */
    export function toISOString(date: Date): string;
    /**
     * Converts the provided string to searchable form.
     * @param string to normalize.
     * @param lowercase flag indicating whether the string should be lower cased.
     * @param normalize flag indicating whether the string should be normalized (accents to be removed
     * from characters).
     * @return searchable string.
     */
    export function toSearchable(string: string, lowercase: boolean, normalize: boolean): string;
    /**
     * For 4.0 jQuery deprecated $.trim in favor of PrimeFaces.trim however that does not handle
     * NULL and jQuery did so this function allows a drop in replacement.
     * @param value the String to trim
     * @return trimmed value or "" if it was NULL
     */
    export function trim(value: string): string;
    /**
     * Updates the class of the given INPUT element to indicate whether the element contains data or not. Used for
     * example in floating labels.
     * @param input The text input to modify
     * @param parent The parent element of the input.
     */
    export function updateFilledState(input: JQuery, parent: JQuery): void;
    /**
     * Generate a RFC-4122 compliant UUID to be used as a unique identifier.
     *
     * See https://www.ietf.org/rfc/rfc4122.txt
     * @return A random UUID.
     */
    export function uuid(): string;
    /**
     * Logs the current PrimeFaces and jQuery version to console.
     */
    export function version(): void;
    /**
     * Logs the given message at the `warn` level.
     * @param log Message to log
     */
    export function warn(log: string): void;
    /**
     * Handles the error case when a widget was requested that is not available. Currently just logs an error
     * message.
     * @param widgetVar Widget variables of a widget
     */
    export function widgetNotAvailable(widgetVar: string): void;
}
declare namespace PrimeFaces {
    /**
     * Represents a deferred render added for a deferred widget.
     * Some widgets need to compute their dimensions based on their parent element(s). This requires that such widgets
     * are not rendered until they have become visible. A widget may not be visible, for example, when it is inside a
     * tab that is not shown when the page is rendered. PrimeFaces provides a global mechanism for widgets to render
     * once they are visible. This is done by keeping a list of widgets that need to be rendered, and checking on every
     * change (AJAX request, tab change etc.) whether any of those have become visible. A widgets should extend
     * `PrimeFaces.widget.DeferredWidget` to make use of this functionality.
     */
    export interface DeferredRender {
        /**
         * Callback that is invoked when the widget _may_ possibly have become visible.
         * Checks whether the widget can be rendered and if so, renders it.
         * @return `true` when the widget was rendered, or `false` when the widget still
         * needs to be rendered later.
         */
        callback(): boolean;
        /**
         * ID of the container that should be visible before the widget can be rendered.
         */
        container: string;
        /**
         * The ID of a deferred widget.
         */
        widget: string;
    }
}
/**
 * An object with some runtime settings, such as the current locale.
 */
declare namespace PrimeFaces.settings {
    /**
     * `true` if the empty string and `null` should be treated the same way, or
     * `false` otherwise.
     */
    export const considerEmptyStringNull: boolean;
    /**
     * The current locale, such as `en`,`en_US`, or `ja`.
     */
    export const locale: string;
    /**
     * `true` if empty (input etc.) fields should be validated, or `false` otherwise.
     */
    export const validateEmptyFields: boolean;
}
/**
 * The object with functionality related to handling resources on the server, such as CSS and JavaScript files.
 */
declare namespace PrimeFaces.resources {
    /**
     * Builds a JSF resource URL for given resource.
     *
     * ```javascript
     * getFacesResource("main.css", "pf", "4.2.0") // => "https://www.primefaces.org/showcase/javax.faces.resource/main.css.xhtml?ln=pf&v=4.2.0"
     * ```
     * @param name The name of the resource, such as `primefaces.js`.
     * @param library The library of the resource, such as `primefaces`.
     * @param version The version of the library, such as `5.1`.
     * @return The JSF resource URL for loading the resource.
     */
    export function getFacesResource(name: string, library: string, version: string): string;
    /**
     * Given a URI, find the name of the script, such as `primefaces-extensions.js`.
     * @param scriptURI The URI of a script
     * @return The name of the script.
     */
    export function getResourceScriptName(scriptURI: string): string;
    /**
     * Gets the resource URI of the first Javascript JS file served as a JSF resource.
     * @return The first JavasScript resource URI.
     */
    export function getResourceScriptURI(): string;
    /**
     * Finds the URL extension of currently included resources, such as `jsf` or `xhtml`.
     *
     * This should only be used if extensions mapping is used, see `PrimeFaces.isExtensionMapping`.
     * @return The URL extension.
     */
    export function getResourceUrlExtension(): string;
    /**
     * Checks if the FacesServlet is mapped with an extension mapping. Common extension mapping are for example:
     *
     * - .jsf
     * - .xhtml
     * @return `true` if the FacesServlet is mapped with an extension mapping, `false` otherwise.
     */
    export function isExtensionMapping(): boolean;
}
/**
 * The object with various utilities needed by PrimeFaces.
 */
declare namespace PrimeFaces.utils {
    /**
     * Creates a new (empty) container for a modal overlay. A modal overlay is an overlay that blocks the content
     * below it. To remove the modal overlay, use `PrimeFaces.utils.removeModal`.
     * @param widget An overlay widget instance.
     * @param overlay The modal overlay element should be a DIV.
     * @param tabbablesCallback A supplier function that return a list of tabbable elements. A
     * tabbable element is an element to which the user can navigate to via the tab key.
     * @return The DOM element for the newly added modal overlay container.
     */
    export function addModal(widget: PrimeFaces.widget.BaseWidget, overlay: JQuery, tabbablesCallback: () => JQuery): JQuery;
    /**
     * An overlay widget is moved in the DOM to the position as specified by the `appendTo` attribute. This function
     * moves the widget to its position in the DOM and removes old elements from previous AJAX updates.
     * @param widget The overlay widget instance.
     * @param overlay The DOM element for the overlay.
     * @param overlayId ID of the overlay, usually the widget ID.
     * @param appendTo The container to which the overlay is appended.
     */
    export function appendDynamicOverlay(widget: PrimeFaces.widget.BaseWidget, overlay: JQuery, overlayId: string, appendTo: JQuery): void;
    /**
     * Blocks the enter key for an event like `keyup` or `keydown`. Useful in filter input events in many
     * components.
     * @param e The key event that occurred.
     * @return `true` if ENTER key was blocked, false if not.
     */
    export function blockEnterKey(e: JQuery.TriggeredEvent): boolean;
    /**
     * Calculates an element offset relative to the current scroll position of the window.
     * @param element An element for which to calculate the scroll position.
     * @return The offset of the given element, relative to the current scroll position of the
     * window.
     */
    export function calculateRelativeOffset(element: JQuery): JQuery.Coordinates;
    /**
     * Cleanup the `detached` overlay.
     *
     * If you update a component, the overlay is rendered as specified in the component tree (XHTML view), but moved
     * to a different container via JavaScript.
     *
     * This means that after an AJAX update, we now have 2 overlays with the same id:
     *
     * 1. The newly rendered overlay, as a child of the element specified by the component tree (XHTML view)
     * 1. The old, detached overlay, as a child of the element specified by `appendTo` attribute
     *
     * We now need to remove the detached overlay. This is done by this function.
     * @param widget The (old) overlay widget instance.
     * @param overlay The DOM element for the overlay.
     * @param overlayId ID of the overlay, usually the widget ID.
     * @param appendTo The container to which the overlay is appended.
     */
    export function cleanupDynamicOverlay(widget: PrimeFaces.widget.DynamicOverlayWidget, overlay: JQuery, overlayId: string, appendTo: JQuery): void;
    /**
     * Count the bytes of the inputtext.
     * borrowed from the ckeditor wordcount plugin
     * @param text Text to count bytes from.
     * @return the byte count
     */
    export function countBytes(text: string): number;
    /**
     * Decode escaped XML into regular string.
     * @param input the input to check if filled
     * @return either the original string or escaped XML
     */
    export function decodeXml(input: string | undefined): string | undefined;
    /**
     * When configuring numeric value like 'showDelay' and the user wants '0' we can't treat 0 as Falsey
     * so we make the value 0.  Otherwise Falsey returns the default value.
     * @param value the original value
     * @param defaultValue the required default value if value is not set
     * @return the calculated value
     */
    export function defaultNumeric(value: number | undefined, defaultValue: number): number;
    /**
     * Disables CSS and jQuery animation.
     */
    export function disableAnimations(): void;
    /**
     * Disables a button from being clicked.
     * @param jq a required jQuery button to disable
     */
    export function disableButton(jq: JQuery): void;
    /**
     * Disables a widget from editing and sets it style as disabled.
     * @param jq a required jQuery element to disable
     * @param input an optional jQuery input to disable (will use jq if null)
     */
    export function disableInputWidget(jq: JQuery, input: JQuery | undefined | null): void;
    /**
     * Enables CSS and jQuery animation.
     */
    export function enableAnimations(): void;
    /**
     * Enables a button.
     * @param jq a required jQuery element to enable
     */
    export function enableButton(jq: JQuery): void;
    /**
     * Enables a widget for editing and sets it style as enabled.
     * @param jq a required jQuery element to enable
     * @param input an optional jQuery input to enable (will use jq if null)
     */
    export function enableInputWidget(jq: JQuery, input: JQuery | undefined | null): void;
    /**
     * Enables scrolling again if previously disabled via `PrimeFaces.utils.preventScrolling`.
     */
    export function enableScrolling(): void;
    /**
     * Enables navigating to an element via the tab key outside an overlay widget. Usually called when a modal
     * overlay is removed. This reverts the changes as made by `PrimeFaces.utils.preventTabbing`.
     * @param widget A modal overlay widget instance.
     * @param id ID of a modal overlay, usually the widget ID.
     */
    export function enableTabbing(widget: PrimeFaces.widget.BaseWidget, id: string): void;
    /**
     * Exclude elements such as buttons, links, inputs from being touch swiped.  Users can always add
     * `class="noSwipe"` to any element to exclude it as well.
     * @return A CSS selector for the elements to be excluded from being touch swiped.
     */
    export function excludedSwipeElements(): string;
    /**
     * Formats the allowTypes regex pattern in a more human-friendly format.
     * @param allowTypes The allowTypes regex pattern to format
     * @return The allowTypes formatted in a more human-friendly format.
     */
    export function formatAllowTypes(allowTypes: string): string;
    /**
     * Formats the given data size in a more human-friendly format, e.g., `1.5 MB` etc.
     * @param bytes File size in bytes to format
     * @return The given file size, formatted in a more human-friendly format.
     */
    export function formatBytes(bytes: number): string;
    /**
     * Finds scrollable parents (not  the document).
     * @param element An element used to find its scrollable parents.
     * @return the list of scrollable parents.
     */
    export function getScrollableParents(element: Element): Element[];
    /**
     * Is this component wrapped in a float label?
     * @param jq An element to check if wrapped in float label.
     * @return true this this JQ has a float label parent
     */
    export function hasFloatLabel(jq: JQuery | undefined | null): boolean;
    /**
     * Ignores unprintable keys on filter input text box. Useful in filter input events in many components.
     * @param e The key event that occurred.
     * @return `true` if the one of the keys to ignore was pressed, or `false` otherwise.
     */
    export function ignoreFilterKey(e: JQuery.TriggeredEvent): boolean;
    /**
     * Is this SPACE or ENTER key. Used throughout codebase to trigger and action.
     * @param e The key event that occurred.
     * @return `true` if the key is an action key, or `false` otherwise.
     */
    export function isActionKey(e: JQuery.TriggeredEvent): boolean;
    /**
     * Checks if the key pressed is cut, copy, or paste.
     * @param e The key event that occurred.
     * @return `true` if the key is cut/copy/paste, or `false` otherwise.
     */
    export function isClipboardKey(e: JQuery.TriggeredEvent): boolean;
    /**
     * Is this CMD on MacOs or CTRL key on other OSes.
     * @param e The key event that occurred.
     * @return `true` if the key is a meta key, or `false` otherwise.
     */
    export function isMetaKey(e: JQuery.TriggeredEvent): boolean;
    /**
     * Checks if a modal with the given ID is currently displayed.
     * @param id The base ID of a modal overlay, usually the widget ID.
     * @return Whether the modal with the given ID is displayed.
     */
    export function isModalActive(id: string): boolean;
    /**
     * Checks if the key pressed is a printable key like 'a' or '4' etc.
     * @param e The key event that occurred.
     * @return `true` if the key is a printable key, or `false` otherwise.
     */
    export function isPrintableKey(e: JQuery.TriggeredEvent): boolean;
    /**
     * Is this scrollable parent a type that should be bound to the window element.
     * @param jq An element to check if should be bound to window scroll.
     * @return true this this JQ should be bound to the window scroll event
     */
    export function isScrollParentWindow(jq: JQuery | undefined | null): boolean;
    /**
     * Killswitch that kills all AJAX requests, running Pollers and IdleMonitors.
     * @see {@link https://github.com/primefaces/primefaces/issues/10299|GitHub Issue 10299}
     */
    export function killswitch(): void;
    /**
     * Helper to open a new URL and if CTRL is held down open in new browser tab.
     * @param event The click event that occurred.
     * @param link The URL anchor link that was clicked.
     */
    export function openLink(event: JQuery.TriggeredEvent, link: JQuery): void;
    /**
     * Prevents the user from scrolling the document BODY element. You can enable scrolling again via
     * `PrimeFaces.utils.enableScrolling`.
     */
    export function preventScrolling(): void;
    /**
     * Given a modal overlay, prevents navigating via the tab key to elements outside of that modal overlay. Use
     * `PrimeFaces.utils.enableTabbing` to restore the original behavior.
     * @param widget An overlay widget instance.
     * @param id ID of a modal overlay widget.
     * @param zIndex The z-index of the modal overlay.
     * @param tabbablesCallback A supplier function that return a list of tabbable elements. A
     * tabbable element is an element to which the user can navigate to via the tab key.
     */
    export function preventTabbing(widget: PrimeFaces.widget.BaseWidget, id: string, zIndex: number, tabbablesCallback: () => JQuery): void;
    /**
     * Queue a microtask if delay is 0 or less and setTimeout if > 0.
     * @param fn the function to call after the delay
     * @param delay the optional delay in milliseconds
     * @return the id associated to the timeout or undefined if no timeout used
     */
    export function queueTask(fn: () => void, delay: number | undefined): number | undefined;
    /**
     * CSS Transition method for overlay panels such as SelectOneMenu/SelectCheckboxMenu/Datepicker's panel etc.
     * @param element An element for which to execute the transition.
     * @param className Class name used for transition phases.
     * @return Two handlers named `show` and `hide` that should be invoked
     * when the element gets shown and hidden. If the given element or className property is `undefined` or `null`,
     * this function returns `null`.
     */
    export function registerCSSTransition(element: JQuery | undefined | null, className: string | undefined | null): PrimeFaces.CssTransitionHandler | null;
    /**
     * Registers a callback that is invoked when a scroll event is triggered on The DOM element for the widget that
     * has a connected overlay.
     * @param widget A widget instance for which to register a scroll handler.
     * @param scrollNamespace A scroll event with a namespace, such as `scroll.widgetId`.
     * @param element A DOM element used to find scrollable parents.
     * @param scrollCallback A callback that is invoked when a scroll event
     * occurs on the widget.
     * @return unbind callback handler
     */
    export function registerConnectedOverlayScrollHandler(widget: PrimeFaces.widget.BaseWidget, scrollNamespace: string, element: JQuery | undefined, scrollCallback: (event: JQuery.TriggeredEvent) => void): PrimeFaces.UnbindCallback;
    /**
     * Sets up an overlay widget. Appends the overlay widget to the element as specified by the `appendTo`
     * attribute. Also makes sure the overlay widget is handled properly during AJAX updates.
     * @param widget An overlay widget instance.
     * @param overlay The DOM element for the overlay.
     * @param overlayId The ID of the overlay, usually the widget ID.
     * @return The overlay that was passed to this function.
     */
    export function registerDynamicOverlay(widget: PrimeFaces.widget.DynamicOverlayWidget, overlay: JQuery, overlayId: string): JQuery;
    /**
     * Registers a callback on the document that is invoked when the user clicks on an element outside the overlay
     * widget.
     * @param widget An overlay widget instance.
     * @param hideNamespace A click event with a namespace to listen to, such as `mousedown.widgetId`.
     * @param overlay The DOM element for the overlay.
     * @param resolveIgnoredElementsCallback The callback which
     * resolves the elements to ignore when the user clicks outside the overlay. The `hideCallback` is not invoked
     * when the user clicks on one those elements.
     * @param hideCallback A callback that is invoked when the
     * user clicks on an element outside the overlay widget.
     * @return Unbind callback handler
     */
    export function registerHideOverlayHandler(widget: PrimeFaces.widget.BaseWidget, hideNamespace: string, overlay: JQuery, resolveIgnoredElementsCallback: ((event: JQuery.TriggeredEvent) => JQuery) | undefined, hideCallback: (event: JQuery.TriggeredEvent, eventTarget: JQuery) => void): PrimeFaces.UnbindCallback;
    /**
     * Registers a callback that is invoked when the window is resized.
     * @param widget A widget instance for which to register a resize handler.
     * @param resizeNamespace A resize event with a namespace to listen to, such as `resize.widgetId`.
     * @param element An element that prevents the callback from being invoked when it is not
     * visible, usually a child element of the widget.
     * @param resizeCallback A callback that is invoked when the window is resized.
     * @param params Optional CSS selector. If given, the callback is invoked only when the resize event
     * is triggered on an element the given selector.
     * @return Unbind callback handler
     */
    export function registerResizeHandler(widget: PrimeFaces.widget.BaseWidget, resizeNamespace: string, element: JQuery | undefined, resizeCallback: (event: JQuery.TriggeredEvent) => void, params?: string): PrimeFaces.UnbindCallback;
    /**
     * Registers a callback that is invoked when a scroll event is triggered on the DOM element for the widget.
     * @param widget A widget instance for which to register a scroll handler.
     * @param scrollNamespace A scroll event with a namespace, such as `scroll.widgetId`.
     * @param scrollCallback A callback that is invoked when a scroll event
     * occurs on the widget.
     * @return unbind callback handler
     */
    export function registerScrollHandler(widget: PrimeFaces.widget.BaseWidget, scrollNamespace: string, scrollCallback: (event: JQuery.TriggeredEvent) => void): PrimeFaces.UnbindCallback;
    /**
     * Removes the overlay from the overlay container as specified by the `appendTo` attribute.
     * @param widget The overlay widget instance.
     * @param overlay The (new) DOM element of the overlay.
     * @param overlayId ID of the the overlay, usually the widget ID.
     * @param appendTo The container to which the overlay is appended.
     */
    export function removeDynamicOverlay(widget: PrimeFaces.widget.DynamicOverlayWidget, overlay: JQuery, overlayId: string, appendTo: JQuery): void;
    /**
     * Given a modal overlay widget, removes the modal overlay element from the DOM. This reverts the changes as
     * made by `PrimeFaces.utils.addModal`.
     * @param widget A modal overlay widget instance.
     * @param overlay The modal overlay element should be a DIV.
     */
    export function removeModal(widget: PrimeFaces.widget.BaseWidget, overlay?: JQuery | null): void;
    /**
     * Finds the element to which the overlay panel should be appended. If none is specified explicitly, append the
     * panel to the body.
     * @param widget A widget that has a panel to be appended.
     * @param target The DOM element that is the target of this overlay
     * @param overlay The DOM element for the overlay.
     * @return The search expression for the element to which the overlay panel should be appended.
     */
    export function resolveAppendTo(widget: PrimeFaces.widget.DynamicOverlayWidget, target: JQuery, overlay: JQuery): string | null;
    /**
     * Finds the container element to which an overlay widget should be appended. This is either the element
     * specified by the widget configurations's `appendTo` attribute, or the document BODY element otherwise.
     * @param widget A widget to be displayed as an overlay.
     * @return The container DOM element to which the overlay is to be appended.
     */
    export function resolveDynamicOverlayContainer(widget: PrimeFaces.widget.DynamicOverlayWidget): JQuery;
    /**
     * This method concatenates the classes into a string according to the condition of the arguments and returns it.
     * @return class
     */
    export function styleClass(): string;
    /**
     * Removes a scroll handler as registered by `PrimeFaces.utils.registerScrollHandler`.
     * @param widget A widget instance for which a scroll handler was registered.
     * @param scrollNamespace A scroll event with a namespace, such as `scroll.widgetId`.
     */
    export function unbindScrollHandler(widget: PrimeFaces.widget.BaseWidget, scrollNamespace: string): void;
    /**
     * Handles floating label CSS if wrapped in a floating label.
     * @param element the to add the CSS classes to
     * @param inputs the input(s) to check if filled
     * @param hasFloatLabel true if this is wrapped in a floating label
     */
    export function updateFloatLabel(element: JQuery | undefined, inputs: JQuery | undefined, hasFloatLabel: boolean | undefined): void;
}
/**
 * This object contains the  widget classes that are currently available. The key is the name of the widget, the
 * value the class (constructor) of the widget. Please note that widgets are usually created by the PrimeFaces
 * framework and should not be created manually.
 *
 * There are a few base classes defined by PrimeFaces that you can use when writing the client-side part of your
 * custom widget:
 *
 * - {@link BaseWidget}: Base class that you should extend if you do not required any advanced functionality.
 * - {@link DeferredWidget}: When you widget needs to be initialized on the client in a way does required the
 * element to be visible, you can use this class as a base. A widget may not be visible, for example, when it is
 * inside a dialog or tab. The deferred widget provides the the method {@link DeferredWidget.addDeferredRender}
 * (to register a listener) and {@link DeferredWidget.renderDeferred} (to render the widget once it is visible).
 * - {@link DynamicOverlayWidget}: When your widget is an overlay with dynamically loaded content, you can use this
 * base class.
 *
 * Note to TypeScript users: you could use these widget classes to check whether a widget instance is of a certain
 * type:
 *
 * <details>
 *
 * <summary>Click to view</summary>
 *
 * ```typescript
 * type Constructor<T> = new (...args: any) => T;
 *
 * function getWidgetName(
 *   widgetType:
 *     PrimeFaces.widget.BaseWidget
 *     | Constructor<PrimeFaces.widget.BaseWidget>
 * ): string {
 *   if (typeof widgetType === "function") {
 *     for (const [name, type] of Object.entries(PrimeFaces.widget)) {
 *       if (type === widgetType) {
 *         return name;
 *       }
 *     }
 *   }
 *   else {
 *     const widgetClass = Object.getPrototypeOf(widgetType);
 *     for (const [name, type] of Object.entries(PrimeFaces.widget)) {
 *       if (
 *         "prototype" in type && widgetClass === type.prototype
 *         || widgetClass === type
 *       ) {
 *         return name;
 *       }
 *     }
 *   }
 *   return "BaseWidget";
 * }
 *
 * function getWidgetOfType<
 *   C extends Constructor<any> = Constructor<PrimeFaces.widget.BaseWidget>
 * >(widgetVar: string, widgetType: C): InstanceType<C> | undefined {
 *   const widget = PF(widgetVar);
 *   if (widget !== undefined && widget !== null) {
 *     if (widget instanceof widgetType) {
 *       // @ts-ignore
 *       return widget;
 *     }
 *     else {
 *       PrimeFaces.error([
 *         `Widget for var '${widgetVar}' of type '${getWidgetName(widget)}'`,
 *         `was found, but expected type '${getWidgetName(widgetType)}'!`
 *       ].join(" "));
 *       return undefined;
 *     }
 *   }
 *   else {
 *     return undefined;
 *   }
 * }
 * ```
 *
 * </details>
 *
 * This function could then be called like this:
 *
 * ```typescript
 * // Automatically inferred to be of type "PrimeFaces.widget.Chart | undefined"
 * const chart = getWidgetByVar("charWidgetVar", PrimeFaces.widget.Chart);
 * ```
 */
declare namespace PrimeFaces.widget {
}
declare namespace PrimeFaces.widget {
    /**
     * A destroy listener for a PrimeFaces widget. It is invoked when the
     * widget is removed, such as during AJAX updates. Use {@link BaseWidget.addDestroyListener} to add a destroy
     * listener.
     * @typeparam TWidget The type of the widget that is being destroyed.
     */
    export type DestroyListener<TWidget> =
    /**
     * @param widget The widget that is being destroyed.
     */
    (this: TWidget, widget: TWidget) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * A callback for a PrimeFaces widget. An optional callback that is
     * invoked after a widget was created successfully, at the end of the {@link BaseWidget.init | init} method. This is
     * usually specified via the `widgetPostConstruct` attribute on the JSF component. Note that this is also called
     * during a `refresh` (AJAX update).
     */
    export type PostConstructCallback =
    /**
     * @param widget The widget that was constructed.
     */
    (this: BaseWidget, widget: BaseWidget) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * An optional callback that is invoked after a widget was refreshed
     * after an AJAX update, at the end of the {@link BaseWidget.refresh | refresh} method. This is usually specified
     * via the `widgetPostRefresh` attribute on the JSF component.
     */
    export type PostRefreshCallback =
    /**
     * @param widget The widget that was refreshed.
     */
    (this: BaseWidget, widget: BaseWidget) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * An optional callback that is invoked before a widget is about to be
     * destroyed, e.g., when the component was removed at the end of an AJAX update. This is called at the beginning
     * of the {@link BaseWidget.destroy | destroy} method. This is usually specified via the `widgetPreDestroy`
     * attribute on the JSF component.
     */
    export type PreDestroyCallback =
    /**
     * @param widget The widget that is about to be destroyed.
     */
    (this: BaseWidget, widget: BaseWidget) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * A refresh listener for a PrimeFaces widget. It is invoked when the
     * widget is reloaded, such as during AJAX updates. Use {@link BaseWidget.addRefreshListener} to add a refresh
     * listener.
     * @typeparam TWidget The type of the widget that is being refreshed.
     */
    export type RefreshListener<TWidget> =
    /**
     * @param widget The widget that is being refreshed.
     */
    (this: TWidget, widget: TWidget) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Base Widget__
     *
     * BaseWidget for the PrimeFaces widgets framework.
     *
     * It provides some common functionality for other widgets. All widgets should inherit from this class, or an
     * appropriate sub class in the following manner:
     *
     * ```javascript
     * class MyWidget extends PrimeFaces.widget.BaseWidget {
     *
     *   init(cfg) {
     *     super.init(cfg);
     *     // custom initialization
     *   }
     *
     *   // more methods required by your widget
     *
     * }
     * ```
     *
     * Or, alternatively, if you need to support old browsers and do not wish to transpile your code:
     *
     * ```javascript
     * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
     *   init: function(cfg) {
     *     this._super(cfg);
     *   }
     * });
     * ```
     *
     * If your widget needs to be visible before it can be rendered, consider using the {@link DeferredWidget} as a
     * base class instead.
     * @typeparam TCfg Defaults to `PrimeFaces.widget.BaseWidgetCfg`. Type of the configuration
     * object for this widget.
     */
    export class BaseWidget<TCfg extends PrimeFaces.widget.BaseWidgetCfg = PrimeFaces.widget.BaseWidgetCfg> {
        /**
         * The configuration of this widget instance. Please note that
         * no property is guaranteed to be present, you should always check for `undefined` before accessing a property.
         * This is partly because the value of a property is not transmitted from the server to the client when it equals
         * the default.
         */
        cfg: PrimeFaces.PartialWidgetCfg<TCfg>;
        /**
         * Creates a new instance of this widget. Please note that you should __NOT__ override this
         * constructor. Instead, override the {@link init} method, which is called at the end of the constructor once the
         * instance is created.
         * @param cfg The widget configuration to be used for this widget
         * instance. This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for
         * this component.
         */
        constructor(cfg: PrimeFaces.PartialWidgetCfg<TCfg>);
        /**
         * Array of registered listeners invoked
         * when this widget is destroyed. You should normally not use modify this directly, use {@link addDestroyListener}
         * instead.
         */
        destroyListeners: PrimeFaces.widget.DestroyListener<BaseWidget>[];
        /**
         * The client-side ID of this widget, with all parent naming containers, such as
         * `myForm:myWidget`. This is also the ID of the container HTML element for this widget. In case the widget needs
         * multiple container elements (such as {@link Paginator}), this may also be an array if IDs.
         */
        id: string | string[];
        /**
         * The jQuery instance of the container element of this widget. In case {@link id} is an array, it
         * will contain multiple elements. Please note that some widgets have got not DOM elements at all, in this case this
         * will be an empty jQuery instance.
         */
        jq: JQuery;
        /**
         * A CSS selector for the container element (or elements, in case {@link id} is an array) of
         * this widget, This is usually an ID selector (that is properly escaped). You can select the container element or
         * elements like this: `$(widget.jqId)`.
         */
        jqId: string;
        /**
         * Array of registered listeners invoked
         * when this widget is refreshed. You should normally not use modify this directly, use {@link addRefreshListener}
         * instead.
         */
        refreshListeners: PrimeFaces.widget.RefreshListener<BaseWidget>[];
        /**
         * The name of the widget variables of this widget. The widget variable can be used to
         * access a widget instance by calling `PF('myWidgetVar')`.
         */
        widgetVar: string;
        /**
         * Lets you register a listener that is called before the component is destroyed.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * all destroy listeners are called. This makes it possible to add listeners from outside the widget code.
         *
         * If you call this method twice with the same listener, it will be registered twice and later also called
         * twice.
         *
         * Note that for this to work, you must not override the `destroy` method; or if you do, call `super`.
         *
         * Also, after this widget was detached is done, all destroy listeners will be unregistered.
         * @param listener A destroy listener to be registered.
         * @since 7.0
         */
        addDestroyListener(listener: PrimeFaces.widget.DestroyListener<this>): void;
        /**
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, all
         * refresh listeners are called. This makes it possible to add listeners from outside the widget code.
         *
         * If you call this method twice with the same listener, it will be registered twice and later also called
         * twice.
         *
         * Note that for this to work, you must not override the `refresh` method; or if you do, call `super`.
         *
         * Also, after the refresh is done, all refresh listeners will be deregistered. If you added the listeners from
         * within this widget, consider adding the refresh listeners not only in the `init` method, but also again in
         * the `refresh` method after calling `super`.
         * @param listener A refresh listener to be registered.
         * @since 7.0.0
         */
        addRefreshListener(listener: PrimeFaces.widget.RefreshListener<this>): void;
        /**
         * Each widget may have one or several behaviors attached to it. This method calls all attached behaviors for
         * the given event name. In case no such behavior exists, this method does nothing and returns immediately.
         *
         * A behavior is a way for associating client-side scripts with UI components that opens all sorts of
         * possibilities, including client-side validation, DOM and style manipulation, keyboard handling, and more.
         * When the behavior is triggered, the configured JavaScript gets executed.
         *
         * Behaviors are often, but not necessarily, AJAX behavior. When triggered, it initiates a request the server
         * and processes the response once it is received. This enables several features such as updating or replacing
         * elements dynamically. You can add an AJAX behavior via
         * `<p:ajax event="name" actionListener="#{...}" onstart="..." />`.
         * @param event The name of an event to call.
         * @param ext Additional configuration that is passed to the
         * AJAX request for the server-side callback.
         * @since 7.0
         */
        callBehavior(event: string, ext?: Partial<PrimeFaces.ajax.ConfigurationExtender>): void;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         */
        destroy(): void;
        /**
         * Each widget may have one or several behaviors attached to it. This method returns the callback function for
         * the given event.
         *
         * __Note__: Do not call the method directly, the recommended way to invoke a behavior is via
         * {@link callBehavior}.
         *
         * A behavior is a way for associating client-side scripts with UI components that opens all sorts of
         * possibilities, including client-side validation, DOM and style manipulation, keyboard handling, and more.
         * When the behavior is triggered, the configured JavaScript gets executed.
         *
         * Behaviors are often, but not necessarily, AJAX behavior. When triggered, it initiates a request the server
         * and processes the response once it is received. This enables several features such as updating or replacing
         * elements dynamically. You can add an AJAX behavior via
         * `<p:ajax event="name" actionListener="#{...}" onstart="..." />`.
         * @param name The name of an event for which to retrieve the behavior.
         * @return The behavior with the given name, or `null` if no such behavior
         * exists.
         */
        getBehavior(name: string): PrimeFaces.Behavior | null;
        /**
         * Each widget has got a container element, this method returns that container. This container element is
         * usually also the element whose ID is the client-side ID of the JSF component.
         * @return The jQuery instance representing the main HTML container element of this widget.
         */
        getJQ(): JQuery;
        /**
         * Gets the closest parent form for this widget.
         * @return A JQuery instance that either contains the form when found, or an empty JQuery instance when
         * the form could not be found.
         * @since 10.0.0
         */
        getParentForm(): JQuery;
        /**
         * Gets the closest parent form ID for this widget lazily so it can be used in AJAX requests.
         * @return Either the form ID or `undefined` if no form can be found.
         * @since 10.0.0
         */
        getParentFormId(): string | undefined;
        /**
         * Each widget may have one or several behaviors attached to it. This method checks whether this widget has got
         * at least one behavior associated with given event name.
         *
         * A behavior is a way for associating client-side scripts with UI components that opens all sorts of
         * possibilities, including client-side validation, DOM and style manipulation, keyboard handling, and more.
         * When the behavior is triggered, the configured JavaScript gets executed.
         *
         * Behaviors are often, but not necessarily, AJAX behavior. When triggered, it initiates a request the server
         * and processes the response once it is received. This enables several features such as updating or replacing
         * elements dynamically. You can add an AJAX behavior via
         * `<p:ajax event="name" actionListener="#{...}" onstart="..." />`.
         * @param event The name of an event to check.
         * @return `true` if this widget has the given behavior, `false` otherwise.
         */
        hasBehavior(event: string): boolean;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Checks if this widget is detached, ie whether the HTML element of this widget is currently contained within
         * the DOM (the HTML body element). A widget may become detached during an AJAX update, and it may remain
         * detached in case the update removed this component from the component tree.
         * @return `true` if this widget is currently detached, or `false` otherwise.
         */
        isDetached(): boolean;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): unknown;
        /**
         * Removes the widget's script block from the DOM. Currently, the ID of this script block consists of the
         * client-side ID of this widget with the prefix `_s`, but this is subject to change.
         * @param clientId The client-side ID of the widget.
         */
        removeScriptElement(clientId: string | string[]): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  BaseWidget| BaseWidget widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified. This configuration is
     * always accessible via the `cfg` property of a widget and consists of key-value pairs. Please note that, in order
     * to save bandwidth, the server only sends a value for a given configuration key when the value differs from the
     * default value. That is, you must expect any configuration value to be absent and make sure you check for its
     * presence before accessing it.
     */
    export interface BaseWidgetCfg {
        /**
         * A map with all behaviors that
         * were defined for this widget. The key is the name of the behavior, the value is the callback function that is
         * invoked when the behavior is called.
         */
        behaviors: Record<string, PrimeFaces.Behavior>;
        /**
         * ID of the form to use for AJAX requests.
         */
        formId?: string;
        /**
         * The client-side ID of the widget, with all parent naming containers, such as
         * `myForm:myWidget`. This is also the ID of the container HTML element for this widget. In case the widget needs
         * multiple container elements (such as {@link Paginator}), this may also be an array if IDs.
         */
        id: string | string[];
        /**
         * An optional callback that is invoked
         * after this widget was created successfully, at the end of the {@link BaseWidget.init | init} method. This is
         * usually specified via the `widgetPostConstruct` attribute on the JSF component. Note that this is also called
         * during a `refresh` (AJAX update).
         */
        postConstruct: PrimeFaces.widget.PostConstructCallback;
        /**
         * An optional callback that is invoked after
         * this widget was refreshed after an AJAX update, at the end of the {@link BaseWidget.refresh | refresh} method.
         * This is usually specified via the `widgetPostRefresh` attribute on the JSF component.
         */
        postRefresh: PrimeFaces.widget.PostRefreshCallback;
        /**
         * An optional callback that is invoked before
         * this widget is about to be destroyed, e.g., when the component was removed at the end of an AJAX update. This is
         * called at the beginning of the {@link BaseWidget.destroy | destroy} method. This is usually specified via the
         * `widgetPreDestroy` attribute on the JSF component.
         */
        preDestroy: PrimeFaces.widget.PreDestroyCallback;
        /**
         * The name of the widget variables of this widget. The widget variable can be used to
         * access a widget instance by calling `PF("myWidgetVar")`.
         */
        widgetVar: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DynamicOverlay Widget__
     *
     * Base class for widgets that are displayed as an overlay. At any given time, several overlays may be active. This
     * requires that the z-index of the overlays is managed globally. This base class takes care of that.
     * @typeparam TCfg Defaults to `DynamicOverlayWidgetCfg`. Type of the configuration object for this widget.
     */
    export class DynamicOverlayWidget<TCfg extends DynamicOverlayWidgetCfg = DynamicOverlayWidgetCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The search expression for the element to which the overlay panel should be appended.
         */
        appendTo: string | null;
        /**
         * The DOM element that is displayed as an overlay with the appropriate `z-index` and
         * `position`. It is usually a child of the `body` element.
         */
        modalOverlay: JQuery;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         * @override
         */
        override destroy(): void;
        /**
         * Disabled modality for this widget and removes the modal overlay element, but does not change whether the
         * overlay is currently displayed.
         * @param overlay The target overlay, if not given default to
         * {@link PrimeFaces.widget.BaseWidget.jq | this.jq}.
         */
        disableModality(overlay?: JQuery | null): void;
        /**
         * Enables modality for this widget and creates the modal overlay element, but does not change whether the
         * overlay is currently displayed.
         * @param overlay The target overlay, if not given default to
         * {@link PrimeFaces.widget.BaseWidget.jq | this.jq}.
         */
        enableModality(overlay?: JQuery | null): void;
        /**
         * This class makes sure a user cannot tab out of the modal and it stops events from targets outside of the
         * overlay element. This requires that we switch back to the modal in case a user tabs out of it. What must
         * be returned by this method are the elements to which the user may switch via tabbing.
         * @return The DOM elements which are allowed to be focused via tabbing. May be an empty `jQuery`
         * instance when the modal contains no tabbable elements, but must not be `undefined`.
         */
        protected getModalTabbables(): JQuery;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg the widget configuraton
         *
         * (from super type BaseWidget) The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         * @param overlay The DOM element for the overlay.
         * @param overlayId The ID of the overlay, usually the widget ID.
         * @param target The DOM element that is the target of this overlay
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>, overlay?: JQuery, overlayId?: string, target?: JQuery): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DynamicOverlayWidget| DynamicOverlayWidget widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DynamicOverlayWidgetCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * `true` to prevent the body from being scrolled, `false` otherwise.
         */
        blockScroll: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Deferred Widget__
     *
     * Base class for widgets that require their container to be visible to initialize properly.
     *
     * For example, a widget may need to know the width and height of its container so that it can resize itself
     * properly.
     *
     * Do not call the {@link render} or {@link _render} method directly in the {@link init} method. Instead, call
     * {@link renderDeferred}. PrimeFaces will then check whether the widget is visible and call the {@link _render}
     * method once it is. Make sure you actually override the {@link _render} method, as the default implementation
     * throws an error.
     *
     * ```javascript
     * class MyWidget extends PrimeFaces.widget.DeferredWidget {
     *   init(cfg) {
     *     super.init(cfg);
     *
     *     // more code if needed
     *     // ...
     *
     *     // Render this widget once its container is visible.
     *     this.renderDeferred();
     *   }
     *
     *   _render() {
     *     // Perform your render logic here, create DOM elements etc.
     *   }
     * }
     * ```
     * @typeparam TCfg Defaults to `DeferredWidgetCfg`. Type of the configuration object for this widget.
     */
    export class DeferredWidget<TCfg extends DeferredWidgetCfg = DeferredWidgetCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         */
        protected _render(): void;
        /**
         * Adds a deferred rendering task for the given widget to the queue.
         * @param widgetId The ID of a deferred widget.
         * @param container The container element that should be visible.
         * @param callback Callback that is invoked when the widget _may_ possibly have become visible.
         * Should return `true` when the widget was rendered, or `false` when the widget still needs to be rendered
         * later.
         */
        protected addDeferredRender(widgetId: string, container: JQuery, callback: () => boolean): void;
        /**
         * Cleans up deferred render tasks. When you extend this class and override this method, make sure to call
         * `super`.
         * @override
         */
        override destroy(): void;
        /**
         * Called after the widget has become visible and after it was rendered. May be overridden, the default
         * implementation is a no-op.
         */
        protected postRender(): void;
        /**
         * This render method to check whether the widget container is visible. Do not override this method, or the
         * deferred widget functionality may not work properly anymore.
         * @return `true` if the widget container is visible, `false` or
         * `undefined` otherwise.
         */
        render(): PrimeFaces.ReturnOrVoid<boolean | undefined>;
        /**
         * Call this method in the {@link init} method if you want deferred rendering support. This method checks
         * whether the container of this widget is visible and call {@link _render} only once it is.
         */
        renderDeferred(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DeferredWidget| DeferredWidget widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that
     * this configuration is usually meant to be read-only and should not be modified.
     */
    export interface DeferredWidgetCfg extends PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Dashboard Widget__
     *
     * Dashboard provides a portal like layout with drag & drop based reorder capabilities.
     *
     * Currently this uses the JQueryUI sortable widget. You can use `$.fn.sortable` to interact with the dashboard
     * programmatically.
     *
     * ```javascript
     * const widget = PF("MyDashboardWidget");
     *
     * // When dragged outside the dashboard: Have the items revert to their new positions using a smooth animation
     * widget.jq.find(".ui-dashboard-column").sortable("option", "revert", true);
     * ```
     * @typeparam TCfg Defaults to `DashboardCfg`. Type of the configuration object for this widget.
     */
    export class Dashboard<TCfg extends DashboardCfg = DashboardCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Sets up all draggable panels.
         */
        private bindDraggable(): void;
        /**
         * Sets up all droppable panels.
         */
        private bindDroppable(): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindSortableEvents(): void;
        /**
         * Disables this dashboard so that it cannot be modified.
         */
        disable(): void;
        /**
         * Enables this dashboard so that it can be modified.
         */
        enable(): void;
        /**
         * Handle dropping a panel either from legacy sortable or responsive draggable.
         * @param widget the Dashboard widget
         * @param e Event that occurred.
         * @param ui the UI element that was dragged
         */
        handleDrop(widget: PrimeFaces.widget.BaseWidget, e: Event, ui: JQuery): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Sets up the responsive drag drop for the dashboard.
         */
        private renderResponsive(): void;
        /**
         * Sets up the sortable for the legacy dashboard.
         */
        private renderSortable(): void;
        /**
         * Sets up the sortable for the legacy dashboard.
         * @param a the first panel
         * @param b the second panel
         */
        private swapPanels(a: HTMLElement, b: HTMLElement): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Dashboard| Dashboard widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DashboardCfg extends JQueryUI.SortableOptions, PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DataGrid Widget__
     *
     * DataGrid displays a collection of data in a grid layout.
     *
     * __DataGrid is deprecated, use DataView instead.__
     * @typeparam TCfg Defaults to `DataGridCfg`. Type of the configuration object for this widget.
     * @deprecated
     */
    export class DataGrid<TCfg extends DataGridCfg = DataGridCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * DOM element of the content container for the data grid.
         */
        content: JQuery;
        /**
         * When pagination is enabled: The paginator widget instance used for
         * paging.
         */
        paginator: PrimeFaces.widget.Paginator;
        /**
         * Retrieves the paginator widget used by this data grid for pagination. You can use this widget to switch to a
         * different page programmatically.
         * @return The paginator widget, or `undefined` when pagination is not
         * enabled.
         */
        getPaginator(): PrimeFaces.widget.Paginator | undefined;
        /**
         * Handles a pagination event by updating the data grid and invoking the appropriate behaviors.
         * @param newState The new pagination state to apply.
         */
        private handlePagination(newState: PrimeFaces.widget.Paginator.PaginationState): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Initializes the paginator, called during widget initialization.
         */
        private setupPaginator(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DataGrid| DataGrid widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DataGridCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * When pagination is enabled: The paginator configuration
         * for the paginator.
         */
        paginator: Partial<PrimeFaces.widget.PaginatorCfg>;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DataList Widget__
     *
     * DataList presents a collection of data in list layout with several display types.
     *
     * __DataList is deprecated, use DataView instead.__
     * @typeparam TCfg Defaults to `DataListCfg`. Type of the configuration object for this widget.
     * @deprecated
     */
    export class DataList<TCfg extends DataListCfg = DataListCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * DOM element of the content container for the data grid.
         */
        content: JQuery;
        /**
         * When pagination is enabled: The paginator widget instance used for
         * paging.
         */
        paginator: PrimeFaces.widget.Paginator;
        /**
         * Retrieves the paginator widget used by this data grid for pagination. You can use this widget to switch to a
         * different page programatically.
         * @return The paginator widget, or `undefined` when pagination is not
         * enabled.
         */
        getPaginator(): PrimeFaces.widget.Paginator | undefined;
        /**
         * Handles a pagination event by updating the data grid and invoking the appropriate behaviors.
         * @param newState The new pagination state to apply.
         */
        private handlePagination(newState: PrimeFaces.widget.Paginator.PaginationState): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Initializes the paginator, called during widget initialization.
         */
        private setupPaginator(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DataList| DataList widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DataListCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * When pagination is enabled: The paginator configuration
         * for the paginator.
         */
        paginator: Partial<PrimeFaces.widget.PaginatorCfg>;
    }
}
declare namespace PrimeFaces.widget.DataScroller {
    /**
     * Defines when more items are loaded by the
     * data scroller. `scroll` loads more items as the user scrolls down the page, `manual` loads more items only when the
     * user click the `more` button.
     */
    export type LoadEvent = "scroll" | "manual";
}
declare namespace PrimeFaces.widget.DataScroller {
    /**
     * Target to listen to for the scroll event.
     * `document` registers a delegated listener on the document element, `inline` registers it on an element of the data
     * scroller.
     */
    export type Mode = "document" | "inline";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DataScroller Widget__
     *
     * DataScroller displays a collection of data with on demand loading using scrolling.
     * @typeparam TCfg Defaults to `DataScrollerCfg`. Type of the configuration object for this widget.
     */
    export class DataScroller<TCfg extends DataScrollerCfg = DataScrollerCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * `true` if all items were loaded and there are no more items to be loaded, or `false`
         * otherwise.
         */
        allLoaded: boolean;
        /**
         * DOM element of the container for the content with the data scroller.
         */
        content: JQuery;
        /**
         * Height in pixels of each row, when virtual scrolling is enabled.
         */
        itemHeight?: number;
        /**
         * DOM element of the list with the data items.
         */
        list: JQuery;
        /**
         * DOM element of the status text or icon shown while loading.
         */
        loadStatus: JQuery;
        /**
         * DOM element of the `more` button for loading more item manually.
         */
        loadTrigger: JQuery;
        /**
         * DOM element of the container with the `more` button for loading more items.
         */
        loaderContainer: JQuery;
        /**
         * `true` if an AJAX request for loading more items is currently process, or `false` otherwise.
         */
        loading: boolean;
        /**
         * Timeout ID of the timer for the scroll animation.
         */
        scrollTimeout?: number;
        /**
         * Whether virtual scrolling is currently active (if enabled at all).
         */
        virtualScrollActive?: boolean;
        /**
         * Sets up the event listeners for the click on the `more` button.
         */
        private bindManualLoader(): void;
        /**
         * Sets up the event listeners for the scroll event, to load more items on-demand.
         */
        private bindScrollListener(): void;
        /**
         * Finds the height of the content, excluding the padding.
         * @return The inner height of the content element.
         */
        private getInnerContentHeight(): number;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Loads more items from the server. Usually triggered either when the user scrolls down or when they click on the
         * `more` button.
         */
        load(): void;
        /**
         * Loads more items and inserts them into the DOM so that the user can see them.
         * @param page The page of the items to load. The items are grouped into pages, each page containts
         * `chunkSize` items.
         * @param callback Callback that is invoked when the new items have been loaded and inserted into the
         * DOM.
         */
        private loadRowsWithVirtualScroll(page: number, callback: () => void): void;
        /**
         * Checks whether more items can be loaded now. Item are not allowed to be loaded when an AJAX request is currently
         * in process, or when all items have been loaded already.
         * @return `true` if more items are allowed to be loaded, `false` otherwise.
         */
        shouldLoad(): boolean;
        /**
         * Inserts newly loaded items into the DOM.
         * @param data New HTML content of the items to insert.
         * @param clear `true` to clear all currently existing items, or `false` otherwise.
         * @param pre `true` to prepend the items, or `false` or `undefined` to append the items to the list of
         * items.
         */
        private updateData(data: string, clear?: boolean, pre?: boolean): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DataScroller| DataScroller widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DataScrollerCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Percentage height of the buffer between the bottom of the page and the scroll position to
         * initiate the load for the new chunk. For example, a value of `10` means that loading happens after the user has
         * scrolled down to at least `90%` of the viewport height.
         */
        buffer: number;
        /**
         * Number of items to load on each load.
         */
        chunkSize: number;
        /**
         * Defines when more items are loaded.
         */
        loadEvent: PrimeFaces.widget.DataScroller.LoadEvent;
        /**
         * Defines the target to listen for scroll event.
         */
        mode: PrimeFaces.widget.DataScroller.Mode;
        /**
         * Number of additional items currently loaded.
         */
        offset: number;
        /**
         * `true` to set the scroll position to the bottom initally and load data from the
         * bottom, or `false` otherwise.
         */
        startAtBottom: boolean;
        /**
         * The total number of items that can be displayed.
         */
        totalSize: number;
        /**
         * Loads data on demand as the scrollbar gets close to the bottom.
         */
        virtualScroll: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DataTable with Frozen Columns Widget__
     * @typeparam TCfg Defaults to `FrozenDataTableCfg`. Type of the configuration object for this widget.
     */
    export class FrozenDataTable<TCfg extends FrozenDataTableCfg = FrozenDataTableCfg> extends PrimeFaces.widget.DataTable<TCfg> {
        /**
         * The DOM element for the frozen body.
         */
        frozenBody: JQuery;
        /**
         * The DOM element for the frozen body TABLE.
         */
        frozenBodyTable: JQuery;
        /**
         * Always `undefined` and not used.
         */
        frozenColgroup: undefined;
        /**
         * The DOM element for the container of the frozen table.
         */
        frozenContainer: JQuery;
        /**
         * The DOM element for the frozen footer.
         */
        frozenFooter: JQuery;
        /**
         * The DOM elements for the frozen columns of the footer.
         */
        frozenFooterCols: JQuery;
        /**
         * The DOM element for the frozen data table footer TABLE.
         */
        frozenFooterTable: JQuery;
        /**
         * The DOM element for the frozen group resizers of the footer.
         */
        frozenGroupResizers: JQuery;
        /**
         * The DOM element for the frozen header.
         */
        frozenHeader: JQuery;
        /**
         * The DOM element for the frozen layout container.
         */
        frozenLayout: JQuery;
        /**
         * The DOM element for the header TBODY.
         */
        frozenTbody: JQuery;
        /**
         * The DOM element for the header TFOOT.
         */
        frozenTfoot: JQuery;
        /**
         * The DOM element for the clone of the frozen THEAD.
         */
        frozenTheadClone: JQuery;
        /**
         * The DOM element for the TABLE of the scrollable body.
         */
        scrollBodyTable: JQuery;
        /**
         * Always `undefined` and not used.
         */
        scrollColgroup: undefined;
        /**
         * The DOM element for the container of the scrollable body.
         */
        scrollContainer: JQuery;
        /**
         * The DOM element for the scrollable columns of the footer.
         */
        scrollFooterCols: JQuery;
        /**
         * The DOM element for the TABLE of the scrollable footer.
         */
        scrollFooterTable: JQuery;
        /**
         * The DOM element for the group resizers of the scrollable body.
         */
        scrollGroupResizers: JQuery;
        /**
         * The DOM element for the TABLE of the scrollable header.
         */
        scrollHeaderTable: JQuery;
        /**
         * The DOM element for the scrollable layout container.
         */
        scrollLayout: JQuery;
        /**
         * The DOM element for the scrollable TFOOT.
         */
        scrollTfoot: JQuery;
        /**
         * The DOM element for the scrollable THEAD.
         */
        scrollThead: JQuery;
        /**
         * The DOM element for the clone of the scrollable THEAD.
         */
        scrollTheadClone: JQuery;
        /**
         * Adds an invisible row for internal purposes.
         * @param body Body of this data table.
         * @param header Header of this data table.
         * @param headerClone Cloned header of this data table, see method `cloneHead`.
         * @param footerTable Footer of this data table.
         * @param columnClass Optional CSS class for the ghost columns.
         */
        protected _addGhostRow(body: JQuery, header: JQuery, headerClone: JQuery, footerTable: JQuery, columnClass?: string): void;
        /**
         * Finds the resizer DOM element that matches the given draggable event params.
         * @param ui Data of the drag event.
         * @param resizers List of all available resizers.
         * @return DOM element of the resizer.
         */
        protected _findGroupResizer(ui: JQueryUI.DraggableEventUIParams, resizers: JQuery): JQuery | null;
        /**
         * Adjusts the width of the given columns to fit the current settings.
         * @param header Header of this data table.
         * @param footerCols The columns to adjust.
         */
        protected _fixColumnWidths(header: JQuery, footerCols: JQuery): void;
        /**
         * Adds and sets up an invisible row for internal purposes.
         * @override
         */
        protected override addGhostRow(): void;
        /**
         * Adds the resizers for change the width of a column of this DataTable.
         * @override
         */
        protected override addResizers(): void;
        /**
         * Adjusts the height of the body of this DataTable for the current scrolling settings.
         * @override
         */
        protected override adjustScrollHeight(): void;
        /**
         * Adjusts the width of the header, body, and footer of this DataTable to fit the current settings.
         * @override
         */
        protected override adjustScrollWidth(): void;
        /**
         * Stores the row which is currently focused.
         * @override
         * @param row Row to set as the focused row.
         */
        protected override assignFocusedRow(row: JQuery): void;
        /**
         * Sets up the event listeners for hovering over a DataTable row.
         * @override
         * @param selector Selector for the row elements. Any hover event that does not reach an element that
         * matches this selector will be ignored.
         */
        protected override bindRowHover(selector: string): void;
        /**
         * Creates and stores a cloned copy of the table head(er) of this DataTable, and sets up some event handlers.
         * @override
         */
        protected override cloneHead(): void;
        /**
         * Collapses the given row, if it is expandable. Use `findRow` to get a row by its index. Does not update the row
         * expansion toggler button.
         * @override
         * @param row Row to collapse.
         */
        protected override collapseRow(row: JQuery): void;
        /**
         * Clones the given row and returns it
         * @param original DOM element of the original row.
         * @return The cloned row.
         */
        copyRow(original: JQuery): JQuery;
        /**
         * Display the given HTML string in the specified row. Called mainly after an AJAX request.
         * @override
         * @param row Row to display.
         * @param content HTML string of the content to add to the row
         */
        protected override displayExpandedRow(row: JQuery, content: string): void;
        /**
         * Finds the group resizer element for the given drag event data.
         * @override
         * @param ui Data of the drag event.
         *
         * (from super type DataTable) Data for the drag event.
         * @return The resizer DOM element.
         */
        protected override findGroupResizer(ui: JQueryUI.DraggableEventUIParams): JQuery | null;
        /**
         * Adjusts the width of the given columns to fit the current settings.
         * @override
         */
        protected override fixColumnWidths(): void;
        /**
         * Adjusts the height of the given rows to fit the current settings.
         * @param scrollRows The scrollable rows to adjust.
         * @param frozenRows The frozen rows to adjust.
         */
        protected fixRowHeights(scrollRows: JQuery, frozenRows: JQuery): void;
        /**
         * Adjusts the height of all rows to fit the current settings.
         */
        fixRowHeightsAll(): void;
        /**
         * Finds the list of row that are currently expanded.
         * @override
         * @return All rows (`TR`) that are currently expanded.
         */
        override getExpandedRows(): JQuery;
        /**
         * Finds the body of this DataTable with the property that the user can focus it.
         * @override
         * @return The body of this DataTable.
         */
        protected override getFocusableTbody(): JQuery;
        /**
         * Finds all editors of a row. Usually each editable column has got an editor.
         * @override
         * @param row A row for which to find its row editors.
         * @return A list of row editors for each editable column of the given row
         */
        protected override getRowEditors(row: JQuery): JQuery;
        /**
         * Retrieves the table body of this DataTable.
         * @override
         * @return DOM element of the table body.
         */
        override getTbody(): JQuery;
        /**
         * Retrieves the table footer of this DataTable.
         * @override
         * @return DOM element of the table footer.
         */
        override getTfoot(): JQuery;
        /**
         * Retrieves the table header of this DataTable.
         * @override
         * @return DOM element of the table header.
         */
        override getThead(): JQuery;
        /**
         * Finds the twin row of the given row. The data table body has got two sets of rows.
         * @param row Row for which to find the twin
         * @return DOM element of the twin row.
         */
        getTwinRow(row: JQuery): JQuery;
        /**
         * When row grouping is enabled, groups all rows accordingly.
         * @override
         */
        protected override groupRows(): void;
        /**
         * Checks whether this DataTable has got any column groups.
         * @override
         * @return `true` if this DataTable has got any column groups, or `false` otherwise.
         */
        protected override hasColGroup(): boolean;
        /**
         * Checks whether the body of this DataTable overflow vertically.
         * @override
         * @return `true` if any content overflow vertically, `false` otherwise.
         */
        protected override hasVerticalOverflow(): boolean;
        /**
         * Highlights the currently focused row (if any) by adding the appropriate CSS class.
         * @override
         */
        protected override highlightFocusedRow(): void;
        /**
         * Highlights row to mark it as selected.
         * @override
         * @param row Row to highlight.
         */
        protected override highlightRow(row: JQuery): void;
        /**
         * Displays row editors in invalid format.
         * @override
         * @param index 0-based index of the row to invalidate.
         */
        override invalidateRow(index: number): void;
        /**
         * Resets the scroll state of the body to a non-scrolled state.
         * @override
         */
        protected override resetVirtualScrollBody(): void;
        /**
         * Resizes this DataTable, row, or columns in response to a drag event of a resizer element.
         * @override
         * @param event Event triggered for the drag.
         * @param ui Data for the drag event.
         */
        protected override resize(event: JQuery.TriggeredEvent, ui: JQueryUI.DraggableEventUIParams): void;
        /**
         * Saves the current column order, used to preserve the state between AJAX updates etc.
         * @override
         */
        protected override saveColumnOrder(): void;
        /**
         * Applies the given scroll width to this DataTable.
         * @override
         * @param width Scroll width in pixels to set.
         */
        protected override setScrollWidth(width: number): void;
        /**
         * Prepares this DataTable for the current scrolling settings and sets up all related event handlers.
         * @override
         */
        protected override setupScrolling(): void;
        /**
         * Shows the row editor(s) for the given row (and hides the normal output display).
         * @override
         * @param row Row for which to show the row editor.
         */
        protected override showRowEditors(row: JQuery): void;
        /**
         * Unhighlights the currently focused row (if any) by adding the appropriate CSS class.
         * @override
         */
        protected override unhighlightFocusedRow(): void;
        /**
         * Removes the highlight of a row so it is no longer marked as selected.
         * @override
         * @param row Row to unhighlight.
         */
        protected override unhighlightRow(row: JQuery): void;
        /**
         * Sets the given HTML string as the content of the body of this DataTable. Afterwards, sets up all required event
         * listeners etc.
         * @override
         * @param data HTML string to set on the body.
         * @param clear Whether the contents of the table body should be removed beforehand.
         */
        protected override updateData(data: string, clear: boolean | undefined): void;
        /**
         * Updates a row with the given content
         * @override
         * @param row Row to update.
         * @param content HTML string to set on the row.
         */
        protected override updateRow(row: JQuery, content: string): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  FrozenDataTable| FrozenDataTable widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface FrozenDataTableCfg extends PrimeFaces.widget.DataTableCfg {
    }
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates whether cell editors are loaded
     * eagerly or on-demand.
     */
    export type CellEditMode = "eager" | "lazy";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates whether multiple rows or only
     * a single row of a DataTable can be selected.
     */
    export type CmSelectionMode = "single" | "multiple";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Callback that is invoked when the user clicks on a row of the
     * DataTable.
     */
    export type OnRowClickCallback =
    /**
     * @param event The click event that occurred.
     * @param row The TR row that was clicked.
     */
    (event: JQuery.TriggeredEvent, row: JQuery) => void;
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates the resize behavior of columns.
     */
    export type ResizeMode = "expand" | "fit";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * When a row is editable: whether to `save` the
     * current contents of the row or `cancel` the row edit and discard all changes.
     */
    export type RowEditAction = "cancel" | "save";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates whether row editors are loaded eagerly
     * or on-demand.
     */
    export type RowEditMode = "eager" | "lazy";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates whether multiple columns of a
     * DataTable can be expanded at the same time, or whether other expanded rows should be collapsed when a new row is
     * expanded.
     */
    export type RowExpandMode = "single" | "multiple";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Either the 0-based index of a row, or the row
     * element (`TR`) itself.
     */
    export type RowSpecifier = number | JQuery;
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates whether rows are selected via
     * radio buttons or via checkboxes.
     */
    export type SelectionMode = "radio" | "checkbox";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates how rows of a DataTable
     * may be selected, when clicking on the row itself (not the checkbox / radiobutton from p:column).
     * `new` always unselects other rows, `add` preserves the currently selected rows, and `none` disables row selection.
     */
    export type SelectionRowMode = "new" | "add" | "none";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Indicates whether a DataTable can be sorted
     * by multiple columns or only by a single column.
     */
    export type SortMode = "single" | "multiple";
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * The available sort order
     * types for the DataTable.
     */
    export type SortOrder = "ASCENDING" | "DESCENDING" | "UNSORTED";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DataTable Widget__
     *
     * DataTable displays data in tabular format.
     * @typeparam TCfg Defaults to `DataTableCfg`. Type of the configuration object for this widget.
     */
    export class DataTable<TCfg extends DataTableCfg = DataTableCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> implements PrimeFaces.widget.ContextMenu.ContextMenuProvider<PrimeFaces.widget.DataTable> {
        /**
         * Whether all available items were  already loaded.
         */
        allLoadedLiveScroll: boolean;
        /**
         * Localized message for sorting a column in ascending order.
         */
        ascMessage?: string;
        /**
         * The DOM element for the body part of the table.
         */
        bodyTable: JQuery;
        /**
         * Cache for the contents of a row. Key is the row index, value the HTML content
         * of the row.
         */
        cacheMap: Record<number, string>;
        /**
         * Number of rows to cache.
         */
        cacheRows: number;
        /**
         * DOM element of the container with the `check all` checkbox in the header.
         */
        checkAllToggler: JQuery;
        /**
         * Clone of the table header.
         */
        clone: JQuery;
        /**
         * Whether column widths are fixed or may be resized.
         */
        columnWidthsFixed: boolean;
        /**
         * DOM element of the table cell for which the context menu was opened.
         */
        contextMenuCell?: JQuery;
        /**
         * Widget with the context menu for the DataTable.
         */
        contextMenuWidget: PrimeFaces.widget.ContextMenu;
        /**
         * Current cell to be edited.
         */
        currentCell: JQuery;
        /**
         * 0-based index of row where the the cursor is located.
         */
        cursorRowMeta: PrimeFaces.widget.DataTable.RowMeta | null;
        /**
         * Localized message for sorting a column in descending order.
         */
        descMessage?: string;
        /**
         * DOM element of the icon that indicates a column is draggable.
         */
        dragIndicatorBottom: JQuery;
        /**
         * DOM element of the icon that indicates a column is draggable.
         */
        dragIndicatorTop: JQuery;
        /**
         * DOM element of the hidden input that holds the row keys of the rows that
         * are expanded. Used to preserve the expansion state during AJAX updates.
         */
        expansionHolder?: JQuery;
        /**
         * List of row indices to expand.
         */
        expansionProcess: number[];
        /**
         * ID as returned by `setTimeout` used during filtering.
         */
        filterTimeout: number;
        /**
         * DOM element of the currently focused row.
         */
        focusedRow: JQuery | null;
        /**
         * Whether the focused row includes the checkbox for selecting the row.
         */
        focusedRowWithCheckbox: boolean;
        /**
         * The DOM elements for the footer columns.
         */
        footerCols: JQuery;
        /**
         * The DOM elements for the footer table.
         */
        footerTable: JQuery;
        /**
         * The DOM element for the header THEAD.
         */
        frozenThead: JQuery;
        /**
         * The DOM elements for the resizer button of each group.
         */
        groupResizers: JQuery;
        /**
         * Whether the table has any column groups.
         */
        hasColumnGroup: boolean;
        /**
         * The DOM elements for the header table.
         */
        headerTable: JQuery;
        /**
         * DOM elements for the `TH` headers of this DataTable.
         */
        headers: JQuery;
        /**
         * Whether to ignore row hover event.
         */
        ignoreRowHoverEvent: boolean;
        /**
         * Whether the writing direction is set to right-to-left.
         */
        isRTL: boolean;
        /**
         * Whether a row toggler was clicked.
         */
        isRowTogglerClicked: boolean;
        /**
         * Whether live scrolling is currently active.
         */
        liveScrollActive: boolean;
        /**
         * List of row keys of the expansion rows that had their content
         * already loaded via AJAX.
         */
        loadedExpansionRows?: string[];
        /**
         * Whether data is currently being loaded due to the live scrolling feature.
         */
        loadingLiveScroll: boolean;
        /**
         * Whether a mousedown event occurred on a row.
         */
        mousedownOnRow: boolean;
        /**
         * INPUT element storing the current column / row order.
         */
        orderStateHolder: JQuery;
        /**
         * The original row index of the row that was clicked.
         */
        originRowMeta: PrimeFaces.widget.DataTable.RowMeta | null;
        /**
         * Localized message for removing the sort order and showing rows in their
         * original order.
         */
        otherMessage?: string;
        /**
         * When pagination is enabled: The paginator widget instance used for
         * paging.
         */
        paginator: PrimeFaces.widget.Paginator;
        /**
         * The current relative vertical scroll position.
         */
        percentageScrollHeight: boolean;
        /**
         * The current relative horizontal scroll position.
         */
        percentageScrollWidth: boolean;
        /**
         * `true` if reflow is enabled, `false` otherwise.
         */
        reflowDD: boolean;
        /**
         * The height of the table viewport, relative to the total height, used for scrolling.
         */
        relativeHeight: number;
        /**
         * A list with the current widths for each resizable column.
         */
        resizableState: string[];
        /**
         * INPUT element storing the current widths for each resizable column.
         */
        resizableStateHolder: JQuery;
        /**
         * The set-timeout timer ID of the timer used for resizing.
         */
        resizeTimeout: number;
        /**
         * The DOM element for the resize helper.
         */
        resizerHelper: JQuery;
        /**
         * Constant height in pixels for each row, when virtual scrolling is enabled.
         */
        rowHeight?: number;
        /**
         * The CSS selector for the table rows.
         */
        rowSelector: string;
        /**
         * The CSS selector for the table rows that can be clicked.
         */
        rowSelectorForRowClick: string;
        /**
         * The DOM element for the scrollable body of the table.
         */
        scrollBody: JQuery;
        /**
         * The DOM element for the scrollable body of the table.
         */
        scrollFooter: JQuery;
        /**
         * The DOM element for the scrollable footer box of the table.
         */
        scrollFooterBox: JQuery;
        /**
         * The DOM element for the scrollable header of the table.
         */
        scrollHeader: JQuery;
        /**
         * The DOM element for the scrollable header box of the table.
         */
        scrollHeaderBox: JQuery;
        /**
         * The current scroll position.
         */
        scrollOffset: number;
        /**
         * INPUT element storing the current scroll position.
         */
        scrollStateHolder: JQuery;
        /**
         * The DOM element for the scrollable TBODY.
         */
        scrollTbody: JQuery;
        /**
         * The set-timeout timer ID of the timer used for scrolling.
         */
        scrollTimeout: number;
        /**
         * CSS attribute for the scrollbar width, eg. `20px`.
         */
        scrollbarWidth: string;
        /**
         * List of row keys for the currently selected rows.
         */
        selection: string[];
        /**
         * ID of the INPUT element storing the currently selected rows.
         */
        selectionHolder: string;
        /**
         * Whether live scrolling is currently enabled.
         */
        shouldLiveScroll: boolean;
        /**
         * Information about how each column is sorted.
         * Key is the column key.
         */
        sortMeta: Record<string, PrimeFaces.widget.DataTable.SortMeta>;
        /**
         * DOM elements for the columns that are sortable.
         */
        sortableColumns: JQuery;
        /**
         * The DOM element for the sticky container of the table.
         */
        stickyContainer: JQuery;
        /**
         * DOM element of the `TBODY` element of this DataTable, if it exists.
         */
        tbody: JQuery;
        /**
         * DOM element of the `TFOOT` element of this DataTable, if it exists.
         */
        tfoot: JQuery;
        /**
         * DOM element of the `THEAD` element of this DataTable, if it exists.
         */
        thead: JQuery;
        /**
         * The DOM element for the cloned table head.
         */
        theadClone: JQuery;
        /**
         * Whether virtual scrolling is currently active.
         */
        virtualScrollActive: boolean;
        /**
         * Map between the sort order names and the multiplier for the comparator.
         */
        protected SORT_ORDER: Record<PrimeFaces.widget.DataTable.SortOrder, -1 | 0 | 1>;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Adds and sets up an invisible row for internal purposes.
         */
        protected addGhostRow(): void;
        /**
         * Adds the resizers for change the width of a column of this DataTable.
         */
        protected addResizers(): void;
        /**
         * Fetches the last row from the backend and inserts a row instead of updating the table itself.
         */
        addRow(): void;
        /**
         * Adds given row to the list of selected rows.
         * @param rowKey Key of the row to add.
         */
        private addSelection(rowKey: number): void;
        /**
         * Adds the given sorting to the list of sort rows. Each sorting describes a column by which to sort. This data
         * table may be sorted by multiple columns.
         * @param meta Sorting to add.
         */
        private addSortMeta(meta: PrimeFaces.widget.DataTable.SortMeta): void;
        /**
         * Adjusts the height of the body of this DataTable for the current scrolling settings.
         */
        protected adjustScrollHeight(): void;
        /**
         * Adjusts the width of the header, body, and footer of this DataTable to fit the current settings.
         */
        protected adjustScrollWidth(): void;
        /**
         * Adds some margin to the scroll body to make it align properly.
         */
        private alignScrollBody(): void;
        /**
         * Applies the width information to the given element.
         * @param element The element to which the width should be applied.
         * @param widthInfo The width information (retrieved using the method {@link getColumnWidthInfo}).
         */
        private applyWidthInfo(element: JQuery, widthInfo: PrimeFaces.widget.DataTable.WidthInfo): void;
        /**
         * Stores the row which is currently focused.
         * @param row Row to set as the focused row.
         */
        protected assignFocusedRow(row: JQuery): void;
        /**
         * Sets up the change event listeners on the column filter elements.
         * @param filter DOM element of a column filter
         */
        private bindChangeFilter(filter: JQuery): void;
        /**
         * Sets up the event listeners for radio buttons contained in this DataTable.
         */
        protected bindCheckboxEvents(): void;
        /**
         * Sets up the 'search' event which for HTML5 text search fields handles the clear 'x' button.
         * @param filter INPUT element of the text filter.
         */
        private bindClearFilterEvent(filter: JQuery): void;
        /**
         * Callback that is invoked when the context menu is initialized. Lets the
         * context menu provider register the appropriate event listeners for when the context menu should be shown and hidden.
         * @override
         * @param menuWidget The widget instance of the
         * context menu.
         * @param targetWidget The widget instance of the target widget that wants
         * to add a context menu.
         * @param targetId ID selector or DOM element of the target, i.e.
         * the element the context menu belongs to.
         * @param cfg The current configuration of the
         * context menu.
         */
        bindContextMenu(menuWidget: PrimeFaces.widget.ContextMenu, targetWidget: PrimeFaces.widget.DataTable, targetId: string, cfg: PrimeFaces.widget.ContextMenuCfg): void;
        /**
         * Binds editor events non-obtrusively.
         */
        private bindEditEvents(): void;
        /**
         * Sets up the enter key event listeners for the text filters on a column.
         * @param filter INPUT element of the text filter.
         */
        private bindEnterKeyFilter(filter: JQuery): void;
        /**
         * Applies events related to row expansion in a non-obtrusive way
         */
        protected bindExpansionEvents(): void;
        /**
         * Sets up all event listeners for the given filter element of a column filter.
         * @param filter DOM element of a column filter.
         */
        private bindFilterEvent(filter: JQuery): void;
        /**
         * Binds the change event listener and renders the paginator
         */
        private bindPaginator(): void;
        /**
         * Sets up the event listeners for radio buttons contained in this DataTable.
         */
        protected bindRadioEvents(): void;
        /**
         * Sets up the event listeners for clicking on a row.
         */
        private bindRowClick(): void;
        /**
         * Sets up all event listeners for event triggered on a row of this DataTable.
         */
        private bindRowEvents(): void;
        /**
         * Sets up the event listeners for hovering over a DataTable row.
         * @param rowSelector Selector for the row elements. Any hover event that does not reach an element that
         * matches this selector will be ignored.
         */
        protected bindRowHover(rowSelector: string): void;
        /**
         * Applies events related to selection in a non-obtrusive way
         */
        private bindSelectionEvents(): void;
        /**
         * Sets up all delegated event listeners on the table body.
         */
        private bindSelectionKeyEvents(): void;
        /**
         * Applies events related to sorting in a non-obtrusive way
         */
        private bindSortEvents(): void;
        /**
         * Sets up the event listeners for the text filters on a column.
         * @param filter INPUT element of the text filter.
         */
        private bindTextFilter(filter: JQuery): void;
        /**
         * Sets up the event handlers for row group events.
         */
        protected bindToggleRowGroupEvents(): void;
        /**
         * Computes the `colspan value for the table rows.
         * @return The computed `colspan` value.
         */
        private calculateColspan(): number;
        /**
         * When the given row is currently being edited, cancel the editing operation and discard the entered data. Use
         * `findRow` to get a row by its index.
         * @param rowEditor A row (`TR`) in edit mode.
         */
        cancelRowEdit(rowEditor: JQuery): void;
        /**
         * Initializes the given cell so that its content can be edited (when row editing is enabled)
         * @param cell A cell of this DataTable to set up.
         */
        private cellEditInit(cell: JQuery): void;
        /**
         * Checks the `select all` checkbox in the header of this DataTable.
         */
        private checkHeaderCheckbox(): void;
        /**
         * Clears all cached rows so that they are loaded from the server the next time they are requested.
         */
        private clearCacheMap(): void;
        /**
         * Clears all table filters and shows all rows that may have been hidden by filters.
         */
        clearFilters(): void;
        /**
         * Clears the saved scrolling position.
         */
        private clearScrollState(): void;
        /**
         * Clears the saved list of selected rows.
         */
        private clearSelection(): void;
        /**
         * Creates and stores a cloned copy of the table head(er) of this DataTable, and sets up some event handlers.
         */
        protected cloneHead(): void;
        /**
         * Clones a table header and removes duplicate IDs.
         * @param thead The head (`THEAD`) of the table to clone.
         * @param table The table to which the head belongs.
         * @return The cloned table head.
         */
        private cloneTableHeader(thead: JQuery, table: JQuery): JQuery;
        /**
         * Collapses all rows that are currently expanded.
         */
        collapseAllRows(): void;
        /**
         * Collapses the given row, if it is expandable. Use `findRow` to get a row by its index. Does not update the row
         * expansion toggler button.
         * @param row Row to collapse.
         */
        protected collapseRow(row: JQuery): void;
        /**
         * Configures the ARIA label for the select all checkbox.
         */
        private configureSelectAllAria(): void;
        /**
         * Disables all cell editors to prevent extra data on form posts.
         * @param element the row or cell to find inputs to enable for editing
         */
        private disableCellEditors(element: JQuery): void;
        /**
         * Disables the `select all` checkbox in the header of this DataTable.
         */
        private disableHeaderCheckbox(): void;
        /**
         * Display the given HTML string in the specified row. Called mainly after an AJAX request.
         * @param row Row to display.
         * @param content HTML string of the content to add to the row
         */
        protected displayExpandedRow(row: JQuery, content: string): void;
        /**
         * When the user wants to discard the edits to a cell, performs the required AJAX request for that.
         * @param cell The cell in edit mode with changes to discard.
         */
        private doCellEditCancelRequest(cell: JQuery): void;
        /**
         * When the users clicks on an editable cell, runs the AJAX request to show the inline editor for the given cell.
         * @param cell The cell to switch to edit mode.
         */
        private doCellEditRequest(cell: JQuery): void;
        /**
         * Sends an AJAX request to handle row save or cancel
         * @param rowEditor The current row editor.
         * @param action Whether to save or cancel the row edit.
         */
        private doRowEditRequest(rowEditor: JQuery, action: PrimeFaces.widget.DataTable.RowEditAction): void;
        /**
         * Enables all cell editors that were previously disabled by the UI and not alreayd disabled from user.
         * @param element the row or cell to find inputs to enable for editing
         */
        private enableCellEditors(element: JQuery): void;
        /**
         * Enables the `select all` checkbox in the header of this DataTable.
         */
        private enableHeaderCheckbox(): void;
        /**
         * Loads next page asynchronously to keep it at viewstate and Updates viewstate
         * @param newState The new values for the current page and the rows
         * per page count.
         */
        private fetchNextPage(newState: PrimeFaces.widget.Paginator.PaginationState): void;
        /**
         * Filters this DataTable. Uses the current values of the filter inputs. This will result in an AJAX request being
         * sent.
         */
        filter(): void;
        /**
         * Finds the saved width of the given column. The width of resizable columns may be saved to restore it after an
         * AJAX update.
         * @param id ID of a column
         * @return The saved width of the given column in pixels. `undefined` when the given column
         * does not exist.
         */
        private findColWidthInResizableState(id: string): string | undefined;
        /**
         * Finds the group resizer element for the given drag event data.
         * @param ui Data for the drag event.
         * @return The resizer DOM element.
         */
        protected findGroupResizer(ui: JQueryUI.DraggableEventUIParams): JQuery | null;
        /**
         * Converts a row specifier to the row element. The row specifier is either a row index or the row element itself.
         *
         * __In case this DataTable has got expandable rows, please not that a new table row is created for each expanded row.__
         * This may result in the given index not pointing to the intended row.
         * @param r The row to convert.
         * @return The row, or an empty JQuery instance of no row was found.
         */
        findRow(r: PrimeFaces.widget.DataTable.RowSpecifier): JQuery;
        /**
         * Invokes the behaviors and event listeners when a column is resized.
         * @param columnHeader Header of the column which was resized.
         */
        private fireColumnResizeEvent(columnHeader: JQuery): void;
        /**
         * Calls the behaviors and event listeners when a row is collapsed.
         * @param row A row of this DataTable.
         */
        private fireRowCollapseEvent(row: JQuery): void;
        /**
         * Sends a row select event on server side to invoke a row select listener if defined.
         * @param rowKey The key of the row that was selected.
         * @param behaviorEvent Name of the event to fire.
         * @param fnShowMenu Optional callback function invoked when the menu was opened.
         */
        private fireRowSelectEvent(rowKey: string, behaviorEvent: string, fnShowMenu?: () => void): void;
        /**
         * Sends a row unselect event on server side to invoke a row unselect listener if defined
         * @param rowKey The key of the row that was deselected.
         * @param behaviorEvent Name of the event to fire.
         */
        private fireRowUnselectEvent(rowKey: string, behaviorEvent: string): void;
        /**
         * Adjusts the width of the given columns to fit the current settings.
         */
        protected fixColumnWidths(): void;
        /**
         * Finds the meta data for a given cell.
         * @param cell A cell for which to get the meta data.
         * @return The meta data of the given cell or NULL if not found
         */
        getCellMeta(cell: JQuery): string;
        /**
         * Retrieves width information of the given column.
         * @param col The column of which the width should be retrieved.
         * @param isIncludeResizeableState Tells whether the width should be retrieved from the resizable state,
         * if it exists.
         * @return The width information of the given column.
         */
        private getColumnWidthInfo(col: JQuery, isIncludeResizeableState: boolean): PrimeFaces.widget.DataTable.WidthInfo;
        /**
         * Finds the list of row that are currently expanded.
         * @return All rows (`TR`) that are currently expanded.
         */
        getExpandedRows(): JQuery;
        /**
         * Finds the body of this DataTable with the property that the user can focus it.
         * @return The body of this DataTable.
         */
        protected getFocusableTbody(): JQuery;
        /**
         * Returns the paginator instance if any exists.
         * @return The paginator instance for this widget, or `undefined` if
         * paging is not enabled.
         */
        getPaginator(): PrimeFaces.widget.Paginator | undefined;
        /**
         * Finds all editors of a row. Usually each editable column has got an editor.
         * @param row A row for which to find its row editors.
         * @return A list of row editors for each editable column of the given row
         */
        protected getRowEditors(row: JQuery): JQuery;
        /**
         * Finds the index and the row key for the given row.
         * @param row The element (`TR`) of a row of this DataTable.
         * @return The meta for the row with the index and the row key.
         */
        getRowMeta(row: JQuery): PrimeFaces.widget.DataTable.RowMeta;
        /**
         * Finds the width of the current scrollbar used for this DataTable.
         * @return The width in pixels of the scrollbar of this DataTable.
         */
        private getScrollbarWidth(): number;
        /**
         * Finds the number of rows that are selected.
         * @return The number of rows that are currently selected.
         */
        getSelectedRowsCount(): number;
        /**
         * Creates the sort order message shown to indicate what the current sort order is.
         * @param ariaLabel Optional label text from an aria attribute.
         * @param sortOrderMessage Sort order message.
         * @return The sort order message to use.
         */
        private getSortMessage(ariaLabel: string | undefined, sortOrderMessage: string): string;
        /**
         * Retrieves the table body of this DataTable.
         * @return DOM element of the table body.
         */
        getTbody(): JQuery;
        /**
         * Retrieves the table footer of this DataTable.
         * @return DOM element of the table footer.
         */
        getTfoot(): JQuery;
        /**
         * Retrieves the table header of this DataTable.
         * @return DOM element of the table header.
         */
        getThead(): JQuery;
        /**
         * Called by `groupRows`, this method performs the grouping of a single set of rows that belong to one row group.
         * @param colIndex Index of the column to group.
         * @param rows Rows to group into one row group.
         */
        private groupRow(colIndex: number, rows: JQuery): void;
        /**
         * When row grouping is enabled, groups all rows accordingly.
         */
        protected groupRows(): void;
        /**
         * Checks whether this DataTable has got any column groups.
         * @return `true` if this DataTable has got any column groups, or `false` otherwise.
         */
        protected hasColGroup(): boolean;
        /**
         * Checks whether the body of this DataTable overflow vertically.
         * @return `true` if any content overflow vertically, `false` otherwise.
         */
        protected hasVerticalOverflow(): boolean;
        /**
         * Highlights the currently focused row (if any) by adding the appropriate CSS class.
         */
        protected highlightFocusedRow(): void;
        /**
         * Highlights row to mark it as selected.
         * @param row Row to highlight.
         */
        protected highlightRow(row: JQuery): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Reflow mode is a responsive mode to display columns as stacked depending on screen size.
         */
        private initReflow(): void;
        /**
         * Initializes the expansion state
         */
        private initRowExpansion(): void;
        /**
         * Displays row editors in invalid format.
         * @param index 0-based index of the row to invalidate.
         */
        protected invalidateRow(index: number): void;
        /**
         * Checks whether the rows of this DataTable are selected via checkboxes.
         * @return `true` if selection mode is set to `checkbox`, or `false` otherwise.
         */
        isCheckboxSelectionEnabled(): boolean;
        /**
         * Checks whether this DataTable has got any rows. When there are no rows, usually the message `no items found` is
         * shown.
         * @return `true` if this DataTable has got no rows, `false` otherwise.
         */
        isEmpty(): boolean;
        /**
         * Checks whether multiples rows may be selected at a time.
         * @return `true` if selection mode is set to `multiple`, or `false` otherwise.
         */
        isMultipleSelection(): boolean;
        /**
         * Checks whether the rows of this DataTable are selected via radio buttons.
         * @return `true` if selection mode is set to `radio`, or `false` otherwise.
         */
        isRadioSelectionEnabled(): boolean;
        /**
         * Checks whether the given row is currently selected.
         * @param rowKey The key of a row from this DataTable.
         * @return `true` if the given row is currently selected, or `false` otherwise.
         */
        isSelected(rowKey: string): boolean;
        /**
         * Checks whether the user may select the rows of this DataTable.
         * @return `true` is rows may be selected, or `false` otherwise.
         */
        isSelectionEnabled(): boolean;
        /**
         * Checks whether only one row may be selected at a time.
         * @return `true` if selection mode is set to `single`, or `false` otherwise.
         */
        isSingleSelection(): boolean;
        /**
         * Serializes the option from the sort meta items.
         * @param option Property of the sort meta to use.
         * @return All values from the current sort meta list for the given option.
         */
        private joinSortMetaOption(option: keyof PrimeFaces.widget.DataTable.SortMeta): string;
        /**
         * Performs the required initialization for making a row editable. Only called on-demand when the row actually needs
         * to be edited.
         * @param row A row of this DataTable.
         */
        private lazyRowEditInit(row: JQuery): void;
        /**
         * Loads the data for the given page and displays it. When some rows exist in the cache, do not reload them from the
         * server.
         * @param newState The new values for the current page and the rows
         * per page count.
         */
        private loadDataWithCache(newState: PrimeFaces.widget.Paginator.PaginationState): void;
        /**
         * Loads the detailed content for the given expandable row.
         * @param row A row with content to load.
         */
        private loadExpandedRowContent(row: JQuery): void;
        /**
         * Use only when live scrolling is enabled: Loads the next set of rows on-the-fly.
         */
        loadLiveRows(): void;
        /**
         * When live scrolling is enabled: Loads the next set of rows via AJAX.
         * @param page 0-based index of the page to load.
         * @param callback Callback that is invoked after the rows have been loaded and inserted into the DOM.
         */
        private loadRowsWithVirtualScroll(page: number, callback: () => void): void;
        /**
         * Makes the rows of this DataTable draggable via JQueryUI.
         */
        private makeRowsDraggable(): void;
        /**
         * Callback for a click event on a row.
         * @param event Click event that occurred.
         * @param rowElement Row that was clicked
         * @param silent `true` to prevent behaviors from being invoked, `false` otherwise.
         */
        private onRowClick(event: JQuery.TriggeredEvent, rowElement: HTMLElement, silent: boolean): void;
        /**
         * Callback for a double click event on a row.
         * @param event Event that occurred.
         * @param row Row that was clicked.
         */
        private onRowDblclick(event: JQuery.TriggeredEvent, row: JQuery): void;
        /**
         * Callback for a right click event on a row. May bring up the context menu
         * @param event Event that occurred.
         * @param rowElement Row that was clicked.
         * @param cmSelMode The current selection mode.
         * @param fnShowMenu Optional callback function invoked when the menu was opened.
         * @return true to hide the native browser context menu, false to display it
         */
        private onRowRightClick(event: JQuery.TriggeredEvent, rowElement: JQuery, cmSelMode: PrimeFaces.widget.DataTable.CmSelectionMode, fnShowMenu?: () => void): boolean;
        /**
         * Switches to the given page by loading the content via AJAX. Compare with `loadDataWithCache`, which first checks
         * whether the data is already cached and loads it from the server only when not found in the cache.
         * @param newState The new values for the current page and the rows
         * per page count.
         */
        private paginate(newState: PrimeFaces.widget.Paginator.PaginationState): void;
        /**
         * Called after an AJAX update. Binds the appropriate event listeners again.
         */
        private postUpdateData(): void;
        /**
         * Removes the current clone of the table header from the DOM, and creates a new clone.
         */
        private reclone(): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Remove given row from the list of selected rows.
         * @param rowKey Key of the row to remove.
         */
        private removeSelection(rowKey: string): void;
        /**
         * Resets the scroll state of the body to a non-scrolled state.
         */
        protected resetVirtualScrollBody(): void;
        /**
         * Resizes this DataTable, row, or columns in response to a drag event of a resizer element.
         * @param event Event triggered for the drag.
         * @param ui Data for the drag event.
         */
        protected resize(event: JQuery.TriggeredEvent, ui: JQueryUI.DraggableEventUIParams): void;
        /**
         * Reads the saved scroll state and applies it. This helps to preserve the current scrolling position during AJAX
         * updates.
         */
        private restoreScrollState(): void;
        /**
         * After the user is done editing a cell, saves the content of the given cell and switches back to view mode.
         * @param cell A cell (`TD`) in edit mode.
         */
        saveCell(cell: JQuery): void;
        /**
         * Saves the current column order, used to preserve the state between AJAX updates etc.
         */
        protected saveColumnOrder(): void;
        /**
         * When the given row is currently being edited, saves the contents of the edited row and switch back to view mode.
         * Use `findRow` to get a row by its index.
         * @param rowEditor A row (`TR`) in edit mode to save.
         */
        saveRowEdit(rowEditor: JQuery): void;
        /**
         * Saves the current scrolling position. This helps to preserve the current scrolling position during AJAX updates.
         */
        private saveScrollState(): void;
        /**
         * Selects all rows of this DataTable so that no rows are selected. This includes all rows on all pages,
         * irrespective of whether they are on the currently shown page.
         */
        selectAllRows(): void;
        /**
         * Select all rows on the currently shown page. Compare with `selectAllRows`.
         */
        selectAllRowsOnPage(): void;
        /**
         * Selects the given checkbox from a row.
         * @param checkbox A checkbox to select.
         */
        private selectCheckbox(checkbox: JQuery): void;
        /**
         * Selects the given radio button from a row.
         * @param radio A radio button to select.
         */
        private selectRadio(radio: JQuery): void;
        /**
         * Selects the given row, according to the current selection mode.
         * @param r A row of this DataTable to select.
         * @param silent `true` to prevent behaviors and event listeners from being invoked, or `false`
         * otherwise.
         */
        selectRow(r: PrimeFaces.widget.DataTable.RowSpecifier, silent?: boolean): void;
        /**
         * Selects the corresponding row of a checkbox based column selection
         * @param checkbox A checkox INPUT element
         * @param event Event that occurred.
         * @param silent `true` to prevent behaviors from being invoked, `false` otherwise.
         */
        private selectRowWithCheckbox(checkbox: JQuery, event: JQuery.TriggeredEvent, silent?: boolean): void;
        /**
         * Selects the corresponding row of a radio based column selection
         * @param radio A radio INPUT element
         */
        private selectRowWithRadio(radio: JQuery): void;
        /**
         * Select the rows between the cursor and the given row.
         * @param row A row of this DataTable.
         * @param silent `true` to prevent behaviors and event listeners from being invoked, or `false`
         * otherwise.
         */
        private selectRowsInRange(row: JQuery, silent?: boolean): void;
        /**
         * Applies the appropriated width to all given column elements.
         * @param columns A list of column elements.
         */
        private setColumnsWidth(columns: JQuery): void;
        /**
         * Applies the given width to this DataTable.
         * @param element Element of the DataTable.
         * @param width New width in pixels to set.
         */
        private setOuterWidth(element: JQuery, width: number): void;
        /**
         * Applies the given scroll width to this DataTable.
         * @param width Scroll width in pixels to set.
         */
        protected setScrollWidth(width: number): void;
        /**
         * Sets up all event listeners required for making column draggable and reorderable.
         */
        private setupDraggableColumns(): void;
        /**
         * Binds filter events to standard filters
         */
        private setupFiltering(): void;
        /**
         * Sets up the event listeners to enable columns to be resized.
         */
        private setupResizableColumns(): void;
        /**
         * Sets up the DataTable and adds all event listeners required for hovering over rows.
         */
        private setupRowHover(): void;
        /**
         * Prepares this DataTable for the current scrolling settings and sets up all related event handlers.
         */
        protected setupScrolling(): void;
        /**
         * Sets up the DataTable and adds all event listener required for selecting rows.
         */
        private setupSelection(): void;
        /**
         * Applies the styling and event listeners required for the sticky headers feature.
         */
        private setupStickyHeader(): void;
        /**
         * When live scrolling (loading more items on-demand) is enabled, checks whether more items are allowed to be loaded
         * right now. Returns `false` when live scroling is disabled or items are currently being loaded already.
         * @return `true` if more items may be loaded, `false` otherwise.
         */
        private shouldLoadLiveScroll(): boolean;
        /**
         * Called in response to a click. Checks whether this DataTable should now be sorted. Returns `false` when there
         * are no items to be sorted, or when no sorting button was clicked.
         * @param event (Click) event that occurred.
         * @param column Column Column of this DataTable on which the event occurred.
         * @return `true` to perform a sorting operation, `false` otherwise.
         */
        private shouldSort(event: JQuery.TriggeredEvent, column: JQuery): boolean;
        /**
         * When cell editing is enabled, shows the cell editor for the given cell that lets the user edit the cell content.
         * @param c A cell (`TD`) of this DataTable to edit.
         */
        showCellEditor(c: JQuery): void;
        /**
         * Shows the cell editors for the given cell.
         * @param cell A cell of this DataTable.
         */
        private showCurrentCell(cell: JQuery): void;
        /**
         * Shows the row editor(s) for the given row (and hides the normal output display).
         * @param row Row for which to show the row editor.
         */
        protected showRowEditors(row: JQuery): void;
        /**
         * Performs a sorting operation on the rows of this DataTable via AJAX
         * @param columnHeader Header of the column by which to sort.
         * @param order `-1` to sort column values in ascending order, `+1` to sort column values in descending
         * order, or `0` to remove the sorting order and display rows in their original order.
         * @param multi `true` if sorting by multiple columns is enabled, or `false` otherwise.
         */
        private sort(columnHeader: JQuery, order: -1 | 0 | 1, multi: boolean): void;
        /**
         * Switch all editable columns of the given row to their editing mode, if editing is enabled on this DataTable.
         * Use `findRow` to get a row by its index.
         * @param row A row (`TR`) to switch to edit mode.
         */
        switchToRowEdit(row: JQuery): void;
        /**
         * Sets the style class on each, depending whether it is an even-numbered or odd-numbered row.
         */
        private syncRowParity(): void;
        /**
         * Moves to the next or previous editable cell when the tab key was pressed.
         * @param cell The currently focused cell
         * @param forward `true` if tabbing forward, `false` otherwise.
         */
        private tabCell(cell: JQuery, forward: boolean): void;
        /**
         * Toggles the `selected all` checkbox in the header of this DataTable. When no rows are selected, this will select
         * all rows. When some rows are selected, this will unselect all rows.
         */
        toggleCheckAll(): void;
        /**
         * Expands a row to display its detailed content
         * @param toggler The row toggler of a row to expand.
         */
        private toggleExpansion(toggler: JQuery): void;
        /**
         * Expands or collapses the given row, depending on whether it is currently collapsed or expanded, respectively.
         * @param row A row (`TR`) to expand or collapse.
         */
        toggleRow(row: JQuery): void;
        /**
         * Removes event listeners needed if refreshing to prevent multiple sort and pagination events.
         *
         * Cancels all current drag and drop events.
         */
        private unbindEvents(): void;
        /**
         * Unchecks the `select all` checkbox in the header of this data table.
         */
        private uncheckHeaderCheckbox(): void;
        /**
         * Unhighlights the currently focused row (if any) by adding the appropriate CSS class.
         */
        protected unhighlightFocusedRow(): void;
        /**
         * Removes the highlight of a row so it is no longer marked as selected.
         * @param row Row to unhighlight.
         */
        protected unhighlightRow(row: JQuery): void;
        /**
         * Unselects all rows of this DataTable so that no rows are selected. This includes all rows on all pages,
         * irrespective of whether they are on the currently shown page.
         */
        unselectAllRows(): void;
        /**
         * Unselect all rows on the currently shown page. Compare with `unselectAllRows`.
         */
        unselectAllRowsOnPage(): void;
        /**
         * Unselects the given checkbox from a row.
         * @param checkbox A checkbox to unselect.
         */
        private unselectCheckbox(checkbox: JQuery): void;
        /**
         * Unselects the given radio button from a row.
         * @param radio A radio button to unselect.
         */
        private unselectRadio(radio: JQuery): void;
        /**
         * Unselects the given row.
         * @param r A row of this DataTable to unselect.
         * @param silent `true` to prevent behaviors and event listeners from being invoked, or `false`
         * otherwise.
         */
        unselectRow(r: PrimeFaces.widget.DataTable.RowSpecifier, silent?: boolean): void;
        /**
         * Unselects the corresponding row of a checkbox based column selection
         * @param checkbox A checkox INPUT element
         * @param event Event that occurred.
         * @param silent `true` to prevent behaviors from being invoked, `false` otherwise.
         */
        private unselectRowWithCheckbox(checkbox: JQuery, event: JQuery.TriggeredEvent, silent?: boolean): void;
        /**
         * Updates the `colspan` attribute of the given row.
         * @param row A row to update.
         * @param colspanValue The new `colspan` value. If not given, computes the value automatically.
         */
        private updateColspan(row: JQuery, colspanValue?: number): void;
        /**
         * Updates some style classes for all columns.
         */
        private updateColumnsView(): void;
        /**
         * Updates the currently selected cell based on where the context menu right click occurred.
         * @param event Event that occurred.
         * @param targetWidget The current widget
         */
        private updateContextMenuCell(event: JQuery.TriggeredEvent, targetWidget: PrimeFaces.widget.DataTable): void;
        /**
         * Sets the given HTML string as the content of the body of this DataTable. Afterwards, sets up all required event
         * listeners etc.
         * @param data HTML string to set on the body.
         * @param clear Whether the contents of the table body should be removed beforehand.
         */
        protected updateData(data: string, clear?: boolean): void;
        /**
         * Updates the colspan attribute for the message shown when no rows are available.
         */
        private updateEmptyColspan(): void;
        /**
         * Updates the `colspan` attributes of all expanded rows.
         */
        private updateExpandedRowsColspan(): void;
        /**
         * Configures the ARIA label for the row expander.
         * @param toggler the toggler button
         */
        private updateExpansionAria(toggler: JQuery): void;
        /**
         * Updates the `check all` checkbox in the header of this DataTable.
         */
        private updateHeaderCheckbox(): void;
        /**
         * Updates and syncs the current pagination state with the server.
         * @param newState The new values for the current page and the rows
         * per page count.
         */
        private updatePageState(newState: PrimeFaces.widget.Paginator.PaginationState): void;
        /**
         * Reflow mode is a responsive mode to display columns as stacked depending on screen size. Updates the reflow for
         * the given header.
         * @param columnHeader Header of a column to update.
         * @param sortOrder Sort order of the column.
         */
        private updateReflowDD(columnHeader: JQuery, sortOrder: number): void;
        /**
         * Computes and saves the resizable state of this DataTable, ie. which columns have got which width. May be used
         * later to restore the current column width after an AJAX update.
         * @param columnHeader Element of a column header of this DataTable.
         * @param nextColumnHeader Element of the column header next to the given column header.
         * @param table The element for this DataTable.
         * @param newWidth New width to be applied.
         * @param nextColumnWidth Width of the column next to the given column header.
         */
        private updateResizableState(columnHeader: JQuery, nextColumnHeader: JQuery, table: JQuery, newWidth: number, nextColumnWidth: number | null): void;
        /**
         * Updates a row with the given content
         * @param row Row to update.
         * @param content HTML string to set on the row.
         */
        protected updateRow(row: JQuery, content: string): void;
        /**
         * Updates the `rowspan` attribute of the given row.
         * @param row A column to update.
         */
        private updateRowspan(row: JQuery): void;
        /**
         * Configures the ARIA label for the row checkbox/radio button.
         * @param row the row key to identify
         */
        private updateSelectionAria(row: JQuery): void;
        /**
         * In multi-sort mode this will add number indicators to let the user know the current
         * sort order. If only one column is sorted then no indicator is displayed and will
         * only be displayed once more than one column is sorted.
         */
        private updateSortPriorityIndicators(): void;
        /**
         * Switches the given cell to its view mode (not editable).
         * @param cell A cell of this DataTable.
         */
        private viewMode(cell: JQuery): void;
        /**
         * Write row expansion state.
         */
        private writeRowExpansions(): void;
        /**
         * Writes selected row ids to state holder
         */
        private writeSelections(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DataTable| DataTable widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DataTableCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * When true columns can be unsorted upon clicking sort.
         */
        allowUnsorting: boolean;
        /**
         * Defines the cell edit behavior.
         */
        cellEditMode: string;
        /**
         * Separator text to use in output mode of editable cells with multiple components.
         */
        cellSeparator: string;
        /**
         * Caches the next page asynchronously.
         */
        clientCache: boolean;
        /**
         * `true` to disable the context menu when the DataTable has got on
         * data row, or `false` otherwise.
         */
        disableContextMenuIfEmpty: boolean;
        /**
         * Disables text selection on row click.
         */
        disabledTextSelection: boolean;
        /**
         * Columns can be reordered with drag & drop when enabled.
         */
        draggableColumns: boolean;
        /**
         * When enabled, rows can be reordered using drag & drop.
         */
        draggableRows: boolean;
        /**
         * Event that triggers row/cell editing.
         */
        editInitEvent: string;
        /**
         * Whether rows may be edited as a whole or whether each
         * cell can be edited individually.
         */
        editMode: PrimeFaces.widget.DataTable.CellEditMode;
        /**
         * Controls in-cell editing.
         */
        editable: boolean;
        /**
         * `true` if rows are expandable, or `false` otherwise.
         */
        expansion: boolean;
        /**
         * `true` if filtering is enabled, or `false` otherwise.
         */
        filter: boolean;
        /**
         * Delay for filtering in milliseconds.
         */
        filterDelay: number;
        /**
         * Event to invoke filtering for input filters.
         */
        filterEvent: string;
        /**
         * The number of frozen columns.
         */
        frozenColumns: number;
        /**
         * Columns are resized live in this mode without using a resize helper.
         */
        liveResize: boolean;
        /**
         * Enables live scrolling.
         */
        liveScroll: boolean;
        /**
         * Percentage of the height of the buffer between the bottom of the page and the
         * scroll position to initiate the load for the new chunk. This value is in the range `0...100`.
         */
        liveScrollBuffer: number;
        /**
         * `true` if sorting by multiple columns is enabled, or `false` otherwise.
         */
        multiSort: boolean;
        /**
         * Whether multiple resize mode is enabled.
         */
        multiViewState: boolean;
        /**
         * `true` to use native radio button and checkbox elements, or `false` otherwise.
         */
        nativeElements: boolean;
        /**
         * Callback that is invoked when the user clicked on
         * a row of the DataTable.
         */
        onRowClick: PrimeFaces.widget.DataTable.OnRowClickCallback;
        /**
         * Reflow mode is a responsive mode to display columns as stacked depending on screen size.
         */
        reflow: boolean;
        /**
         * Enables column resizing.
         */
        resizableColumns: boolean;
        /**
         * Defines the resize behavior.
         */
        resizeMode: PrimeFaces.widget.DataTable.ResizeMode;
        /**
         * CSS selector for the draggable handle.
         */
        rowDragSelector: string;
        /**
         * Defines the row edit.
         */
        rowEditMode: PrimeFaces.widget.DataTable.RowEditMode;
        /**
         * Defines row expand mode.
         */
        rowExpandMode: PrimeFaces.widget.DataTable.RowExpandMode;
        /**
         * Adds hover effect to rows. Hover is always on when selection is enabled.
         */
        rowHover: boolean;
        /**
         * CSS selector find finding the rows of this DataTable.
         */
        rowSelector: string;
        /**
         * Saves the changes in cell editing on blur, when set to false changes are
         * discarded.
         */
        saveOnCellBlur: boolean;
        /**
         * Scroll viewport height.
         */
        scrollHeight: string;
        /**
         * Maximum number of rows that may be loaded via live scrolling.
         */
        scrollLimit: number;
        /**
         * Number of additional rows to load in each live scroll.
         */
        scrollStep: number;
        /**
         * Scroll viewport width.
         */
        scrollWidth: string;
        /**
         * Makes data scrollable with fixed header.
         */
        scrollable: boolean;
        /**
         * Enables row selection.
         */
        selectionMode: PrimeFaces.widget.DataTable.SelectionMode;
        /**
         * When using a paginator and selection mode is `checkbox`, the select all
         * checkbox in the header will select all rows on the current page if `true`, or all rows on all pages if `false`.
         * Default is `true`.
         */
        selectionPageOnly: boolean;
        /**
         * Defines row selection mode when clicking on the row itself.
         */
        selectionRowMode: PrimeFaces.widget.DataTable.SelectionRowMode;
        /**
         * IDs of the columns by which to order. Order by the first column, then by the
         * second, etc.
         */
        sortMetaOrder: string[];
        /**
         * `true` if sorting is enabled on the DataTable, `false` otherwise.
         */
        sorting: boolean;
        /**
         * Sticky header stays in window viewport during scrolling.
         */
        stickyHeader: boolean;
        /**
         * Selector to position on the page according to other fixing elements on the top of the
         * table.
         */
        stickyTopAt: string;
        /**
         * The value of the `tabindex` attribute for this DataTable.
         */
        tabindex: string;
        /**
         * Loads data on demand as the scrollbar gets close to the bottom.
         */
        virtualScroll: boolean;
    }
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Describes the width information of a DOM element.
     */
    export interface WidthInfo {
        /**
         * Tells whether the width includes the border-box or not.
         */
        isOuterWidth: boolean;
        /**
         * The width of the element. It's either a unit-less numeric pixel value or a
         * string containing the width including an unit.
         */
        width: number | string;
    }
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Describes a sorting operation of the DataTable. The
     * items of the DataTable may be sorted by multiple column, in which case the sorting operation is describes by a list
     * of these objects.
     */
    export interface SortMeta {
        /**
         * ID of the column to sort by.
         */
        col: string;
        /**
         * Whether to sort the items by the column value in an ascending or descending order.
         */
        order: -1 | 1;
    }
}
declare namespace PrimeFaces.widget.DataTable {
    /**
     * Describes the meta information of row, such as its index and
     * its row key.
     */
    export interface RowMeta {
        /**
         * The 0-based index of the row in the DataTable.
         */
        index: number;
        /**
         * The unique key of the row. `undefined` when no key was defined for the rows.
         */
        key: string | undefined;
    }
}
declare namespace PrimeFaces.widget.DataView {
    /**
     * The layout mode the data view. `grid` displays the
     * item in a grid with cards, `list` displays the items in a vertical list.
     */
    export type Layout = "grid" | "list";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DataView Widget__
     *
     * DataView displays data in grid or list layout.
     * @typeparam TCfg Defaults to `DataViewCfg`. Type of the configuration object for this widget.
     */
    export class DataView<TCfg extends DataViewCfg = DataViewCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * DOM elements of the buttons for switching the layout (grid or list).
         */
        buttons: JQuery;
        /**
         * DOM element of the content container for the data grid.
         */
        content: JQuery;
        /**
         * DOM element of the data view header.
         */
        header: JQuery;
        /**
         * DOM element of the container with the layout switch buttons.
         */
        layoutOptions: JQuery;
        /**
         * When pagination is enabled: The paginator widget instance used for
         * paging.
         */
        paginator: PrimeFaces.widget.Paginator;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Retrieves the paginator widget used by this data grid for pagination. You can use this widget to switch to a
         * different page programatically.
         * @return The paginator widget, or `undefined` when pagination is not
         * enabled.
         */
        getPaginator(): PrimeFaces.widget.Paginator | undefined;
        /**
         * Handles a pagination event by updating the data grid and invoking the appropriate behaviors.
         * @param newState The new pagination state to apply.
         */
        private handlePagination(newState: PrimeFaces.widget.Paginator.PaginationState): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Loads the content with the data items for the selected layout (grid or list).
         * @param layout The current layout of this data view.
         */
        private loadLayoutContent(layout: PrimeFaces.widget.DataView.Layout): void;
        /**
         * Switches this data view to the given layout (grid or list).
         *
         * ```javascript
         * const widget = PF("MyDataView");
         * // Switch to grid layout
         * widget.select(widget.buttons.eq(1));
         * ```
         * @param button One of the layout switch buttons (`.ui-button`).
         */
        select(button: JQuery): void;
        /**
         * Initializes the paginator, called during widget initialization.
         */
        private setupPaginator(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DataView| DataView widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DataViewCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * When pagination is enabled: The paginator configuration
         * for the paginator.
         */
        paginator: Partial<PrimeFaces.widget.PaginatorCfg>;
    }
}
/**
 * Namespace for the JQuery UI PrimeFaces date picker.
 *
 * The Prime date picker is a JQuery UI widget used to select a date featuring display modes, paging, localization,
 * AJAX selection and more.
 */
declare namespace JQueryPrimeDatePicker {
    /**
     * Defines the selection mode, whether one or multiple dates can be selected.
     */
    export type SelectionMode = "single" | "multiple" | "range";
    /**
     * Defines the view mode, whether a date or month is to be selected.
     */
    export type ViewMode = "date" | "month";
    /**
     * Defines the hour format or the clock convention.
     * - `12`: Time convention in which the 24 hours of the day are divided into two periods: `a.m.` and `p.m.`.
     * - `24`: Time convention in which the 24 hours of the day are divided into one period.
     */
    export type ClockConvention = "12" | "24";
    /**
     * Represents the available periods of a 12 hour clock convention.
     */
    export type HalfDayPeriod = "AM" | "PM";
    /**
     * Represents whether a numerical value is to be decremented or incremented.
     */
    export type AlterationMode = "DECREMENT" | "INCREMENT";
    /**
     * Specifies which part of the time is changed:
     * - `-1`: The time is not changed.
     * - `0`: Changes the hour.
     * - `1`: Changes the minutes.
     * - `2`: Changes the second.
     * - `3`: Changes the millisecond.
     * - `4`: Changes the half day period.
     */
    export type ChangeTimeType = -1 | 0 | 1 | 2 | 3 | 4;
    /**
     * Represents a one dimensional direction:
     * - `-1`: Backwards.
     * - `+1`: Forwards.
     */
    export type OneDimensionalDirection = -1 | 1;
    /**
     * Javascript function that takes a date object and returns the content for the date cell.
     */
    export type DateTemplate =
    /**
     * @param monthNameOrDate Either the name of a month or a date to be rendered.
     * @return An HTML snippets with the contents of the date cell.
     */
    (this: PickerInstance, monthNameOrDate: string | DayInstantSelectableRelative) => string;
    /**
     * Base callback that is only passed the current date picker instance.
     */
    export type BaseCallback = (this: PickerInstance) => void;
    /**
     * Base callback that, in addition to the current date picker instance, is also passed the event that occured.
     */
    export type BaseEventCallback =
    /**
     * @param event The event that occurred.
     */
    (this: PickerInstance, event: JQuery.TriggeredEvent) => void;
    /**
     * Callbacks for when a value has changed.
     */
    export type MutationCallback<T extends unknown[]> =
    /**
     * @param newValues The value or set of values that represent the new state.
     */
    (this: PickerInstance, ...newValues: T) => void;
    /**
     * Callback for when a value has changed. It is also passed the event that occurred.
     */
    export type MutationEventCallback<T extends unknown[]> =
    /**
     * @param event The event that occurred.
     * @param newValues The value or set of values that represent the new state.
     */
    (this: PickerInstance, event: JQuery.TriggeredEvent, ...newValues: T) => void;
    /**
     * A cardinal number, i.e. a number that represents an amount of something. Some common examples include the number
     * of days in a month or the number of seconds in a year.
     */
    export type Cardinal = number;
    // Of-types representing a temporal index, or partial information of an instant
    /**
     * Integer value representing the month, beginning with `0` for `January` and ending with `11` for `December`.
     */
    export type MonthOfTheYear = number;
    /**
     * Integer value representing the month, beginning with `1` for `January` and ending with `12` for `December`.
     */
    export type MonthOfTheYearOneBased = number;
    /**
     * Integer value representing the day of the month. `1` represents the first day of the month.
     */
    export type DayOfTheMonth = number;
    /**
     * Integer value representing the day of the week, starting with `0` for `Sunday` and ending with `6` for
     * `Saturday`.
     */
    export type DayOfTheWeek = number;
    /**
     * Integer value representing the hour of the day. `0` represents midnight.
     */
    export type HourOfTheDay = number;
    /**
     * Integer value representing the minute segment of a time. `0` represents 0 minutes past the hour.
     */
    export type MinuteOfTheHour = number;
    /**
     * Integer value representing the second segment of a time. `0` represents 0 seconds past the minute.
     */
    export type SecondOfTheMinute = number;
    /**
     * Integer value representing the millisecond segment of a time. `0` represents 0 milliseconds past the second.
     */
    export type MillisecondOfTheSecond = number;
    /**
     * Represents the time of a day.
     */
    export interface TimeOfTheDay {
        /**
         * The hour of the day, between 0 and 23.
         */
        hour: HourOfTheDay;
        /**
         * The minute of the hour, between 0 and 59.
         */
        minute: MinuteOfTheHour;
        /**
         * The second of the minute, between 0 and 59.
         */
        second: SecondOfTheMinute;
        /**
         * The millisecond of the second, between 0 and 999.
         */
        millisecond: MillisecondOfTheSecond;
    }
    // Instants representing an instant in time (to a certain precision)
    /**
     * An integer value representing the year. All values represent the actual year.
     */
    export type YearInstant = number;
    /**
     * Represents a month of a particular year.
     */
    export interface MonthInstant {
        /**
         * The month of the year.
         */
        month: MonthOfTheYear;
        /**
         * The year of the month instant.
         */
        year: YearInstant;
    }
    /**
     * Represents a day of a particular month and year.
     */
    export interface DayInstant {
        /**
         * The day of the month.
         */
        day: DayOfTheMonth;
        /**
         * The month of the year.
         */
        month: MonthOfTheYear;
        /**
         * The year of the month instant.
         */
        year: YearInstant;
    }
    /**
     * Represents a {@link DayInstant} and whether that day can be selected by the user.
     */
    export interface DayInstantSelectable extends DayInstant {
        /**
         * Whether the day can be selected as the currently selected value of the date picker.
         */
        selectable: boolean;
    }
    /**
     * Represents a {@link DayInstantSelectable} and additionally indicates whether the day is the current date.
     */
    export interface DayInstantSelectableRelative extends DayInstantSelectable {
        /**
         * Whether this day is today (equal to the current date).
         */
        today: boolean;
    }
    /**
     * Represents a list of all days in a particular month.
     */
    export interface DayListInMonth extends MonthInstant {
        /**
         * An array with the weeks of the month, each week being an array containing the days of that week.
         */
        dates: DayInstantSelectableRelative[][];
        /**
         * 0-based index of the month in the year.
         */
        index: MonthOfTheYear;
    }
    /**
     * Localized strings for various messages displayed by the date or time picker.
     */
    export interface PickerLocale {
        /**
         * Index of the day that represents the first day of the week.
         */
        firstDayOfWeek: number;
        /**
         * Names of the weekdays, starting at `Sunday`.
         */
        dayNames: [
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        /**
         * Short names of the weekdays, starting at `Sunday` (`Sun`).
         */
        dayNamesShort: [
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        /**
         * Extra short names of the weekdays, starting at `Sunday` (`Su`).
         */
        dayNamesMin: [
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        /**
         * Names of the months in the year, starting at `January`.
         */
        monthNames: [
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        /**
         * Short names of the months in the year, starting at `January` (`Jan`).
         */
        monthNamesShort: [
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string,
            string
        ];
        /**
         * Name of `today` button for navigating to the current date.
         */
        today: string;
        /**
         * Name of `clear` button for clearing the selected date or time.
         */
        clear: string;
    }
    /**
     * Represents the available options for the date or time picker.
     */
    export interface PickerOptions {
        /**
         * The ID of this widget, usually the ID of the DOM element.
         */
        id: string | string[];
        /**
         * The name of this widget.
         */
        name: string | null;
        /**
         * The default date that is shown when none was specified.
         */
        defaultDate: string | Date | Date[] | null;
        /**
         * The date that is shown in the picker.
         */
        viewDate: string | Date | Date[] | null;
        /**
         * Inline style of the component.
         */
        style: string | null;
        /**
         * Style class of the component.
         */
        styleClass: string | null;
        /**
         * Whether the date picker is rendered inline or as an overlay.
         */
        inline: boolean;
        /**
         * Defines the selection mode, whether one or multiple dates can be selected.
         */
        selectionMode: SelectionMode;
        /**
         * Separator for joining start and end dates on range selection mode, such as `-`.
         */
        rangeSeparator: string;
        /**
         * ID of the input element that stores the selected date or time.
         */
        inputId: string | null;
        /**
         * Inline style of the input element. Used when mode is popup.
         */
        inputStyle: string | null;
        /**
         * Style class of the input element. Used when mode is popup.
         */
        inputStyleClass: string | null;
        /**
         * Whether an input is required.
         */
        required: boolean;
        /**
         * Whether the input is set to `readonly`.
         */
        readOnlyInput: boolean;
        /**
         * Whether the input is disabled.
         */
        disabled: boolean;
        /**
         * Tabindex of the date picker button
         */
        tabIndex: string | null;
        /**
         * Specifies a short hint.
         */
        placeholder: string | null;
        /**
         * Whether to show an icon to display the picker in an overlay
         */
        showIcon: boolean;
        /**
         * Icon of the date picker element that toggles the visibility in popup mode.
         */
        icon: string;
        /**
         * Whether the date picker overlay is shown when the element focused.
         */
        showOnFocus: boolean;
        /**
         * Separator for joining the hour and minute part of a time, defaults to `:`.
         */
        timeSeparator: string;
        /**
         * Whether the current input is a  valid date / time.
         */
        valid: boolean;
        /**
         * Whether to keep the invalid inputs in the field or not.
         */
        keepInvalid: boolean;
        /**
         * Number of months to display concurrently.
         */
        numberOfMonths: Cardinal;
        /**
         * Defines the view mode, whether a date or month is to be selected.
         */
        view: ViewMode;
        /**
         * Activates touch friendly mode
         */
        touchUI: boolean;
        /**
         * Specifies if the time picker should be displayed
         */
        showTime: boolean;
        /**
         * Shows only time picker without date.
         */
        timeOnly: boolean;
        /**
         * Whether to show the seconds in time picker. Default is `false`.
         */
        showSeconds: boolean;
        /**
         * Whether to show the milliseconds in time picker. Default is `false`.
         */
        showMilliseconds: boolean;
        /**
         * Defines the hour format, either 12 hour mode or 24 hour mode.
         */
        hourFormat: ClockConvention;
        /**
         * Hour steps.
         */
        stepHour: Cardinal;
        /**
         * Minute steps.
         */
        stepMinute: Cardinal;
        /**
         * Second steps.
         */
        stepSecond: Cardinal;
        /**
         * Millisecond steps.
         */
        stepMillisecond: Cardinal;
        /**
         * The cutoff year for determining the century for a date. Any dates entered with a year value less than or
         * equal to the cutoff year are considered to be in the current century, while those greater than it are deemed
         * to be in the previous century.
         */
        shortYearCutoff: string;
        /**
         * Defines if the popup should be hidden when a time is selected.
         */
        hideOnDateTimeSelect: boolean;
        /**
         * Custom localized settings for the {@link locale}.
         */
        userLocale: Partial<PickerLocale> | null;
        /**
         * Localized strings for various messages displayed by the date or time picker.
         */
        locale: Partial<PickerLocale>;
        /**
         * Date format to be used for parsing and formatting dates, such as `mm/dd/yy`.
         */
        dateFormat: string;
        /**
         * The range of years displayed in the year drop-down in (`nnnn:nnnn`) format such as (`2000:2020`). Default
         * value is `displayed_date - 10 : displayed_date + 10`.
         */
        yearRange: string | null;
        /**
         * Inline style of the container element.
         */
        panelStyle: string | null;
        /**
         * Style class of the container element.
         */
        panelStyleClass: string | null;
        /**
         * Style class of the individual date elements.
         */
        dateStyleClasses: string | null;
        /**
         * Whether to show the month navigator
         */
        monthNavigator: boolean;
        /**
         * Whether to show the year navigator
         */
        yearNavigator: boolean;
        /**
         * List of dates that should be disabled.
         */
        disabledDates: string[] | null;
        /**
         * List of dates that should be enabled.
         */
        enabledDates: string[] | null;
        /**
         * List of week day indexes that should be disabled.
         */
        disabledDays: DayOfTheWeek | null;
        /**
         * Sets date picker's minimum selectable value. Also used for validation on the server-side.
         */
        minDate: string | null;
        /**
         * Sets date picker's maximum selectable value. Also used for validation on the server-side.
         */
        maxDate: string | null;
        /**
         * Defines the maximum number of selectable dates in multiple selection mode.
         */
        maxDateCount: Cardinal | null;
        /**
         * Displays days belonging to other months.
         */
        showOtherMonths: boolean;
        /**
         * Enables selection of days belonging to other months.
         */
        selectOtherMonths: boolean;
        /**
         * Whether to display buttons at the footer.
         */
        showButtonBar: boolean;
        /**
         * Style class for the button that switches to the current date.
         */
        todayButtonStyleClass: string;
        /**
         * Style class for the button that clear the selected date or time.
         */
        clearButtonStyleClass: string;
        /**
         * Appends the dialog to the element defined by the CSS selector.
         */
        appendTo: string | null;
        /**
         * Javascript function that takes a date object and returns the content for the date cell.
         */
        dateTemplate: DateTemplate | null;
        /**
         * Whether an input is rendered for the time, or a text element only.
         */
        timeInput: boolean;
        /**
         * Client side callback to execute when input element receives focus.
         */
        onFocus: BaseEventCallback | null;
        /**
         * Client side callback to execute when input element loses focus.
         */
        onBlur: BaseEventCallback | null;
        /**
         * Client side callback to execute when data was entered into the input field.
         */
        onInput: BaseEventCallback | null;
        /**
         * Client side callback to execute when text within input element is selected by user.
         */
        onSelect: MutationEventCallback<[
            Date
        ]> | null;
        /**
         * Client side callback to execute when the selected date has changed.
         */
        onChange: BaseEventCallback | null;
        /**
         * Client side callback to execute when the displayed date changes.
         */
        onViewDateChange: MutationEventCallback<[
            Date
        ]> | null;
        /**
         * Client side callback to execute when the button to switch to the current date was clicked.
         */
        onTodayButtonClick: BaseEventCallback | null;
        /**
         * Client side callback to execute when the button to clear the selected date or time was clicked.
         */
        onClearButtonClick: BaseEventCallback | null;
        /**
         * Client side callback to execute before the date picker overlay is shown.
         */
        onBeforeShow: BaseCallback | null;
        /**
         * Client side callback to execute before the date picker overlay is hidden.
         */
        onBeforeHide: BaseCallback | null;
        /**
         * Client side callback to execute when the selected month has changed.
         */
        onMonthChange: MutationCallback<[
            MonthOfTheYearOneBased,
            YearInstant
        ]> | null;
        /**
         * Client side callback to execute when the panel with the date picker was created.
         */
        onPanelCreate: BaseCallback | null;
        /**
         * Client side callback to execute when the selected year has changed.
         */
        onYearChange: MutationCallback<[
            MonthOfTheYear,
            YearInstant
        ]> | null;
    }
    /**
     * Base interface for the {@link PickerInstance} that contains all methods that are available via the JQuery
     * wrapper, see {@link JQuery.datePicker}.
     */
    export interface PickerWidgetMethods {
        // =========================
        // === Getters / setters ===
        // =========================
        /**
         * Changes the selected date of this date picker to the given date, and updates the UI.
         * @param date The new date to set.
         */
        setDate(date: string | Date): void;
        /**
         * Retrieves the currently selected date of this date picker.
         * @return The currently selected date.
         */
        getDate(): Date | "" | null;
        // ==========
        // === UI ===
        // ==========
        /**
         * If the date picker is shown in an overlay panel, adjusts the position of that overlay panel so that is shown
         * at its proper location.
         */
        alignPanel(): void;
        /**
         * Clears the select time of the time picker.
         */
        clearTimePickerTimer(): void;
        /**
         * Shows the overlay panel with the date picker.
         */
        showOverlay(): void;
        /**
         * Hides the overlay panel with the date picker.
         */
        hideOverlay(): void;
        /**
         * Makes the overlay panel a modal dialog so that other elements in the page cannot be interacted with while the
         * overlay date picker is shown.
         */
        enableModality(): void;
        /**
         * Removes the modality feature from the overlay panel so that other elements in the page can be interacted with
         * while the overlay date picker is shown.
         */
        disableModality(): void;
        /**
         * Adjust the UI so that the given date now appears selected.
         * @param event The event that triggered the selection, such as a mouse click.
         * @param dateMeta The date that is to be selected.
         */
        selectDate(event: JQuery.TriggeredEvent, dateMeta: DayInstantSelectable): void;
        /**
         * Changes the current date of the navigation, i.e. the dates or times that are displayed from which the user
         * can select an option.
         * @param newViewDate New view date to set.
         */
        setNavigationState(newViewDate: Date): void;
        /**
         * @return Whether the date picker panel is currently displayed.
         */
        isPanelVisible(): boolean;
        /**
         * When the time picker up or down arrows are clicked and the mouse button is held down for a prolonged period
         * of time: repeatedly increment the minute or hour.
         * @param event Event that occurred, such as a click event.
         * @param interval Amount of time between successive increments.
         * @param type Which part of the time is to be incremented or decremented (hour, minute, second, or millisecond).
         * @param direction Whether to increment or decrement the time part.
         */
        repeat(event: JQuery.TriggeredEvent, interval: Cardinal, type: ChangeTimeType, direction: OneDimensionalDirection): void;
        /**
         * Updates the time display so that is shows the given time.
         * @param event Event that occurred.
         * @param hour Current hour.
         * @param minute Current minute.
         * @param second Current second.
         * @param millisecond Current millisecond.
         */
        updateTime(event: JQuery.TriggeredEvent, hour: HourOfTheDay, minute: MinuteOfTheHour, second: SecondOfTheMinute, millsecond: MillisecondOfTheSecond): void;
        /**
         * After a time was entered, updates the time display so that is shows the given time.
         * @param event Event that occurred.
         * @param newDateTime The time to display.
         */
        updateTimeAfterInput(event: JQuery.TriggeredEvent, newDateTime: Date): void;
        /**
         * Updates the year navigator element that lets the user choose a year so that it reflects the current settings.
         */
        updateYearNavigator(): void;
        /**
         * Updates the currently displayed date range.
         * @param event Event that occurred.
         * @param value The date to be displayed.
         */
        updateViewDate(event: JQuery.TriggeredEvent, value: Date): void;
        /**
         * Updates the hidden input field and saves the currently selected value.
         * @param event Event that occurred.
         * @param value Date that is selected.
         */
        updateModel(event: JQuery.TriggeredEvent | null, value: Date | Date[] | null): void;
        // ===========================
        // === Date and time logic ===
        // ===========================
        /**
         * Parses a string that either represents a date time, a list of date times, or a date time range, depending on
         * the current {@link PickerOptions.selectionMode} setting.
         *
         * When the given value is a date time, a list of date times, or a date time range already, returns that value
         * unchanged.
         * @param text The string with the date time, date times, or date time range to parse.
         * @return The parsed date time, date times, or date time range.
         */
        parseValue(option: string | Date | Date[]): Date | Date[];
        /**
         * Parses a string that either represent a date time, a list of date times, or a date time range, depending on
         * the current {@link PickerOptions.selectionMode} setting.
         * @param text The string with the date time, date times, or date time range to parse.
         * @return The parsed date time, date times, or date time range.
         */
        parseValueFromString(text: string): Date | Date[];
        /**
         * Parses a string with a time (but no date).
         * @param value The time string to parse.
         * @param ampm Whether it is currently before or after midday.
         * @return The parses date.
         */
        parseTime(value: string, ampm?: HalfDayPeriod): TimeOfTheDay;
        /**
         * Parses a string with a date (but no time).
         * @param value The date string to parse.
         * @param format The format to use for parsing.
         * @return The parsed date.
         */
        parseDate(value: string, format: string): Date;
        /**
         * Parses a string with a date and a time.
         * @param text The date time string to parse.
         * @return The parsed date.
         */
        parseDateTime(text: string): Date;
        /**
         * Parses a texual representation of a date that is stored in the {@link PickerOptions.disabledDates} list.
         * @param option Value to parse as a date.
         * @return The parsed date.
         */
        parseOptionValue(option: string): Date;
        /**
         * Parses the textual representation of a date as stored in {@link PickerOptions.minDate} and
         * {@link PickerOptions.maxDate}.
         * @param option A textual representation of a date to parse.
         * @return The parsed date.
         */
        parseMinMaxValue(option: string | Date): Date | Date[];
        /**
         * Formats the given point in time as a string with a date and a time.
         * @param date A point in time to format.
         * @return A textual representation of the given point in time, with a date and a time part.
         */
        formatDateTime(date: Date | undefined): string | null;
        /**
         * Formats the given point in time as a string, omitting the time part.
         * @param date A point in time to format.
         * @param format Date format to use.
         * @return A textual representation of the given point in time, including the date but omitting the time part.
         */
        formatDate(date: Date | undefined, format: string): string;
        /**
         * Formats the given point in time as a string, omitting the date part.
         * @param date A point in time to parse.
         * @return A text represent of the given point in time, including the time part but omitting the date part.
         */
        formatTime(date: Date | undefined): string;
        /**
         * Converts a date object to an ISO date (date only, no time) string. Useful to check if a dates matches with a
         * date sent from the backend without needing to parse the backend date first.
         * @param date Date to convert.
         * @return The data as an ISO date string.
         */
        toISODateString(date: Date): string;
        /**
         * Finds the day of the week index that represents the first day of the week for the given month.
         * @param month Month to check.
         * @param year Year to check.
         * @return The day of the week index that represents the first day of the week for the given month.
         */
        getFirstDayOfMonthIndex(month: MonthOfTheYear, year: YearInstant): DayOfTheWeek;
        /**
         * Finds the day of the week index that represents sunday.
         * @return The day of the week index that represents sunday.
         */
        getSundayIndex(): number;
        /**
         * Finds the number of day in the given month.
         * @param month Month to check.
         * @param year Year to check.
         * @return The number of days in the given month.
         */
        getDaysCountInMonth(month: MonthOfTheYear, year: YearInstant): Cardinal;
        /**
         * Finds the number of day in month before the given month.
         * @param month Month to check.
         * @param year Year to check.
         * @return The number of days in month before the given month.
         */
        getDaysCountInPrevMonth(month: MonthOfTheYear, year: YearInstant): Cardinal;
        /**
         * Finds the month of the year index and year index of the month preceding the given month.
         * @param month Month to check.
         * @param year Year to check.
         * @return The month before the given month.
         */
        getPreviousMonthAndYear(month: MonthOfTheYear, year: YearInstant): MonthInstant;
        /**
         * Finds the month of the year index and year index of the month succeeding the given month.
         * @param month Month to check.
         * @param year Year to check.
         * @return The month after the given month.
         */
        getNextMonthAndYear(month: MonthOfTheYear, year: YearInstant): MonthInstant;
        /**
         * Finds the formatted date or time string that is to be shown as the currently selected date or time.
         * @return The currently selected date or time, formatted according to the current options.
         */
        getValueToRender(): string;
        /**
         * Creates a list of super short day names in a week.
         * @return A list with the super sort day names in a week.
         */
        createWeekDaysMin(): string[];
        /**
         * Creates a list of short day names in a week.
         * @return A list with the sort day names in a week.
         */
        createWeekDaysShort(): string[];
        /**
         * Creates a list of long day names in a week.
         * @return A list with the long day names in a week.
         */
        createWeekDays(): string[];
        /**
         * Creates a list of all days in the year, starting at the given month.
         * @param month Month where to start. Months before that are not included in the returned list.
         * @param year Year to check.
         * @return A list with all days in the year, starting at the given month.
         */
        createMonths(month: MonthOfTheYear, year: YearInstant): DayListInMonth[];
        /**
         * Creates a list of all days in the given month.
         * @param month A month to check.
         * @param year A year to check.
         * @param index Index that will be included in the return value.
         * @return All days in the given month.
         */
        createMonth(month: MonthOfTheYear, year: YearInstant, index: number): DayListInMonth;
        /**
         * @param value A value to check whether it is a Date instance.
         * @return `true` if the value is an instance of `Date`, and `false` otherwise.
         */
        isDate(value: unknown): value is Date;
        /**
         * Checks whether thee given day can be selected.
         * @param day A day to check.
         * @param month A month to check.
         * @param year A year to check.
         * @param otherMonth Whether the given month belongs to another month other than the currently displayed month.
         * See {@link PickerOptions.selectOtherMonths}.
         */
        isSelectable(day: DayOfTheMonth, month: MonthOfTheYear, year: YearInstant, otherMonth: boolean): boolean;
        /**
         * Checks whether the given day is selected.
         * @param dateMeta Day to check.
         * @return Whether the given day is selected.
         */
        isSelected(dateMeta: DayInstantSelectable): boolean;
        /**
         * Checks whether the {@link PickerOptions.selectionMode} is currently set to `single`.
         * @return Whether only a single date can be selected.
         */
        isSingleSelection(): boolean;
        /**
         * Checks whether the {@link PickerOptions.selectionMode} is currently set to `range`.
         * @return Whether a range of dates can be selected.
         */
        isRangeSelection(): boolean;
        /**
         * Checks whether the {@link PickerOptions.selectionMode} is currently set to `multiple`.
         * @return Whether multiple dates can be selected.
         */
        isMultipleSelection(): boolean;
        /**
         * Checks whether the given month is currently selected.
         * @param month A month to check.
         * @return Whether the given month is currently selected.
         */
        isMonthSelected(month: MonthOfTheYear): boolean;
        /**
         * Checks whether the given date equals the other given date.
         * @param value First date for the comparison.
         * @param dateMeta Other date for the comparison.
         * @return `true` if both given values represent the same date, or `false` otherwise.
         */
        isDateEquals(value: Date | undefined, dateMeta: DayInstantSelectable): boolean;
        /**
         * Checks whether the given date lies in the given range.
         * @param start Start point of the date range.
         * @param end End point of the date range.
         * @param dateMeta Date to check whether it lies in the given range.
         * @return `true` if the given date lies in the range `[start, end]` (inclusive), or `false` otherwise.
         */
        isDateBetween(start: Date | undefined, end: Date | undefined, dateMeta: DayInstantSelectable): boolean;
        /**
         * Checks whether the given date is equal to the current date.
         * @param today The date of today.
         * @param day Day to check.
         * @param month Month to check.
         * @param year Year to check.
         * @return `true` if the given `today` represents the same date as the given `day`, `month`, and `year`.
         */
        isToday(today: Date, day: DayOfTheMonth, month: MonthOfTheYear, year: YearInstant): boolean;
        /**
         * Checks whether the given date is currently disabled and cannot be selected.
         * @param day Day to check.
         * @param month Month to check.
         * @param year Year to check.
         * @return Whether the given date is currently disabled and cannot be selected.
         */
        isDateDisabled(day: DayOfTheMonth, month: MonthOfTheYear, year: YearInstant): boolean;
        /**
         * Checks whether the given day is currently disabled and cannot be selected.
         * @param day Day to check.
         * @param month Month to check.
         * @param year Year to check.
         * @return Whether the given day is currently disabled and cannot be selected.
         */
        isDayDisabled(day: DayOfTheMonth, month: MonthOfTheYear, year: YearInstant): boolean;
        /**
         * Checks whether the year of the currently displayed month page is equal to the year of the
         * {@link PickerOptions.minDate}.
         * @return Whether the year of the currently displayed month page is equal to the year of the
         * {@link PickerOptions.minDate}.
         */
        isInMinYear(): boolean;
        /**
         * Checks whether the year of the currently displayed month page is equal to the year of the
         * {@link PickerOptions.maxDate}.
         * @return Whether the year of the currently displayed month page is equal to the year of the
         * {@link PickerOptions.maxDate}.
         */
        isInMaxYear(): boolean;
        daylightSavingAdjust(date: Date): Date;
        populateTime(value: Date, timeString: string, ampm?: HalfDayPeriod): void;
        validateTime(hour: HourOfTheDay, minute: MinuteOfTheHour, second: SecondOfTheMinute, millisecond: MillisecondOfTheSecond, value: Date, direction: AlterationMode): boolean;
        // =================
        // === Rendering ===
        // =================
        /**
         * Creates the HTML snippet for the trigger button and saves it in this instance.
         */
        renderTriggerButton(): void;
        /**
         * Creates the HTML snippet for the date picker panel and saves it in this instance.
         */
        renderDatePickerPanel(): void;
        /**
         * Creates the HTML snippet for the panel elements.
         * @return The rendered HTML snippet.
         */
        renderPanelElements(): string;
        /**
         * Creates the HTML snippet for the date view that shows the current month page.
         * @return The rendered HTML snippet.
         */
        renderDateView(): string;
        /**
         * Creates the HTML snippet for the month view with the days in the current month.
         * @return The rendered HTML snippet.
         */
        renderMonthView(): string;
        /**
         * Creates the HTML snippet for the time picker that lets the user select an hour, minute, and second.
         * @return The rendered HTML snippet.
         */
        renderTimePicker(): string;
        /**
         * Creates the HTML snippet for the button bar with the today and clear buttons.
         * @return The rendered HTML snippet.
         */
        renderButtonBar(): string;
        /**
         * Creates the HTML snippet for the month part of the month view.
         * @param index Month to use.
         * @return The rendered HTML snippet.
         */
        renderMonthViewMonth(index: MonthOfTheYear): string;
        /**
         * Creates the HTML snippet for the month list part of the month view.
         * @return The rendered HTML snippet.
         */
        renderMonthViewMonths(): string;
        /**
         * Creates the HTML snippet for the given days.
         * @param monthsMetadata List of days to render.
         * @return The rendered HTML snippet.
         */
        renderMonths(monthsMetadata: DayListInMonth[]): string;
        /**
         * Creates the HTML snippet for the given days in a month.
         * @param monthMetadata List of days to render
         * @param index Month to which the days belong.
         * @return The rendered HTML snippet.
         */
        renderMonth(monthMetadata: DayListInMonth, index: MonthOfTheYear): string;
        /**
         * Creates the HTML snippet for the button for navigating to the previous month.
         * @return The rendered HTML snippet.
         */
        renderBackwardNavigator(): string;
        /**
         * Creates the HTML snippet for the button for navigating to the next month.
         * @return The rendered HTML snippet.
         */
        renderForwardNavigator(): string;
        /**
         * Creates the HTML snippet for a title bar that shows the given month.
         * @param month Month to use.
         * @param index 0-based index of the month in the year.
         * @return The rendered HTML snippet.
         */
        renderTitleMonthElement(month: MonthOfTheYear, index: MonthOfTheYear): string;
        /**
         * Creates the HTML snippet for a title bar that shows the given year.
         * @param year Year to use.
         * @param index 0-based index of the month in the year.
         * @return The rendered HTML snippet.
         */
        renderTitleYearElement(year: YearInstant, index: MonthOfTheYear): string;
        /**
         * Creates the HTML snippet for the options elements of the select element in the title bar that lets the user
         * switch to another month.
         * @param name Whether to create the options for months or years.
         * @param options List of month names.
         * @return The rendered HTML snippet.
         */
        renderTitleOptions(name: "month", options: string[]): string;
        /**
         * Creates the HTML snippet for the options elements of the select element in the title bar that lets the user
         * switch to another year.
         * @param name Whether to create the options for months or years.
         * @param options List of year numbers to use as options.
         * @return The rendered HTML snippet.
         */
        renderTitleOptions(name: "year", options: YearInstant[]): string;
        /**
         * Creates the HTML snippet for the title bar of the given month.
         * @param monthMetadata Month to use.
         * @return The rendered HTML snippet.
         */
        renderTitle(monthMetadata: DayListInMonth): string;
        /**
         * Creates the HTML snippet for the names of the given days.
         * @param weekDaysMin List of super short week day names.
         * @param weekDays List of long week day names.
         * @return The rendered HTML snippet.
         */
        renderDayNames(weekDaysMin: string[], weekDays: string[]): string;
        /**
         * Creates the HTML snippet for the days in the given week.
         * @param weekDates List of days in the week.
         * @return The rendered HTML snippet.
         */
        renderWeek(weekDates: DayInstantSelectableRelative[]): string;
        /**
         * Creates the HTML snippet for the content of a date cell with a single day.
         * @param date Date to render.
         * @param dateClass Style class to apply.
         * @return The rendered HTML snippet.
         */
        renderDateCellContent(date: DayInstantSelectableRelative, dateClass: string): string;
        /**
         * Creates the HTML snippet for the given dates.
         * @param monthMetadata List of dates to render.
         * @return The rendered HTML snippet.
         */
        renderDates(monthMetadata: DayListInMonth): string;
        /**
         * Creates the HTML snippet for the date view grid of the given month.
         * @param monthMetadata Month to use.
         * @param weekDaysMin List of super short week day names.
         * @param weekDays List of long week names.
         * @return The rendered HTML snippet.
         */
        renderDateViewGrid(monthMetadata: DayListInMonth, weekDaysMin: string[], weekDays: string[]): string;
        /**
         * Creates the HTML snippet for the hour picker for selecting an hour.
         * @return The rendered HTML snippet.
         */
        renderHourPicker(): string;
        /**
         * Creates the HTML snippet for the minute picker for selecting a minute.
         * @return The rendered HTML snippet.
         */
        renderMinutePicker(): string;
        /**
         * Creates the HTML snippet for the second picker for selecting a second.
         * @return The rendered HTML snippet.
         */
        renderSecondPicker(): string;
        /**
         * Creates the HTML snippet for the millisecond picker for selecting a millisecond.
         * @return The rendered HTML snippet.
         */
        renderMillisecondPicker(): string;
        /**
         * Creates the HTML snippet for the picker that lets the user choose between `a.m.` and `p.m.`.
         * @return The rendered HTML snippet.
         */
        renderAmPmPicker(): string;
        /**
         * Creates the HTML snippet for separator between hours, minutes, and seconds (such as a colon).
         * @return The rendered HTML snippet.
         */
        renderSeparator(): string;
        /**
         * Creates the HTML snippet for separator before fractional seconds (such as a dot).
         * @return The rendered HTML snippet.
         */
        renderFractionSeparator(): string;
        /**
         * Creates the HTML snippet for container with the up and down button.
         * @param containerClass Style class for the container.
         * @param text Text to shown in the time element container.
         * @param type Whether to render the time elements of a hour, minute, second, or millisecond.
         * @return The rendered HTML snippet.
         */
        renderTimeElements(containerClass: string, text: string, type: ChangeTimeType): string;
        /**
         * Creates the HTML snippet for the button to increment the hour, minutes, second, or millisecond.
         * @return The rendered HTML snippet.
         */
        renderTimePickerUpButton(): string;
        /**
         * Creates the HTML snippet for the button to decrement the hour, minutes, second, or millisecond.
         * @return The rendered HTML snippet.
         */
        renderTimePickerDownButton(): string;
        // ======================
        // === Event handling ===
        // ======================
        /**
         * Adds the event listener for click events to the document.
         */
        bindDocumentClickListener(): void;
        /**
         * Removes the event listener for click events from the document.
         */
        unbindDocumentClickListener(): void;
        /**
         * Callback that is invoked when the date input was clicked.
         * @param event Event that occurred.
         */
        onInputClick(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the date input was focused.
         * @param event Event that occurred.
         */
        onInputFocus(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the date input lost focus.
         * @param event Event that occurred.
         */
        onInputBlur(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a key was pressed in the date input.
         * @param event Event that occurred.
         */
        onInputKeyDown(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the user made an input.
         * @param event Event that occurred.
         */
        onUserInput(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the trigger button was pressed.
         * @param event Event that occurred.
         */
        onButtonClick(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the date picker panel was clicked.
         * @param event Event that occurred.
         */
        onPanelClick(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a different month was selected in the dropdown menu in the title bar.
         * @param event Event that occurred.
         */
        onMonthDropdownChange(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a different year was selected in the dropdown menu in the title bar.
         * @param event Event that occurred.
         */
        onYearDropdownChange(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a different month was selected by clicking on that month.
         * @param event Event that occurred.
         * @param month Month that was selected.
         */
        onMonthSelect(event: JQuery.TriggeredEvent, month: MonthOfTheYear): void;
        /**
         * Callback that is invoked when the left mouse button was pressed down while the cursor is over the time picker
         * element.
         * @param event Event that occurred.
         * @param type Whether the hour, minute, second, or millisecond was clicked.
         * @param direction Whether the up or down button was clicked.
         */
        onTimePickerElementMouseDown(event: JQuery.TriggeredEvent, type: ChangeTimeType, direction: OneDimensionalDirection): void;
        /**
         * Callback that is invoked when the left mouse button was release while the cursor is over the time picker
         * element.
         * @param event Event that occurred.
         */
        onTimePickerElementMouseUp(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a date was selected by clicking on it.
         * @param event Event that occurred.
         * @param dateMeta Day that was clicked.
         */
        onDateSelect(event: JQuery.TriggeredEvent, dateMeta: DayInstantSelectable): void;
        /**
         * Callback that is invoked when the today button was pressed.
         * @param event Event that occurred.
         */
        onTodayButtonClick(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the clear button was pressed.
         * @param event Event that occurred.
         */
        onClearButtonClick(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a value was entered in the hour input.
         * @param input Hour input element.
         * @param event Event that occurred.
         */
        handleHoursInput(input: HTMLElement, event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a value was entered in the minute input.
         * @param input Minute input element.
         * @param event Event that occurred.
         */
        handleMinutesInput(input: HTMLElement, event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a value was entered in the second input.
         * @param input Second input element.
         * @param event Event that occurred.
         */
        handleSecondsInput(input: HTMLElement, event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when a value was entered in the millisecond input.
         * @param input Millisecond input element.
         * @param event Event that occurred.
         */
        handleMillisecondsInput(input: HTMLElement, event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the up button of the hour input was pressed.
         * @param event Event that occurred.
         */
        incrementHour(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the down button of the hour input was pressed.
         * @param event Event that occurred.
         */
        decrementHour(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the up button of the minute input was pressed.
         * @param event Event that occurred.
         */
        incrementMinute(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the down button of the minute input was pressed.
         * @param event Event that occurred.
         */
        decrementMinute(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the up button of the second input was pressed.
         * @param event Event that occurred.
         */
        incrementSecond(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the down button of the second input was pressed.
         * @param event Event that occurred.
         */
        decrementSecond(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the up button of the millisecond input was pressed.
         * @param event Event that occurred.
         */
        incrementMillisecond(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the down button of the millisecond input was pressed.
         * @param event Event that occurred.
         */
        decrementMillisecond(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when button for navigating to the previous month was pressed.
         * @param event Event that occurred.
         */
        navBackward(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when button for navigating to the next month was pressed.
         * @param event Event that occurred.
         */
        navForward(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the button for switching between `a.m.` and `p.m.` was pressed.
         * @param event Event that occurred.
         */
        toggleAmPm(event: JQuery.TriggeredEvent): void;
        // ============
        // === Misc ===
        // ============
        /**
         * Joins the given style class names for use in the HTML class attribute.
         * @param classes List of style classes.
         * @return The given classes, joined with a space.
         */
        getClassesToAdd(classes: string[]): string;
        /**
         * Escapes characters that have a special meaning in HTML, so that the given value can be used safely as a value
         * in an HTML attribute or text node.
         * @param value Value to escape.
         * @return The given value, escaped for use in an HTML attribute or text node.
         */
        escapeHTML(value: string): string;
    }
    /**
     * The date picker instance used by the JQuery UI widget. You can retrieve it via `$(...).datePicker("instance")`.
     */
    export interface PickerInstance extends PickerWidgetMethods, JQueryUI.WidgetCommonProperties {
        /**
         * The current options of this widget instance.
         */
        options: PickerOptions;
        /**
         * Namespace for event triggered by this widget.
         */
        eventNamespace: string;
        bindings: JQuery;
        /**
         * The DOM element that is hoverable.
         */
        hoverable: JQuery;
        /**
         * The DOM element that is focusable.
         */
        focusable: JQuery;
        /**
         * The container element of the date picker.
         */
        container: JQuery;
        /**
         * The hidden input field containing the currently selected value.
         */
        inputfield: JQuery;
        /**
         * The currently selected date time, list of date times, or date time range.
         */
        value: Date | Date[];
        /**
         * The currently displayed date or dates.
         */
        viewDate: Date | Date[];
        /**
         * Number of microseconds since January 1, 1970.
         */
        ticksTo1970: Cardinal;
        /**
         * The panel element of the date picker.
         */
        panel: JQuery;
        /**
         * List of all days in the current year.
         */
        monthsMetadata: DayListInMonth[];
        /**
         * Mask for the modal overlay.
         */
        mask: JQuery | null;
        /**
         * Handler for CSS transitions used by this date picker.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * Trigger button that opens or closes the date picker.
         */
        triggerButton?: JQuery;
        /**
         * Whether a custom year range was specified.
         */
        hasCustomYearRange: boolean;
    }
}
interface JQuery {
    /**
     * Initializes the date picker on the current elements.
     * @param options Optional settings for configuring the date picker.
     * @return This JQuery instance for chaining.
     */
    datePicker(...options: Partial<JQueryPrimeDatePicker.PickerOptions>[]): this;
    /**
     * Retrieves the current date picker instance.
     * @param method Name of the method to call on the widget.
     * @return The current date picker instance, or `undefined` if the widget was not yet initialized.
     */
    datePicker(method: "instance"): JQueryPrimeDatePicker.PickerInstance | undefined;
    /**
     * Removes the widget functionality completely. This will return the element back to its pre-init state.
     * @param method Name of the method to call on the widget.
     * @return This JQuery instance for chaining.
     */
    datePicker(method: "destroy"): this;
    /**
     * Returns a JQuery object containing the original element or other relevant generated elements.
     * @param method Name of the method to call on the widget.
     * @return A JQuery object with the original element or other relevant generated elements.
     */
    datePicker(method: "widget"): JQuery;
    /**
     * Finds the current option of the date picker widget.
     * @param method Name of the method to call on the widget.
     * @return The current options of this date picker widget.
     */
    datePicker(method: "option"): JQueryPrimeDatePicker.PickerOptions;
    /**
     * Updates the current options with the given options.
     * @param method Name of the method to call on the widget.
     * @return This JQuery instance for chaining.
     */
    datePicker(method: "option", options: Partial<JQueryPrimeDatePicker.PickerOptions>): this;
    /**
     * Finds the value of the given option.
     * @typeparam K Name of an option to retrieve.
     * @param method Name of the method to call on the widget.
     * @param optionName Name of an option to retrieve.
     * @return The value of the given option.
     */
    datePicker<K extends keyof JQueryPrimeDatePicker.PickerOptions>(method: "option", optionName: K): JQueryPrimeDatePicker.PickerOptions[K];
    /**
     * Sets the value of the given option to the given value.
     * @typeparam K Name of an option to set.
     * @param method Name of the method to call on the widget.
     * @param optionName Name of an option to set.
     * @param optionValue Value of the option to set.
     * @return This JQuery instance for chaining.
     */
    datePicker<K extends keyof JQueryPrimeDatePicker.PickerOptions>(method: "option", optionName: K, optionValue: JQueryPrimeDatePicker.PickerOptions[K]): this;
    /**
     * Calls a method of the {@link JQueryPrimeDatePicker.PickerInstance} and return the result.
     * @typeparam K Name of the method to call on the widget.
     * @param method Name of the method to call on the widget.
     * @param args Arguments as required by the {@link JQueryPrimeDatePicker.PickerInstance} method.
     * @return The value as returned by the {@link JQueryPrimeDatePicker.PickerInstance} method. Return this JQuery
     * instance for chaining when the instance method has no return value.
     * @see {@link PickerWidgetMethods}
     */
    datePicker<K extends keyof JQueryPrimeDatePicker.PickerWidgetMethods>(method: K, ...args: Parameters<JQueryPrimeDatePicker.PickerWidgetMethods[K]>): PrimeFaces.ToJQueryUIWidgetReturnType<JQueryPrimeDatePicker.PickerInstance, ReturnType<JQueryPrimeDatePicker.PickerWidgetMethods[K]>, this>;
}
declare namespace PrimeFaces.widget.DatePicker {
    /**
     * User-defined callback invoked before the date picker overlay is
     * shown.
     */
    export type PreShowCallback =
    /**
     * @param datePicker The current
     * date picker instance.
     */
    (this: PrimeFaces.widget.DatePickerCfg, datePicker: JQueryPrimeDatePicker.PickerInstance) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DatePicker Widget__
     *
     * DatePicker is an input component used to select a date featuring display modes, paging, localization, ajax selection
     * and more.
     *
     * DatePicker is designed to replace the old {@link Calendar|p:calendar} component.
     *
     * To interact with the calendar, use the methods of this widget, or for more advanced usages, use the `datePicker`
     * JQueryUI widget plugin, for example:
     *
     * ```javascript
     * PF("datePickerWidget").getDate();
     * PF("datePickerWidget").jq.datePicker("getDate");
     *
     * PF("datePickerWidget").setDate(new Date());
     * PF("datePickerWidget").jq.datePicker("setDate", new Date());
     *
     * PF("datePickerWidget").jq.datePicker("enableModality");
     * ```
     * @typeparam TCfg Defaults to `DatePickerCfg`. Type of the configuration object for this widget.
     */
    export class DatePicker<TCfg extends DatePickerCfg = DatePickerCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the hidden input element with the selected date.
         */
        input: JQuery;
        /**
         * The DOM element for the inline picker or the input.
         */
        jqEl: JQuery;
        /**
         * The DOM element for the panel with the datepicker.
         */
        panel?: JQuery;
        /**
         * Whether focus should be put on the input again.
         */
        refocusInput: boolean;
        /**
         * The DOM element for the trigger button if using showIcon.
         */
        triggerButton: JQuery;
        /**
         * The date that is displayed in the date picker.
         */
        viewDateOption: Date | Date[];
        /**
         * Initializes the mask on the input if using a mask and not an inline picker.
         */
        private applyMask(): void;
        /**
         * Sets up the event listener for when the Clear button is selected.
         */
        private bindClearButtonListener(): void;
        /**
         * Sets up the event listeners for when the date picker is closed.
         */
        private bindCloseListener(): void;
        /**
         * Sets up the event listener for when another date was selected.
         */
        private bindDateSelectListener(): void;
        /**
         * Callback for after the overlay panel is created.
         */
        private bindPanelCreationListener(): void;
        /**
         * Sets up the event listener for when the date picker changes to a different month or year page.
         */
        private bindViewChangeListener(): void;
        /**
         * Initializes the localized messages for the currently configured language.
         */
        private configureLocale(): void;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         * @override
         */
        override destroy(): void;
        /**
         * Disables the datepicker, so that the user can no longer select any date.
         */
        disable(): void;
        /**
         * Enables the datepicker, so that the user can select a date.
         */
        enable(): void;
        /**
         * Fires the close event when the date picker is closed.
         */
        private fireCloseEvent(): void;
        /**
         * Triggers the event for when another date was selected.
         */
        private fireDateSelectEvent(): void;
        /**
         * Triggers the event for when the date picker changed to a different month or year page.
         * @param date The date to which the date picker changed.
         */
        private fireViewChangeEvent(date: Date): void;
        /**
         * Gets the currently selected date value of the date picker.
         * @return The date, if one is currently selected. The empty `string` or `null` when no date
         * is selected.
         */
        getDate(): Date | string | null;
        /**
         * Gets the displayed visible calendar date. This refers to the currently displayed month page.
         * @return The currently displayed date or dates.
         */
        getViewDate(): Date | Date[];
        /**
         * Checks whether a date is selected.
         * @return true if a date is selected.
         */
        hasDate(): boolean;
        /**
         * Hide the popup panel.
         */
        hide(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Sets the date value the date picker.
         * @param date The new date for this widget.
         */
        setDate(date: Date | string): void;
        /**
         * Sets the disabled dates.
         * @param disabledDates The dates to disable.
         */
        setDisabledDates(disabledDates: string[] | Date[]): void;
        /**
         * Sets the disabled days.
         * @param disabledDays The days to disable.
         */
        setDisabledDays(disabledDays: number[]): void;
        /**
         * Sets the enabled dates.
         * @param enabledDates The dates to enable.
         */
        setEnabledDates(enabledDates: string[] | Date[]): void;
        /**
         * Sets the displayed visible calendar date. This refers to the currently displayed month page.
         * @param date The date to be shown in the calendar.
         */
        setViewDate(date: string | Date | Date[]): void;
        /**
         * Shows the popup panel.
         */
        show(): void;
        /**
         * Update panel.
         */
        private updatePanel(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DatePicker| DatePicker widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DatePickerCfg extends JQueryPrimeDatePicker.PickerOptions, PrimeFaces.widget.BaseWidgetCfg {
        /**
         * The search expression for the element to which the overlay panel should be appended.
         */
        appendTo: string;
        /**
         * Tabindex of the datepicker button
         */
        buttonTabindex: string;
        /**
         * When enabled, input receives focus after a value is picked.
         */
        focusOnSelect: boolean;
        /**
         * Whether the datepicker is rendered inline or as an overlay.
         */
        inline: boolean;
        /**
         * Applies a mask using the pattern.
         */
        mask: string;
        /**
         * Clears the field on blur when incomplete input is entered
         */
        maskAutoClear: boolean;
        /**
         * Placeholder in mask template.
         */
        maskSlotChar: string;
        /**
         * Callback invoked after the datepicker panel was created.
         */
        onPanelCreate: JQueryPrimeDatePicker.BaseCallback;
        /**
         * User-defined callback that may be overridden by the
         * user. Invoked before the date picker overlay is shown.
         */
        preShow: PrimeFaces.widget.DatePicker.PreShowCallback;
        /**
         * Separator for joining start and end dates when selection mode is `range`, defaults
         * to `-`.
         */
        rangeSeparator: string;
        /**
         * Separator for joining hour and minute, defaults to `:`.
         */
        timeSeparator: string;
        /**
         * Icon of the datepicker element that toggles the visibility in popup mode.
         */
        triggerButtonIcon: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Diagram Widget__
     *
     * Diagram is generic component to create visual elements and connect them on a web page. SVG is used on modern browsers
     * and VML on IE 8 and below. Component is highly flexible in terms of api, events and theming.
     * @typeparam TCfg Defaults to `DiagramCfg`. Type of the configuration object for this widget.
     */
    export class Diagram<TCfg extends DiagramCfg = DiagramCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * The JSPlumb instance for this diagram.
         */
        canvas: import("jsplumb").jsPlumbInstance;
        /**
         * Internal state whether the connection was changed before a connect event.
         */
        connectionChanged: boolean;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Sets up all event listeners for this widget.
         */
        private bindEvents(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Initializes the lines connecting the diagram nodes.
         */
        private initConnections(): void;
        /**
         * Initializes the end points of the lines connection diagram nodes.
         */
        private initEndPoints(): void;
        /**
         * Callback for the event when two nodes are connected with each other.
         * @param info Details about the connection that was made.
         */
        private onConnect(info: PrimeFaces.widget.Diagram.ConnectionInfo): void;
        /**
         * Callback for the event when the connection between two nodes was changed.
         * @param info Details about the connection that was changed.
         */
        private onConnectionChange(info: PrimeFaces.widget.Diagram.UpdateConnectionInfo): void;
        /**
         * Callback for the event when two nodes are disconnected from each other.
         * @param info Details about the connection that was severed.
         */
        private onDisconnect(info: PrimeFaces.widget.Diagram.ConnectionInfo): void;
        /**
         * Callback for when the location of a diagram element has changed.
         * @param info Details about the element that was changed.
         */
        private onUpdateElementPosition(info: PrimeFaces.widget.Diagram.UpdateElementInfo): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link Diagram|diagram widget}. You can
     * access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DiagramCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * Overlays for the connections (arrows, labels, etc.)
         */
        connectionOverlays: import("jsplumb").OverlaySpec[];
        /**
         * List of initial connections to be established between nodes.
         */
        connections: {
            uuids: string[];
        }[];
        /**
         * Whether connections can be severed by the user.
         */
        connectionsDetachable: boolean;
        /**
         * Constrains dragging to within the bounds of the specified element or region.
         */
        containment: string;
        /**
         * Connector (straight lines, bezier curves, etc.) to use
         * by default.
         */
        defaultConnector: import("jsplumb").ConnectorSpec;
        /**
         * A list of
         * endpoints (ports) of all diagram nodes.
         */
        endPoints: (import("jsplumb").EndpointOptions & {
            element: import("jsplumb").ElementGroupRef;
        })[];
        /**
         * Paint style to use when hovering.
         */
        hoverPaintStyle: import("jsplumb").PaintStyle;
        /**
         * Maximum number of allowed connections (per node).
         */
        maxConnections: number;
        /**
         * Paint style to use when not hovering.
         */
        paintStyle: import("jsplumb").PaintStyle;
    }
}
declare namespace PrimeFaces.widget.Diagram {
    /**
     * Details about an element when its location
     * was changed.
     */
    export interface UpdateElementInfo {
        /**
         * ID of the element that was changed.
         */
        elementId: string;
        /**
         * New horizontal position of the element.
         */
        x: number;
        /**
         * New vertical position of the element.
         */
        y: number;
    }
}
declare namespace PrimeFaces.widget.Diagram {
    /**
     * Details about a connection between
     * two nodes that was changed.
     */
    export interface UpdateConnectionInfo extends PrimeFaces.widget.Diagram.ConnectionInfo {
        /**
         * Previous UUID of the point (port) where the connections
         * started.
         */
        originalSourceEndpoint: string;
        /**
         * Previous ID of the source node where the connection started.
         */
        originalSourceId: string;
        /**
         * Previous UUID of the point (port) where the connections
         * ended.
         */
        originalTargetEndpoint: string;
        /**
         * Previous ID of the target node where the connection ended.
         */
        originalTargetId: string;
    }
}
declare namespace PrimeFaces.widget.Diagram {
    /**
     * Details about a connection between two nodes that
     * was either established or dissolved.
     */
    export interface ConnectionInfo {
        /**
         * UUID of the point (port) where the connections starts.
         */
        sourceEndpoint: string;
        /**
         * ID of the source node where the connection starts.
         */
        sourceId: string;
        /**
         * UUID of the point (port) where the connections ends.
         */
        targetEndpoint: string;
        /**
         * ID of the target node where the connection end.
         */
        targetId: string;
    }
}
declare namespace PrimeFaces.widget.Dialog {
    /**
     * Client-side callback to invoke when the dialog is closed, see
     * {@link DialogCfg.onHide}.
     */
    export type OnHideCallback = (this: PrimeFaces.widget.Dialog) => void;
}
declare namespace PrimeFaces.widget.Dialog {
    /**
     * Client-side callback to invoke when the dialog is opened, see
     * {@link DialogCfg.onShow}
     */
    export type OnShowCallback = (this: PrimeFaces.widget.Dialog) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Dialog Widget__
     *
     * Dialog is a panel component that is displayed as an overlay on top of other elements on the current page. Optionally,
     * the dialog may be modal and block the user from interacting with elements below the dialog.
     * @typeparam TCfg Defaults to `DialogCfg`. Type of the configuration object for this widget.
     */
    export class Dialog<TCfg extends DialogCfg = DialogCfg> extends PrimeFaces.widget.DynamicOverlayWidget<TCfg> {
        /**
         * DOM element of the icon for closing this dialog, when this dialog is closable (an `x` by
         * default).
         */
        closeIcon: JQuery;
        /**
         * DOM element of the container for the content of this dialog.
         */
        content: JQuery;
        /**
         * Element that was focused before the dialog was opened.
         */
        focusedElementBeforeDialogOpened: HTMLElement;
        /**
         * DOM element of the container with the footer of this dialog.
         */
        footer: JQuery;
        /**
         * DOM elements of the title bar icons of this dialog.
         */
        icons: JQuery;
        /**
         * The DOM element of the overlay that is put over iframes during a resize.
         */
        iframeFix?: JQuery;
        /**
         * The native DOM element instance of the container element of this widget (same element as the
         * `jq` property).
         */
        jqEl: HTMLElement;
        /**
         * The last known vertical scrolling position.
         */
        lastScrollTop?: number;
        /**
         * Whether the dialog content was already loaded (when dynamic loading via AJAX is
         * enabled.)
         */
        loaded?: boolean;
        /**
         * DOM element of the icon for maximizing this dialog, when this dialog can be maximized.
         */
        maximizeIcon: JQuery;
        /**
         * Whether the dialog is currently maximized.
         */
        maximized?: boolean;
        /**
         * DOM element clone of the JQ to be used for minimizing.
         */
        minimizeClone: JQuery;
        /**
         * DOM element of the icon for minimizing this dialog, when this dialog can be minimized.
         */
        minimizeIcon: JQuery;
        /**
         * Whether the dialog is currently minimized.
         */
        minimized?: boolean;
        /**
         * The DOM element of the parent that contains this dialog, i.e the element to which the dialog
         * was appended.
         */
        parent: JQuery;
        /**
         * Whether the position of the dialog was already set. If not, it must be set the
         * next time the dialog is shown.
         */
        positionInitialized: boolean;
        /**
         * The DOM element of the resize icons for resizing the dialog, if resizing is
         * enabled.
         */
        resizers?: JQuery;
        /**
         * The client-side state of the dialog such as its width
         * and height. The client-side state can be preserved during AJAX updates by sending it to the server.
         */
        state?: PrimeFaces.widget.Dialog.ClientState;
        /**
         * DOM element of the title bar container of this dialog.
         */
        titlebar: JQuery;
        /**
         * Performs the client-side actions needed to actually show this dialog. Compare to `show`, which loads the dialog
         * content from the server if required, then call this method.
         * @param duration Durations are given in milliseconds; higher values indicate slower
         * animations, not faster ones. The strings `fast` and `slow` can be supplied to indicate durations of 200 and 600
         * milliseconds, respectively.
         */
        protected _show(duration?: number | string): void;
        /**
         * Applies all `ARIA` attributes to the contents of this dialog.
         */
        protected applyARIA(): void;
        /**
         * Puts focus on the first element that can be focused.
         */
        protected applyFocus(): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        protected bindEvents(): void;
        /**
         * Sets up the event listeners for handling resize events.
         */
        protected bindResizeListener(): void;
        /**
         * Docks this dialog to the given docking zone. The docking zone is usually at the bottom of the screen and displays
         * a list of minimized dialogs.
         * @param zone Zone to dock to.
         */
        protected dock(zone: JQuery): void;
        /**
         * Makes this dialog fit the current browser window, if the `fitViewport` option is enabled.
         */
        protected fitViewport(): void;
        /**
         * This class makes sure a user cannot tab out of the modal and it stops events from targets outside of the
         * overlay element. This requires that we switch back to the modal in case a user tabs out of it. What must
         * be returned by this method are the elements to which the user may switch via tabbing.
         * @override
         * @return The DOM elements which are allowed to be focused via tabbing.
         *
         * (from super type DynamicOverlayWidget) The DOM elements which are allowed to be focused via tabbing. May be an empty `jQuery`
         * instance when the modal contains no tabbable elements, but must not be `undefined`.
         */
        protected override getModalTabbables(): JQuery;
        /**
         * Hide the dialog with an optional animation lasting for the given duration.
         * @param duration Durations are given in milliseconds; higher values indicate slower
         * animations, not faster ones. The strings `fast` and `slow` can be supplied to indicate durations of 200 and 600
         * milliseconds, respectively.
         */
        hide(duration?: number | string): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg the widget configuraton
         *
         * (from super type BaseWidget) The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Positions this dialog on the screen as specified by the widget configuration.
         */
        protected initPosition(): void;
        /**
         * Computes and applies the correct size for this dialog, according to the current configuration.
         */
        protected initSize(): void;
        /**
         * Checks whether this dialog is opened and visible. This method returns `true` irrespective of whether this dialog
         * is minimized, maximized, or shown normally. Returns `false` only when this dialog is closed.
         * @return `true` if this dialog is currently being shown, `false` otherwise.
         */
        isVisible(): boolean;
        /**
         * Loads the content of the dialog via AJAx, if this dialog is `dynamic` and the the content has not yet been
         * loaded.
         */
        protected loadContents(): void;
        /**
         * Moves this dialog to the top so that it is positioned above other elements and overlays.
         */
        moveToTop(): void;
        /**
         * Called when this dialog was closed. Invokes the appropriate behaviors and event listeners.
         * @param event Unused.
         * @param ui Unused.
         */
        protected onHide(event?: unknown, ui?: unknown): void;
        /**
         * Called after this dialog became visible. Triggers the behaviors and registered event listeners.
         */
        protected postShow(): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Called when this dialog is minimized. Restores the original position of this dialog.
         */
        protected removeMinimize(): void;
        /**
         * Resets the dialog position as specified by the `position` property of this widget configuration.
         */
        protected resetPosition(): void;
        /**
         * Restores the state as saved by `saveState`, usually called after an AJAX update.
         */
        protected restoreState(): void;
        /**
         * Puts focus on the element that opened this dialog.
         */
        protected returnFocus(): void;
        /**
         * Saves the current state of this dialog, such as its width and height. Used for example to preserve that state
         * during AJAX updates.
         */
        protected saveState(): void;
        /**
         * Sets up all event listeners required to make this dialog draggable.
         */
        protected setupDraggable(): void;
        /**
         * Sets up all event listeners required to make this dialog resizable.
         */
        protected setupResizable(): void;
        /**
         * Displays this dialog. In case the `dynamic` option is enabled and the content was not yet loaded, this may result
         * in an AJAX request to the sever to retrieve the content. Also triggers the show behaviors registered for this
         * dialog.
         * @param duration Durations are given in milliseconds; higher values indicate slower
         * animations, not faster ones. The strings `fast` and `slow` can be supplied to indicate durations of 200 and 600
         * milliseconds, respectively.
         */
        show(duration?: number | string): void;
        /**
         * Toggle maxification, as if the user had clicked the maximize button. If this dialog is not yet maximized,
         * maximizes it. If this dialog is already maximized, reverts it back to its orignal size.
         */
        toggleMaximize(): void;
        /**
         * Toggles minification, as if the user had clicked the minimize button. If this dialog is not yet minimized,
         * minimizes it.  If this dialog is already minimized, restores its original position.
         */
        toggleMinimize(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Dialog| Dialog widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DialogCfg extends PrimeFaces.widget.DynamicOverlayWidgetCfg {
        /**
         * Whether the dialog is positioned absolutely.
         */
        absolutePositioned: boolean;
        /**
         * A search expression for the element to which the dialog is appended. Defaults to the
         * body.
         */
        appendTo: string;
        /**
         * Whether to prevent the document from scrolling when the dialog is visible.
         */
        blockScroll: boolean;
        /**
         * Only relevant for dynamic="true": Defines if activating the dialog should load the contents from server again. For cache="true" (default) the dialog content is only loaded once.
         */
        cache: boolean;
        /**
         * Whether the dialog can be closed by the user.
         */
        closable: boolean;
        /**
         * Whether the dialog is closed when the user presses the escape button.
         */
        closeOnEscape: boolean;
        /**
         * Whether the dialog is draggable.
         */
        draggable: boolean;
        /**
         * Whether lazy loading of the content via AJAX is enabled.
         */
        dynamic: boolean;
        /**
         * Dialog size might exceed the viewport if the content is taller than viewport in terms
         * of height. When this is set to `true`, automatically adjust the height to fit the dialog within the viewport.
         */
        fitViewport: boolean;
        /**
         * The height of the dialog in pixels.
         */
        height: number;
        /**
         * Effect to use when hiding the dialog.
         */
        hideEffect: string;
        /**
         * One or more CSS classes for the iframe within the dialog.
         */
        iframeStyleClass: string;
        /**
         * The title of the iframe with the dialog.
         */
        iframeTitle: string;
        /**
         * Whether the dialog is maximizable.
         */
        maximizable: boolean;
        /**
         * The minimum height of the dialog in pixels.
         */
        minHeight: number;
        /**
         * The minimum width of the dialog in pixels.
         */
        minWidth: number;
        /**
         * Whether the dialog is minimizable.
         */
        minimizable: boolean;
        /**
         * Whether the dialog is modal and blocks the main content and other dialogs.
         */
        modal: boolean;
        /**
         * Position of the dialog relative to the target.
         */
        my: string;
        /**
         * Client-side callback to invoke when the dialog is
         * closed.
         */
        onHide: PrimeFaces.widget.Dialog.OnHideCallback;
        /**
         * Client-side callback to invoke when the dialog is opened.
         */
        onShow: PrimeFaces.widget.Dialog.OnShowCallback;
        /**
         * Defines where the dialog should be displayed
         */
        position: string;
        /**
         * Whether the dialog can be resized by the user.
         */
        resizable: boolean;
        /**
         * Use ResizeObserver to automatically adjust dialog-height after e.g. AJAX-updates. Resizeable must be set to false to use this option. (Known limitation: Dialog does not automatically resize yet when resizing the browser-window.)
         */
        resizeObserver: boolean;
        /**
         * Can be used together with resizeObserver = true. Centers the dialog again after it was resized to ensure the whole dialog is visible onscreen.
         */
        resizeObserverCenter: boolean;
        /**
         * Whether the dialog is responsive. In responsive mode, the dialog adjusts itself based
         * on the screen width.
         */
        responsive: boolean;
        /**
         * Effect to use when showing the dialog
         */
        showEffect: string;
        /**
         * One or more CSS classes for the dialog.
         */
        styleClass: string;
        /**
         * The width of the dialog in pixels.
         */
        width: number;
    }
}
declare namespace PrimeFaces.widget.Dialog {
    /**
     * The client-side state of the dialog such as its width
     * and height. The client-side state can be preserved during AJAX updates by sending it to the server.
     */
    export interface ClientState {
        /**
         * The total height in pixels of the content area of the dialog.
         */
        contentHeight: number;
        /**
         * The total width in pixels of the content area of the dialog..
         */
        contentWidth: number;
        /**
         * The total height of the dialog in pixels, including the header and its content.
         */
        height: number;
        /**
         * Vertical and horizontal offset of the top-left corner of the dialog.
         */
        offset?: JQuery.Coordinates;
        /**
         * The total width of the dialog in pixels, including the header and its content.
         */
        width: number;
        /**
         * Horizontal scroll position of the window.
         */
        windowScrollLeft: number;
        /**
         * Vertical scroll position of the window.
         */
        windowScrollTop: number;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces ConfirmDialog Widget__
     *
     * ConfirmDialog is a replacement to the legacy JavaScript confirmation box. Skinning, customization and avoiding popup
     * blockers are notable advantages over the classic JavaScript confirmation box.
     * @typeparam TCfg Defaults to `ConfirmDialogCfg`. Type of the configuration object for this widget.
     */
    export class ConfirmDialog<TCfg extends ConfirmDialogCfg = ConfirmDialogCfg> extends PrimeFaces.widget.Dialog<TCfg> {
        /**
         * DOM element of the icon displayed next to the confirmation message.
         */
        icon: JQuery;
        /**
         * DOM element of the confirmation message displayed in this confirm dialog.
         */
        message: JQuery;
        /**
         * DOM element of the title bar text.
         */
        title: JQuery;
        /**
         * Puts focus on the first element that can be focused.
         * @override
         */
        protected override applyFocus(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg the widget configuraton
         *
         * (from super type BaseWidget) The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Shows the given message in this confirmation dialog.
         * @param msg Message to show.
         */
        showMessage(msg: Partial<PrimeFaces.widget.ConfirmDialog.ConfirmDialogMessage>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the
     * {@link  ConfirmDialog| ConfirmDialog widget}. You can access this configuration via
     * {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this configuration is usually meant to be
     * read-only and should not be modified.
     */
    export interface ConfirmDialogCfg extends PrimeFaces.widget.DialogCfg {
    }
}
declare namespace PrimeFaces.widget.ConfirmDialog {
    /**
     * Interface for the message that
     * is shown in the confirm dialog.
     */
    export interface ConfirmDialogMessage {
        /**
         * Optional code that is run before the message is shown. Must be valid JavaScript code.
         * It is evaluated via {@link PrimeFaces.csp.eval}.
         */
        beforeShow?: string;
        /**
         * If `true`, the message is escaped for HTML. If `false`, the message is
         * interpreted as an HTML string.
         */
        escape: boolean;
        /**
         * Header of the dialog message.
         */
        header: string;
        /**
         * Optional icon that is shown to the left of the confirm dialog. When not given, defaults to
         * `ui-icon-alert`. Must be a style class of some icon font.
         */
        icon?: string;
        /**
         * Main content of the dialog message.
         */
        message: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Dynamic Dialog Widget__
     *
     * Used by the dialog framework for displaying other JSF views or external pages in a dialog on the current.
     * @typeparam TCfg Defaults to `DynamicDialogCfg`. Type of the configuration object for this widget.
     */
    export class DynamicDialog<TCfg extends DynamicDialogCfg = DynamicDialogCfg> extends PrimeFaces.widget.Dialog<TCfg> {
        /**
         * Performs the client-side actions needed to actually show this dialog. Compare to `show`, which loads the dialog
         * content from the server if required, then call this method.
         * @override
         */
        protected override _show(): void;
        /**
         * Computes and applies the correct size for this dialog, according to the current configuration.
         * @override
         */
        protected override initSize(): void;
        /**
         * Displays this dialog. In case the `dynamic` option is enabled and the content was not yet loaded, this may result
         * in an AJAX request to the sever to retrieve the content. Also triggers the show behaviors registered for this
         * dialog.
         * @override
         */
        override show(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DynamicDialog| DynamicDialog widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DynamicDialogCfg extends PrimeFaces.widget.DialogCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Dock Widget__
     *
     * Dock component mimics the well known dock interface of Mac OS X.
     * @typeparam TCfg Defaults to `DockCfg`. Type of the configuration object for this widget.
     */
    export class Dock<TCfg extends DockCfg = DockCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM elements for the dock items.
         */
        items: JQuery;
        /**
         * The DOM elements for the clickable anchors of the dock items.
         */
        links: JQuery;
        /**
         * Sets up the hover and click event listeners required for the dock.
         */
        private bindAnimations(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Dock| Dock widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DockCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether opening and closing the dock should be animated.
         */
        animate: string;
        /**
         * The duration for the opening and closing animation in milliseconds.
         */
        animationDuration: string;
        /**
         * Whether to block scrolling of the document.
         */
        blockScroll: boolean;
    }
}
declare namespace PrimeFaces.widget.Draggable {
    /**
     * Callback for when dragging starts. See also {@link
     * DraggableCfg.onStart}.
     */
    export type OnStartCallback =
    /**
     * @param event The drag event that occurred.
     * @param ui Details about the drag event.
     */
    (this: PrimeFaces.widget.Draggable, event: JQuery.TriggeredEvent, ui: JQueryUI.DraggableEventUIParams) => void;
}
declare namespace PrimeFaces.widget.Draggable {
    /**
     * Callback for when dragging ends. See also {@link
     * DraggableCfg.onStop}.
     */
    export type OnStopCallback =
    /**
     * @param event The drag event that occurred.
     * @param ui Details about the drag event.
     */
    (this: PrimeFaces.widget.Draggable, event: JQuery.TriggeredEvent, ui: JQueryUI.DraggableEventUIParams) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Draggable Widget__
     *
     * Drag&Drop utilities of PrimeFaces consists of two components; Draggable and Droppable.
     * @typeparam TCfg Defaults to `DraggableCfg`. Type of the configuration object for this widget.
     */
    export class Draggable<TCfg extends DraggableCfg = DraggableCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element of the target that should be made draggable.
         */
        jqTarget: JQuery;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Draggable| Draggable widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DraggableCfg extends JQueryUI.DraggableOptions, PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Callback for when dragging starts.
         */
        onStart: PrimeFaces.widget.Draggable.OnStartCallback;
        /**
         * Callback for when dragging ends.
         */
        onStop: PrimeFaces.widget.Draggable.OnStopCallback;
        /**
         * ID of the target of this draggable.
         */
        target: string;
    }
}
declare namespace PrimeFaces.widget.Droppable {
    /**
     * Callback for when an items is dropped. See also {@link
     * DroppableCfg.onDrop}.
     */
    export type OnDropCallback =
    /**
     * @param event The drop event that occurred.
     * @param ui Details about the drop event.
     */
    (this: PrimeFaces.widget.Droppable, event: JQuery.TriggeredEvent, ui: JQueryUI.DroppableEventUIParam) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Droppable Widget__
     *
     * Drag&Drop utilities of PrimeFaces consists of two components; Draggable and Droppable.
     * @typeparam TCfg Defaults to `DroppableCfg`. Type of the configuration object for this widget.
     */
    export class Droppable<TCfg extends DroppableCfg = DroppableCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element of the target that should be made droppable.
         */
        jqTarget: JQuery;
        /**
         * Sets up the vent listener for when an item is dropped.
         */
        private bindDropListener(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Droppable| Droppable widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DroppableCfg extends JQueryUI.DroppableOptions, PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Callback for when an items is dropped.
         */
        onDrop: PrimeFaces.widget.Droppable.OnDropCallback;
        /**
         * ID of the target of this droppable.
         */
        target: string;
    }
}
declare namespace PrimeFaces.widget.Effect {
    /**
     * Function that runs the effect when invoked. See also
     * {@link EffectCfg.fn}.
     */
    export type EffectCallback = () => void;
}
declare namespace PrimeFaces.widget.Effect {
    /**
     * Internal function that runs the effect after a given
     * delay (via `setTimeout`) when invoked. See also {@link PrimeFaces.widget.Effect.runner}.
     */
    export type EffectRunner = () => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Effect Widget__
     *
     * Effect component is based on the jQuery UI effects library.
     * @typeparam TCfg Defaults to `EffectCfg`. Type of the configuration object for this widget.
     */
    export class Effect<TCfg extends EffectCfg = EffectCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Internal function that runs the effect after a given delay (via
         * `setTimeout`) when invoked.
         */
        runner: PrimeFaces.widget.Effect.EffectRunner;
        /**
         * The DOM element with the source for the effect.
         */
        source: JQuery;
        /**
         * ID of the current `setTimeout` for scheduling the effect.
         */
        timeoutId: number | undefined;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Effect| Effect widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface EffectCfg extends JQueryUI.EffectOptions, PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Delay between effect repetitions in milliseconds.
         */
        delay: number;
        /**
         * Event that triggers the effect. Defaults to `load` (page load).
         */
        event: string;
        /**
         * Function that runs the effect when invoked.
         */
        fn: PrimeFaces.widget.Effect.EffectCallback;
        /**
         * ID of the source element for the effect.
         */
        source: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Fieldset Widget__
     *
     * Fieldset is a grouping component as an extension to html fieldset.
     * @typeparam TCfg Defaults to `FieldsetCfg`. Type of the configuration object for this widget.
     */
    export class Fieldset<TCfg extends FieldsetCfg = FieldsetCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element with the content of this fieldset.
         */
        content: JQuery;
        /**
         * The DOM element with the legend of this fieldset.
         */
        legend: JQuery;
        /**
         * When dynamic loading is enabled, whether the content was already loaded.
         */
        loaded: boolean;
        /**
         * The DOM element with the hidden input field for the state of this fieldset.
         */
        stateHolder: JQuery;
        /**
         * The DOM element with the toggler for hiding or showing the content of this fieldset.
         */
        toggler: JQuery;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Loads the contents of this fieldset panel dynamically via AJAX, if dynamic loading is enabled.
         */
        private loadContents(): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Toggles the content of this fieldset (collapsed or expanded).
         * @param e Optional event that triggered the toggling.
         */
        toggle(e?: JQuery.TriggeredEvent): void;
        /**
         * Updates the visual toggler state and saves its state
         * @param collapsed If this fieldset is now collapsed or expanded.
         */
        private updateToggleState(collapsed: boolean): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Fieldset| Fieldset widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface FieldsetCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether this fieldset is currently collapsed (content invisible) or expanded (content
         * visible).
         */
        collapsed: boolean;
        /**
         * `true` to load the content via AJAX when the fieldset panel is opened, `false` to load
         * the content immediately.
         */
        dynamic: boolean;
        /**
         * Toggle duration in milliseconds.
         */
        toggleSpeed: number;
        /**
         * Whether the content of this fieldset can be toggled between expanded and collapsed.
         */
        toggleable: boolean;
    }
}
declare namespace PrimeFaces {
    /**
     * Fetches the resource at the given URL and prompts the user to download that file, without leaving the current
     * page. This function is usually called by server-side code to download a data URI or a data from a remote URL.
     * @param url URL pointing to the resource to download.
     * @param mimeType The MIME content-type of the file to download. It helps the browser present friendlier
     * information about the download to the user, encouraging them to accept the download.
     * @param fileName The name of the file to be created. Note that older browsers (like FF3.5, Ch5) do not
     * honor the file name you provide, instead they automatically name the downloaded file.
     * @param cookieName Name of the file download cookie (by default `primefaces.download`). This function
     * makes sure the cookie is set properly when the download finishes.
     */
    export function download(url: string, mimeType: string, fileName: string, cookieName: string): void;
}
/**
 * Namespace for the jQuery BlueImp File Upload plugin.
 *
 * File Upload widget with multiple file selection, drag&drop support, progress bars, validation and preview images,
 * audio and video for jQuery.
 *
 * Supports cross-domain, chunked and resumable file uploads and client-side image resizing.
 *
 * Works with any server-side platform (PHP, Python, Ruby on Rails, Java, Node.js, Go etc.) that supports standard HTML
 * form file uploads.
 *
 * See https://github.com/blueimp/jQuery-File-Upload
 */
declare namespace JQueryFileUpload {
    /**
     * The method to use for uploading files.
     */
    type HttpUploadMethod = "POST" | "PUT" | "PATCH";
    /**
     * Enumeration of common HTTP methods that can be used with the jQuery file upload plugin.
     */
    type HttpMethod = "GET" | "HEAD" | "POST" | "PUT" | "DELETE" | "CONNECT" | "OPTIONS" | "TRACE" | "PATCH";
    /**
     * Represents the response of a successful or failed request.
     * @typeparam TResponse Type of the response.
     * @typeparam TFailure Type of the error.
     */
    type ResponseObject<TResponse = unknown, TFailure = unknown> = ResponseSuccess<TResponse> | ResponseFailure<TFailure>;
    /**
     * Plain callback for event listeners. They only received the event that occurred.
     */
    type PlainCallback =
    /**
     * @param event The event that triggered this callback.
     */
    (event: JQuery.TriggeredEvent) => void;
    /**
     * A callback for an event that also receives some additional data, in addition to the event.
     */
    type DataCallback<TData> =
    /**
     * @param event The event that triggered this callback.
     * @param data Additional data passed to this callback. Specific to the event.
     */
    (event: JQuery.TriggeredEvent, data: TData) => void;
    /**
     * A callback for an event that receives some additional data and may also return a boolean that usually defines
     * whether the action that triggered this event is pursued further.
     */
    type DataCallbackBool<TData> =
    /**
     * @param event The event that triggered this callback.
     * @param data Additional data passed to this callback. Specific to the event.
     * @return A boolean flag that indicates how event handling proceeds.
     */
    (event: JQuery.TriggeredEvent, data: TData) => boolean;
    /**
     * A processing action that is available.
     */
    type ProcessAction =
    /**
     * @param data A copy of the data object that is passed to the add callback, with data.files referencing the files
     * array.
     * @param options The options object of the current process action.
     * @return Either the data object, or a Promise object which resolves or rejects with the data object as argument.
     */
    (this: FileUpload, data: ProcessActionData, options: JQueryFileUpload.FileUploadOptions) => ProcessActionData | JQuery.Promise<ProcessActionData>;
    /**
     * Constructor for a new file upload instance.
     */
    type FileUploadConstructor =
    /**
     * @param options Options for the widget.
     * @param Element where the widget is created.
     */
    new (options?: Partial<FileUploadOptions>, element?: HTMLFormElement) => FileUpload;
    /**
     * All the options that can be used to configure the file upload widget.
     */
    interface FileUploadOptions extends JQuery.AjaxSettings {
        // ==============================
        // === Configuration settings ===
        // ==============================
        /**
         * The HTTP request method for the file uploads.
         *
         * "PUT" and "PATCH" are only supported by browser supporting XHR file uploads, as iframe transport uploads rely
         * on standard HTML forms which only support "POST" file uploads.
         *
         * If the type is defined as "PUT" or "PATCH", the iframe transport will send the files via "POST" and transfer
         * the original method as "_method" URL parameter.
         *
         * Note: As was noted above, it's a common practice to use "_method" to transfer the type of your request. For
         * example, "Ruby on Rails" framework uses a hidden input with the name "_method" within each form, so it will
         * likely override the value that you will set here.
         */
        type: HttpUploadMethod;
        /**
         * The type of data that is expected back from the server.
         *
         * __Note__: The UI version of the File Upload plugin sets this option to "json" by default.
         */
        dataType: string;
        /**
         * The drop target jQuery object, by default the complete document.
         *
         * Set to null or an empty jQuery collection to disable drag & drop support.
         */
        dropZone: Element | Element[] | JQuery | string | null;
        /**
         * The paste target jQuery object, by the default the complete document.
         *
         * Set to a jQuery collection to enable paste support:
         */
        pasteZone: Element | Element[] | JQuery | string | null;
        /**
         * The file input field(s), that are listened to for change events.
         *
         * If undefined, it is set to the file input fields inside of the widget element on plugin initialization.
         *
         * Set to null to disable the change listener.
         */
        fileInput: Element | Element[] | JQuery | string | null;
        /**
         * By default, the file input field is replaced with a clone after each input field change event.
         *
         * This is required for iframe transport queues and allows change events to be fired for the same file
         * selection, but can be disabled by setting this option to false.
         */
        replaceFileInput: boolean;
        /**
         * The parameter name for the file form data (the request argument name).
         *
         * If undefined or empty, the name property of the file input field is used, or "files[]" if the file input name
         * property is also empty.
         */
        paramName: string | string[];
        /**
         * Allows to set the accept-charset attribute for the iframe upload forms.
         *
         * If `formAcceptCharset` is not set, the accept-charset attribute of the file upload widget form is used, if
         * available.
         */
        formAcceptCharset: string;
        /**
         * By default, each file of a selection is uploaded using an individual request for XHR type uploads.
         *
         * Set to false to upload file selections in one request each.
         */
        singleFileUploads: boolean;
        /**
         * To limit the number of files uploaded with one XHR request, set this option to an integer greater than 0.
         */
        limitMultiFileUploads: number;
        /**
         * This option limits the number of files uploaded with one XHR request to keep the request size under
         * or equal to the defined limit in bytes.
         */
        limitMultiFileUploadSize: number;
        /**
         * Multipart file uploads add a number of bytes to each uploaded file, therefore this option adds an
         * overhead for each file used in the {@link limitMultiFileUploadSize} configuration.
         */
        limitMultiFileUploadSizeOverhead: number;
        /**
         * Set this option to true to issue all file upload requests in a sequential order.
         */
        sequentialUploads: boolean;
        /**
         * To limit the number of concurrent uploads, set this option to an integer greater than 0.
         */
        limitConcurrentUploads: number;
        /**
         * Set this option to true to force iframe transport uploads, even if the browser is capable of XHR file
         * uploads.
         *
         * This can be useful for cross-site file uploads, if the `Access-Control-Allow-Origin` header cannot be set for
         * the server-side upload handler which is required for cross-site XHR file uploads.
         */
        forceIframeTransport: boolean;
        /**
         * This option is only used by the iframe transport and allows to override the URL of the initial iframe src.
         */
        initialIframeSrc: string;
        /**
         * Set this option to the location of a redirect URL on the origin server (the server that hosts the file upload
         * form), for cross-domain iframe transport uploads.
         *
         * If set, this value is sent as part of the form data to the upload server.
         *
         * The upload server is supposed to redirect the browser to this URL after the upload completes and append the
         * upload information as URL encoded JSON string to the redirect URL, e.g., by replacing the `%s` character
         * sequence.
         */
        redirect: string;
        /**
         * The parameter name for the redirect url, sent as part of the form data and set to `redirect` if this option
         * is empty.
         */
        redirectParamName: string;
        /**
         * Set this option to the location of a postMessage API on the upload server, to enable cross-domain postMessage
         * transport uploads.
         */
        postMessage: string;
        /**
         * By default, XHR file uploads are sent as multipart/form-data.
         *
         * The iframe transport is always using multipart/form-data.
         *
         * If this option is set to `false`, the file content is streamed to the server URL instead of sending a RFC
         * 2388 multipart message for XMLHttpRequest file uploads. Non-multipart uploads are also referred to as HTTP
         * PUT file upload.
         *
         * __Note__: Additional form data is ignored when the multipart option is set to `false`. Non-multipart uploads
         * (multipart: false) are broken in Safari 5.1.
         */
        multipart: boolean;
        /**
         * To upload large files in smaller chunks, set this option to a preferred maximum chunk size. If set to `0`,
         * `null` or `undefined`, or the browser does not support the required Blob API, files will be uploaded as a
         * whole.
         *
         * For chunked uploads to work in Mozilla Firefox < 7, the multipart option has to be set to `false`. This is
         * due to Gecko 2.0 (Firefox 4 etc.) adding blobs with an empty filename when building a multipart upload
         * request using the FormData interface (fixed in FF 7.0).
         *
         * Several server-side frameworks (including PHP and Django) cannot handle multipart file uploads with empty
         * filenames.
         *
         * __Note__: If this option is enabled and {@link singleFileUploads} is set to false, only the first file of a
         * selection will be uploaded.
         */
        maxChunkSize: number;
        /**
         * When a non-multipart upload or a chunked multipart upload has been aborted, this option can be used to resume
         * the upload by setting it to the size of the already uploaded bytes. This option is most useful when modifying
         * the options object inside of the {@link add} or {@link send} callbacks, as the options are cloned for each
         * file upload.
         */
        uploadedBytes: number;
        /**
         * By default, failed (abort or error) file uploads are removed from the global progress calculation. Set this
         * option to `false` to prevent recalculating the global progress data.
         */
        recalculateProgress: boolean;
        /**
         * Interval in milliseconds to calculate and trigger progress events.
         */
        progressInterval: number;
        /**
         * Interval in milliseconds to calculate progress bitrate.
         */
        bitrateInterval: number;
        /**
         * By default, uploads are started automatically when adding files.
         *
         * Please note that in the basic File Upload plugin, this option is set to `true` by default.
         */
        autoUpload: boolean;
        /**
         * Additional form data to be sent along with the file uploads can be set using this option, which accepts an
         * array of objects with name and value properties, a function returning such an array, a FormData object (for
         * XHR file uploads), or a simple object.
         *
         * The form of the first file input is given as parameter to the function.
         *
         * Note: Additional form data is ignored when the multipart option is set to false.
         */
        formData: FormData | NameValuePair<FormDataEntryValue>[] | Record<string, FormDataEntryValue> | ((this: unknown, form: JQuery<HTMLFormElement>) => NameValuePair<FormDataEntryValue>[]);
        // ===============================
        // === File processing options ===
        // ===============================
        /**
         * A list of file processing actions.
         */
        processQueue: ProcessingQueueItem[];
        // =====================
        // === Image options ===
        // =====================
        /**
         * Disable parsing and storing the image header.
         */
        disableImageHead: boolean;
        /**
         * Disable parsing EXIF data.
         */
        disableExif: boolean;
        /**
         * Disable parsing the EXIF Thumbnail.
         */
        disableExifThumbnail: boolean;
        /**
         * Disable parsing the EXIF Sub IFD (additional EXIF info).
         */
        disableExifSub: boolean;
        /**
         * Disable parsing EXIF GPS data.
         */
        disableExifGps: boolean;
        /**
         * Disable parsing image meta data (image head and EXIF data).
         */
        disableImageMetaDataLoad: boolean;
        /**
         * Disables saving the image meta data into the resized images.
         */
        disableImageMetaDataSave: boolean;
        /**
         * The regular expression for the types of images to load, matched against the file type.
         */
        loadImageFileTypes: RegExp;
        /**
         * The maximum file size of images to load.
         */
        loadImageMaxFileSize: number;
        /**
         * Don't revoke the object URL created to load the image.
         */
        loadImageNoRevoke: boolean;
        /**
         * Disable loading and therefore processing of images.
         */
        disableImageLoad: boolean;
        /**
         * The maximum width of resized images.
         */
        imageMaxWidth: number;
        /**
         * The maximum height of resized images.
         */
        imageMaxHeight: number;
        /**
         * The minimum width of resized images.
         */
        imageMinWidth: number;
        /**
         * The minimum height of resized images.
         */
        imageMinHeight: number;
        /**
         * Define if resized images should be cropped or only scaled.
         */
        imageCrop: boolean;
        /**
         * Defines the image orientation (1-8) or takes the orientation value from Exif data if set to true.
         */
        imageOrientation: 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | boolean;
        /**
         * If set to true, forces writing to and saving images from canvas, even if the original image fits the maximum
         * image constraints.
         */
        imageForceResize: boolean;
        /**
         * Disables the resize image functionality.
         */
        disableImageResize: boolean;
        /**
         * Sets the quality parameter given to the canvas.toBlob() call when saving resized images.
         */
        imageQuality: number;
        /**
         * Sets the type parameter given to the canvas.toBlob() call when saving resized images.
         */
        imageType: string;
        /**
         * The maximum width of the preview images.
         */
        previewMaxWidth: number;
        /**
         * The maximum height of the preview images.
         */
        previewMaxHeight: number;
        /**
         * The minimum width of preview images.
         */
        previewMinWidth: number;
        /**
         * The minimum height of preview images.
         */
        previewMinHeight: number;
        /**
         * Define if preview images should be cropped or only scaled.
         */
        previewCrop: boolean;
        /**
         * Defines the preview orientation (1-8) or takes the orientation value from Exif data if set to true.
         */
        previewOrientation: 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | boolean;
        /**
         * Create the preview using the Exif data thumbnail.
         */
        previewThumbnail: boolean;
        /**
         * Define if preview images should be resized as canvas elements.
         */
        previewCanvas: boolean;
        /**
         * Define the name of the property that the preview element is stored as on the File object.
         */
        imagePreviewName: string;
        /**
         * Disables image previews.
         */
        disableImagePreview: boolean;
        // =============================
        // === Audio preview options ===
        // =============================
        /**
         * The regular expression for the types of audio files to load, matched against the file type.
         */
        loadAudioFileTypes: RegExp;
        /**
         * The maximum file size of audio files to load.
         */
        loadAudioMaxFileSize: number;
        /**
         * Define the name of the property that the preview element is stored as on the File object.
         */
        audioPreviewName: string;
        /**
         * Disable audio previews.
         */
        disableAudioPreview: boolean;
        // =============================
        // === Video preview options ===
        // =============================
        /**
         * The regular expression for the types of video files to load, matched against the file type.
         */
        loadVideoFileTypes: RegExp;
        /**
         * The maximum file size of video files to load.
         */
        loadVideoMaxFileSize: number;
        /**
         * Define the name of the property that the preview element is stored as on the File object.
         */
        videoPreviewName: string;
        /**
         * Disable video previews.
         */
        disableVideoPreview: boolean;
        // ==========================
        // === Validation options ===
        // ==========================
        /**
         * The regular expression for allowed file types, matches against either file type or file name as only browsers
         * with support for the File API report the file type.
         */
        acceptFileTypes: RegExp;
        /**
         * The maximum allowed file size in bytes.
         *
         * __Note__: This option has only an effect for browsers supporting the File API.
         */
        maxFileSize: number;
        /**
         * The minimum allowed file size in bytes.
         *
         * __Note__: This option has only an effect for browsers supporting the File API.
         */
        minFileSize: number;
        /**
         * This option limits the number of files that are allowed to be uploaded using this widget.
         *
         * By default, unlimited file uploads are allowed.
         *
         * __Note__: The maxNumberOfFiles option depends on the getNumberOfFiles option, which is defined by the UI and
         * AngularJS implementations.
         */
        maxNumberOfFiles: number;
        /**
         * Disables file validation.
         */
        disableValidation: boolean;
        // ==========================
        // === UI version options ===
        // ==========================
        /**
         * Callback to retrieve the list of files from the server response.
         *
         * Is given the data argument of the done callback, which contains the result property. Must return an array.
         */
        getFilesFromResponse: null | ((data: JQueryAjaxCallbackData & ResponseObject) => File[]);
        /**
         * This option is a function that returns the current number of files selected and uploaded.
         *
         * It is used in the {@link maxNumberOfFiles} validation.
         */
        getNumberOfFiles: null | (() => number);
        /**
         * The container for the files listed for upload / download.
         *
         * Is transformed into a jQuery object if set as DOM node or string.
         */
        filesContainer: Element | Element[] | JQuery | string;
        /**
         * By default, files are appended to the files container.
         *
         * Set this option to true, to prepend files instead.
         */
        prependFiles: boolean;
        /**
         * The upload template function.
         */
        uploadTemplate: null | ((data: TemplateData) => JQuery | string);
        /**
         * The ID of the upload template, given as parameter to the tmpl() method to set the uploadTemplate option.
         */
        uploadTemplateId: string;
        /**
         * The upload template function.
         */
        downloadTemplate: null | ((data: TemplateData) => JQuery | string);
        /**
         * The ID of the download template, given as parameter to the tmpl() method to set the downloadTemplate option.
         */
        downloadTemplateId: string;
        // =======================
        // === Basic callbacks ===
        // =======================
        /**
         * The add callback can be understood as the callback for the file upload request queue. It is invoked as soon
         * as files are added to the fileupload widget - via file input selection, drag & drop or add API call.
         *
         * The data parameter allows to override plugin options as well as define ajax settings
         *
         * data.files holds a list of files for the upload request: If the singleFileUploads option is enabled (which is
         * the default), the add callback will be called once for each file in the selection for XHR file uploads, with
         * a data.files array length of one, as each file is uploaded individually. Else the add callback will be called
         * once for each file selection.
         *
         * The upload starts when the submit method is invoked on the data parameter.
         *
         * data.submit() returns a Promise object and allows to attach additional handlers using jQuery's Deferred
         * callbacks.
         *
         * The default add callback submits the files if the autoUpload option is set to true (the default for the basic
         * plugin). It also handles processing of files before upload, if any process handlers have been registered.
         */
        add: DataCallback<AddCallbackData> | null;
        /**
         * Callback for the submit event of each file upload.
         *
         * If this callback returns `false`, the file upload request is not started.
         */
        submit: DataCallbackBool<AddCallbackData> | null;
        /**
         * Callback for the start of each file upload request.
         *
         * If this callback returns `false`, the file upload request is aborted.
         */
        send: DataCallbackBool<ProgressCallbackData> | null;
        /**
         * Callback for upload progress events.
         */
        progress: DataCallback<ProgressCallbackData> | null;
        /**
         * Callback for global upload progress events.
         */
        progressall: DataCallback<ProgressAllCallbackData> | null;
        /**
         * Callback for uploads start, equivalent to the global ajaxStart event (but for file upload requests only).
         */
        start: PlainCallback | null;
        /**
         * Callback for uploads stop, equivalent to the global ajaxStop event (but for file upload requests only).
         */
        stop: PlainCallback | null;
        /**
         * Callback for change events of the fileInput collection.
         */
        change: DataCallback<ChangeCallbackData> | null;
        // ===========================
        // === Drop zone callbacks ===
        // ===========================
        /**
         * Callback for paste events to the drop zone collection.
         */
        paste: DataCallback<DropzoneCallbackData> | null;
        /**
         * Callback for drop events of the drop zone collection.
         */
        drop: DataCallback<DropzoneCallbackData> | null;
        /**
         * Callback for dragover events of the drop zone collection.
         *
         * To prevent the plugin from calling the `preventDefault()` method on the original dragover event object and
         * setting the `dataTransfer.dropEffect` to copy, call the `preventDefault()` method on the event object of the
         * `fileuploaddragover` callback.
         *
         * __Note__: The file upload plugin only provides a dragover callback, as it is used to make drag&drop callbacks
         * work. If you want callbacks for additional drag events, simply bind to these events with JQuery's native
         * event binding mechanism on your dropZone element.
         */
        dragover: DataCallback<DropzoneCallbackData> | null;
        // =======================
        // === Chunk callbacks ===
        // =======================
        /**
         * Callback before the start of each chunk upload request, before form data is initialized.
         */
        chunkbeforesend: null | DataCallback<ChunkCallbackData & ResponseObject> | null;
        /**
         * Callback for the start of each chunk upload request, after form data is initialized.
         *
         * If this callback returns `false`, the chunk upload request is aborted.
         */
        chunksend: DataCallbackBool<ChunkCallbackData & ResponseObject> | null;
        /**
         * Callback for successful chunk upload requests.
         */
        chunkdone: DataCallback<ChunkCallbackData & ResponseObject> | null;
        /**
         * Callback for failed (abort or error) chunk upload requests
         */
        chunkfail: DataCallback<ChunkCallbackData & ResponseObject> | null;
        /**
         * Callback for completed (success, abort or error) chunk upload requests.
         */
        chunkalways: DataCallback<ChunkCallbackData & ResponseObject> | null;
        // ================================
        // === JQuery wrapper callbacks ===
        // ================================
        /**
         * Callback for successful upload requests. This callback is the equivalent to the success callback provided by
         * jQuery ajax() and will also be called if the server returns a JSON response with an error property.
         */
        done: DataCallback<JQueryAjaxCallbackData & ResponseObject>;
        /**
         * Callback for failed (abort or error) upload requests. This callback is the equivalent to the error callback
         * provided by jQuery ajax() and will not be called if the server returns a JSON response with an error
         * property, as this counts as successful request due to the successful HTTP response.
         */
        fail: DataCallback<JQueryAjaxCallbackData & ResponseObject>;
        /**
         * Callback for completed (success, abort or error) upload requests. This callback is the equivalent to the
         * complete callback provided by jQuery ajax().
         */
        always: DataCallback<JQueryAjaxCallbackData & ResponseObject>;
        // =================================
        // === File processing callbacks ===
        // =================================
        /**
         * Callback for the start of the fileupload processing queue.
         */
        processstart: PlainCallback | null;
        /**
         * Callback for the stop of the fileupload processing queue.
         */
        processstop: PlainCallback | null;
        /**
         * Callback for the start of an individual file processing queue.
         */
        process: DataCallback<ProcessCallbackData> | null;
        /**
         * Callback for the successful end of an individual file processing queue.
         */
        processdone: DataCallback<ProcessCallbackData> | null;
        /**
         * Callback for the failure of an individual file processing queue.
         */
        processfail: DataCallback<ProcessCallbackData> | null;
        /**
         * Callback for the end (done or fail) of an individual file processing queue.
         */
        processalways: DataCallback<ProcessCallbackData> | null;
        // ============================
        // === UI version callbacks ===
        // ============================
        /**
         * Callback for file deletion events.
         *
         * __Note__ Since the UI version already sets this callback option, it is recommended to use the event binding
         * method to attach additional event listeners.
         */
        destroy: DataCallback<DestroyCallbackData> | null;
        /**
         * The destroyed callback is the equivalent to the destroy callback and is triggered after files have been
         * deleted, the transition effects have completed and the download template has been removed.
         */
        destroyed: DataCallback<DestroyCallbackData> | null;
        /**
         * The added callback is the equivalent to the add callback and is triggered after the upload template has been
         * rendered and the transition effects have completed.
         */
        added: DataCallback<AddCallbackData> | null;
        /**
         * The sent callback is the equivalent to the send callback and is triggered after the send callback has run and
         * the files are about to be sent.
         */
        sent: DataCallbackBool<ProgressCallbackData> | null;
        /**
         * The completed callback is the equivalent to the done callback and is triggered after successful uploads after
         * the download template has been rendered and the transition effects have completed.
         */
        completed: DataCallback<JQueryAjaxCallbackData & ResponseObject> | null;
        /**
         * The failed callback is the equivalent to the fail callback and is triggered after failed uploads after the
         * download template has been rendered and the transition effects have completed.
         */
        failed: DataCallback<JQueryAjaxCallbackData & ResponseObject> | null;
        /**
         * The finished callback is the equivalent to the always callback and is triggered after both completed and
         * failed uploads after the equivalent template has been rendered and the transition effects have completed.
         */
        finished: DataCallback<FileUploadOptions> | null;
        /**
         * The started callback is the equivalent to the start callback and is triggered after the start callback has
         * run and the transition effects called in the start callback have completed.
         *
         * __Note__: Unlike the start callback, which is always called before all send callbacks, the started callback
         * might be called after the sent callbacks, depending on the duration of the transition effects in those
         * callbacks.
         */
        started: PlainCallback | null;
        /**
         * The stopped callback is the equivalent to the stop callback and is triggered after the stop callback has run
         * and the transition effects called in the stop callback and all done callbacks have completed.
         *
         * The stopped callback is therefore always triggered after each completed, failed and finished callback is
         * done.
         */
        stopped: PlainCallback | null;
    }
    // =====================
    // === Callback data ===
    // =====================
    /**
     * Data passed to the drop zone event handlers.
     */
    interface DropzoneCallbackData {
        /**
         * An list of files.
         */
        files: File[];
    }
    /**
     * Represents the data passed to file related event handlers.
     */
    interface ChangeCallbackData {
        /**
         * The input element of type file.
         */
        fileInput?: JQuery;
        /**
         * The cloned input element of type file.
         */
        fileInputClone?: JQuery;
        /**
         * List of files currently selected.
         */
        files: File[];
        /**
         * Form element that holds the file input.
         */
        form?: JQuery;
        /**
         * Original list of files that were present.
         */
        originalFiles: File[];
    }
    /**
     * The data for the callback when a file is added.
     */
    interface AddCallbackData extends ChangeCallbackData, ConvenienceMethods {
        /**
         * Name of the file input or inputs.
         */
        paramName?: string | string[];
    }
    /**
     * The data for when a file is processed.
     */
    interface ProcessCallbackData extends ChangeCallbackData, ConvenienceMethods {
        /**
         * Name of the file input or inputs.
         */
        paramName?: string | string[];
    }
    /**
     * The data for the callback when the widget is destroyed.
     */
    interface DestroyCallbackData {
        /**
         * Download row
         */
        context: JQuery;
        /**
         * Deletion url
         */
        url: string;
        /**
         * Deletion request type, e.g., `DELETE`.
         */
        type: HttpMethod;
        /**
         * Deletion response type, e.g., `json`.
         */
        dataType: string;
    }
    /**
     * Represents the data passed to the global progress event handler.
     */
    interface ProgressAllCallbackData extends UploadProgress {
    }
    /**
     * Represents the data passed to various progress related event handlers.
     */
    interface ProgressCallbackData extends FileUploadOptions, UploadProgress {
    }
    /**
     * Represents the callback data that is passed to various chunk related event handlers.
     */
    interface ChunkCallbackData extends FileUploadOptions {
        /**
         * The blob of the chunk that is being sent.
         */
        blob: Blob;
        /**
         * Length of the chunk that is being sent.
         */
        chunkSize: number;
        /**
         * Content range of the chunk being sent.
         */
        contentRange: string;
        /**
         * The error that occurred, if any.
         */
        errorThrown?: unknown;
    }
    /**
     * Represents the data that is passed to various event handlers simulating JQuery AJAX handlers.
     */
    interface JQueryAjaxCallbackData extends FileUploadOptions {
    }
    // =============================
    // === Misc. data structures ===
    // =============================
    /**
     * The data passed to the template methods {@link FileUploadOptions.uploadTemplate} and
     * {@link FileUploadOptions.downloadTemplate}.
     */
    interface TemplateData {
        /**
         * List of files to be rendered.
         */
        files: File[];
        /**
         * Formats a file size as a human readable string.
         * @param size File size to format, in bytes.
         * @return Humand-readable string of a file size.
         */
        formatFileSize(size: number): string;
        /**
         * Current options of the upload widget.
         */
        options: FileUploadOptions;
    }
    /**
     * Represents an item in the upload queue.
     */
    interface ProcessingQueueItem {
        /**
         * MIME types a file must have to be accepted.
         */
        acceptFileTypes?: RegExp | string;
        /**
         * Actin to run.
         */
        action: string;
        /**
         * Whether the action is always run.
         */
        always?: boolean;
        /**
         * Whether a prefix is added.
         */
        prefix?: boolean;
        /**
         * List of recognized file types.
         */
        fileTypes?: string;
        /**
         * Whether this action is disabled.
         */
        disabled?: string;
    }
    /**
     * @typeparam V Type of the property value.
     * @typeparam V Type of the property name.
     * An object for a single property, with a name and value entry.
     */
    interface NameValuePair<V, K = string> {
        /**
         * The name of the property.
         */
        name: K;
        /**
         * The value of the property.
         */
        value: V;
    }
    /**
     * Additional methods for the data objects that are passed to the event handlers.
     */
    interface ConvenienceMethods {
        /**
         * Aborts the file upload.
         * @return A promise that resolves once the file upload is aborted.
         */
        abort(): JQuery.Promise<void>;
        /**
         * Adds the handlers to the process queue and returns the process queue.
         * @param resolveFunc Additional handler for chaining to the promise.
         * @param rejectFunc Additional handler for chaining to the promise.
         * @return The process queue promise.
         */
        process<T>(resolveFunc: (value?: this) => JQuery.Promise<T> | JQuery.Thenable<T> | T, rejectFunc?: (reason?: unknown) => JQuery.Promise<T> | JQuery.Thenable<T> | T): JQuery.Promise<T>;
        /**
         * Adds the handlers to the process queue and returns the process queue.
         * @param resolveFunc Additional handler for chaining to the promise.
         * @param rejectFunc Additional handler for chaining to the promise.
         * @return The process queue promise.
         */
        process<T>(resolveFunc: undefined, rejectFunc: (reason?: unknown) => JQuery.Promise<T> | JQuery.Thenable<T> | T): JQuery.Promise<T>;
        /**
         * Retrieves the process queue.
         * @return The process queue promise.
         */
        process(): JQuery.Promise<this>;
        /**
         * Checks whether any upload is being processed.
         * @return Whether any upload is being processed.
         */
        processing(): boolean;
        /**
         * Retrieves the details about the current upload progress.
         * @return Details about the current upload progress.
         */
        progress(): UploadProgress;
        /**
         * Retrieves the current response object.
         * @return The current response object with the response info.
         */
        response(): ResponseObject | Record<string, unknown>;
        /**
         * Submits the form.
         * @return A promise that resolves when submission is done.
         */
        submit(): JQuery.Promise<unknown>;
        /**
         * Finds the current state of the file upload request.
         * @return A promise that resolves with the current state of the file upload request.
         */
        state(): JQuery.Promise<string>;
    }
    /**
     * Represents the progress of one or more file uploads
     */
    interface UploadProgress {
        /**
         * Amount of data already loaded.
         */
        loaded: number;
        /**
         * Total amount of data to be sent.
         */
        total: number;
        /**
         * Bitrate at which the data is transferred.
         */
        bitrate: number;
    }
    /**
     * Represents a response for a successful request.
     * @typeparam TResponse Type of the response.
     */
    interface ResponseSuccess<TResponse = unknown> {
        /**
         * The underlying JQuery XHR object.
         */
        jqXHR: JQuery.jqXHR;
        /**
         * The result of the request.
         */
        result: TResponse;
        /**
         * The status text of the request.
         */
        textStatus: string;
    }
    /**
     * Represents a response for a failed request.
     * @typeparam TFailure Type of the error.
     */
    interface ResponseFailure<TFailure = unknown> {
        /**
         * The reason why the request failed.
         */
        errorThrown: TFailure;
        /**
         * The underlying JQuery XHR object.
         */
        jqXHR: JQuery.jqXHR;
        /**
         * The status text of the request.
         */
        textStatus: string;
    }
    /**
     * Represents the data passed to the {@link FileUpload.processActions|$.blueimp.fileupload.prototype.processActions}
     * handlers.
     */
    interface ProcessActionData extends AddCallbackData {
        /**
         * The index of the current file to be processed.
         */
        index: number;
    }
    /**
     * Represents the options for adding or uploading files programmatically.
     */
    interface FileUploadData extends Partial<FileUploadOptions> {
        /**
         * An array or array-like list of File or Blob objects.
         */
        files: ArrayLike<File | Blob>;
    }
    /**
     * Interface for an instance of the jQuery BlueImp File Upload plugin.
     *
     * The widget can be initialized on a file upload in the following way:
     *
     * ```javascript
     * $('#fileupload').fileupload();
     * ```
     *
     * File Upload widget with multiple file selection, drag&drop support, progress bars, validation and preview images,
     * audio and video for jQuery.
     *
     * Supports cross-domain, chunked and resumable file uploads and client-side image resizing.
     *
     * Works with any server-side platform (PHP, Python, Ruby on Rails, Java, Node.js, Go etc.) that supports standard
     * HTML form file uploads.
     *
     * See https://github.com/blueimp/jQuery-File-Upload
     */
    interface FileUpload extends JQueryUI.WidgetCommonProperties {
        /**
         * Namespace for the events of this widget.
         */
        eventNamespace: string;
        /**
         * List of focusable elements.
         */
        focusable: JQuery;
        /**
         * List of hoverable elements.
         */
        hoverable: JQuery;
        /**
         * Current options for this file upload widget.
         */
        options: FileUploadOptions;
        /**
         * This actions that are defined as a property of `$.blueimp.fileupload.prototype.processActions`.
         */
        processActions: Record<string, ProcessAction>;
        /**
         * Sets the given option to the given value.
         * @typeparam K Name of an option to set.
         * @param optionName Name of an option to set.
         * @return The value of the given option
         */
        option<K extends keyof JQueryFileUpload.FileUploadOptions>(optionName: K, optionValue: JQueryFileUpload.FileUploadOptions[K]): this;
        /**
         * Retrieves the value of the given option.
         * @param method The method to call on this file upload instance.
         * @param optionName Name of an option to retrieve.
         * @return The value of the given option.
         */
        option<K extends keyof JQueryFileUpload.FileUploadOptions>(optionName: K): JQueryFileUpload.FileUploadOptions[K];
        /**
         * Sets the given options on this file upload instance.
         * @param options Options to apply.
         * @return This JQuery instance for chaining.
         */
        option(options: Partial<JQueryFileUpload.FileUploadOptions>): this;
        /**
         * Return the current set of options. This includes default options.
         * @return An object with all options.
         */
        option(): JQueryFileUpload.FileUploadOptions;
        /**
         * To remove the file upload widget functionality from the element node, call the destroy method. This will also
         * remove any added event listeners.
         * @return This JQuery instance for chaining.
         */
        destroy(): undefined;
        /**
         * As other widgets based on jQuery UI Widget, the file upload widget can also be enabled.
         */
        enable(): this;
        /**
         * As other widgets based on jQuery UI Widget, the file upload widget can also be disabled.
         */
        disable(): this;
        /**
         * Finds the overall progress of all uploads.
         * @return The overall progress of all uploads.
         */
        progress(): JQueryFileUpload.UploadProgress;
        /**
         * Finds the number of currently active uploads.
         * @return The number of active uploads.
         */
        active(): number;
        /**
         * Usually, file uploads are invoked by selecting files via file input button or by dropping files on the drop zone.
         *
         * However it is also possible to upload files programmatically for browsers with support for XHR file uploads.
         *
         * This uploads files by adding them to the upload queue, the same way that files are added via the file input
         * button or drag & drop.
         * @param filesAndOptions A list of files to add to this widget. You can also override options of this widget.
         * @return This JQuery instance for chaining.
         */
        add(filesAndOptions: JQueryFileUpload.FileUploadData): void;
        /**
         * Usually, file uploads are invoked by selecting files via file input button or by dropping files on the drop zone.
         *
         * However it is also possible to upload files programmatically for browsers with support for XHR file uploads.
         *
         * This sends the file directly to the server.
         *
         * __Note__: The send API method sends the given files directly, without splitting them up into multiple requests.
         * So if your files argument is made up of 3 files, it will still only send one request. If the multipart option is
         * true, it will still send all 3 files as part of one multipart request, else it will only send the first file. So
         * if you need to send files with multiple requests, either call the send API method multiple times, or use the add
         * API method instead.
         * @param filesAndOptions A list of files to add to this widget. You can also override options of this widget.
         * @return A jqXHR object that allows to bind callbacks to the AJAX file upload requests.
         */
        send(filesAndOptions: JQueryFileUpload.FileUploadData): JQuery.jqXHR;
    }
    // ==========================
    // === Blue Imp Namespace ===
    // ==========================
    /**
     * Namespace for JQuery plugins created by https://github.com/blueimp
     */
    interface BlueImpNamespace {
        /**
         * The constructor for the {@link FileUpload} widget.
         *
         * The widget can be initialized on a file upload in the following way:
         *
         * ```javascript
         * $('#fileupload').fileupload();
         * ```
         */
        fileupload: FileUploadConstructor;
    }
}
interface JQuery {
    /**
     * The file upload widget is initialized by calling the fileupload method on a jQuery collection with the target
     * HTML element.
     *
     * The target element is usually a container element holding the file upload form, or the file upload form itself,
     * but it can also be just the file input element itself for a customized UI and if a URL is provided as options
     * parameter.
     * @param settings Options for configuring the file upload.
     * @return This JQuery instance for chaining.
     */
    fileupload(settings?: Partial<JQueryFileUpload.FileUploadOptions>): this;
    /**
     * Sets the given option to the given value.
     * @typeparam K Name of an option to set.
     * @param method The method to call on this file upload instance.
     * @param optionName Name of an option to set.
     * @return The value of the given option
     */
    fileupload<K extends keyof JQueryFileUpload.FileUploadOptions>(method: "option", optionName: K, optionValue: JQueryFileUpload.FileUploadOptions[K]): this;
    /**
     * Retrieves the value of the given option.
     * @typeparam K Name of an option to retrieve.
     * @param method The method to call on this file upload instance.
     * @param optionName Name of an option to retrieve.
     * @return The value of the given option.
     */
    fileupload<K extends keyof JQueryFileUpload.FileUploadOptions>(method: "option", optionName: K): JQueryFileUpload.FileUploadOptions[K];
    /**
     * Sets the given options on this file upload instance.
     * @param method The method to call on this file upload instance.
     * @param options Options to apply.
     * @return This JQuery instance for chaining.
     */
    fileupload(method: "option", options: Partial<JQueryFileUpload.FileUploadOptions>): this;
    /**
     * Return the current set of options. This includes default options.
     * @param method The method to call on this file upload instance.
     * @return An object with all options.
     */
    fileupload(method: "option"): JQueryFileUpload.FileUploadOptions;
    /**
     * To remove the file upload widget functionality from the element node, call the destroy method. This will also
     * remove any added event listeners.
     * @param method The method to call on this file upload instance.
     * @return This JQuery instance for chaining.
     */
    fileupload(method: "destroy"): this;
    /**
     * As other widgets based on jQuery UI Widget, the file upload widget can also be enabled.
     * @param method The method to call on this file upload instance.
     */
    fileupload(method: "enable"): this;
    /**
     * As other widgets based on jQuery UI Widget, the file upload widget can also be disabled.
     * @param method The method to call on this file upload instance.
     */
    fileupload(method: "disable"): this;
    /**
     * Finds the overall progress of all uploads.
     * @param method The method to call on this file upload instance.
     * @return The overall progress of all uploads.
     */
    fileupload(method: "progress"): JQueryFileUpload.UploadProgress;
    /**
     * Finds the number of currently active uploads.
     * @param method The method to call on this file upload instance.
     * @return The number of active uploads.
     */
    fileupload(method: "active"): number;
    /**
     * Usually, file uploads are invoked by selecting files via file input button or by dropping files on the drop zone.
     *
     * However it is also possible to upload files programmatically for browsers with support for XHR file uploads.
     *
     * This uploads files by adding them to the upload queue, the same way that files are added via the file input
     * button or drag & drop.
     * @param method The method to call on this file upload instance.
     * @param filesAndOptions A list of files to add to this widget. You can also override options of this widget.
     * @return This JQuery instance for chaining.
     */
    fileupload(method: "add", filesAndOptions: JQueryFileUpload.FileUploadData): this;
    /**
     * Usually, file uploads are invoked by selecting files via file input button or by dropping files on the drop zone.
     *
     * However it is also possible to upload files programmatically for browsers with support for XHR file uploads.
     *
     * This sends the file directly to the server.
     *
     * __Note__: The send API method sends the given files directly, without splitting them up into multiple requests.
     * So if your files argument is made up of 3 files, it will still only send one request. If the multipart option is
     * true, it will still send all 3 files as part of one multipart request, else it will only send the first file. So
     * if you need to send files with multiple requests, either call the send API method multiple times, or use the add
     * API method instead.
     * @param method The method to call on this file upload instance.
     * @param filesAndOptions A list of files to add to this widget. You can also override options of this widget.
     * @return A jqXHR object that allows to bind callbacks to the AJAX file upload requests.
     */
    fileupload(method: "send", filesAndOptions: JQueryFileUpload.FileUploadData): JQuery.jqXHR;
}
declare namespace JQuery {
    interface TypeToTriggeredEventMap<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.add|FileUploadOptions.add}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadadd: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.added|FileUploadOptions.added}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadadded: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.always|FileUploadOptions.always}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadalways: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.change|FileUploadOptions.change}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadchange: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.chunkalways|FileUploadOptions.chunkalways}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadchunkalways: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.chunkbeforesend|FileUploadOptions.chunkbeforesend}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadchunkbeforesend: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.chunkdone|FileUploadOptions.chunkdone}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadchunkdone: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.chunkfail|FileUploadOptions.chunkfail}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadchunkfail: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.chunksend|FileUploadOptions.chunksend}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadchunksend: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.completed|FileUploadOptions.completed}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadcompleted: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.destroy|FileUploadOptions.destroy}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploaddestroy: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.destroyed|FileUploadOptions.destroyed}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploaddestroyed: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.done|FileUploadOptions.done}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploaddone: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.dragover|FileUploadOptions.dragover}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploaddragover: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.drop|FileUploadOptions.drop}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploaddrop: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.fail|FileUploadOptions.fail}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadfail: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.failed|FileUploadOptions.failed}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadfailed: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.finished|FileUploadOptions.finished}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadfinished: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.paste|FileUploadOptions.paste}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadpaste: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.process|FileUploadOptions.process}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprocess: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.processalways|FileUploadOptions.processalways}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprocessalways: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.processdone|FileUploadOptions.processdone}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprocessdone: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.processfail|FileUploadOptions.processfail}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprocessfail: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.processstart|FileUploadOptions.processstart}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprocessstart: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.processstop|FileUploadOptions.processstop}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprocessstop: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.progress|FileUploadOptions.progress}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprogress: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.progressall|FileUploadOptions.progressall}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadprogressall: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.send|FileUploadOptions.send}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadsend: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.sent|FileUploadOptions.sent}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadsent: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.start|FileUploadOptions.start}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadstart: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.started|FileUploadOptions.started}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadstarted: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.stop|FileUploadOptions.stop}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadstop: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.stopped|FileUploadOptions.stopped}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadstopped: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.fileupload|jQuery BlueImp File Upload plugin}.
         *
         * See {@link JQueryFileUpload.FileUploadOptions.submit|FileUploadOptions.submit}, the callback passed via
         * the options, for more details on the parameters passed to the callback.
         */
        fileuploadsubmit: JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
    }
}
interface JQueryStatic {
    blueimp: JQueryFileUpload.BlueImpNamespace;
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * Callback invoked when file was selected and is added to this
     * widget. See also {@link FileUploadCfg.onAdd}.
     */
    export type OnAddCallback =
    /**
     * @param file The file that was selected for the upload.
     * @param callback Callback that needs to be
     * invoked with the file that should be added to the upload queue.
     */
    (this: PrimeFaces.widget.FileUpload, file: File, callback: (processedFile: File) => void) => void;
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * Callback that is invoked when a file upload was canceled. See
     * also {@link FileUploadCfg.oncancel}.
     */
    export type OnCancelCallback = (this: PrimeFaces.widget.FileUpload) => void;
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * Callback that is invoked after a file was uploaded to the
     * server successfully. See also {@link FileUploadCfg.oncomplete}.
     */
    export type OnCompleteCallback =
    /**
     * @param pfArgs The additional
     * arguments from the jQuery XHR requests.
     * @param data Details about
     * the uploaded file or files.
     */
    (this: PrimeFaces.widget.FileUpload, pfArgs: PrimeFaces.ajax.PrimeFacesArgs, data: JQueryFileUpload.JQueryAjaxCallbackData) => void;
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * Callback that is invoked when a file could not be uploaded to
     * the server. See also {@link FileUploadCfg.onerror}.
     */
    export type OnErrorCallback =
    /**
     * @param jqXHR The XHR object from the HTTP request.
     * @param textStatus The HTTP status text of the failed request.
     * @param pfArgs The additional arguments
     * from the jQuery XHR request.
     */
    (this: PrimeFaces.widget.FileUpload, jqXHR: JQuery.jqXHR, textStatus: string, pfArgs: PrimeFaces.ajax.PrimeFacesArgs) => void;
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * Callback that is invoked at the beginning of a file upload,
     * when a file is sent to the server. See also {@link FileUploadCfg.onstart}.
     */
    export type OnStartCallback = (this: PrimeFaces.widget.FileUpload) => void;
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * Callback to execute before the files are sent.
     * If this callback returns false, the file upload request is not started. See also {@link FileUploadCfg.onupload}.
     */
    export type OnUploadCallback = (this: PrimeFaces.widget.FileUpload) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces FileUpload Widget__
     *
     * FileUpload goes beyond the browser input `type="file"` functionality and features an HTML5 powered rich solution with
     * graceful degradation for legacy browsers.
     * @typeparam TCfg Defaults to `FileUploadCfg`. Type of the configuration object for this widget.
     */
    export class FileUpload<TCfg extends FileUploadCfg = FileUploadCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the bar with the buttons of this widget.
         */
        buttonBar: JQuery;
        /**
         * The DOM element for the button for canceling a file upload.
         */
        cancelButton: JQuery;
        /**
         * The DOM element for the button for selecting a file.
         */
        chooseButton: JQuery;
        /**
         * The DOM element for the button to clear the file upload messages (which inform the
         * user about whether a file was uploaded).
         */
        clearMessageLink: JQuery;
        /**
         * The DOM element for the content of this widget.
         */
        content: JQuery;
        /**
         * Custom drop zone to use for drag and drop.
         */
        customDropZone: string;
        /**
         * Amount of dragover on drop zone and its children.
         */
        dragoverCount: number;
        /**
         * Drop zone to use for drag and drop.
         */
        dropZone: string;
        /**
         * Current index where to add files.
         */
        fileAddIndex: number;
        /**
         * ID of the current file.
         */
        fileId: string;
        /**
         * List of currently selected files.
         */
        files: File[];
        /**
         * The DOM element for the table tbody with the files.
         */
        filesTbody: JQuery;
        /**
         * The DOM element for the form containing this upload widget.
         */
        form: JQuery;
        /**
         * The DOM element of the container with the file upload messages which inform the user
         * about whether a file was uploaded.
         */
        messageContainer: JQuery;
        /**
         * The DOM element of the UL list element with the file upload messages which inform the user
         * about whether a file was uploaded.
         */
        messageList: JQuery;
        /**
         * Selector for the button for canceling a file upload.
         */
        rowCancelActionSelector: string;
        /**
         * Options for the BlueImp jQuery file upload plugin.
         */
        ucfg: JQueryFileUpload.FileUploadOptions;
        /**
         * The DOM element for the button for starting the file upload.
         */
        uploadButton: JQuery;
        /**
         * Number of currently uploaded files.
         */
        uploadedFileCount: number;
        /**
         * Regular expression that matches image files for which a preview can be shown.
         */
        IMAGE_TYPES: RegExp;
        /**
         * Adds a file selected by the user to this upload widget.
         * @param file A file to add.
         * @param data The data from the selected file.
         */
        private addFileToRow(file: File, data: JQueryFileUpload.AddCallbackData): void;
        /**
         * Sets up all events listeners for this file upload widget.
         */
        private bindEvents(): void;
        /**
         * Clears this file upload field, i.e. removes all uploaded files.
         */
        clear(): void;
        /**
         * Removes all error messages that are shown for this widget.
         */
        clearMessages(): void;
        /**
         * Creates the HTML post data for uploading the selected files.
         * @return Parameters to post when upload the files.
         */
        private createPostData(): PrimeFaces.ajax.RequestParameter;
        /**
         * Creates a unique identifier (file key) for a given file. That identifier consists e.g. of the name of the
         * uploaded file, its last modified-attribute etc. This is used by the server to identify uploaded files.
         * @param file A file for which to create an identifier.
         * @return An identifier for the given file.
         */
        private createXFileId(file: File): string;
        /**
         * Disabled the given file upload button.
         * @param btn Button to disabled.
         */
        private disableButton(btn: JQuery): void;
        /**
         * Enables the given file upload button.
         * @param btn Button to enable.
         */
        private enableButton(btn: JQuery): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Called after a file was added to this upload widget. Takes care of the UI buttons.
         * @param data Data of the selected file.
         */
        private postSelectFile(data: JQueryFileUpload.AddCallbackData): void;
        /**
         * Removes the given uploaded file from this upload widget.
         * @param file File to remove from this widget.
         */
        private removeFile(file: PrimeFaces.widget.FileUpload.UploadFile): void;
        /**
         * Removes a row with an uploaded file form this upload widget.
         * @param row Row of an uploaded file to remove.
         */
        private removeFileRow(row: JQuery): void;
        /**
         * Removes the given uploaded file from this upload widget.
         * @param files Files to remove from this widget.
         */
        private removeFiles(files: PrimeFaces.widget.FileUpload.UploadFile[]): void;
        /**
         * Displays the current error messages on this widget.
         */
        private renderMessages(): void;
        /**
         * Brings up the native file selection dialog.
         */
        show(): void;
        /**
         * Shows the given error message
         * @param msg Error message to show.
         */
        private showMessage(msg: PrimeFaces.widget.FileUpload.UploadMessage): void;
        /**
         * Uploads the selected files to the server.
         */
        private upload(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  FileUpload| FileUpload widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface FileUploadCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Regular expression for accepted file types.
         */
        allowTypes: RegExp;
        /**
         * When set to true, selecting a file starts the upload process implicitly.
         */
        auto: boolean;
        /**
         * Whether this file upload is disabled.
         */
        disabled: boolean;
        /**
         * Whether drag and drop is enabled.
         */
        dnd: boolean;
        /**
         * Custom drop zone to use for drag and drop.
         */
        dropZone: string;
        /**
         * Global AJAX requests are listened to by `ajaxStatus`. When `false`, `ajaxStatus` will not
         * get triggered.
         */
        global: boolean;
        /**
         * To upload large files in smaller chunks, set this option to a preferred maximum chunk
         * size. If set to `0`, `null` or `undefined`, or the browser does not support the required Blob API, files will be
         * uploaded as a whole.
         */
        maxChunkSize: number;
        /**
         * Only for chunked file upload: Amount of retries when upload gets interrupted due to
         * e.g. an unstable network connection.
         */
        maxRetries: number;
        /**
         * Message template to use when displaying file validation errors.
         */
        messageTemplate: string;
        /**
         * Callback invoked when an uploaded file is added.
         */
        onAdd: PrimeFaces.widget.FileUpload.OnAddCallback;
        /**
         * Callback that is invoked when a file upload was
         * canceled.
         */
        oncancel: PrimeFaces.widget.FileUpload.OnCancelCallback;
        /**
         * Callback that is invoked after a file was
         * uploaded to the server successfully.
         */
        oncomplete: PrimeFaces.widget.FileUpload.OnCompleteCallback;
        /**
         * Callback that is invoked when a file could not be
         * uploaded to the server.
         */
        onerror: PrimeFaces.widget.FileUpload.OnErrorCallback;
        /**
         * Callback that is invoked at the beginning of a file
         * upload, when a file is sent to the server.
         */
        onstart: PrimeFaces.widget.FileUpload.OnStartCallback;
        /**
         * Callback to execute before the files are sent.
         * If this callback returns false, the file upload request is not started.
         */
        onupload: PrimeFaces.widget.FileUpload.OnUploadCallback;
        /**
         * Width for image previews in pixels.
         */
        previewWidth: number;
        /**
         * Component(s) to process in fileupload request.
         */
        process: string;
        /**
         * Server-side path which provides information to resume chunked file upload.
         */
        resumeContextPath: string;
        /**
         * Only for chunked file upload: (Base) timeout in milliseconds to wait until the next
         * retry. It is multiplied with the retry count. (first retry: `retryTimeout * 1`, second retry: `retryTimeout * 2`,
         * ...)
         */
        retryTimeout: number;
        /**
         * `true` to upload files one after each other, `false` to upload in parallel.
         */
        sequentialUploads: boolean;
        /**
         * Component(s) to update after fileupload completes.
         */
        update: string;
    }
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * A error message for a file upload widget.
     */
    export interface UploadMessage {
        /**
         * The name of the uploaded file.
         */
        filename: string;
        /**
         * The size of the uploaded file in bytes.
         */
        filesize: number;
        /**
         * A short summary of this message.
         */
        summary: string;
    }
}
declare namespace PrimeFaces.widget.FileUpload {
    /**
     * Represents an uploaded file added to the upload
     * widget.
     */
    export interface UploadFile {
        /**
         * Row of an uploaded file.
         */
        row: JQuery;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Simple FileUpload Widget__
     * @typeparam TCfg Defaults to `SimpleFileUploadCfg`. Type of the configuration object for this widget.
     */
    export class SimpleFileUpload<TCfg extends SimpleFileUploadCfg = SimpleFileUploadCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Number of concurrent active Ajax requests.
         */
        ajaxCount?: number;
        /**
         * The DOM element for the button for selecting a file.
         */
        button: JQuery;
        /**
         * The DOM element for the UI display.
         */
        display: JQuery;
        /**
         * The DOM element of the (closest) form that contains this file upload.
         */
        form: JQuery;
        /**
         * The DOM element for the file input element.
         */
        input: JQuery;
        /**
         * Sets up all events listeners for this file upload widget.
         */
        private bindEvents(): void;
        /**
         * Sets up the global event listeners on the button.
         */
        private bindTriggers(): void;
        /**
         * Clears the currently selected file.
         */
        clear(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Brings up the native file selection dialog.
         */
        show(): void;
        /**
         * Uploads all selected files via AJAX.
         */
        private upload(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the
     * {@link  SimpleFileUpload| SimpleFileUpload widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SimpleFileUploadCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether this file upload is disabled.
         */
        disabled: boolean;
        /**
         * Wheter the filename should be displayed.
         */
        displayFilename: boolean;
        /**
         * Global AJAX requests are listened to by `ajaxStatus`. When `false`, `ajaxStatus` will not
         * get triggered.
         */
        global: boolean;
        /**
         * Message template to use when displaying file validation errors.
         */
        messageTemplate: string;
        /**
         * Whether to apply theming to the simple upload widget.
         */
        skinSimple: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Button Widget__
     *
     * Button is an extension to the standard h:button component with skinning capabilities.
     * @typeparam TCfg Defaults to `ButtonCfg`. Type of the configuration object for this widget.
     */
    export class Button<TCfg extends ButtonCfg = ButtonCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Enables this button so that the user cannot press it.
         */
        disable(): void;
        /**
         * Enables this button so that the user can press it.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Button| Button widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ButtonCfg extends PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces CascadeSelect Widget__
     *
     * CascadeSelect CascadeSelect displays a nested structure of options.
     * @typeparam TCfg Defaults to `CascadeSelectCfg`. Type of the configuration object for this widget.
     */
    export class CascadeSelect<TCfg extends CascadeSelectCfg = CascadeSelectCfg> extends PrimeFaces.widget.DynamicOverlayWidget<TCfg> {
        /**
         * The DOM element for the content in the available selectable options.
         */
        contents: JQuery;
        /**
         * Unbind callback for the hide overlay handler.
         */
        hideOverlayHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM element for the hidden input with the current value.
         */
        input: JQuery;
        /**
         * The DOM elements for the the available selectable options.
         */
        items: JQuery;
        /**
         * The DOM element for the wrapper with the container with the available selectable
         * options.
         */
        itemsWrapper: JQuery;
        /**
         * The DOM element for the label indicating the currently selected option.
         */
        label: JQuery;
        /**
         * The DOM element for the overlay panel with the available selectable options.
         */
        panel: JQuery;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * Handler for CSS transitions used by this widget.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * The DOM elements for the buttons that can trigger (hide or show) the overlay panel with the
         * available selectable options.
         */
        triggers: JQuery;
        /**
         * Align the overlay panel with the available options.
         */
        private alignPanel(): void;
        /**
         * Adjust the width of the overlay panel.
         */
        private alignPanelWidth(): void;
        /**
         * Align the sub overlay panel with the available options.
         * @param subpanel Sub panel element of the cascade select panel.
         * @param parentPanel Parent panel element of the sub panel element.
         */
        private alignSubPanel(subpanel: JQuery, parentPanel: JQuery): void;
        /**
         * Sets up all event listeners that are required by this widget.
         */
        private bindEvents(): void;
        /**
         * Sets up all panel event listeners
         */
        private bindPanelEvents(): void;
        /**
         * Deactivate siblings and active children of an item.
         * @param item Cascade select panel element.
         */
        private deactivateItems(item: JQuery): void;
        /**
         * Disables this widget so that the user cannot select any option.
         */
        disable(): void;
        /**
         * Enables this widget so that the user can select an option.
         */
        enable(): void;
        /**
         * Fired when the browser viewport is resized or scrolled.  In Mobile environment we don't want to hider the overlay
         * we want to re-align it.  This is because on some mobile browser the popup may force the browser to trigger a
         * resize immediately and close the overlay. See GitHub #7075.
         */
        private handleViewportChange(): void;
        /**
         * Hides the overlay panel with the available options.
         */
        hide(): void;
        /**
         * Hides the panel of a group item.
         * @param item Dom element of the cascadeselect.
         */
        hideGroup(item: JQuery): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg the widget configuraton
         *
         * (from super type BaseWidget) The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Brings up the overlay panel with the available options.
         */
        show(): void;
        /**
         * Removes some event listeners when this widget was disabled.
         */
        private unbindEvents(): void;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  CascadeSelect| CascadeSelect widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface CascadeSelectCfg extends PrimeFaces.widget.DynamicOverlayWidgetCfg {
        /**
         * Appends the overlay to the element defined by search expression. Defaults to the document
         * body.
         */
        appendTo: string;
        /**
         * If true, disables the component.
         */
        disabled: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces CommandButton Widget__
     *
     * CommandButton is an extended version of standard commandButton with AJAX and theming.
     * @typeparam TCfg Defaults to `CommandButtonCfg`. Type of the configuration object for this widget.
     */
    export class CommandButton<TCfg extends CommandButtonCfg = CommandButtonCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Number of concurrent active Ajax requests.
         */
        ajaxCount?: number;
        /**
         * Sets up the global event listeners on the button.
         */
        private bindTriggers(): void;
        /**
         * Disables this button so that the user cannot press the button anymore.
         */
        disable(): void;
        /**
         * Enables this button so that the user can press the button.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  CommandButton| CommandButton widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface CommandButtonCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * When set to `true` this button is only enabled after successful client side validation, otherwise classic behaviour. Used together with p:clientValidator.
         */
        validateClientDynamic: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces CommandLink Widget__
     *
     * CommandLink is an extended version of standard commandLink with AJAX and theming.
     * @typeparam TCfg Defaults to `CommandLinkCfg`. Type of the configuration object for this widget.
     */
    export class CommandLink<TCfg extends CommandLinkCfg = CommandLinkCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Number of concurrent active Ajax requests.
         */
        ajaxCount?: number;
        /**
         * Timestamp of the Ajax request that started the animation.
         */
        ajaxStart?: number;
        /**
         * Sets up the global event listeners on the link.
         */
        private bindTriggers(): void;
        /**
         * Disables this link so that the user cannot click the link anymore.
         */
        disable(): void;
        /**
         * Enables this link so that the user can click the link.
         */
        enable(): void;
        /**
         * Ends the AJAX disabled state.
         * @param widget the widget.
         */
        endAjaxDisabled(widget?: PrimeFaces.widget.BaseWidget): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  CommandLink| CommandLink widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface CommandLinkCfg extends PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces DefaultCommand Widget__
     *
     * Which command to submit the form with when enter key is pressed a common problem in web apps not just specific to
     * JSF. Browsers tend to behave differently as there doesn’t seem to be a standard and even if a standard exists,
     * IE probably will not care about it. There are some ugly workarounds like placing a hidden button and writing
     * JavaScript for every form in your app. `DefaultCommand` solves this problem by normalizing the command (e.g. button
     * or link) to submit the form with on enter key press.
     * @typeparam TCfg Defaults to `DefaultCommandCfg`. Type of the configuration object for this widget.
     */
    export class DefaultCommand<TCfg extends DefaultCommandCfg = DefaultCommandCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the target button or link.
         */
        jqTarget: JQuery;
        /**
         * The ancestor component to enable multiple default commands in a form.
         */
        scope: JQuery | null;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  DefaultCommand| DefaultCommand widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface DefaultCommandCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Identifier of the ancestor component to enable multiple default commands in a form.
         */
        scope: string;
        /**
         * Identifier of the default command component.
         */
        target: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces InputText Widget__
     *
     * InputText is an extension to standard inputText with skinning capabilities.
     * @typeparam TCfg Defaults to `InputTextCfg`. Type of the configuration object for this widget.
     */
    export class InputText<TCfg extends InputTextCfg = InputTextCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the counter that informs the user about the number of characters they can
         * still enter before they reach the limit.
         */
        counter: JQuery;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Replaces all line breaks with a Window-style line break (carriage return + line feed).
         * @param text Text to normalize.
         * @return The given text, with all line breaks replaced with carriage return + line feed.
         */
        private normalizeNewlines(text: string): string;
        /**
         * Updates the counter value that keeps count of how many more characters the user can enter before they reach the
         * limit.
         */
        private updateCounter(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  InputText| InputText widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface InputTextCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * ID of the label component to display remaining and entered characters.
         */
        counter: string;
        /**
         * Template text to display in counter, default value is `{0}`.
         */
        counterTemplate: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces InputTextarea Widget__
     *
     * InputTextarea is an extension to standard inputTextarea with autoComplete, autoResize, remaining characters counter
     * and theming features.
     * @typeparam TCfg Defaults to `InputTextareaCfg`. Type of the configuration object for this widget.
     */
    export class InputTextarea<TCfg extends InputTextareaCfg = InputTextareaCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * The DOM element for the counter that informs the user about the number of
         * characters they can still enter before they reach the limit.
         */
        counter?: JQuery | null;
        /**
         * The DOM elements in the autocomplete panel that the user can select.
         */
        items?: JQuery;
        /**
         * The DOM element for the overlay panel with the autocomplete suggestions, when
         * autocomplete is enabled.
         */
        panel?: JQuery;
        /**
         * The keyword or search term the autocomplete method receives as input.
         */
        query?: string;
        /**
         * The internal timeout ID of the most recent timeout that was started.
         */
        timeout?: number | null;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Aligns the search window panel of the autocomplete feature.
         */
        alignPanel(): void;
        /**
         * Applies the value of the max length setting, counting line breaks correctly.
         */
        private applyMaxlength(): void;
        /**
         * Sets up all event listeners for the various events required by this widget.
         */
        private bindDynamicEvents(): void;
        /**
         * Clears the timeout that was set up by the autocomplete feature.
         */
        private clearTimeout(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Finds the keyword to be used for the autocomplete search.
         * @return The keyword or search term the autocomplete method receives as input.
         */
        private extractQuery(): string;
        /**
         * Hides the search window panel of the autocomplete feature.
         */
        private hide(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Callback that is invoked when the user has selected one of the suggested tokens.
         * @param event Event that triggered the item selection (usually a click or enter press).
         * @param itemValue Value of the suggestion that was selected.
         */
        private invokeItemSelectBehavior(event: JQuery.TriggeredEvent, itemValue: string): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Performs an autocomplete search for the given search term. Opens the windows with the suggestions.
         * @param query Search term to search for.
         */
        search(query: string): void;
        /**
         * Sets up the server-side auto complete functionality that suggests tokens while the user types.
         */
        private setupAutoComplete(): void;
        /**
         * Initializes the auto resize functionality that resize this textarea depending on the entered text.
         */
        private setupAutoResize(): void;
        /**
         * Adjust the search window panel of the autocomplete in case this widget is inside a dialog overlay.
         */
        private setupDialogSupport(): void;
        /**
         * Brings up the search window panel of the autocomplete feature.
         */
        private show(): void;
        /**
         * Updates the counter value that keeps count of how many more characters the user can enter before they reach the
         * limit.
         */
        private updateCounter(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  InputTextarea| InputTextarea widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface InputTextareaCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * Enables autocompletion that suggests tokens to the user as they type.
         */
        autoComplete: boolean;
        /**
         * Enables auto growing when being typed.
         */
        autoResize: boolean;
        /**
         * ID of the label component to display remaining and entered characters.
         */
        counter: string;
        /**
         * Template text to display in counter, default value is `{0}`.
         */
        counterTemplate: string;
        /**
         * Maximum number of characters that may be entered in this field.
         */
        maxlength: number;
        /**
         * Number of characters to be typed to run a query.
         */
        minQueryLength: number;
        /**
         * Delay in milliseconds before sending each query.
         */
        queryDelay: number;
        /**
         * Height of the viewport for autocomplete suggestions.
         */
        scrollHeight: number;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces LinkButton Widget__
     *
     * LinkButton a simple link, which is styled as a button and integrated with JSF navigation model.
     * @typeparam TCfg Defaults to `LinkButtonCfg`. Type of the configuration object for this widget.
     */
    export class LinkButton<TCfg extends LinkButtonCfg = LinkButtonCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the link that is a child of the button.
         */
        link: JQuery;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Disables this link button so that it cannot be clicked.
         */
        disable(): void;
        /**
         * Enables this link button so that it can be clicked.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  LinkButton| LinkButton widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface LinkButtonCfg extends PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces MultiSelectListbox Widget__
     *
     * MultiSelectListbox is used to select an item from a collection of listboxes that are in parent-child relationship.
     * @typeparam TCfg Defaults to `MultiSelectListboxCfg`. Type of the configuration object for this widget.
     */
    export class MultiSelectListbox<TCfg extends MultiSelectListboxCfg = MultiSelectListboxCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The hidden input field storing the selected value.
         */
        input: JQuery;
        /**
         * The DOM elements in all boxes that can be selected.
         */
        items: JQuery;
        /**
         * The DOM element for the root box with no children.
         */
        root: JQuery;
        /**
         * Sets up all event listeners that are required by this widget.
         */
        private bindEvents(): void;
        /**
         * Disabled this list box so that the user cannot select items anymore.
         */
        disable(): void;
        /**
         * Enables this list box so that the user can select an item.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Selects the item with the given value, expanding and showing all parent boxes as neccessary.
         * @param value Value of the item to be shown.
         */
        preselect(value: string): void;
        /**
         * Shows the given box with a group of options.
         * @param item The box to be shown.
         */
        private showOptionGroup(item: JQuery): void;
        /**
         * Triggers the change behavior, invoked after an item was selected or deselected.
         */
        private triggerChange(): void;
        /**
         * Removes some of the event listener that were registered by `bindEvents`. Called when this widget is disabled.
         */
        private unbindEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  MultiSelectListbox| MultiSelectListbox widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface MultiSelectListboxCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * If true, disables the component.
         */
        disabled: boolean;
        /**
         * Effect to use when showing a group of items.
         */
        effect: string;
        /**
         * Displays label of a group at header section of the children items.
         */
        showHeaders: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Password Widget__
     *
     * Password component is an extended version of standard inputSecret component with theme integration and strength
     * indicator.
     * @typeparam TCfg Defaults to `PasswordCfg`. Type of the configuration object for this widget.
     */
    export class Password<TCfg extends PasswordCfg = PasswordCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for mask/unmask icon
         */
        icon: JQuery;
        /**
         * The DOM element for the informational text regarding how strong the password is.
         */
        infoText: JQuery;
        /**
         * The DOM element for the gauge giving visual feedback regarding how strong the password is.
         */
        meter: JQuery;
        /**
         * The DOM element for the overlay panel with the hint regarding how strong the password is.
         */
        panel: JQuery;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * Handler for CSS transitions used by this widget.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * Align the panel with the password strength indicator so that it is next to the password input field.
         */
        private align(): void;
        /**
         * Sets up all panel event listeners
         */
        private bindPanelEvents(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Hides the panel with the password strength indicator.
         */
        hide(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Returns a normalized value between `0` and `1.5` that indicates how much bigger the first input x is compared
         * to the other input y. `0` means that x is much smaller than `y`, a value of `1.5` mean that `x` is much larger
         * than `y`.
         * @param x First input, must be a non-negative number.
         * @param y Second input, must be a positive number
         * @return A value between `0` and `1.5` that indicates how big `x` is compared to `y`.
         */
        private normalize(x: number, y: number): number;
        /**
         * Sets up the overlay panel informing the user about how good the password their typed is.
         */
        private setupFeedback(): void;
        /**
         * Sets up the ability to unmask and remask the password.
         */
        private setupUnmasking(): void;
        /**
         * Brings up the panel with the password strength indicator.
         */
        show(): void;
        /**
         * Computes a numerical score that estimates how strong the given password is. The returned value can range from `0`
         * (very weak) to `128` (very strong). This test takes into account whether the password has got a certain minimal
         * length and whether it contains characters from certain character classes.
         * @param password A password to check for its strength.
         * @return A value between `0` and `128` that indicates how good the password is, with `0` indicating a
         * very weak password and `128` indicating a very strong password.
         */
        testStrength(password: string): number;
        /**
         * Toggle masking and unmasking the password.
         */
        toggleMask(): void;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Password| Password widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface PasswordCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Enables strength indicator.
         */
        feedback: boolean;
        /**
         * Text of the hint when the password is judged to be good.
         */
        goodLabel: string;
        /**
         * Event hiding the feedback overlay. Default is 'blur'.
         */
        hideEvent: string;
        /**
         * Displays feedback inline rather than using a popup.
         */
        inline: boolean;
        /**
         * Label of the password prompt.
         */
        promptLabel: string;
        /**
         * Event displaying the feedback overlay. Default is 'focus'.
         */
        showEvent: string;
        /**
         * Text of the hint when the password is judged to be strong.
         */
        strongLabel: string;
        /**
         * Whether or not this password can be unmasked/remasked.
         */
        unmaskable: boolean;
        /**
         * Text of the hint when the password is judged to be weak.
         */
        weakLabel: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectBooleanButton Widget__
     *
     * SelectBooleanButton is used to select a binary decision with a toggle button.
     * @typeparam TCfg Defaults to `SelectBooleanButtonCfg`. Type of the configuration object for this widget.
     */
    export class SelectBooleanButton<TCfg extends SelectBooleanButtonCfg = SelectBooleanButtonCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Whether this button is disabled.
         */
        disabled: boolean;
        /**
         * The DOM element for the icon with the button.
         */
        icon: JQuery;
        /**
         * The DOM element for the hidden input field storing the value of this widget.
         */
        input: JQuery;
        /**
         * Calculated Off label value either set by user or locale.
         */
        offLabel: string;
        /**
         * Calculated On label value either set by user or locale.
         */
        onLabel: string;
        /**
         * Turns this button to its on state, which corresponds to checking the underlying checkbox.
         */
        check(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Toggles the state of this button, i.e. turning it on if it is off and vice-versa. Corresponds to checking or
         * unchecking the underlying checkbox.
         */
        toggle(): void;
        /**
         * Turns this button to its off state, which corresponds to unchecking the underlying checkbox.
         */
        uncheck(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectBooleanButton| SelectBooleanButton widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectBooleanButtonCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Icon to display when button is unselected.
         */
        offIcon: string;
        /**
         * Label to display when button is unselected.
         */
        offLabel: string;
        /**
         * Icon to display when button is selected.
         */
        onIcon: string;
        /**
         * Label to display when button is selected.
         */
        onLabel: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectBooleanCheckbox Widget__
     *
     * SelectBooleanCheckbox is an extended version of the standard checkbox with theme integration.
     * @typeparam TCfg Defaults to `SelectBooleanCheckboxCfg`. Type of the configuration object for this widget.
     */
    export class SelectBooleanCheckbox<TCfg extends SelectBooleanCheckboxCfg = SelectBooleanCheckboxCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the box with the checkbox.
         */
        box: JQuery;
        /**
         * Whether this checkbox is disabled.
         */
        disabled: boolean;
        /**
         * The DOM element for the checked or unchecked checkbox icon.
         */
        icon: JQuery;
        /**
         * The DOM element for the hidden input field storing the current value of this widget.
         */
        input: JQuery;
        /**
         * The DOM element for the label of the checkbox.
         */
        itemLabel: JQuery;
        /**
         * Checks this checkbox, if it is not checked already.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        check(silent?: boolean): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Checks whether this checkbox is currently checked.
         * @return `true` if this checkbox is checked, or `false` otherwise.
         */
        isChecked(): boolean;
        /**
         * Resets the input.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        resetValue(silent?: boolean): void;
        /**
         * Checks this checkbox if it is currently unchecked, or unchecks it otherwise.
         */
        toggle(): void;
        /**
         * Unchecks this checkbox, if it is not unchecked already .
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        uncheck(silent?: boolean): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectBooleanCheckbox| SelectBooleanCheckbox widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectBooleanCheckboxCfg extends PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget.SelectCheckboxMenu {
    /**
     * A function for filtering the options of a select list
     * box.
     */
    export type FilterFunction =
    /**
     * @param itemLabel The label of the currently selected
     * text.
     * @param filterValue The value to search for.
     * @return `true` if the item label matches the filter
     * value, or `false` otherwise.
     */
    (itemLabel: string, filterValue: string) => boolean;
}
declare namespace PrimeFaces.widget.SelectCheckboxMenu {
    /**
     * Available modes for filtering the options of a select list box. When `custom` is set, a `filterFunction` must be
     * specified.
     */
    export type FilterMatchMode = "startsWith" | "contains" | "endsWith" | "custom";
}
declare namespace PrimeFaces.widget.SelectCheckboxMenu {
    /**
     * Callback that is invoked when a checkbox option was
     * checked or unchecked. See also {@link SelectCheckboxMenuCfg.onChange}.
     */
    export type OnChangeCallback = (this: PrimeFaces.widget.SelectCheckboxMenu) => void;
}
declare namespace PrimeFaces.widget.SelectCheckboxMenu {
    /**
     * Callback that is invoked when the overlay panel is
     * brought up. See also {@link SelectCheckboxMenuCfg.onHide}.
     */
    export type OnHideCallback = (this: PrimeFaces.widget.SelectCheckboxMenu) => void;
}
declare namespace PrimeFaces.widget.SelectCheckboxMenu {
    /**
     * Callback that is invoked when the overlay panel is
     * hidden. See also {@link SelectCheckboxMenuCfg.onShow}.
     */
    export type OnShowCallback = (this: PrimeFaces.widget.SelectCheckboxMenu) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectCheckboxMenu Widget__
     *
     * SelectCheckboxMenu is a multi select component that displays options in an overlay.
     * @typeparam TCfg Defaults to `SelectCheckboxMenuCfg`. Type of the configuration object for this widget.
     */
    export class SelectCheckboxMenu<TCfg extends SelectCheckboxMenuCfg = SelectCheckboxMenuCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the checkboxes that can be selected.
         */
        checkboxes: JQuery;
        /**
         * The DOM element for the button that closes the overlay panel with the select
         * options (when the panel is shown).
         */
        closer?: JQuery;
        /**
         * The DOM element for the default label.
         */
        defaultLabel: JQuery;
        /**
         * Whether this widget is currently disabled.
         */
        disabled: boolean;
        /**
         * DOM element of the input element used for entering a filter text, if filtering is
         * enabled.
         */
        filterInput?: JQuery;
        /**
         * DOM element of the wrapper that holds the input element used for entering a
         * filter text, if filtering is enabled.
         */
        filterInputWrapper?: JQuery;
        /**
         * The filter that was selected and is
         * currently used.
         */
        filterMatcher: PrimeFaces.widget.SelectCheckboxMenu.FilterFunction;
        /**
         * Map between the available filter types and the filter implementation.
         */
        filterMatchers: Record<PrimeFaces.widget.SelectCheckboxMenu.FilterMatchMode, PrimeFaces.widget.SelectCheckboxMenu.FilterFunction>;
        /**
         * The DOM elements for the headers of each option group.
         */
        groupHeaders: JQuery;
        /**
         * DOM element of the header of the overlay panel with the available select items,
         * when the overlay panel is shown.
         */
        header?: JQuery;
        /**
         * Unbind callback for the hide overlay handler.
         */
        hideOverlayHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM elements for the hidden inputs for each checkbox option.
         */
        inputs: JQuery;
        /**
         * When loading the panel with the available options lazily: if they have been loaded
         * already.
         */
        isDynamicLoaded: boolean;
        /**
         * The DOM element for the container with the available checkbox options.
         */
        itemContainer: JQuery;
        /**
         * The DOM element for the wrapper with the container with the available checkbox
         * options.
         */
        itemContainerWrapper: JQuery;
        /**
         * The DOM elements for the the available checkbox options.
         */
        items: JQuery;
        /**
         * The DOM element for the hidden input element that that can be selected via pressing
         * tab.
         */
        keyboardTarget: JQuery;
        /**
         * The DOM element for the label indicating the currently selected option.
         */
        label: JQuery;
        /**
         * The DOM element for the container with the label indicating the currently selected
         * option.
         */
        labelContainer: JQuery;
        /**
         * ID of the label element that indicates the currently selected option.
         */
        labelId: string;
        /**
         * The DOM element for the icon for bringing up the overlay panel.
         */
        menuIcon: JQuery;
        /**
         * The DOM element for the container with the tags representing the selected options.
         */
        multiItemContainer: JQuery;
        /**
         * The DOM element for the overlay panel with the available checkbox options.
         */
        panel: JQuery;
        /**
         * ID of the DOM element for the overlay panel with the available checkbox options.
         */
        panelId: JQuery;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * Tab index of this widget.
         */
        tabindex: string;
        /**
         * The wrapping DOM element of the toggler for selecting or unselecting all options
         * in the overlay panel with all selected items (when the overlay panel is shown).
         */
        toggler?: JQuery;
        /**
         * The DOM element with the toggler checkbox for selecting or unselecting all
         * options in the overlay panel with all selected items (when the overlay panel is shown).
         */
        togglerBox?: JQuery;
        /**
         * The DOM element for the hidden input of the Select All checkbox.
         */
        togglerCheckboxInput?: JQuery;
        /**
         * Handler for CSS transitions used by this widget.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * The DOM elements for the buttons that can trigger (hide or show) the overlay panel with the
         * available checkbox options.
         */
        triggers: JQuery;
        /**
         * Whether the width of the overlay panel was aligned already.
         */
        widthAligned: boolean;
        /**
         * Creates the overlay panel with the checkboxes for the selectable option, and also sets up the event listeners
         * for the panel.
         */
        private _renderPanel(): void;
        /**
         * Align the overlay panel with the available checkbox options so that is is positioned next to the the button.
         */
        alignPanel(): void;
        /**
         * Sets up the event listeners for hovering over the checkboxes. Adds the appropriate hover style classes.
         * @param item A checkbox for which to add the event listeners.
         */
        private bindCheckboxHover(item: JQuery): void;
        /**
         * Sets up the keyboard event listeners for the given checkbox options.
         * @param items Checkbo options for which to add the event listeners.
         */
        private bindCheckboxKeyEvents(items: JQuery): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Sets up the event listeners for all keyboard related events other than the overlay panel, such as pressing space
         * to bring up the overlay panel.
         */
        private bindKeyEvents(): void;
        /**
         * When multi select mode is enabled: Sets up the event listeners for the overlay panel.
         */
        private bindMultipleModeEvents(): void;
        /**
         * Sets up the event listeners for the overlay panel with the selectable checkbox options.
         */
        private bindPanelContentEvents(): void;
        /**
         * Sets up all panel event listeners
         */
        private bindPanelEvents(): void;
        /**
         * Sets up the event listeners for all keyboard related events of the overlay panel, such as pressing space to
         * toggle a checkbox.
         */
        private bindPanelKeyEvents(): void;
        /**
         * Selects the given checkbox option.
         * @param checkbox Checkbox option to select.
         * @param updateInput If `true`, update the hidden input field with the current value of this widget.
         */
        private check(checkbox: JQuery, updateInput: boolean): void;
        /**
         * Selects all available options.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        checkAll(silent?: boolean): void;
        /**
         * Configures the ARIA label for the select all checkbox.
         */
        private configureSelectAllAria(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectCheckboxMenu.FilterFunction` that matches the given option when it
         * contains the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the contains the filter value, or `false` otherwise.
         */
        containsFilter(value: string, filter: string): boolean;
        /**
         * When multi mode is enabled: Creates a tag for the given item that was checked.
         * @param item The checkbox item that was checked.
         */
        private createMultipleItem(item: JQuery): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectCheckboxMenu.FilterFunction` that matches the given option when it
         * ends with the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the options ends with the filter value, or `false` otherwise.
         */
        endsWithFilter(value: string, filter: string): boolean;
        /**
         * Filters the available options in the overlay panel by the given search value. Note that this does not bring up
         * the overlay panel, use `show` for that.
         * @param value A value against which the available options are matched.
         */
        filter(value: string): void;
        /**
         * Triggers the select behavior, if any, when a checkbox option was selected or unselected.
         * @param checked Whether the checkbox option is now checked.
         */
        private fireToggleSelectEvent(checked: boolean): void;
        /**
         * Fired when the browser viewport is resized or scrolled.  In Mobile environment we don't want to hider the overlay
         * we want to re-align it.  This is because on some mobile browser the popup may force the browser to trigger a
         * resize immediately and close the overlay. See GitHub #7075.
         */
        private handleViewportChange(): void;
        /**
         * Hides the overlay panel with the available checkbox options.
         */
        hide(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Has the panel been loaded with checkbox data yet?
         * @return `true` when the panel has been loaded with checkbox items
         */
        isLoaded(): boolean;
        /**
         * Callback that is invoked after the overlay panel with the checkbox options was hidden.
         */
        private postHide(): void;
        /**
         * Callback that is invoked after the overlay panel with the checkbox options was made visible.
         */
        private postShow(): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Mark trigger and descandants of trigger as a trigger for a primefaces overlay.
         */
        private registerTrigger(): void;
        /**
         * When multi mode is enabled: Removes all visible tags with the same value as the given checkbox item.
         * @param item Checkbox item that was unchecked.
         */
        private removeMultipleItem(item: JQuery): void;
        /**
         * Creates the individual checkboxes for each selectable option in the overlay panel.
         */
        private renderItems(): void;
        /**
         * Create the label to display values
         */
        private renderLabel(): void;
        /**
         * Creates the overlay panel with the checkboxes for the selectable option.
         */
        private renderPanel(): void;
        /**
         * Resets the input.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        resetValue(silent?: boolean): void;
        /**
         * Checks the checkbox option with the given value.
         * @param value Value of the option to check.
         */
        selectValue(value: string): void;
        /**
         * Finds and stores the filter function which is to be used for filtering the options of this select checkbox menu.
         */
        private setupFilterMatcher(): void;
        /**
         * Brings up the overlay panel with the available checkbox options.
         */
        show(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectCheckboxMenu.FilterFunction` that matches the given option when it
         * starts with the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the options starts with the filter value, or `false` otherwise.
         */
        startsWithFilter(value: string, filter: string): boolean;
        /**
         * Select or unselect the given checkbox option.
         * @param checkbox One of the checkbox options of this widget to toggle.
         * @param input (optional) input element representing the value of the checkbox
         */
        toggleItem(checkbox: JQuery, input: JQuery): void;
        /**
         * Bring up the overlay panel if its not showing or hide it if it is showing.
         */
        togglePanel(): void;
        /**
         * Toggles either selecting all items or unselecting all items.
         * @param selectAll true to select all items and false to uncheck all items
         */
        toggleSelection(selectAll: boolean): void;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
        /**
         * Unselects the given checkbox option.
         * @param checkbox Checkbox option to unselect.
         * @param updateInput If `true`, update the hidden input field with the current value of this widget.
         */
        private uncheck(checkbox: JQuery, updateInput: boolean): void;
        /**
         * Unselects all available options.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        uncheckAll(silent?: boolean): void;
        /**
         * When multi mode is disabled: Upates the label that indicates the currently selected item.
         */
        private updateLabel(): void;
        /**
         * Updates the `select all` / `unselect all` toggler so that it reflects the currently selected options.
         */
        private updateToggler(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectCheckboxMenu| SelectCheckboxMenu widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectCheckboxMenuCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * The search expression for the element to which the overlay panel should be appended.
         */
        appendTo: string;
        /**
         * Defines if filtering would be case sensitive.
         */
        caseSensitive: boolean;
        /**
         * Defines if dynamic loading is enabled for the element's panel. If the value is `true`,
         * the overlay is not rendered on page load to improve performance.
         */
        dynamic: boolean;
        /**
         * Label to be shown in updateLabel mode when no item is selected. If not set the label is
         * shown.
         */
        emptyLabel: string;
        /**
         * `true` if the options can be filtered, or `false` otherwise.
         */
        filter: boolean;
        /**
         * A custom filter function that is used
         * when `filterMatchMode` is set to `custom`.
         */
        filterFunction: PrimeFaces.widget.SelectCheckboxMenu.FilterFunction;
        /**
         * Mode of the filter. When set to
         * `custom`, a `filterFunction` must be specified.
         */
        filterMatchMode: PrimeFaces.widget.SelectCheckboxMenu.FilterMatchMode;
        /**
         * Defines if filtering would be done using normalized values.
         */
        filterNormalize: boolean;
        /**
         * Placeholder text to show when filter input is empty.
         */
        filterPlaceholder: string;
        /**
         * Initial height of the overlay panel in pixels.
         */
        initialHeight: number;
        /**
         * Separator for joining item labels if updateLabel is set to true. Default is `,`.
         */
        labelSeparator: string;
        /**
         * Whether to show selected items as multiple labels.
         */
        multiple: boolean;
        /**
         * Callback that is invoked when a checkbox
         * option was checked or unchecked.
         */
        onChange: PrimeFaces.widget.SelectCheckboxMenu.OnChangeCallback;
        /**
         * Callback that is invoked when the overlay
         * panel is brought up.
         */
        onHide: PrimeFaces.widget.SelectCheckboxMenu.OnHideCallback;
        /**
         * Callback that is invoked when the overlay
         * panel is hidden.
         */
        onShow: PrimeFaces.widget.SelectCheckboxMenu.OnShowCallback;
        /**
         * Inline style of the overlay panel.
         */
        panelStyle: string;
        /**
         * Style class of the overlay panel
         */
        panelStyleClass: string;
        /**
         * Renders panel content on client.
         */
        renderPanelContentOnClient: boolean;
        /**
         * Maximum height of the overlay panel.
         */
        scrollHeight: number;
        /**
         * Label to be shown in updateLabel mode when one or more items are selected. If not
         * set the label is shown.
         */
        selectedLabel: string;
        /**
         * When enabled, the header of overlay panel is displayed.
         */
        showHeader: boolean;
        /**
         * When enabled, the selected items are displayed on the label.
         */
        updateLabel: boolean;
    }
}
declare namespace PrimeFaces.widget.SelectListbox {
    /**
     * A function for filtering the options of a select list box.
     */
    export type FilterFunction =
    /**
     * @param itemLabel The label of the currently selected text.
     * @param filterValue The value to search for.
     * @return `true` if the item label matches the filter value,
     * or `false` otherwise.
     */
    (itemLabel: string, filterValue: string) => boolean;
}
declare namespace PrimeFaces.widget.SelectListbox {
    /**
     * Available modes for filtering the options of a select list box. When `custom` is set, a `filterFunction` must be
     * specified.
     */
    export type FilterMatchMode = "startsWith" | "contains" | "endsWith" | "custom";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectListbox Widget__
     *
     * Base class for the `SelectManyMenu` and `SelectOneListBox` widgets. Contains some common functionality such as
     * filtering and working with SELECT and OPTION elements.
     * @typeparam TCfg Defaults to `SelectListboxCfg`. Type of the configuration object for this widget.
     */
    export class SelectListbox<TCfg extends SelectListboxCfg = SelectListboxCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * All available items, including disabled options. These are not form elements, but the DOM
         * elements presented to the user.
         */
        allItems: JQuery;
        /**
         * `true` if this many select element is disabled, `false` if enabled, `undefined`
         * if the state is not known.
         */
        disabled?: boolean;
        /**
         * DOM element of the input element used for entering a filter text, if filtering
         * is enabled.
         */
        filterInput?: JQuery;
        /**
         * The filter that was selected and is currently
         * used.
         */
        filterMatcher: PrimeFaces.widget.SelectListbox.FilterFunction;
        /**
         * Map between the available filter types and the filter implementation.
         */
        filterMatchers: Record<PrimeFaces.widget.SelectListbox.FilterMatchMode, PrimeFaces.widget.SelectListbox.FilterFunction>;
        /**
         * The hidden INPUT or SELECT element.
         */
        input: JQuery;
        /**
         * All available items, excluding disabled options. These are not form elements, but the DOM
         * elements presented to the user.
         */
        items: JQuery;
        /**
         * Container of the list element.
         */
        listContainer: JQuery;
        /**
         * The element that contains the available items.
         */
        listElement: JQuery;
        /**
         * A list of the available OPTION elements of the hidden SELECT element.
         */
        options: JQuery<HTMLOptionElement>;
        /**
         * Sets up all event listeners for this widget instance.
         */
        protected bindEvents(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectListbox.FilterFunction` that matches the given option when it
         * contains the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the contains the filter value, or `false` otherwise.
         */
        containsFilter(value: string, filter: string): boolean;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectListbox.FilterFunction` that matches the given option when it ends
         * with the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the options ends with the filter value, or `false` otherwise.
         */
        endsWithFilter(value: string, filter: string): boolean;
        /**
         * Filters the options of this select list box by the given search value.
         * @param value Current value of the filter.
         */
        filter(value: string): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Select the given item as the currently selected option. Does not unselect other items that are already selected.
         * @param item An OPTION element to set as the selected element.
         */
        selectItem(item: JQuery): void;
        /**
         * Finds and stores the filter function which is to be used for filtering the options of this select list box.
         */
        private setupFilterMatcher(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectListbox.FilterFunction` that matches the given option when it starts
         * with the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the options starts with the filter value, or `false` otherwise.
         */
        startsWithFilter(value: string, filter: string): boolean;
        /**
         * Unselects all items that are currently selected.
         */
        unselectAll(): void;
        /**
         * Unselect the given items. Does not change other selected items.
         * @param item Item to unselect.
         */
        unselectItem(item: JQuery): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectListbox| SelectListbox widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectListboxCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * `true` if filtering is case-sensitive, `false` otherwise.
         */
        caseSensitive: boolean;
        /**
         * `true` if the options can be filtered, or `false` otherwise.
         */
        filter: boolean;
        /**
         * A custom filter function that is used when
         * `filterMatchMode` is set to `custom`.
         */
        filterFunction: PrimeFaces.widget.SelectListbox.FilterFunction;
        /**
         * Mode of the filter. When set to `custom`
         *  a `filterFunction` must be specified.
         */
        filterMatchMode: PrimeFaces.widget.SelectListbox.FilterMatchMode;
        /**
         * Defines if filtering would be done using normalized values.
         */
        filterNormalize: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectManyButton Widget__
     *
     * SelectManyButton is a multi select component using button UI.
     * @typeparam TCfg Defaults to `SelectManyButtonCfg`. Type of the configuration object for this widget.
     */
    export class SelectManyButton<TCfg extends SelectManyButtonCfg = SelectManyButtonCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM elements for the selectable buttons.
         */
        buttons: JQuery;
        /**
         * `true` if this many select element is disabled, `false` if enabled, `undefined`
         * if the state is not known.
         */
        disabled?: boolean;
        /**
         * The DOM elements for the hidden input fields of type checkbox storing which buttons are
         * selected.
         */
        inputs: JQuery;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Selects the given button option.
         * @param button A button of this widget to select.
         */
        select(button: JQuery): void;
        /**
         * Unselects the given button option.
         * @param button A button of this widget to unselect.
         */
        unselect(button: JQuery): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectManyButton| SelectManyButton widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectManyButtonCfg extends PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectManyCheckbox Widget__
     *
     * SelectManyCheckbox is an extended version of the standard SelectManyCheckbox.
     * @typeparam TCfg Defaults to `SelectManyCheckboxCfg`. Type of the configuration object for this widget.
     */
    export class SelectManyCheckbox<TCfg extends SelectManyCheckboxCfg = SelectManyCheckboxCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * `true` if this many select element is disabled, `false` if enabled, `undefined`
         * if the state is not known.
         */
        disabled?: boolean;
        /**
         * The (cloned) DOM elements for the non-disabled hidden input fields of type checkbox
         * storing the value of this widget.
         */
        enabledInputs: JQuery;
        /**
         * The (DOM elements for the hidden input fields of type checkbox storing the value of
         * this widget. In case of layout 'custom', this is are the visible inputs.
         */
        inputs: JQuery;
        /**
         * The DOM elements for the label texts of each radio button.
         */
        labels: JQuery;
        /**
         * The DOM elements for the hidden input fields of type checkbox storing the value of
         * this widget. It's only used in layout 'custom'.
         */
        originalInputs: JQuery;
        /**
         * The DOM elements for the checkbox icons shown on the UI.
         */
        outputs: JQuery;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Checks the given checkbox and associated input.
         * @param input the input.
         * @param checkbox the checbkox.
         */
        private check(input: JQuery, checkbox: JQuery): void;
        /**
         * Check all available options.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        checkAll(silent?: boolean): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Calls the behavior for when a checkbox was clicked.
         * @param input Checkbox input that was clicked.
         * @param event (Click) event that was triggered.
         */
        private fireClickEvent(input: JQuery, event: JQuery.TriggeredEvent): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Resets the input.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        resetValue(silent?: boolean): void;
        /**
         * Toggles the given checkbox and associated input.
         * @param input the input.
         * @param checkbox the checbkox.
         * @param event event that was triggered.
         */
        private toggle(input: JQuery, checkbox: JQuery, event: JQuery.TriggeredEvent): void;
        /**
         * Unchecks the given checkbox and associated input.
         * @param input the input.
         * @param checkbox the checbkox.
         */
        private uncheck(input: JQuery, checkbox: JQuery): void;
        /**
         * Uncheck all available options.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        uncheckAll(silent?: boolean): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectManyCheckbox| SelectManyCheckbox widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectManyCheckboxCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether a custom HTML snippet needs to be used for the individual select items.
         */
        custom: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectManyMenu Widget__
     *
     * SelectManyMenu is an extended version of the standard SelectManyMenu.
     * @typeparam TCfg Defaults to `SelectManyMenuCfg`. Type of the configuration object for this widget.
     */
    export class SelectManyMenu<TCfg extends SelectManyMenuCfg = SelectManyMenuCfg> extends PrimeFaces.widget.SelectListbox<TCfg> {
        /**
         * Whether a checkbox of the select menu was clicked. Reset after the list
         * box was clicked.
         */
        checkboxClick?: boolean;
        /**
         * DOM elements of the checkboxes, if checkboxes are enabled.
         */
        checkboxes?: JQuery;
        /**
         * The last clicked item, used for selecting a range by holding down the shift
         * key.
         */
        cursorItem?: JQuery;
        /**
         * Sets up all event listeners for this widget instance.
         * @override
         */
        protected override bindEvents(): void;
        /**
         * Focus the item.
         * @param item The item to focus.
         */
        focus(item: JQuery): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Handle keyboard events.
         * @param event the DOM onKeyDown event
         */
        private onKeyDown(event: Event): void;
        /**
         * Selects all available items of this select many menu.
         */
        selectAll(): void;
        /**
         * Select the given checkbox. Does not unselect any other checkboxes that are currently selected.
         * @param chkbox A CHECKBOX element to select.
         */
        selectCheckbox(chkbox: JQuery): void;
        /**
         * Select the given item as the currently selected option. Does not unselect other items that are already selected.
         * @override
         * @param item An OPTION element to set as the selected element.
         */
        override selectItem(item: JQuery): void;
        /**
         * Unselects all items that are currently selected.
         * @override
         */
        override unselectAll(): void;
        /**
         * Unselects the given checkbox. Does not modify any other checkboxes.
         * @param chkbox A CHECKBOX element to unselect.
         */
        unselectCheckbox(chkbox: JQuery): void;
        /**
         * Unselect the given items. Does not change other selected items.
         * @override
         * @param item Item to unselect.
         */
        override unselectItem(item: JQuery): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectManyMenu| SelectManyMenu widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectManyMenuCfg extends PrimeFaces.widget.SelectListboxCfg {
        /**
         * Whether the select many menu is initially disabled.
         */
        disabled: boolean;
        /**
         * Whether the meta key (`SHIFT` or `CTRL`) must be held down to select multiple
         * items.
         */
        metaKeySelection: boolean;
        /**
         * When set to `true`, a checkbox is displayed next to each item.
         */
        showCheckbox: boolean;
    }
}
declare namespace PrimeFaces.widget.SelectOneButton {
    /**
     * Callback that is invoked when the value of this widget has
     * changed. See also {@link SelectOneButtonCfg.change}.
     */
    export type ChangeCallback = (this: PrimeFaces.widget.SelectOneButton) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectOneButton Widget__
     *
     * SelectOneButton is an input component to do a single select.
     * @typeparam TCfg Defaults to `SelectOneButtonCfg`. Type of the configuration object for this widget.
     */
    export class SelectOneButton<TCfg extends SelectOneButtonCfg = SelectOneButtonCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the button optios the user can select.
         */
        buttons: JQuery;
        /**
         * The DOM element for the hidden input fields storing the value of this widget.
         */
        inputs: JQuery;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Disables this input components so that the user cannot select an option anymore.
         */
        disable(): void;
        /**
         * Enables this input components so that the user can select an option.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Selects the given button option. If another button option is selected already, it will be unselected.
         * @param button A button of this widget to select.
         */
        select(button: JQuery): void;
        /**
         * Trigger the change behavior when the value of this widget has changed.
         */
        private triggerChange(): void;
        /**
         * Unselects the given button option.
         * @param button A button of this widget to unselect.
         */
        unselect(button: JQuery): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectOneButton| SelectOneButton widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectOneButtonCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Callback that is invoked when the value of this
         * widget has changed.
         */
        change: PrimeFaces.widget.SelectOneButton.ChangeCallback;
        /**
         * Whether selection can be cleared.
         */
        unselectable: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectOneListbox Widget__
     *
     * SelectOneListbox is an extended version of the standard selectOneListbox component.
     * @typeparam TCfg Defaults to `SelectOneListboxCfg`. Type of the configuration object for this widget.
     */
    export class SelectOneListbox<TCfg extends SelectOneListboxCfg = SelectOneListboxCfg> extends PrimeFaces.widget.SelectListbox<TCfg> {
        /**
         * The DOM element for the button select item currently focused.
         */
        focusedItem: JQuery;
        /**
         * The DOM element for the hidden input field storing the selected item.
         */
        input: JQuery;
        /**
         * The DOM element for the select items the user can select.
         */
        items: JQuery;
        /**
         * Sets up all event listeners for this widget instance.
         * @override
         */
        protected override bindEvents(): void;
        /**
         * Sets up the event listeners for keyboard related events.
         */
        private bindKeyEvents(): void;
        /**
         * Removes the outline around the listbox with the select options.
         */
        private removeOutline(): void;
        /**
         * Select the item.
         * @param item The item to focus.
         */
        select(item: JQuery): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectOneListbox| SelectOneListbox widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectOneListboxCfg extends PrimeFaces.widget.SelectListboxCfg {
        /**
         * Whether this widget is currently disabled.
         */
        disabled: boolean;
    }
}
declare namespace PrimeFaces.widget.SelectOneMenu {
    /**
     * Duration of toggle animation of the
     * overlay panel.
     */
    export type EffectSpeed = "slow" | "normal" | "fast";
}
declare namespace PrimeFaces.widget.SelectOneMenu {
    /**
     * A function for filtering the options of a select list box.
     */
    export type FilterFunction =
    /**
     * @param itemLabel The label of the currently selected text.
     * @param filterValue The value to search for.
     * @return `true` if the item label matches the filter value,
     * or `false` otherwise.
     */
    (itemLabel: string, filterValue: string) => boolean;
}
declare namespace PrimeFaces.widget.SelectOneMenu {
    /**
     * Available modes for filtering the options of a select list box. When `custom` is set, a `filterFunction` must be
     * specified.
     */
    export type FilterMatchMode = "startsWith" | "contains" | "endsWith" | "custom";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectOneMenu Widget__
     *
     * SelectOneMenu is an extended version of the standard SelectOneMenu.
     * @typeparam TCfg Defaults to `SelectOneMenuCfg`. Type of the configuration object for this widget.
     */
    export class SelectOneMenu<TCfg extends SelectOneMenuCfg = SelectOneMenuCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * Whether the value of this widget was changed from its original value.
         */
        changed: boolean;
        /**
         * The DOM element for the input field that lets the user enter a custom value which does not
         * have to match one of the available options.
         */
        customInput: JQuery;
        /**
         * The custom value that was entered by the user which does not have to match one the
         * available options.
         */
        customInputVal: string;
        /**
         * Whether this widget is currently disabled.
         */
        disabled: boolean;
        /**
         * The DOM element for the input field that lets the user enter a search term to filter the
         * list of available options.
         */
        filterInput: JQuery;
        /**
         * The filter that was selected and is
         * currently used.
         */
        filterMatcher: PrimeFaces.widget.SelectOneMenu.FilterFunction;
        /**
         * Map between the available filter types and the filter implementation.
         */
        filterMatchers: Record<PrimeFaces.widget.SelectOneMenu.FilterMatchMode, PrimeFaces.widget.SelectOneMenu.FilterFunction>;
        /**
         * The hidden input that can be focused via the tab key etc. (only used with editable="true")
         */
        focusInput: JQuery;
        /**
         * Is this component wrapped in a float label.
         */
        hasFloatLabel: boolean;
        /**
         * Unbind callback for the hide overlay handler.
         */
        hideOverlayHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM element for the hidden input with the current value.
         */
        input: JQuery;
        /**
         * Whether the contents of the overlay panel were loaded.
         */
        isDynamicLoaded: boolean;
        /**
         * Whether the current process is handling the tab key.
         */
        isTabbing: boolean;
        /**
         * The DOM elements for the available selectable options.
         */
        items?: JQuery;
        /**
         * The DOM element for the container with the available selectable options.
         */
        itemsContainer?: JQuery;
        /**
         * The DOM element for the wrapper with the container with the available selectable options.
         */
        itemsWrapper: JQuery;
        /**
         * The DOM element used as target for keyboard - events.
         */
        keyboardTarget: JQuery;
        /**
         * The DOM element for the label indicating the currently selected option.
         */
        label: JQuery;
        /**
         * The DOM element for the label connected to he SelectOneMenu.
         * up.
         */
        labeledBy: JQuery;
        /**
         * The DOM element for the icon for bringing up the overlay panel.
         */
        menuIcon: JQuery;
        /**
         * The number of option groups.
         */
        optGroupsSize: number;
        /**
         * The DOM elements for the available selectable options.
         */
        options: JQuery;
        /**
         * The DOM element for the overlay panel with the available selectable options.
         */
        panel: JQuery;
        /**
         * ID of the DOM element for the overlay panel with the available selectable options.
         */
        panelId: JQuery;
        /**
         * The adjusted width of the overlay panel.
         */
        panelWidthAdjusted: number;
        /**
         * The DOM element for the selected option that is shown before the overlay panel is brought.
         */
        preShowValue: JQuery;
        /**
         * Whether an AJAX request for the autocompletion items is currently in progress.
         */
        querying?: boolean;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * ID of the timeout for the delay of the filter input in the overlay panel.
         */
        searchTimer: number;
        /**
         * Letters typed for selection. (#4682, only used with editable="false")
         */
        searchValue: number;
        /**
         * Handler for CSS transitions used by this widget.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * The DOM elements for the buttons that can trigger (hide or show) the overlay panel with the
         * available selectable options.
         */
        triggers: JQuery;
        /**
         * The current value of this select one menu.
         */
        value: string;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Brings up the overlay panel with the available selectable options. Compared this `show`, this does not ensure
         * the the overlay panel is loaded already (when dynamic loading is enabled).
         */
        private _show(): void;
        /**
         * Align the overlay panel with the available selectable options so that is is positioned next to the the button.
         */
        alignPanel(): void;
        /**
         * Adjust the width of the overlay panel.
         */
        private alignPanelWidth(): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Sets up the event listeners for the filter input in the overlay panel.
         */
        private bindFilterEvents(): void;
        /**
         * Sets up the event listeners if this is bound to a floating label.
         */
        private bindFloatLabel(): void;
        /**
         * Sets up the event listeners for the selectable items.
         */
        private bindItemEvents(): void;
        /**
         * Sets up the event listeners for all keyboard related events other than the overlay panel, such as pressing space
         * to bring up the overlay panel.
         */
        private bindKeyEvents(): void;
        /**
         * Sets up all panel event listeners
         */
        private bindPanelEvents(): void;
        /**
         * Removes focus from this widget.
         */
        blur(): void;
        /**
         * Invokes the given method after making sure that the overlay panel was loaded (in case dynamic mode is enabled).
         * @param handleMethod Callback method to
         * invoke after the dynamic overlay panel was loaded.
         * @param event An event that is passed to the callback.
         */
        private callHandleMethod(handleMethod: (this: PrimeFaces.widget.SelectOneMenu, event: JQuery.TriggeredEvent) => void, event: JQuery.TriggeredEvent): void;
        /**
         * Adjusts the value of the aria attributes for the given selectable option.
         * @param item An option for which to set the aria attributes.
         */
        private changeAriaValue(item: JQuery): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectOneMenu.FilterFunction` that matches the given option when it
         * contains the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the contains the filter value, or `false` otherwise.
         */
        containsFilter(value: string, filter: string): boolean;
        /**
         * Disables this widget so that the user cannot select any option.
         */
        disable(): void;
        /**
         * Loads the overlay panel with the selectable options, if dynamic mode is enabled.
         */
        private dynamicPanelLoad(): void;
        /**
         * Enables this widget so that the user can select an option.
         */
        enable(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectOneMenu.FilterFunction` that matches the given option when it ends
         * with the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the options ends with the filter value, or `false` otherwise.
         */
        endsWithFilter(value: string, filter: string): boolean;
        /**
         * Filters the available options in the overlay panel by the given search value. Note that this does not bring up
         * the overlay panel, use `show` for that.
         * @param value A value against which the available options are matched.
         */
        filter(value: string): void;
        /**
         * Puts focus on this widget.
         */
        focus(): void;
        /**
         * Puts focus on the filter input in the overlay panel.
         * @param timeout Amount of time in milliseconds to wait before attempting to focus the input.
         */
        focusFilter(timeout?: number): void;
        /**
         * Finds the element for the currently select option of this select one menu.
         * @return The DOM element that represents the currently selected option.
         */
        getActiveItem(): JQuery;
        /**
         * Finds the label of the option with the given value.
         * @param value The value of a selectable option.
         * @return The label of the option with the given value.
         */
        private getLabelToDisplay(value: string): string;
        /**
         * Finds the label of the currently selected item, if any.
         * @return The label of the currently selected item. Empty string if none is selected.
         */
        getSelectedLabel(): string;
        /**
         * Finds the value of the currently selected item, if any.
         * @return The value of the currently selected item. Empty string if none is selected.
         */
        getSelectedValue(): string;
        /**
         * Callback for when the enter key was pressed. Brings up the overlay panel or accepts the highlighted option.
         * @param event The event of the keypress.
         */
        private handleEnterKey(event: JQuery.TriggeredEvent): void;
        /**
         * Callback for when the escape key was pressed. Hides the overlay panel.
         * @param event The event of the keypress.
         */
        private handleEscapeKey(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that adjusts the label, invoked when the selected option has changed.
         * @param event The event that triggered the change.
         */
        private handleLabelChange(event: JQuery.TriggeredEvent): void;
        /**
         * Callback for when the space key was pressed. Brings up or hides the overlay panel.
         * @param event The event of the keypress.
         */
        private handleSpaceKey(event: JQuery.TriggeredEvent): void;
        /**
         * Callback for when the tab key was pressed. Selects the next option.
         * @param event The keyboard event for the TAB.
         */
        private handleTabKey(event: JQuery.TriggeredEvent): void;
        /**
         * Fired when the browser viewport is resized or scrolled.  In Mobile environment we don't want to hider the overlay
         * we want to re-align it.  This is because on some mobile browser the popup may force the browser to trigger a
         * resize immediately and close the overlay. See GitHub #7075.
         */
        private handleViewportChange(): void;
        /**
         * Hides the overlay panel with the available selectable options.
         */
        hide(): void;
        /**
         * Highlight the given selectable option.
         * @param item Option to highlight.
         */
        private highlightItem(item: JQuery): void;
        /**
         * Highlights the next option after the currently highlighted option in the overlay panel.
         * @param event The event of the keypress.
         */
        private highlightNext(event: JQuery.TriggeredEvent): void;
        /**
         * Highlights the previous option before the currently highlighted option in the overlay panel.
         * @param event The event of the keypress.
         */
        private highlightPrev(event: JQuery.TriggeredEvent): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Finds and initializes the DOM elements that make up this widget.
         */
        private initContents(): void;
        /**
         * Finds all options that match the given search string.
         * @param text The search string against which to match the options.
         * @return All selectable options that match (contain) the given search string.
         */
        private matchOptions(text: string): JQuery;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Renders panel content based on hidden select.
         * @param initContentsAndBindItemEvents `true` to call {@link initContents} and {@link bindItemEvents}
         * after rendering, `false` otherwise.
         */
        private renderPanelContentFromHiddenSelect(initContentsAndBindItemEvents: boolean): void;
        /**
         * Renders panel HTML code for one select item (group).
         * @param item An option (group) for which to render HTML code.
         * @param isGrouped Indicates whether the item is part of a group.
         * @return The rendered HTML string.
         */
        private renderSelectItem(item: JQuery, isGrouped?: boolean): string;
        /**
         * Renders panel HTML-code for all select items.
         * @param parentItem A parent item (select, optgroup) for which to render HTML code.
         * @param isGrouped Indicated whether the elements of the parent item should be marked as grouped.
         * @return The rendered HTML string.
         */
        private renderSelectItems(parentItem: JQuery, isGrouped?: boolean): string;
        /**
         * Resets the input.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        resetValue(silent?: boolean): void;
        /**
         * Finds the index of the given selectable option.
         * @param item One of the available selectable options.
         * @return The index of the given item.
         */
        resolveItemIndex(item: JQuery): number;
        /**
         * Unselect the selected item, if any, and select the `please select` option.
         */
        revert(): void;
        /**
         * Callback for when the user selects an item with the mouse.
         * @param item The option to select.
         * @param silent `true` to suppress triggering event listeners, or `false` otherwise.
         */
        private selectItem(item: JQuery, silent?: boolean): void;
        /**
         * Selects the option with the given value.
         * @param value Value of the option to select.
         */
        selectValue(value: string): void;
        /**
         * Sets the label text that indicates the currently selected item to the item with the given value.
         * @param value Value of the item that was selected.
         */
        private setLabel(value: string): void;
        /**
         * Sets the querying state.
         * @param state Querying state to set.
         */
        private setQuerying(state: boolean): void;
        /**
         * Finds and stores the filter function which is to be used for filtering the options of this select one menu.
         */
        private setupFilterMatcher(): void;
        /**
         * Brings up the overlay panel with the available selectable options.
         */
        show(): void;
        /**
         * Implementation of a `PrimeFaces.widget.SelectOneMenu.FilterFunction` that matches the given option when it starts
         * with the given search text.
         * @param value Text of an option.
         * @param filter Value of the filter.
         * @return `true` when the text of the options starts with the filter value, or `false` otherwise.
         */
        startsWithFilter(value: string, filter: string): boolean;
        /**
         * Adjust the value of the title attribute to match selected option.
         * @param option The option that was selected.
         */
        private syncTitle(option: JQuery): void;
        /**
         * Triggers the event listeners when the value of this widget changed.
         * @param edited Whether the value was edited by the user. If it was, checks which option is now selected.
         */
        private triggerChange(edited: boolean): void;
        /**
         * Removes some event listeners when this widget was disabled.
         */
        private unbindEvents(): void;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
        /**
         * Handles floating label CSS if wrapped in a floating label.
         * @param input the input
         */
        private updateFloatLabel(input: JQuery | undefined): void;
        /**
         * Updates the style class of the label that indicates the currently selected item.
         * @param add `true` if a placeholder should be displayed, or `false` otherwise.
         */
        updatePlaceholderClass(add: boolean): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectOneMenu| SelectOneMenu widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectOneMenuCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * `true` if the label of the selected item should always be set on the visible
         * input, `false` otherwise.
         */
        alwaysDisplayLabel: boolean;
        /**
         * Appends the overlay to the element defined by search expression. Defaults to the document
         * body.
         */
        appendTo: string;
        /**
         * Calculates a fixed width based on the width of the maximum option label. Possible values: `auto`,
         * `true`, `false`.
         */
        autoWidth: boolean;
        /**
         * Defines if filtering would be case sensitive.
         */
        caseSensitive: boolean;
        /**
         * Defines if dynamic loading is enabled for the element's panel. If the value is `true`,
         * the overlay is not rendered on page load to improve performance.
         */
        dynamic: boolean;
        /**
         * When true, the input field becomes editable.
         */
        editable: boolean;
        /**
         * `true` if the options can be filtered, or `false` otherwise.
         */
        filter: boolean;
        /**
         * A custom filter function that is used
         * when `filterMatchMode` is set to `custom`.
         */
        filterFunction: PrimeFaces.widget.SelectOneMenu.FilterFunction;
        /**
         * Mode of the filter. When set to
         * `custom` a `filterFunction` must be specified.
         */
        filterMatchMode: PrimeFaces.widget.SelectOneMenu.FilterMatchMode;
        /**
         * Defines if filtering would be done using normalized values.
         */
        filterNormalize: boolean;
        /**
         * Initial height of the overlay panel in pixels.
         */
        initialHeight: number;
        /**
         * Text of the label for the input.
         */
        label: string;
        /**
         * Displays label of the element in a custom template. Valid placeholder is `{0}`,
         * which is replaced with the value of the currently selected item.
         */
        labelTemplate: string;
        /**
         * Renders panel content on client.
         */
        renderPanelContentOnClient: boolean;
        /**
         * Updates the title of the component with the description of the selected item.
         */
        syncTooltip: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SelectOneRadio Widget__
     *
     * SelectOneRadio is an extended version of the standard SelectOneRadio with theme integration.
     * @typeparam TCfg Defaults to `SelectOneRadioCfg`. Type of the configuration object for this widget.
     */
    export class SelectOneRadio<TCfg extends SelectOneRadioCfg = SelectOneRadioCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM elements for the active radio icons shown on the UI .
         */
        checkedRadio: JQuery;
        /**
         * The (cloned) DOM elements for the non-disabled hidden input fields of type radio storing
         * the value of this widget.
         */
        enabledInputs: JQuery;
        /**
         * Whether custom is used with a facet.
         */
        facet: boolean;
        /**
         * The (cloned) DOM elements for the hidden input fields of type radio storing the value of this
         * widget.
         */
        inputs: JQuery;
        /**
         * The DOM elements for the label texts of each radio button.
         */
        labels: JQuery;
        /**
         * The DOM elements for the hidden radio input fields of type checkbox storing the value
         * of this widget.
         */
        originalInputs: JQuery;
        /**
         * The DOM elements for the radio icons shown on the UI.
         */
        outputs: JQuery;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Disables a given radio button option of this widget.
         * @param index Index of the radio button option to disable.
         */
        disable(index: number): void;
        /**
         * Enables a given radio button option of this widget.
         * @param index Index of the radio button option to enable.
         */
        enable(index: number): void;
        /**
         * Calls the behavior for when a radio button options was clicked.
         * @param input Radio button input that was clicked.
         * @param event (Click) event that was triggered.
         */
        private fireClickEvent(input: JQuery, event: JQuery.TriggeredEvent): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Selects the given radio button option. If another radio button option is selected already, it will be unselected.
         * @param radio A radio button of this widget to select.
         */
        select(radio: JQuery): void;
        /**
         * Sets aria-checked attribute.
         * @param input of which to set aria-checked attribute.
         * @param checked state to set.
         */
        private setAriaChecked(input: JQuery, checked: boolean): void;
        /**
         * Removes some of the event listeners added by `bindEvents`. Called when this widget is disabled.
         * @param input Radio input element for which to remove the listeners.
         */
        private unbindEvents(input: JQuery): void;
        /**
         * Unselects the given radio button option.
         * @param radio A radio button of this widget to unselect.
         */
        unselect(radio: JQuery): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SelectOneRadio| SelectOneRadio widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SelectOneRadioCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether a custom layout is enabled.
         */
        custom: boolean;
        /**
         * Whether the radio group is readonly.
         */
        readonly: boolean;
        /**
         * Unselectable mode when true clicking a radio again will clear the selection.
         */
        unselectable: boolean;
    }
}
declare namespace PrimeFaces.widget.SplitButton {
    /**
     * A filter function that takes a term and returns whether the
     * search term matches the value.
     */
    export type FilterFunction =
    /**
     * @param value A value to check.
     * @param query A search term against which the value is checked.
     * @return `true` if the search term matches the value, or `false`
     * otherwise.
     */
    (value: string, query: string) => string;
}
declare namespace PrimeFaces.widget.SplitButton {
    /**
     * Available modes for filtering the options of the available buttons actions of a split button. When `custom` is set, a
     * `filterFunction` must be specified.
     */
    export type FilterMatchMode = "startsWith" | "contains" | "endsWith" | "custom";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces SplitButton Widget__
     *
     * SplitButton displays a command by default and additional ones in an overlay.
     * @typeparam TCfg Defaults to `SplitButtonCfg`. Type of the configuration object for this widget.
     */
    export class SplitButton<TCfg extends SplitButtonCfg = SplitButtonCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Number of concurrent active Ajax requests.
         */
        ajaxCount?: number;
        /**
         * The DOM element for the main button.
         */
        button: JQuery;
        /**
         * The DOM element for the filter input field
         */
        filterInput: JQuery;
        /**
         * The current filter function.
         */
        filterMatcher: PrimeFaces.widget.SplitButton.FilterFunction;
        /**
         * A map of all flter functions. The
         * key is the name of the filter function.
         */
        filterMatchers: Record<string, PrimeFaces.widget.SplitButton.FilterFunction>;
        /**
         * Unbind callback for the hide overlay handler.
         */
        hideOverlayHandler?: PrimeFaces.UnbindCallback;
        /**
         * The DOM element for the additional buttons actions.
         */
        menu: JQuery;
        /**
         * The DOM element for the button that triggers the overlay panel with the additional buttons
         * actions.
         */
        menuButton: JQuery;
        /**
         * The prefix shared ny the different IDs of the components of this widget.
         */
        menuId: string;
        /**
         * The DOM element for the container of the additional buttons actions.
         */
        menuitemContainer: JQuery;
        /**
         * The DOM elements for the individual additional button actions.
         */
        menuitems: JQuery;
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * Handler for CSS transitions used by this widget.
         */
        transition?: PrimeFaces.CssTransitionHandler | null;
        /**
         * Align the overlay panel with the additional buttons actions.
         */
        alignPanel(): void;
        /**
         * Sets up all event listeners required by this widget.
         */
        private bindEvents(): void;
        /**
         * Sets up the event listeners for filtering the available buttons actions via a search field.
         */
        private bindFilterEvents(): void;
        /**
         * Sets up all panel event listeners
         */
        private bindPanelEvents(): void;
        /**
         * A filter function that takes a value and a search and returns true if the value contains the search term.
         * @param value Value to be filtered
         * @param filter Filter or search term to apply.
         * @return `true` if the given value contains the search term, or `false` otherwise.
         */
        containsFilter(value: string, filter: string): boolean;
        /**
         * Disables this button so that the user cannot press the button anymore.
         */
        disable(): void;
        /**
         * Enables this button so that the user can press the button.
         */
        enable(): void;
        /**
         * A filter function that takes a value and a search and returns true if the value ends with the search term.
         * @param value Value to be filtered
         * @param filter Filter or search term to apply.
         * @return `true` if the given value ends with the search term, or `false` otherwise.
         */
        endsWithFilter(value: string, filter: string): boolean;
        /**
         * Filters the overlay panel with the additional buttons actions, leaving only the buttons that match the given
         * search term.
         * @param value Search term for filtering.
         */
        filter(value: string): void;
        /**
         * Callback that is invoked when the enter key is pressed. When overlay panel with the additional buttons actions is
         * shown, activates the selected buttons action. Otherwise, opens the overlay panel.
         * @param event Keyboard event of the enter press.
         */
        private handleEnterKey(event: JQuery.TriggeredEvent): void;
        /**
         * Callback that is invoked when the escape key is pressed while the overlay panel with the additional buttons
         * actions is shown. Hides that overlay panel.
         */
        private handleEscapeKey(): void;
        /**
         * Hides the overlay panel with the additional buttons actions.
         */
        private hide(): void;
        /**
         * Highlights the next button action, usually when the user navigates via the keyboard arrows.
         * @param event Keyboard arrow event that caused the next item to be highlighted.
         */
        private highlightNext(event: JQuery.TriggeredEvent): void;
        /**
         * Highlights the previous button action, usually when the user navigates via the keyboard arrows.
         * @param event Keyboard arrow event that caused the previous item to be highlighted.
         */
        private highlightPrev(event: JQuery.TriggeredEvent): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Creates the filter functions for filtering the button actions.
         */
        private setupFilterMatcher(): void;
        /**
         * Shows the overlay panel with the additional buttons actions.
         */
        private show(): void;
        /**
         * A filter function that takes a value and a search and returns true if the value starts with the search term.
         * @param value Value to be filtered
         * @param filter Filter or search term to apply.
         * @return `true` if the given value starts with the search term, or `false` otherwise.
         */
        startsWithFilter(value: string, filter: string): boolean;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  SplitButton| SplitButton widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface SplitButtonCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * The search expression for the element to which the overlay panel should be appended.
         */
        appendTo: string;
        /**
         * Whether this input is currently disabled.
         */
        disabled: boolean;
        /**
         * Whether client side filtering feature is enabled.
         */
        filter: boolean;
        /**
         * Custom JavaScript function for filtering the
         * available split button actions.
         */
        filterFunction: PrimeFaces.widget.SplitButton.FilterFunction;
        /**
         * Defines if the filter should receive focus on overlay popup.
         */
        filterInputAutoFocus: boolean;
        /**
         * Match mode for filtering, how the search
         * term is matched against the items.
         */
        filterMatchMode: PrimeFaces.widget.SplitButton.FilterMatchMode;
        /**
         * Defines if filtering would be done using normalized values.
         */
        filterNormalize: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Galleria Widget__
     *
     * Galleria is a content gallery component.
     * @typeparam TCfg Defaults to `GalleriaCfg`. Type of the configuration object for this widget.
     */
    export class Galleria<TCfg extends GalleriaCfg = GalleriaCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * The DOM element for the caption below the image.
         */
        primeGalleriaWidget: JQuery;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Cleans up deferred render tasks. When you extend this class and override this method, make sure to call
         * `super`.
         * @override
         */
        override destroy(): void;
        /**
         * Hides content on fullscreen mode.
         */
        hide(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Moves to the next content that comes after the currently shown content.
         */
        next(): void;
        /**
         * Moves to the previous content that comes before the currently shown content.
         */
        prev(): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Shows content on fullscreen mode.
         */
        show(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Galleria| Galleria widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface GalleriaCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * Index of the first item.
         */
        activeIndex: number;
        /**
         * Items are displayed with a slideshow in autoPlay mode.
         */
        autoPlay: boolean;
        /**
         * When enabled, item is changed on indicator item's hover.
         */
        changeItemOnIndicatorHover: boolean;
        /**
         * Defines if scrolling would be infinite.
         */
        circular: boolean;
        /**
         * Close icon on fullscreen mode.
         */
        closeIcon: string;
        /**
         * Whether to display the component on fullscreen.
         */
        fullScreen: boolean;
        /**
         * Position of indicators. Valid values are "bottom", "top", "left" and "right".
         */
        indicatorsPosition: string;
        /**
         * Number of items per page.
         */
        numVisible: number;
        /**
         * A model of options for responsive design.
         */
        responsiveOptions: {
            breakpoint: string;
            numVisible: number;
        }[];
        /**
         * Whether to display caption container.
         */
        showCaption: boolean;
        /**
         * Whether to display indicator container.
         */
        showIndicators: boolean;
        /**
         * When enabled, indicator container is displayed on item container.
         */
        showIndicatorsOnItem: boolean;
        /**
         * Whether to display navigation buttons in item container.
         */
        showItemNavigators: boolean;
        /**
         * Whether to display navigation buttons on item container's hover.
         */
        showItemNavigatorsOnHover: boolean;
        /**
         * Whether to display navigation buttons in thumbnail container.
         */
        showThumbnailNavigators: boolean;
        /**
         * Whether to display thumbnail container.
         */
        showThumbnails: boolean;
        /**
         * Position of thumbnails. Valid values are "bottom", "top", "left" and "right".
         */
        thumbnailsPosition: string;
        /**
         * Time in milliseconds to scroll items.
         */
        transitionInterval: number;
        /**
         * Height of the viewport in vertical layout.
         */
        verticalViewPortHeight: string;
    }
}
declare namespace PrimeFaces.widget.GMap {
    /**
     * Javascript callback to execute when a point on map is clicked.
     * See also {@link GMapCfg.onPointClick}.
     */
    export type OnPointClickCallback =
    /**
     * @param event The
     * mouse or click event that occurred.
     */
    (event: google.maps.MapMouseEvent | google.maps.IconMouseEvent) => void;
}
declare namespace PrimeFaces.widget.GMap {
    /**
     * An overlay shape that extends the shapes and markers as defined by the maps API. Adds an ID property for identifying
     * the shape or marker.
     */
    export type Overlay = (google.maps.Marker | google.maps.Circle | google.maps.Polyline | google.maps.Polygon | google.maps.Rectangle) & PrimeFaces.widget.GMap.IdProviding;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Google Maps Widget__
     *
     * GMap is a map component integrated with Google Maps API V3.
     * @typeparam TCfg Defaults to `GMapCfg`. Type of the configuration object for this widget.
     */
    export class GMap<TCfg extends GMapCfg = GMapCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * The current google maps instance.
         */
        map: google.maps.Map;
        /**
         * The currently selected and active overlay shape.
         */
        selectedOverlay: PrimeFaces.widget.GMap.Overlay;
        /**
         * The spherical portion of the earth's surface that is currently shown in the map
         * viewport.
         */
        viewport: google.maps.LatLngBounds;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Adds an overlay shape (circle, polyline, or polygon) to this map.
         * @param overlay Overlay shape to add to this map.
         */
        private addOverlay(overlay: PrimeFaces.widget.GMap.Overlay): void;
        /**
         * Adds all overlay shapes (circle, polyline, or polygon) to this map.
         * @param overlays A list of overlay shapes to add to this map.
         */
        addOverlays(overlays: PrimeFaces.widget.GMap.Overlay[]): void;
        /**
         * Triggers a resize event and reapplies the current zoom level, redrawing the map. Useful when the browser viewport
         * was resized etc.
         */
        checkResize(): void;
        /**
         * Adds the overlay for a circle shape.
         */
        private configureCircles(): void;
        /**
         * Adds some event listeners for click events and sets up some behaviors.
         */
        private configureEventListeners(): void;
        /**
         * Adds and sets up all configured markers for the gmap.
         */
        private configureMarkers(): void;
        /**
         * Sets up the event listeners for when a point on the map was selected.
         */
        private configurePointSelectListener(): void;
        /**
         * Adds the overlay for a polygonal shape.
         */
        private configurePolygons(): void;
        /**
         * Adds the overlay for a polyline shape.
         */
        private configurePolylines(): void;
        /**
         * Adds the overlay for a rectangular shape.
         */
        private configureRectangles(): void;
        /**
         * Sets up the event listeners for when the state of this map has changed.
         */
        private configureStateChangeListener(): void;
        /**
         * Adjusts (zooms out) the viewport of this map so that it fully shows the given shape.
         * @param overlay A shape for which to adjust the viewport.
         */
        private extendView(overlay: PrimeFaces.widget.GMap.Overlay): void;
        /**
         * Calls the behavior for when a marker was dragged.
         * @param event Event that occurred.
         * @param marker The marker that was dragged.
         */
        private fireMarkerDragEvent(event: google.maps.MapMouseEvent | google.maps.IconMouseEvent, marker: google.maps.MarkerOptions): void;
        /**
         * Triggers the behavior for when an overlay shape was selected.
         * @param event The event that occurred.
         * @param overlay The shape that was selected.
         * @param clickCount whether it was single or double click
         */
        private fireOverlaySelectEvent(event: google.maps.MapMouseEvent | google.maps.IconMouseEvent, overlay: PrimeFaces.widget.GMap.Overlay, clickCount: number): void;
        /**
         * Triggers the behavior for when a point on the map was selected.
         * @param event The event that triggered the point selection.
         * @param clickCount whether it was single or double click
         */
        private firePointSelectEvent(event: google.maps.MapMouseEvent | google.maps.IconMouseEvent, clickCount: number): void;
        /**
         * Triggers the behavior for when the state of this map has changed.
         * @param event The event that triggered the state change.
         */
        private fireStateChangeEvent(event: never): void;
        /**
         * Sets the map viewport to contain the given bounds.
         * @param bounds The new bounds
         * @param padding Optional padding around the bounds.
         * @see https://developers.google.com/maps/documentation/javascript/reference/map?hl=uk#Map.fitBounds
         */
        fitBounds(bounds: google.maps.LatLngBounds | google.maps.LatLngBoundsLiteral, padding?: number | google.maps.Padding): void;
        /**
         * Finds the geocode for the given address and calls the server-side `geocode` behavior, if such a behavior exists.
         * Use `<p:ajax event="geocode" listener="#{geocodeView.onGeocode}" update="@this" />` on the component to define a
         * behavior.
         * @param address Address for which to find a geocode.
         */
        geocode(address: string): void;
        /**
         * The info window that can be displayed to provide detailed information when a marker is selected.
         * @return The current info window instance, if any exists.
         */
        getInfoWindow(): google.maps.InfoWindow | undefined;
        /**
         * Returns the current google maps instance.
         * @return The current map instance.
         */
        getMap(): google.maps.Map;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Writes the given HTML content into the info window.
         * @param content HTML content for the info window.
         */
        private loadWindow(content: string): void;
        /**
         * Loads the contents of the info window from the server and open the info window.
         * @param responseXML The XML that was returned by the AJAX request made to fetch the contents of the
         * info window.
         * @return `true` if the info window load was initiated successfully, or `false` otherwise.
         */
        private openWindow(responseXML: XMLDocument): boolean;
        /**
         * Attempts to find an address for the given lattitude and longitude, and calls the `reverseGeocode` behavior with
         * the result. Use `<p:ajax event="reverseGeocode" listener="#{geocodeView.onReverseGeocode}" update="@this" />` on
         * the component to define a behavior.
         * @param lat Latitude to look up, specified in degrees within the range `[-90, 90]`.
         * @param lng Longitude to look up, specified in degrees within the range `[-180, 180]`.
         */
        reverseGeocode(lat: number, lng: number): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  GMap| GMap widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface GMapCfg extends PrimeFaces.widget.DeferredWidgetCfg, google.maps.MapOptions {
        /**
         * List of overlay circular shapes added
         * to this map.
         */
        circles: (google.maps.Circle & PrimeFaces.widget.GMap.IdProviding)[];
        /**
         * Defines if center and zoom should be calculated automatically to contain all markers on
         * the map.
         */
        fitBounds: boolean;
        /**
         * The current info window instance, if any info window was created yet.
         */
        infoWindow: google.maps.InfoWindow;
        /**
         * HTML string with the contents of the info window, as fetched from the server.
         */
        infoWindowContent: string;
        /**
         * A list of markers to display on the
         * map.
         */
        markers: (google.maps.Marker & PrimeFaces.widget.GMap.IdProviding)[];
        /**
         * Javascript callback to execute when a point on
         * map is clicked.
         */
        onPointClick: PrimeFaces.widget.GMap.OnPointClickCallback;
        /**
         * List of overlay polygonal shapes
         * added to this map.
         */
        polygons: (google.maps.Polygon & PrimeFaces.widget.GMap.IdProviding)[];
        /**
         * List of overlay polyline shapes
         * added to this map.
         */
        polylines: (google.maps.Polyline & PrimeFaces.widget.GMap.IdProviding)[];
        /**
         * List of overlay rectangular
         * shapes added to this map.
         */
        rectangles: (google.maps.Rectangle & PrimeFaces.widget.GMap.IdProviding)[];
    }
}
declare namespace PrimeFaces.widget.GMap {
    /**
     * Interface for objects that provide an ID that uniquely
     * identifies the object.
     */
    export interface IdProviding {
        /**
         * The ID that uniquely identifies this object.
         */
        id: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Growl Widget__
     *
     * Growl is based on the Mac’s growl notification widget and used to display FacesMessages in an overlay.
     * @typeparam TCfg Defaults to `GrowlCfg`. Type of the configuration object for this widget.
     */
    export class Growl<TCfg extends GrowlCfg = GrowlCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Appends a message to the current displayed messages.
         * @param msg A message to translate into an HTML element.
         */
        add(msg: PrimeFaces.FacesMessage): void;
        /**
         * Appends all messages to the current displayed messages.
         * @param msgs The messages to translate into HTML elements.
         */
        addAll(msgs: PrimeFaces.FacesMessage[]): void;
        /**
         * Sets up all event listeners for the given message, such as for closing the message when the close icon clicked.
         * @param message The message for which to set up the event listeners
         */
        private bindEvents(message: JQuery): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Removes all growl messages that are currently displayed.
         */
        removeAll(): void;
        /**
         * Removes the given message from the screen, if it is currently displayed.
         * @param message The message to remove, an HTML element with the class `ui-growl-item-container`.
         */
        removeMessage(message: JQuery): void;
        /**
         * Creates the HTML elements for the given faces message, and adds it to the DOM.
         * @param msg A message to translate into an HTML element.
         */
        private renderMessage(msg: PrimeFaces.FacesMessage): void;
        /**
         * Starts a timeout that removes the given message after a certain delay (as defined by this widget's
         * configuration).
         * @param message The message to remove, an HTML element with the class `ui-growl-item-container`.
         */
        private setRemovalTimeout(message: JQuery): void;
        /**
         * Displays the given messages in the growl window represented by this growl widget.
         * @param msgs Messages to display in this growl
         */
        show(msgs: PrimeFaces.FacesMessage[]): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Growl| Growl widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface GrowlCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * `true` to treat the message's summary and details as plain text, `false` to treat them as
         * an HTML string.
         */
        escape: boolean;
        /**
         * Defines if previous messages should be kept on a new message is shown.
         */
        keepAlive: boolean;
        /**
         * Duration in milliseconds to display non-sticky messages.
         */
        life: number;
        /**
         * List of messages that are shown initially when the widget is loaded or
         * refreshed.
         */
        msgs: PrimeFaces.FacesMessage[];
        /**
         * Specifies if the message should stay instead of hidden automatically.
         */
        sticky: boolean;
    }
}
/**
 * Namespace for the masked input jQuery Hotkeys plugin.
 *
 * jQuery Hotkeys is a plug-in that lets you easily add and remove handlers for keyboard events anywhere in your
 * code supporting almost any key combination. The official webpage is available at
 * https://github.com/tzuryby/jquery.hotkeys
 *
 * Usage is as follows:
 *
 * ```javascript
 * $(expression).on(types.keys, handler);
 * $(expression).off(types.keys, handler);
 *
 * $(document).on('keydown.ctrl_a', fn);
 *
 * // e.g. replace '$' sign with 'EUR'
 * $('input.foo').on('keyup.$', function(){
 *   this.value = this.value.replace('$', 'EUR');
 * });
 * ```
 */
declare namespace JQueryHotkey {
    /**
     * Represents some global data required by the hotkey plugin, such as the version of the library and the default
     * options.
     */
    export interface GlobalSettings {
        /**
         * The version of this library.
         */
        version: string;
        /**
         * A map of keys that may be part of a hotkey combination. The map key is the character code of the key, the
         * mapped value is the name of the key, such as `capslock` or `+`.
         */
        specialKeys: Record<number, string>;
        /**
         * A map between the character a key produces normally and the character it produces while the shift key is
         * pressed. Both the map key and the mapped value is the name of the key, e.g., `;` or `3`.
         */
        shiftNums: Record<string, string>;
        /**
         * A list of values for the `type` attribute of an HTML INPUT element that are allowed to trigger a hotkey.
         * Excludes `button`, `checkbox`, `file`, `hidden`, `image`, `password`, `radio`, `reset`, `search`, `submit`, and
         * `url`,
         */
        textAcceptingInputTypes: HTMLInputElement["type"];
        /**
         * A RegExp that the tag name of an HTML element needs to match for hotkeys to be available for that element.
         * Defaults to TEXTAREA, INPUT, and SELECT elements.
         */
        textInputTypes: RegExp;
        /**
         * The default settings for the hotkey plugin.
         */
        options: HotkeyOptions;
    }
    /**
     * The options for the jQuery Hotkeys plugin you may pass when initializing this plugin on an input element.
     */
    export interface HotkeyOptions {
        /**
         * `true` to exclude all elements with a tag name not matched by `jQuery.hotkeys.textInputTypes`, or `false`
         * otherwise.
         */
        filterInputAcceptingElements: boolean;
        /**
         * `true` to exclude all input elements with a type attribute not set to one of the values contained in
         * `jQuery.hotkeys.textAcceptingInputTypes`, or `false` otherwise.
         */
        filterTextInputs: boolean;
        /**
         * `true` to exclude input elements that are content-editable (i.e. have the `contenteditable` attribute set),
         * or `false` otherwise.
         */
        filterContentEditable: boolean;
    }
}
interface JQueryStatic {
    /**
     * The global settings for the jQuery Hotkeys plugin.
     */
    hotkeys: JQueryHotkey.GlobalSettings;
}
/**
 * Namespace for the jQuery Idle Timer plugin.
 *
 * Fires a custom event when the user is "idle". Idle is defined by not...
 * - moving the mouse
 * - scrolling the mouse wheel
 * - using the keyboard
 *
 * You can use this plugin via `$.idleTimer()` or `$(document).idleTimer()`.
 *
 * See https://github.com/thorst/jquery-idletimer
 */
declare namespace JQueryIdleTimer {
    /**
     * Interface with all the methods supported by the jQuery Idle Timer plugin. These metods are available on JQuery
     * instances as well as on the static JQuery object:
     *
     * ```javascript
     * $.idleTimer(); // shortcut for the below
     * $( document ).idleTimer();
     * ```
     */
    export interface IdleTimerMethods {
        /**
         * There are two ways to instantiate. Either statically, or on an element. Element bound timers will only watch for
         * events inside of them. You may just want page-level activity, in which case you may set up your timers on
         * `document`, `document.documentElement`, and `document.body`.
         *
         * ```javascript
         * $(function() {
         *   // binds to document - shorthand
         *   $.idleTimer();
         *
         *   // binds to document - explicit
         *   $( document ).idleTimer();
         *
         *   // bind to different element
         *   $( "#myTextArea" ).idleTimer();
         * });
         * ```
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return this JQuery instance for chaining.
         */
        idleTimer(element?: Document | HTMLElement, id?: string): this;
        /**
         * There are two ways to instantiate. Either statically, or on an element. Element bound timers will only watch for
         * events inside of them. You may just want page-level activity, in which case you may set up your timers on
         * `document`, `document.documentElement`, and `document.body`.
         *
         * ```javascript
         * $(function() {
         *   // binds to document - shorthand
         *   $.idleTimer(1000);
         *
         *   // binds to document - explicit
         *   $( document ).idleTimer(1000);
         *
         *   // bind to different element
         *   $( "#myTextArea" ).idleTimer(1000);
         * });
         * ```
         * @param idleTimeoutMillis The timeout period in milliseconds.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return this JQuery instance for chaining.
         */
        idleTimer(idleTimeoutMillis: number, element?: Document | HTMLElement, id?: string): this;
        /**
         * There are two ways to instantiate. Either statically, or on an element. Element bound timers will only watch for
         * events inside of them. You may just want page-level activity, in which case you may set up your timers on
         * `document`, `document.documentElement`, and `document.body`.
         *
         * ```javascript
         * $(function() {
         *   // binds to document - shorthand
         *   $.idleTimer({
         *     timeout:10000,
         *     idle:true
         *   });
         *
         *   // binds to document - explicit
         *   $( document ).idleTimer({
         *     timeout:10000,
         *     idle:true
         *   });
         *
         *   // bind to different element
         *   $( "#myTextArea" ).idleTimer({
         *     timeout:10000,
         *     idle:true
         *   });
         * });
         * ```
         * @param options The options for this idle timer. Any options not specified explicitly are set to their default
         * values.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return this JQuery instance for chaining.
         */
        idleTimer(options: Partial<JQueryIdleTimer.IdleTimerOptions>, element?: Document | HTMLElement, id?: string): this;
        /**
         * Stop the timer, removes data, removes event bindings to come back from this you will need to instantiate again.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return this JQuery instance for chaining.
         */
        idleTimer(method: "destroy", element?: Document | HTMLElement, id?: string): this;
        /**
         * Saves the remaining time, and stops the timer.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return this JQuery instance for chaining.
         */
        idleTimer(method: "pause", element?: Document | HTMLElement, id?: string): this;
        /**
         * Starts the timer with remaining time saved when `pause` was called.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return this JQuery instance for chaining.
         */
        idleTimer(method: "resume", element?: Document | HTMLElement, id?: string): this;
        /**
         * Restore initial idle state, and restart the timer.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return this JQuery instance for chaining.
         */
        idleTimer(method: "reset", element?: Document | HTMLElement, id?: string): this;
        /**
         * Get time left until idle. If currently idle, returns 0.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return The time in milliseconds until the user goes idle. If user is already idle, returns `0`.
         */
        idleTimer(method: "getRemainingTime", element?: Document | HTMLElement, id?: string): number;
        /**
         * Get time elapsed since the user went idle or active.
         * - If currently idle, how long the user has been idle.
         * - If currently active, how long the user has been active.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return How long the user has been idle or active, in milliseconds.
         */
        idleTimer(method: "getElapsedTime", element?: Document | HTMLElement, id?: string): number;
        /**
         * Get time the last `active.idleTimer` event was fired.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return A timestamp (milliseconds since 1 January 1970 UTC) for when the most recent time the user went from idle
         * to active.
         */
        idleTimer(method: "getLastActiveTime", element?: Document | HTMLElement, id?: string): number;
        /**
         * Get whether the user is currently idle.
         * @param method The method to be invoked on this idle timer instance.
         * @param element Element to watch, defaults to the document.
         * @param id Unique ID for this idle timer, to support multiple timers on the same page.
         * @return `true` if the user is currently idle, or `false` if the user is currently active.
         */
        idleTimer(method: "isIdle", element?: Document | HTMLElement, id?: string): boolean;
    }
    /**
     * Optional settings that can be specified when creating an idle timer.
     */
    export interface IdleTimerOptions {
        /**
         * List of events that constitute an activity by the user. Defaults to
         *
         * ```
         * mousemove keydown wheel DOMMouseScroll mousewheel mousedown touchstart touchmove MSPointerDown MSPointerMove
         * ```
         */
        events: string;
        /**
         * Indicates if the user is currently idle. Defaults to `false`.
         */
        idle: boolean;
        /**
         * The timeout period in milliseconds. Defaults to `30000`.
         */
        timeout: number;
        /**
         * If set, the use a local storage key to sync activity across browser tabs or windows.
         */
        timerSyncId: string;
    }
    /**
     * The event that is triggered when the user comes back.
     */
    export interface ActiveEvent<
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TDelegateTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TData = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TCurrentTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TTarget = any> extends JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        type: "active";
    }
    /**
     * The event that is triggered when the user goes idle.
     */
    export interface IdleEvent<
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TDelegateTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TData = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TCurrentTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TTarget = any> extends JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        type: "idle";
    }
}
// Extend $.fn.* plugins
interface JQuery extends JQueryIdleTimer.IdleTimerMethods {
}
// Extend $.* static properties on the JQuery object
interface JQueryStatic extends JQueryIdleTimer.IdleTimerMethods {
}
// Extend available event types
declare namespace JQuery {
    interface TypeToTriggeredEventMap<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        /**
         * Triggered by the {@link JQuery.idleTimer|jQuery Idle Timer plugin}.
         *
         * Fired when the user becomes active again.
         *
         * Usually receives the following additional arguments:
         * - `elem` (JQuery): The element that the event was triggered on
         * - `obj` (object): A copy of the internal data used by idleTimer
         * - `triggerevent` (JQuery.TriggeredEvent): The initial event that triggered the element to become active.
         */
        ["active.idleTimer"]: JQueryIdleTimer.ActiveEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.idleTimer|jQuery Idle Timer plugin}.
         *
         * Fired when the user goes idle.
         *
         * Usually receives the following additional arguments:
         * - `elem` (JQuery): The element that the event was triggered on.
         * - `obj` (object): A copy of the internal data used by idleTimer.
         */
        ["idle.idleTimer"]: JQueryIdleTimer.IdleEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
    }
}
declare namespace PrimeFaces.widget.IdleMonitor {
    /**
     * Client side callback to execute when the user comes back. See
     * also {@link IdleMonitorCfg.onactive}.
     */
    export type OnActiveCallback = (this: PrimeFaces.widget.IdleMonitor) => void;
}
declare namespace PrimeFaces.widget.IdleMonitor {
    /**
     * Client side callback to execute when the user goes idle. See
     * also {@link IdleMonitorCfg.onidle}.
     */
    export type OnIdleCallback = (this: PrimeFaces.widget.IdleMonitor) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces IdleMonitor Widget__
     *
     * IdleMonitor watches user actions on a page and notify callbacks in case they go idle or active again.
     * @typeparam TCfg Defaults to `IdleMonitorCfg`. Type of the configuration object for this widget.
     */
    export class IdleMonitor<TCfg extends IdleMonitorCfg = IdleMonitorCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The set-interval ID used for the timer.
         */
        timer: number;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         * @override
         */
        override destroy(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Pauses the monitor, saving the remaining time.
         */
        pause(): void;
        /**
         * Resets the monitor and restarts the timer.
         */
        reset(): void;
        /**
         * Resumes the monitor, with the remaining time as it was saved when the monitor was paused.
         */
        resume(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  IdleMonitor| IdleMonitor widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface IdleMonitorCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * When set to true, the lastAccessed state will be shared between all browser
         * windows for the same servlet context.
         */
        multiWindowSupport: boolean;
        /**
         * Client side callback to execute when the user
         * comes back.
         */
        onactive: PrimeFaces.widget.IdleMonitor.OnActiveCallback;
        /**
         * Client side callback to execute when the user goes
         * idle.
         */
        onidle: PrimeFaces.widget.IdleMonitor.OnIdleCallback;
        /**
         * Time to wait in milliseconds until deciding if the user is idle. Default is 5 minutes.
         */
        timeout: number;
    }
}
/**
 * [JuxtaposeJS](https://juxtapose.knightlab.com) is a simple, open source tool for creating before/after image sliders.
 * Just provide two image URLs and Juxtapose will do the rest of the work for you. Below are instructions for
 * implementing Juxtapose with HTML and Javascript but we also have a tool that lets you make a slider without needing
 * to know any code.
 *
 * Used by the `ImageCompare` widget.
 */
declare namespace juxtapose {
    /**
     * The direction in which the slider should move.
     */
    export type SlideMode = "horizontal" | "vertical";
    /**
     * Represents an image to be used for the comparison. Must contain at least the `src` property.
     */
    export interface ImageSpecification {
        /**
         * URL to the image.
         */
        src: string;
        /**
         * Optional label text for the image.
         */
        label?: string;
        /**
         * Optional credit text for the image.
         */
        credit?: string;
    }
    /**
     * Represents an image with some metadata that is displayed in the image slider.
     */
    export interface Graphic {
        /**
         * Credit text for the image. `false` when there is none.
         */
        credit: string | false;
        /**
         * Image element of this image.
         */
        image: HTMLImageElement;
        /**
         * Whether this image was loaded.
         */
        loaded: boolean;
        /**
         * Label text for the image. `false` when there is none.
         */
        label: string | false;
    }
    /**
     * Represents the configuration that may be specified when a new image slider is created.
     */
    export interface JXSliderConfiguration {
        /**
         * Whether the slider should be animated to make it look smoother.
         */
        animate: boolean;
        /**
         * Instance of the image handler for the left image.
         */
        imgAfter: Graphic;
        /**
         * Instance of the image handler for the right image.
         */
        imgBefore: Graphic;
        /**
         * The direction in which the slider should move. Defaults to `horizontal`.
         */
        mode: SlideMode;
        /**
         * Whether the specified labels should be shown on the images.
         */
        showLabels: boolean;
        /**
         * Whether the specified credit should be shown on the images.
         */
        showCredits: boolean;
        /**
         * Initial position of the slider, as a percentage, eg. `50%`.
         */
        startingPosition: string;
        /**
         * Whether the image slider should be responsive to support different screen sizes.
         */
        makeResponsive: boolean;
    }
    /**
     * A list of instantiated sliders created on the current page. Can be used to access instances when you did not
     * save them.
     */
    export const sliders: JXSlider[];
    /**
     * Represents an instantiated image slider.
     */
    export class JXSlider {
        /**
         * Container for the slider control (`.jx-control`).
         */
        control: JQuery;
        /**
         * The draggable slider element (`.jx-controller`).
         */
        controller: JQuery;
        /**
         * Container element with controller and the left and right arrow.
         */
        handler: JQuery;
        /**
         * Element for the left slider arrow (`.jx-arrow`).
         */
        leftArrow: JQuery;
        /**
         * Container element with the left image (`.jx-image`).
         */
        leftImage: JQuery;
        /**
         * Current configuration of this slider.
         */
        options: JXSliderConfiguration;
        /**
         * Element for the right slider arrow (`.jx-arrow`).
         */
        rightArrow: JQuery;
        /**
         * Container element with the right image (`.jx-image`).
         */
        rightImage: JQuery;
        /**
         * Container element for the image slider component (`.jx-slider`).
         */
        slider: JQuery;
        /**
         * Current position of the image slider, eg. `50.00%`.
         */
        sliderPosition: string;
        /**
         * Wrapper element with the image slider.
         */
        wrapper: JQuery;
        /**
         * @param selector CSS selector for the element you want to turn into a slider.
         * @param leftAndRightImage The two images to be shown in the comparison.
         * @param options Lets you set additional options for the image slider.
         */
        constructor(elementSelector: string, leftAndRightImage: [
            ImageSpecification,
            ImageSpecification
        ], options: Partial<JXSliderConfiguration>);
        /**
         * Display the given label text for the left or right image. Adds to the exiting label(s).
         * @param image The image for which to display the label. Should be either the `leftImage` or `rightImage`
         * property of this image slider instance.
         * @param labelText Label text to display.
         */
        displayLabel(image: string, labelText: string): void;
        /**
         * Display the given label text for the left or right image. Adds to the existing credits
         * @param image The image for which to display the credits. Should be either the `leftImage` or `rightImage`
         * property of this image slider instance.
         * @param creditsText Label text to display.
         */
        displayLabel(image: string, creditsText: string): void;
        /**
         * Finds the current position of this slider.
         * @return The current position of this slider, eg. `50.00%`.
         */
        getPosition(): string;
        /**
         * Moves this slider to the specified position, optionally animating the move.
         * @param percentage A number in the range `[0...100]`. May also be a string in the format `50.00%` or `50`.
         * Where you want to set the handle, relative to the left side of the slider.
         * @param animate `true` to animate the transition, `false` to move this slider immediately.
         */
        updateSlider(percentage: number | string, animate?: boolean): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces ImageCompare Widget__
     *
     * ImageCompare provides a user interface to compare two images. Uses the `ImageCompare` library. To access an instance
     * of the image slider programatically, you can use the list of instantiated sliders:
     *
     * ```javascript
     * const widget = PF("myImageCompareWidget");
     * const slider = juxtapose.sliders.filter(slider => slider.wrapper === a.getJQ().get(0))[0];
     *
     * // Smoothly move the slider to the right.
     * slider.updateSlider("80%", true);
     * ```
     * @typeparam TCfg Defaults to `ImageCompareCfg`. Type of the configuration object for this widget.
     */
    export class ImageCompare<TCfg extends ImageCompareCfg = ImageCompareCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  ImageCompare| ImageCompare widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ImageCompareCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * URL of the image to the left
         */
        leftImage: string;
        /**
         * URL of the image to the right
         */
        rightImage: string;
        /**
         * Whether image links are shown completely or abbreviated.
         */
        showFullLinks: boolean;
    }
}
/**
 * Namespace for Cropper jQuery plugin It is used by the {@link PrimeFaces.widget.ImageCropper} widget.
 *
 * This namespace contains several required interfaces and types.
 */
declare namespace JQueryCropper {
    /**
     * Possible actions for the `cropstart` event.
     *
     * - `crop`: create a new crop box
     * - `move`: move the canvas (image wrapper)
     * - `zoom`: zoom in / out the canvas (image wrapper) by touch.
     * - `e`: resize the east side of the crop box
     * - `w`: resize the west side of the crop box
     * - `s`: resize the south side of the crop box
     * - `n`: resize the north side of the crop box
     * - `se`: resize the southeast side of the crop box
     * - `sw`: resize the southwest side of the crop box
     * - `ne`: resize the northeast side of the crop box
     * - `nw`: resize the northwest side of the crop box
     * - `all`: move the crop box (all directions)
     */
    export type CropAction = "crop" | "move" | "zoom" | "e" | "w" | "s" | "n" | "se" | "sw" | "ne" | "nw" | "all";
    /**
     * Details about the `cropstart`, `cropmove`, and `cropend` event.
     */
    export interface CropPhaseDetail {
        /**
         * The original event that occurred and triggered the crop to start, continue, or end.
         */
        originalEvent: MouseEvent | TouchEvent | PointerEvent;
        /**
         * The type of crop action that started.
         */
        action: CropAction;
    }
    /**
     * Details about the `zoom` event.
     */
    export interface ZoomDetail {
        /**
         * The original event that occurred and triggered the zoom.
         */
        originalEvent: WheelEvent | TouchEvent;
        /**
         * The old (current) ratio of the canvas
         */
        oldRatio: number;
        /**
         * The new (next) ratio of the canvas (`canvasData.width` / `canvasData.naturalWidth`)
         */
        ratio: number;
    }
    /**
     * Event that is fired when the crop box starts to change, keeps changing, or stopped changing.
     */
    export interface CropPhaseEvent<K extends "cropstart" | "cropmove" | "cropend",
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TDelegateTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TData = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TCurrentTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TTarget = any> extends JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        type: K;
        originalEvent: CustomEvent<CropPhaseDetail>;
    }
    /**
     * This event fires when the canvas (image wrapper) or the crop box changed.
     */
    export interface CropEvent<
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TDelegateTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TData = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TCurrentTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TTarget = any> extends JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        type: "crop";
        originalEvent: CustomEvent<Cropper.Data>;
    }
    /**
     * This event fires when a cropper instance starts to zoom in or zoom out its canvas (image wrapper).
     */
    export interface ZoomEvent<
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TDelegateTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TData = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TCurrentTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TTarget = any> extends JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        type: "zoom";
        originalEvent: CustomEvent<ZoomDetail>;
    }
}
interface JQuery {
    /**
     * Creates a new image {@link Cropper} on the current elements.
     * @param options Optional settings for the image cropper.
     * @return this jQuery instance for chaining.
     */
    cropper(options?: Cropper.Options): this;
    /**
     * Invokes a method on the {@link Cropper} instance and returns the result.
     * @typeparam K Name of a method that exists on the {@link Cropper} instance.
     * @param method Name of a method that exists on the {@link Cropper} instance.
     * @param args Arguments required by the given cropper method.
     * @return The value as it returned by the cropper instance method.
     */
    cropper<K extends PrimeFaces.MatchingKeys<Cropper, (...args: never[]) => void>>(method: K, ...args: Parameters<Cropper[K]>): PrimeFaces.ToJQueryUIWidgetReturnType<Cropper, ReturnType<Cropper[K]>, this>;
}
declare namespace JQuery {
    interface TypeToTriggeredEventMap<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        /**
         * Triggered by the {@link JQuery.cropper|JQuery Cropper plugin}.
         *
         * This event fires when the canvas (image wrapper) or the crop box starts to change.
         */
        cropstart: JQueryCropper.CropPhaseEvent<"cropstart", TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.cropper|JQuery Cropper plugin}.
         *
         * This event fires when the canvas (image wrapper) or the crop box is changing.
         */
        cropmove: JQueryCropper.CropPhaseEvent<"cropmove", TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.cropper|JQuery Cropper plugin}.
         *
         * This event fires when the canvas (image wrapper) or the crop box stops to change.
         */
        cropend: JQueryCropper.CropPhaseEvent<"cropend", TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.cropper|JQuery Cropper plugin}.
         *
         * This event fires when the canvas (image wrapper) or the crop box changed.
         */
        crop: JQueryCropper.CropEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
        /**
         * Triggered by the {@link JQuery.cropper|JQuery Cropper plugin}.
         *
         * This event fires when a cropper instance starts to zoom in or zoom out its canvas (image wrapper).
         */
        zoom: JQueryCropper.ZoomEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces ImageCropper Widget__
     *
     * ImageCropper allows cropping a certain region of an image. A new image is created containing the cropped area and
     * assigned to a `CroppedImage` instanced on the server side. Uses CropperJS - to interact with the image cropper
     * programmatically, use the Cropper JQuery plugin. For example:
     *
     * ```javascript
     * PF("myImageCropperWidget").image.cropper("rotate", 90);
     * ```
     * @typeparam TCfg Defaults to `ImageCropperCfg`. Type of the configuration object for this widget.
     */
    export class ImageCropper<TCfg extends ImageCropperCfg = ImageCropperCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * The current {@link Cropper} instance.
         */
        cropper: Cropper;
        /**
         * Whether the image is currently being cropped.
         */
        cropping?: boolean;
        /**
         * DOM element of the image element to crop. You can use this element to access the {@link Cropper}.
         */
        image: JQuery;
        /**
         * DOM element of the hidden INPUT element that stores the selected crop area.
         */
        jqCoords: JQuery;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Clears the crop box.
         */
        clear(): void;
        /**
         * Cleans up deferred render tasks. When you extend this class and override this method, make sure to call
         * `super`.
         * @override
         */
        override destroy(): void;
        /**
         * Disables (freezes) the cropper.
         */
        disable(): void;
        /**
         * Enables (unfreezes) the cropper.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Callback for when a crop was performed.
         * @param event The crop event that occurred.
         */
        private onCrop(event: JQueryCropper.CropEvent): void;
        /**
         * Reset the image and crop box to their initial states.
         */
        reset(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  ImageCropper|ImageCropper widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ImageCropperCfg extends Cropper.Options, PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * ID of the IMAGE element.
         */
        image: string;
        /**
         * Initial coordinates of the cropper area (x, y, width,
         * height).
         */
        initialCoords: [
            number,
            number,
            number,
            number
        ];
    }
}
/**
 * Namespace for the JQuery Cycle plugin, accessible via `$.fn.cycle`. It is used by the `ImageSwitch` widget.
 *
 * This namespace contains several required interfaces and types.
 */
declare namespace JQueryCycle {
    /**
     * Transition callback for the image switch.
     */
    export type CallbackAfter =
    /**
     * @param currSlideElement Current image before the transition
     * @param nextSlideElement Next image to be shown
     * @param options Current image switch options
     * @param forwardFlag `true` if switching the next, `false` if switching to the previous element.
     */
    (this: JQuery, currSlideElement: JQuery, nextSlideElement: JQuery, options: Configuration, forwardFlag: boolean) => void;
    /**
     * Transition callback for the image switch
     */
    export type CallbackBefore =
    /**
     * @param currSlideElement Current image before the transition
     * @param nextSlideElement Next image to be shown
     * @param options Current image switch options
     * @param forwardFlag `true` if switching the next, `false` if switching to the previous element.
     */
    (this: JQuery, currSlideElement: JQuery, nextSlideElement: JQuery, options: Configuration, forwardFlag: boolean) => void;
    /**
     * Callback invoked when the slideshow terminates (use with autostop or nowrap options)
     */
    export type CallbackEnd =
    /**
     * @param options Current image switch options
     */
    (options: Configuration) => void;
    /**
     * Function used to control the transition
     */
    export type CallbackFxFn =
    /**
     * @param currSlideElement Current image before the transition
     * @param nextSlideElement Next image to be shown
     * @param options Current image switch options
     * @param afterCallback Callback that should be invoked once the transition to the next element is complete.
     * @param forwardFlag `true` if switching the next, `false` if switching to the previous element.
     */
    (currSlideElement: JQuery, nextSlideElement: JQuery, options: Configuration, afterCallback: CallbackAfter, forwardFlag: boolean) => void;
    /**
     * Callback fn for pager events
     */
    export type CallbackOnPagerEvent =
    /**
     * @param index 0-based index of the current image.
     * @param slideElement Element that need to be slided.
     */
    (index: number, slideElement: JQuery) => void;
    /**
     * Callback when switching to the next or previous image (prev/next events).
     */
    export type CallbackOnPrevNextEvent =
    /**
     * @param isNext `true` if switching to the next image, `false` if switching to the previous image.
     * @param index 0-based index of the current image.
     * @param slideElement Element that need to be slided.
     */
    (isNext: boolean, zeroBasedSlideIndex: number, slideElement: JQuery) => void;
    /**
     * Callback for determining per-slide timeout value
     */
    export type CallbackTimeoutFn =
    /**
     * @param currSlideElement Current image before the transition
     * @param nextSlideElement Next image to be shown
     * @param options Current image switch options
     * @param forwardFlag `true` if switching the next, `false` if switching to the previous element.
     * @return Number in milliseconds for the next timeout.
     */
    (currSlideElement: JQuery, nextSlideElement: JQuery, options: Configuration, forwardFlag: boolean) => number;
    /**
     * Callback fn invoked to update the active pager link (adds/removes activePagerClass style)
     */
    export type CallbackUpdateActivePagerLink =
    /**
     * @param pager The current pager
     * @param currSlideElement Current image before the transition
     * @param activePagerClass The CSS class to be added to active pagers.
     */
    (pager: JQuery, currSlideElement: JQuery, activePagerClass: string) => void;
    /**
     * Represents a two dimensional point
     */
    export interface Point {
        /**
         * The x coordinate of the point in pixels.
         */
        top: number;
        /**
         * The y coordinate of the point in pixels.
         */
        left: number;
    }
    /**
     * Optional settings for the image switch jQuery plugin that can be passed when initializing an image switch
     * instance.
     */
    export interface Configuration {
        /**
         * Class name used for the active pager link
         */
        activePagerClass: string;
        /**
         * Transition callback (this context set to element that was shown)
         */
        after: null | CallbackAfter;
        /**
         * Allows or prevents click event on pager anchors from bubbling
         */
        allowPagerClickBubble: boolean;
        /**
         * An object of CSS properties and values that the animation will move toward. Properties that define how the
         * slide animates in.
         */
        animIn: null | Record<string, string | number>;
        /**
         * An object of CSS properties and values that the animation will move toward. Properties that define how the
         * slide animates out
         */
        animOut: null | Record<string, string | number>;
        /**
         * A number other than `0` to end slideshow after X transitions (where X == slide count)
         */
        autostop: number;
        /**
         * Number of transitions (optionally used with autostop to define X)
         */
        autostopCount: number;
        /**
         * `true` to start slideshow at last slide and move backwards through the stack
         */
        backwards: boolean;
        /**
         * Transition callback (this context set to element to be shown)
         */
        before: null | CallbackBefore;
        /**
         * `true` if clearType corrections should be applied (for IE)
         */
        cleartype: boolean;
        /**
         * Set to true to disable extra cleartype fixing (leave false to force background color setting on slides)
         */
        cleartypeNoBg: boolean;
        /**
         * Resize container to fit largest slide
         */
        containerResize: number;
        /**
         * `true` to start next transition immediately after current one completes
         */
        continuous: number;
        /**
         * Properties that defined the state of the slide after transitioning out
         */
        cssAfter: null | Record<string, string | number>;
        /**
         * Properties that define the initial state of the slide before transitioning in
         */
        cssBefore: null | Record<string, string | number>;
        /**
         * Additional delay (in ms) for first transition (hint: can be negative)
         */
        delay: number;
        /**
         * Easing for "in" transition
         */
        easeIn: null | string;
        /**
         * Easing for "out" transition
         */
        easeOut: null | string;
        /**
         * Easing method for both in and out transitions
         */
        easing: null | string;
        /**
         * Callback invoked when the slideshow terminates (use with autostop or nowrap options)
         */
        end: null | CallbackEnd;
        /**
         * Force fast transitions when triggered manually (via pager or prev/next); value == time in ms
         */
        fastOnEvent: number;
        /**
         * Force slides to fit container
         */
        fit: number;
        /**
         * Name of transition effect (or comma separated names, ex: 'fade,scrollUp,shuffle')
         */
        fx: string;
        /**
         * Function used to control the transition
         */
        fxFn: null | CallbackFxFn;
        /**
         * Container height (if the 'fit' option is true, the slides will be set to this height as well)
         */
        height: string;
        /**
         * Causes manual transition to stop an active transition instead of being ignored
         */
        manualTrump: boolean;
        /**
         * Data attribute that holds the option data for the slideshow
         */
        metaAttr: string;
        /**
         * Selector for element to use as event trigger for next slide
         */
        next: null | string;
        /**
         * `true` to prevent slideshow from wrapping
         */
        nowrap: number;
        /**
         * Callback fn for pager events
         */
        onPagerEvent: null | CallbackOnPagerEvent;
        /**
         * Callback fn for prev/next events
         */
        onPrevNextEvent: null | CallbackOnPrevNextEvent;
        /**
         * Selector for element to use as pager container
         */
        pager: null | string;
        /**
         * Name of event which drives the pager navigation
         */
        pagerEvent: string;
        /**
         * `true` to enable "pause on hover"
         */
        pause: number;
        /**
         * `true` to pause when hovering over pager link
         */
        pauseOnPagerHover: number;
        /**
         * Selector for element to use as event trigger for previous slide
         */
        prev: null | string;
        /**
         * Event which drives the manual transition to the previous or next slide
         */
        prevNextEvent: string;
        /**
         * `true` for random, false for sequence (not applicable to shuffle fx)
         */
        random: number;
        /**
         * Valid when multiple effects are used; true to make the effect sequence random
         */
        randomizeEffects: number;
        /**
         * Requeue the slideshow if any image slides are not yet loaded
         */
        requeueOnImageNotLoaded: boolean;
        /**
         * Delay in milliseconds for requeue.
         */
        requeueTimeout: number;
        /**
         * Causes animations to transition in reverse (for effects that support it such as scrollHorz/scrollVert/shuffle)
         */
        rev: number;
        /**
         * Coordinates for shuffle animation, ex: `{ top:15, left: 200 }`
         */
        shuffle: null | Point;
        /**
         * CSS selector for selecting slides (if something other than all children is required)
         */
        slideExpr: null | string;
        /**
         * Force slide width/height to fixed size before every transition
         */
        slideResize: number;
        /**
         * Speed of the transition in milliseconds (any valid fx speed value)
         */
        speed: number;
        /**
         * Speed of the 'in' transition in milliseconds
         */
        speedIn: null | number;
        /**
         * Speed of the 'out' transition in milliseconds
         */
        speedOut: null | number;
        /**
         * Zero-based index of the first slide to be displayed
         */
        startingSlide: number;
        /**
         * `true` if in/out transitions should occur simultaneously
         */
        sync: number;
        /**
         * Milliseconds between slide transitions (0 to disable auto advance)
         */
        timeout: number;
        /**
         * Callback for determining per-slide timeout value
         */
        timeoutFn: null | CallbackTimeoutFn;
        /**
         * Callback fn invoked to update the active pager link (adds/removes activePagerClass style)
         */
        updateActivePagerLink: null | CallbackUpdateActivePagerLink;
        /**
         * Container width (if the 'fit' option is true, the slides will be set to this width as well)
         */
        width: null | number;
    }
}
interface JQuery {
    cycle(options: Partial<JQueryCycle.Configuration>): this;
    /**
     * Starts slideshow mode.
     * @param method Name of the method to invoke on the image cycle instance.
     * @return this for chaining.
     */
    cycle(method: "resume"): this;
    /**
     * Stops slideshow mode.
     * @param method Name of the method to invoke on the image cycle instance.
     * @return this for chaining.
     */
    cycle(method: "stop"): this;
    /**
     * Stops or starts slideshow mode.
     * @param method Name of the method to invoke on the image cycle instance.
     * @return this for chaining.
     */
    cycle(method: "toggle"): this;
    /**
     * Switches to the next image.
     * @param method Name of the method to invoke on the image cycle instance.
     * @return this for chaining.
     */
    cycle(method: "next"): this;
    /**
     * Pauses slideshow mode.
     * @param method Name of the method to invoke on the image cycle instance.
     * @return this for chaining.
     */
    cycle(method: "pause"): this;
    /**
     * Switches to the previous image.
     * @param method Name of the method to invoke on the image cycle instance.
     * @return this for chaining.
     */
    cycle(method: "prev"): this;
    /**
     * Removes the slideshow from this element.
     * @param method Name of the method to invoke on the image cycle instance.
     * @return this for chaining.
     */
    cycle(method: "destroy"): this;
    /**
     * Switches to the image with given index.
     * @param index 0-based index of the image to switch to.
     * @return this for chaining.
     */
    cycle(index: number): this;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces ImageSwitch Widget__
     *
     * ImageSwitch is an image gallery component with 25+ effects. Use the JQuery cycle plugin. You can also interact with
     * the imageswitch programatically via `$.fn.cycle`:
     *
     * ```javascript
     * const widget = PF("myImageSwitchWidget");
     * widget.getJQ().cycle("resume");
     * ```
     * @typeparam TCfg Defaults to `ImageSwitchCfg`. Type of the configuration object for this widget.
     */
    export class ImageSwitch<TCfg extends ImageSwitchCfg = ImageSwitchCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Switches to the next image.
         */
        next(): void;
        /**
         * Pauses slideshow mode.
         */
        pauseSlideshow(): void;
        /**
         * Switches to the previous image.
         */
        previous(): void;
        /**
         * Starts slideshow mode.
         */
        resumeSlideshow(): void;
        /**
         * Stops slideshow mode.
         */
        stopSlideshow(): void;
        /**
         * Switches to the image with given index.
         * @param index 0-based index of the image to switch to.
         */
        switchTo(index: number): void;
        /**
         * Stops or starts slideshow mode.
         */
        toggleSlideshow(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  ImageSwitch| ImageSwitch widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface ImageSwitchCfg extends JQueryCycle.Configuration, PrimeFaces.widget.BaseWidgetCfg {
    }
}
declare namespace PrimeFaces.widget.Inplace {
    /**
     * Available effect types for the transition
     * between the display and the inline content of the inline widget.
     */
    export type EffectType = "fade" | "none" | "slide";
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Inplace Widget__
     *
     * Inplace provides easy inplace editing and inline content display. Inplace
     * consists of two members, a display element that is the initially clickable
     * label and an inline element that is the hidden content which is displayed
     * when the display element is toggled.
     * @typeparam TCfg Defaults to `InplaceCfg`. Type of the configuration object for this widget.
     */
    export class Inplace<TCfg extends InplaceCfg = InplaceCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element with the container of the label that is shown when the content or inplace
         * editor is hidden.
         */
        content: JQuery;
        /**
         * The DOM element with the container of the content or inplace editor that is shown when this
         * inline widget is toggled.
         */
        display: JQuery;
        /**
         * The DOM element with the inplace editor, if one exists.
         */
        editor: JQuery;
        /**
         * When an inplace editor exists and it is currently active: discard changes that were made and hides the inplace
         * editor.
         * @param e The (click) event which triggered the cancellation. Currently unused.
         */
        cancel(e?: JQuery.TriggeredEvent): void;
        /**
         * Fetches the content element, which is the container element with the inline content or inplace editor.
         * @return The content element with the inplace editor.
         */
        getContent(): JQuery;
        /**
         * Fetches the display element, which is the container with the label or description shown when the inline content
         * is not displayed.
         * @return The display element or label when the editor is not shown.
         */
        getDisplay(): JQuery;
        /**
         * Leaves editing mode and hides the inplace editor.
         */
        hide(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Callback that is invoked when the inline content or inplace editor is shown or hidden. Puts focus on the
         * appropriate element and makes sure the inline content is rendered correctly.
         */
        private postShow(): void;
        /**
         * When an inplace editor exists and it is currently active: saves the content of the editor and hides the inplace
         * editor.
         * @param e The (click) event which triggered the saving. Currently unused.
         */
        save(e?: JQuery.TriggeredEvent): void;
        /**
         * Switches to editing mode and displays the inplace editor.
         */
        show(): void;
        /**
         * Hides the label and shows the inline content or inplace editor; or vice versa.
         * @param elToShow Element to show, either the label or the inplace editor.
         * @param elToHide Element to hide, either the label or the inplace editor.
         */
        private toggle(elToShow: JQuery, elToHide: JQuery): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Inplace| Inplace widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface InplaceCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Whether this inplace widget is disabled. If disabled, switching to the content or
         * inplace editor is not possible.
         */
        disabled: boolean;
        /**
         * `true` to add save and cancel buttons to the inline content, or `false` otherwise. Usually
         * used when the inline content is a form field.
         */
        editor: boolean;
        /**
         * Effect to be used when toggling.
         */
        effect: PrimeFaces.widget.Inplace.EffectType;
        /**
         * Speed of the effect in milliseconds.
         */
        effectSpeed: number;
        /**
         * Name of the client side event to display inline content.
         */
        event: string;
        /**
         * Defines if inplace content is toggleable or not.
         */
        toggleable: boolean;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces InputMask Widget__
     *
     * InputMask forces an input to fit in a defined mask template.
     * @typeparam TCfg Defaults to `InputMaskCfg`. Type of the configuration object for this widget.
     */
    export class InputMask<TCfg extends InputMaskCfg = InputMaskCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Is this component wrapped in a float label.
         */
        hasFloatLabel: boolean;
        /**
         * Applys the mask to the input.
         */
        private applyMask(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Returns the current value of this input field including the mask like "12/31/1999".
         * @return The current value of this input field with mask.
         */
        getValue(): string;
        /**
         * Returns the current value of this input field without the mask like "12311999".
         * @return The current value of this input field without mask.
         */
        getValueUnmasked(): string;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Sets the value of this input field to the given value. If the value does not fit the mask, it is adjusted
         * appropriately.
         * @param value New value to set on this input field
         */
        setValue(value: string): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  InputMask| InputMask widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface InputMaskCfg extends Inputmask.Options, PrimeFaces.widget.BaseWidgetCfg {
        /**
         * The mask template to use.
         */
        mask: string;
    }
}
declare namespace PrimeFaces.widget.InputNumber {
    /**
     * Alias for the AutoNumeric
     * options, required for technical reasons.
     */
    export type AutoNumericOptions = import("autonumeric").Options;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces InputNumber Widget__
     *
     * InputNumber formats input fields with numeric strings. It supports currency symbols, minimum and maximum value,
     * negative numbers, and a lot of round methods.
     * @typeparam TCfg Defaults to `InputNumberCfg`. Type of the configuration object for this widget.
     */
    export class InputNumber<TCfg extends InputNumberCfg = InputNumberCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The current AutoNumeric instance.
         */
        autonumeric: import("autonumeric");
        /**
         * Whether this widget is currently disabled, i.e. whether the user can enter a number.
         */
        disabled: boolean;
        /**
         * The DOM element for the hidden input field with the current value of this widget.
         */
        hiddenInput: JQuery;
        /**
         * The initial, numerical value that is displayed, such as `0.0` or `5.3`.
         */
        initialValue: string;
        /**
         * The DOM element for the visible input field with autoNumeric.
         */
        input: JQuery;
        /**
         * Always `undefined`.
         */
        plugOptArray: undefined;
        /**
         * Clean up this widget and remove events from the DOM.
         */
        private _cleanup(): void;
        /**
         * Binds input listener which fixes a browser autofill issue.
         * See: https://github.com/autoNumeric/autoNumeric/issues/536
         */
        private bindInputEvents(): void;
        /**
         * Wraps the events on the external (visible) input to copy the value to the hidden input.
         * @return The original value of the hidden input.
         */
        private copyValueToHiddenInput(): number;
        /**
         * Will be called after an AJAX request if the widget container will be detached.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. When the element is removed from the DOM by the update, the DOM element is detached from the DOM and
         * this method gets called.
         *
         * Please note that instead of overriding this method, you should consider adding a destroy listener instead
         * via {@link addDestroyListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method just calls all destroy listeners.
         * @override
         */
        override destroy(): void;
        /**
         * Enables this input field, so that the user cannot enter data.
         */
        disable(): void;
        /**
         * Enables this input field, so that the user can enter data.
         */
        enable(): void;
        /**
         * Finds the current value, which is the raw numerical value without any formatting applied.
         * @return The current value of this input number widget, in its string representation according to the
         * configured format.
         */
        getValue(): string;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Sets the value of this input number widget to the given value. Makes sure that the number is formatted correctly.
         * @param value The new value to set. If a number, it will be formatted appropriately. If the
         * empty string, resets the value. Any other string is parsed into a number and then the number is set.
         */
        setValue(value: number | string): void;
        /**
         * Writes the given value to the hidden input field that stores the actual value of this widget.
         * @param value A value to set on the hidden input.
         */
        private setValueToHiddenInput(value: string): void;
        /**
         * Wraps the events on the external (visible) input to copy the value to the hidden input, before calling the
         * callback.
         */
        private wrapEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  InputNumber| InputNumber widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface InputNumberCfg extends PrimeFaces.widget.BaseWidgetCfg, PrimeFaces.widget.InputNumber.AutoNumericOptions {
        /**
         * Whether this widget is initially disabled.
         */
        disabled: boolean;
        /**
         * Always undefined.
         */
        pluginOptions: undefined;
        /**
         * The initial, numerical value that is displayed, such as `0.0` or `5.3`.
         */
        valueToRender: string;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces InputSwitch Widget__
     *
     * InputSwitch is used to select a boolean value.
     * @typeparam TCfg Defaults to `InputSwitchCfg`. Type of the configuration object for this widget.
     */
    export class InputSwitch<TCfg extends InputSwitchCfg = InputSwitchCfg> extends PrimeFaces.widget.DeferredWidget<TCfg> {
        /**
         * The DOM element for the handle that lets the user change the state of this input switch by
         * dragging it.
         */
        handle: JQuery;
        /**
         * The DOM element for the hidden input field storing the current value of this widget.
         */
        input: JQuery;
        /**
         * The DOM element for the container with the elements for displaying the off state.
         */
        offContainer: JQuery;
        /**
         * The DOM element for the label with the text for when this input switch is turned off.
         */
        offLabel: JQuery;
        /**
         * Offset to the left of all switch parts, depends on the handler width.
         */
        offset: number;
        /**
         * The DOM element for the container with the elements for displaying the on state.
         */
        onContainer: JQuery;
        /**
         * The DOM element for the label with the text for when this input switch is turned on.
         */
        onLabel: JQuery;
        /**
         * Sets up all event listeners required by this widget.
         */
        private _bindEvents(): void;
        /**
         * Performs the UI updates for when this input switch is turned on, such as animating the transition.
         */
        private _checkUI(): void;
        /**
         * This render method is called by this deferred widget once the widget container has become visible. You may
         * now proceed with widget initialization.
         *
         * __Must be overridden__, or an error will be thrown.
         * @override
         */
        protected override _render(): void;
        /**
         * Performs the UI updates for when this input switch is turned off, such as animating the transition.
         */
        private _uncheckUI(): void;
        /**
         * Turns this input switch on, if not already switched on.
         */
        check(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Toggles this input switch, i.e. switch it from on to off or from off to on.
         */
        toggle(): void;
        /**
         * Turns this input switch off, if not already switched off.
         */
        uncheck(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  InputSwitch| InputSwitch widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface InputSwitchCfg extends PrimeFaces.widget.DeferredWidgetCfg {
        /**
         * Text for off label
         */
        offLabel: string;
        /**
         * Text for on label
         */
        onLabel: string;
        /**
         * Flag to show the labels or not
         */
        showLabels: boolean;
    }
}
/**
 * Autosize is a small, stand-alone script to automatically adjust textarea height to fit text. Attaches autosize to the
 * given element or elements.
 *
 * See http://www.jacklmoore.com/autosize
 * @param element The TEXTAREA element to which to attach autosize.
 */
declare function autosize<TElement extends autosize.ElementOrElements>(element: TElement): TElement;
/**
 * Autosize is a small, stand-alone script to automatically adjust textarea height to fit text. This contains a few more
 * utility methods for updating or removing autosize.
 *
 * See http://www.jacklmoore.com/autosize
 */
declare namespace autosize {
    /**
     * An element or a list of elements accepted by the autosize functions.
     */
    type ElementOrElements = Element | ArrayLike<Element>;
    /**
     * Removes Autosize and reverts any changes it made to the TEXTAREA element.
     * @param element The TEXTAREA element to which autosize is attached.
     */
    export function destroy<TElement extends ElementOrElements>(element: TElement): TElement;
    /**
     * Triggers the height adjustment for an assigned textarea element.
     *
     * Autosize will automatically adjust the textarea height on keyboard and window resize events.
     *
     * There is no efficient way for Autosize to monitor for when another script has changed the TEXTAREA value or for
     * changes in layout that impact the textarea element.
     * @param element The TEXTAREA element to which autosize is attaached.
     */
    export function update<TElement extends ElementOrElements>(element: TElement): TElement;
}
/**
 * Namespace for the Browser JQueryUI plugin, available as `$.browser`.
 *
 * Contains some additional types and interfaces required for the typings.
 */
declare namespace JQueryBrowser {
    /**
     * Represents the retrieved information about the current browser, such as the type of browser and the OS.
     */
    export interface BrowserInformation {
        /**
         * `true` if the current browser is an Android browser, `false` otherwise.
         */
        android?: boolean;
        /**
         * `true` if the current browser is a BB browser, `false` otherwise.
         */
        bb?: boolean;
        /**
         * `true` if the current browser is a Blackberry browser, `false` otherwise.
         */
        blackberry?: boolean;
        /**
         * `true` if the current browser is a Chrome browser, `false` otherwise.
         */
        chrome?: boolean;
        /**
         * `true` if the current browser is a Chromium OS browser, `false` otherwise.
         */
        cros?: boolean;
        /**
         * `true` if the current browser is a desktop browser, `false` otherwise.
         */
        desktop?: boolean;
        /**
         * `true` if the current browser is an Edge browser, `false` otherwise.
         */
        edge?: boolean;
        /**
         * `true` if the current browser is an IE mobile browser, `false` otherwise.
         */
        iemobile?: boolean;
        /**
         * `true` if the current browser is an iPad browser, `false` otherwise.
         */
        ipad?: boolean;
        /**
         * `true` if the current browser is an iPhone browser, `false` otherwise.
         */
        iphone?: boolean;
        /**
         * `true` if the current browser is an iPod browser, `false` otherwise.
         */
        ipod?: boolean;
        /**
         * `true` if the current browser is a Kindle browser, `false` otherwise.
         */
        kindle?: boolean;
        /**
         * `true` if the current browser is a Linux browser, `false` otherwise.
         */
        linux?: boolean;
        /**
         * `true` if the current browser is a Mac browser, `false` otherwise.
         */
        mac?: boolean;
        /**
         * `true` if the current browser is a mobile browser, `false` otherwise.
         */
        mobile?: boolean;
        /**
         * `true` if the current browser is a Mozilla browser, `false` otherwise.
         */
        mozilla?: boolean;
        /**
         * `true` if the current browser is an MS Edge browser, `false` otherwise.
         */
        msedge?: boolean;
        /**
         * `true` if the current browser is the Internet Explorer browser, `false` otherwise.
         */
        msie?: boolean;
        /**
         * The name of the current browser.
         */
        name: string;
        /**
         * `true` if the current browser is an Opera browser, `false` otherwise.
         */
        opera?: boolean;
        /**
         * `true` if the current browser is an Opr browser, `false` otherwise.
         */
        opr?: boolean;
        /**
         * The current platform (operating system).
         */
        platform: string;
        /**
         * `true` if the current browser is a PlayBook browser, `false` otherwise.
         */
        playbook?: boolean;
        /**
         * `true` if the current browser is an RV browser, `false` otherwise.
         */
        rv?: boolean;
        /**
         * `true` if the current browser is a Safari browser, `false` otherwise.
         */
        safari?: boolean;
        /**
         * `true` if the current browser is a Silk browser, `false` otherwise.
         */
        silk?: boolean;
        /**
         * The version string of the current browser.
         */
        version?: string;
        /**
         * The version number of the current browser.
         */
        versionNumber?: number;
        /**
         * `true` if the current browser is a Webkit browser, `false` otherwise.
         */
        webkit?: boolean;
        /**
         * `true` if the current browser is a Windows browser, `false` otherwise.
         */
        win?: boolean;
        /**
         * `true` if the current browser is a Windows Phone browser, `false` otherwise.
         */
        "windows phone"?: boolean;
    }
    /**
     * Analyzes the given user agent string and returns information about the browser it represents.
     * @param userAgent A user agent string, such as the one in `window.navigator.userAgent`.
     * @return Details about the given browser.
     */
    export interface BrowserInspector extends BrowserInformation {
        uaMatch(userAgent: string): BrowserInformation;
    }
}
interface JQueryStatic {
    /**
     * Contains information about the current browser environment, such as which browser the user is using etc.
     */
    browser: JQueryBrowser.BrowserInspector;
}
/**
 * Contains information about the current browser environment, such as which browser the user is using etc.
 */
declare const jQBrowser: JQueryBrowser.BrowserInspector;
/**
 * Namespace for the caretposition JQueryUI plugin.
 *
 * Contains some additional types and interfaces required for the typings.
 */
declare namespace JQueryCaretposition {
    /**
     * Represents a position of an element.
     */
    export interface CaretPosition {
        /**
         * Position from the top in pixels.
         */
        top: number;
        /**
         * Position from the left in pixels.
         */
        left: number;
    }
}
interface JQuery {
    /**
     * Finds the computed value of a CSS style property.
     * @param styleName Name of a style.
     * @return The value of the given style.
     */
    getComputedStyle(styleName: string): string;
    /**
     * Copies the given CSS style property from this element to the target element.
     * @param target Target element to which to copy the style.
     * @param styleName Name of the CSS style property.
     */
    cloneStyle(target: JQuery | HTMLElement, styleName: string): void;
    /**
     * Copies all CSS style properties from this element to the target element
     * @param target Target element to which to copy the CSS style properties.
     */
    cloneAllStyle(target: JQuery | HTMLElement): void;
    /**
     * Finds the position of the cursor for an INPUT or TEXTAREA element.
     * @return The current position of the cursor, in characters from the first character of the first line.
     */
    getCursorPosition(): number;
    /**
     * Finds the position of the cursor for an INPUT or TEXTAREA element in pixels, relative to the top left corner
     * of the INPUT or TEXTAREA element.
     * @return The current position of the cursor in pixels, relative to the top left of the element.
     */
    getCaretPosition(): JQueryCaretposition.CaretPosition;
}
/**
 * Namespace for the jQuery Mouse Wheel plugin.
 *
 * A jQuery plugin that adds cross-browser mouse wheel support with delta normalization.
 *
 * In order to use the plugin, simply bind the mousewheel event to an element.
 *
 * See https://github.com/jquery/jquery-mousewheel
 */
declare namespace JQueryMousewheel {
    /**
     * The event that is triggered when the mousewheel is rotated.
     */
    export interface MousewheelEvent<
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TDelegateTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TData = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TCurrentTarget = any,
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    TTarget = any> extends JQuery.TriggeredEvent<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        type: "mousewheel";
        /**
         * The normalized horizontal distance the mouse wheel has moved.
         *
         * Multiply the {@link deltaFactor} by {@link  deltaX} or {@link deltaY} to get the scroll distance that the
         * browser has reported.
         */
        deltaX: number;
        /**
         * The normalized vertical distance the mouse wheel has moved.
         *
         * Multiply the {@link deltaFactor} by {@link  deltaX} or {@link deltaY} to get the scroll distance that the
         * browser has reported.
         */
        deltaY: number;
        /**
         * The normalization factor for converting the normalized delta to the absolute delta reported by the browser.
         *
         * Multiply the {@link deltaFactor} by {@link  deltaX} or {@link deltaY} to get the scroll distance that the
         * browser has reported.
         */
        deltaFactor: number;
    }
}
interface JQuery {
    /**
     * Triggers a `mousewheel` event on this element.
     * @return this for chaining.
     */
    mousewheel(): this;
    /**
     * Registers the given event listener for the `mousewheel` event.
     * @param handler Callback to invoke when the event occurs.
     * @return this for chaining.
     */
    mousewheel(handler: JQuery.TypeEventHandler<HTMLElement, undefined, HTMLElement, HTMLElement, "mousewheel">): this;
    /**
     * Removes the given event listener for the `mousewheel` event.
     * @param handler Callback to remove.
     * @return this for chaining.
     */
    unmousewheel(handler: JQuery.TypeEventHandler<HTMLElement, undefined, HTMLElement, HTMLElement, "mousewheel">): this;
}
declare namespace JQuery {
    interface TypeToTriggeredEventMap<TDelegateTarget, TData, TCurrentTarget, TTarget> {
        /**
         * Triggered by the {@link JQuery.idleTimer|jQuery Idle Timer plugin}.
         *
         * Fired when the user becomes active again.
         *
         * Usually receives the following additional arguments:
         * - `elem` (JQuery): The element that the event was triggered on
         * - `obj` (object): A copy of the internal data used by idleTimer
         * - `triggerevent` (JQuery.TriggeredEvent): The initial event that triggered the element to become active.
         */
        ["mousewheel"]: JQueryMousewheel.MousewheelEvent<TDelegateTarget, TData, TCurrentTarget, TTarget>;
    }
}
/**
 * Namespace for the rangy JQueryUI plugin.
 *
 * Contains some additional types and interfaces required for the typings.
 *
 * The rangy JQueryUI plugin adds a few methods to jQuery instances:
 *
 * - {@link JQuery.collapseSelection}
 * - {@link JQuery.deleteSelectedText}
 * - {@link JQuery.deleteText}
 * - {@link JQuery.extractSelectedText}
 * - {@link JQuery.getSelection}
 * - {@link JQuery.insertText}
 * - {@link JQuery.replaceSelectedText}
 * - {@link JQuery.setSelection}
 * - {@link JQuery.surroundSelectedText}
 */
declare namespace JQueryRangy {
    /**
     * Represents a range of text of an element. When no text is selected, the `length` may be `0`.
     */
    export interface Range {
        /**
         * Index of the character that marks the beginning of the range.
         */
        start: number;
        /**
         * Index of the character that marks the beginning of the range, exclusive. This is one character after the
         * last selected character.
         */
        end: number;
        /**
         * Length of the range (`end - start`)
         */
        length: number;
        /**
         * The selected text.
         */
        text: string;
    }
}
interface JQuery {
    /**
     * Returns the selected text of an INPUT or TEXTAREA element.
     * @param element
     * @return The selected range, or `undefined` if this DOM element is not supported.
     */
    getSelection(): JQueryRangy.Range | undefined;
    /**
     * Selects the text at given range of an INPUT or TEXTAREA element.
     * @param startOffset 0-based index pointing to the start of the range to select (inclusive).
     * @param endOffset 0-based index pointing to the end of the range to select (exclusive).
     * @return this for chaining.
     */
    setSelection(startOffset: number, endOffset: number): this;
    /**
     * Unselects all selected text of an INPUT or TEXTAREA element.
     * @return this for chaining.
     */
    collapseSelection(): this;
    /**
     * Deletes the currently selected text of an INPUT or TEXTAREA element.
     * @return this for chaining.
     */
    deleteSelectedText(): this;
    /**
     * Deletes the text at the given range of an INPUT or TEXTAREA element, optionally moving the cursor to the start
     * of the range.
     * @param startOffset 0-based index pointing to the start of the range to select (inclusive).
     * @param endOffset 0-based index pointing to the end of the range to select (exclusive).
     * @param moveSelection `true` to position the cursor at the start of the given range, or `false` otherwise.
     * @return this for chaining.
     */
    deleteText(startOffset: number, endOffset: number, moveSelection?: boolean): this;
    /**
     * Removes and returns the currently selected text of an INPUT or TEXTAREA element.
     * @return The text that was selected. Empty string if no text is selected. `undefined` if this DOM element is not
     * supported.
     */
    extractSelectedText(): string | undefined;
    /**
     * Inserts the text at the given position of an INPUT or TEXTAREA element, optionally moving the cursor to the end
     * if the inserted text.
     * @param text Text to insert.
     * @param index 0-based index where to insert the text.
     * @param moveSelection `true` to move the cursor to the end of the inserted text, or `false` otherwise.
     * @return this for chaining.
     */
    insertText(text: string, index: number, moveSelection?: boolean): this;
    /**
     * Replaces the currently selected text of an INPUT or TEXTAREA element with the given text. When no text is
     * selected, insert the given text at the current cursor position.
     * @param text Text to replace the current selection.
     * @return this for chaining.
     */
    replaceSelectedText(text: string): this;
    /**
     * For an INPUT or TEXTAREA element: Inserts the given prefix at the beginning of the currently selected text, and
     * the given suffix at the end of the currently selected text. When no text is selected, inserts the given prefix
     * and suffix at the current position of the cursor.
     * @param prefix Text to insert at the beginning of the currently selected text.
     * @param suffix Text to insert at the end of the currently selected text.
     * @return this for chaining.
     */
    surroundSelectedText(prefix: string, suffix: string): this;
}
interface JQuery {
    /**
     * Prevents the user from selection any text of this element.
     * @return this for chaining.
     */
    disableSelection(): this;
    /**
     * Allows the user to select text of this element again.
     * @return this for chaining.
     */
    enableSelection(): this;
    /**
     * Gets the z-index of this element.
     * @return The z-index of this element.
     */
    zIndex(): number;
    /**
     * Sets the z-index of this element the given value.
     * @param zIndex The z-index to set.
     * @return The z-index of this element.
     */
    zIndex(zIndex: number): this;
}
// Nothing to document, touch punch simply monkey patches some internal JQueryUI methods.
/**
 * Namespace for the jQuery keypad plugin.
 *
 * This plugin attaches a popup keyboard to a text field for mouse-driven entry or adds an inline keypad in a division or
 * span.
 *
 * See http://keith-wood.name/keypad.html and https://github.com/kbwood/keypad
 */
declare namespace JQueryKeypad {
    /**
     * A function that is called after the keypad is constructed but before it is displayed.
     */
    export type BeforeShowListener =
    /**
     * @param division The division to be displayed
     * @param instance The current keypad instance controlling the keyboard.
     */
    (this: HTMLInputElement | HTMLTextAreaElement, division: JQuery, instance: KeypadInstance) => void;
    /**
     * A function that is called when a key is pressed on the keypad.
     */
    export type KeypressListener =
    /**
     * @param currentKey The value of they key that was pressed.
     * @param inputFieldValue The current value of the input field.
     * @param instance The current keypad instance controlling the keyboard.
     */
    (this: HTMLInputElement | HTMLTextAreaElement, currentKey: string, inputFieldValue: string, instance: KeypadInstance) => void;
    /**
     * A function that is called when the keypad is closed.
     */
    export type CloseListener =
    /**
     * @param inputFieldValue The current value of the input field.
     * @param instance The current keypad instance controlling the keyboard.
     */
    (this: HTMLInputElement | HTMLTextAreaElement, inputFieldValue: string, instance: KeypadInstance) => void;
    /**
     * Optional settings that can be passed when creating a new keypad instance to customize its behavior.
     */
    export interface KeypadSettings {
        /**
         * Control when the keypad is displayed:
         * - `focus` for only on focus of the text field
         * - `button` for only on clicking the trigger button
         * - `both` for both focus and button
         */
        showOn: "focus" | "button" | "both";
        /**
         * The URL of an image to use for the trigger button.
         */
        buttonImage: string;
        /**
         * Set to true to indicate that the trigger image should appear by itself and not on a button.
         */
        buttonImageOnly: boolean;
        /**
         * Set which animation is used to display the keypad. Choose from three standard animations `show`, `fadeIn`, or
         * `slideDown`; or use one of the jQuery UI effects if you include that package. For no animation use the empty
         * string.
         * @since 1.2.3 - use empty string for no animation.
         */
        showAnim: "show" | "fadeIn" | "slideDown" | string;
        /**
         * If you use one of the jQuery UI effects for the animation above, you can supply additional options for that
         * effect via this setting:
         *
         * ```javascript
         * $(selector).keypad({
         *   showAnim: "clip",
         *   showOptions: {
         *     direction: "horizontal"
         *   }
         * });
         * ```
         */
        showOptions: Record<string, unknown> | null;
        /**
         * Control the speed of the show/hide animation with this setting: `slow`, `normal`, `fast`, or a number of
         * milliseconds.
         */
        duration: "slow" | "normal" | "fast" | number;
        /**
         * Text that is to be appended to all affected fields, such as to describe the presence or purpose of the
         * keypad.
         */
        appendText: string;
        /**
         * Set to `true` to add ThemeRoller classes to the keypad to allow the keypad to integrate with other UI
         * components using a theme.
         * @since 1.3.0
         */
        useThemeRoller: boolean;
        /**
         * One popup keypad is shared by all instances, so this setting allows you to apply different CSS styling by
         * adding an extra class to the keypad for each instance.
         */
        keypadClass: string;
        /**
         * Text that is displayed at the top of the keypad. The value may include HTML markup.
         */
        prompt: string;
        /**
         * Set the layout of the keypad by defining the characters present on each row. All alphabetic characters should
         * be in lower case. Make use of the keypad constants to add special features into the layout:
         *
         * - $.keypad.CLOSE - the button to close the keypad
         * - $.keypad.CLEAR - the button to clear the input field
         * - $.keypad.BACK - the button to erase the previous character
         * - $.keypad.SHIFT - the button to toggle between upper/lower case characters
         * - $.keypad.SPACE_BAR - an extended space character
         * - $.keypad.ENTER - the button to add a carriage return
         * - $.keypad.TAB - the button to add a tab character
         * - $.keypad.SPACE - blank space equivalent to one key
         * - $.keypad.HALF_SPACE - blank space equivalent to half a key
         *
         * Use the regional settings to set what is shown for each of these buttons.
         * @since 1.2.0 - added `SPACE_BAR`.
         * @since 1.2.4 - added `ENTER`.
         * @since 1.4.0 - added `TAB`.
         */
        layout: string[];
        /**
         * The character that separates the text content of the keys, used in conjunction with the layout setting.
         *
         * By default it is blank, so each key contains only one character.
         *
         * ```javascript
         * $(selector).keypad({
         *   separator: "|",
         *   layout: [
         *     "ACT|NSW|NT",
         *     "QLD|SA|TAS",
         *     "VIC|WA"
         *   ]
         * });
         * ```
         * @since 1.2.0
         */
        separator: string;
        /**
         * When using an inline keypad you can set this field to an input field to have the keypad update it
         * automatically. The value can be either the element itself, a jQuery wrapper around the element, or the jQuery
         * selector for it.
         *
         * When not given, uses the onKeypress callback instead.
         * @since 1.2.1
         */
        target: HTMLElement | JQuery | string | null;
        /**
         * Set to `true` to indicate that only the keypad can be used to enter text, or to `false` to allow normal
         * entry as well. This option does not apply to inline keypads.
         */
        keypadOnly: boolean;
        /**
         * Set to `true` to indicate that the alphabetic characters in the layout should be randomised for each display.
         * The `isAlphabetic` setting determines which characters are alphabetic.
         */
        randomiseAlphabetic: boolean;
        /**
         * Set to `true` to indicate that the numeric characters in the layout should be randomised for each display. The
         * `isNumeric` setting determines which characters are numeric.
         */
        randomiseNumeric: boolean;
        /**
         * Set to `true` to indicate that the non-alphanumeric characters in the layout should be randomised for each
         * display.
         */
        randomiseOther: boolean;
        /**
         * Set to `true` to indicate that all characters in the layout should be randomised for each display. When
         * `true`, this setting overrides the other randomized settings.
         * @since 1.0.2
         */
        randomiseAll: boolean;
        /**
         * A function that is called after the keypad is constructed but before it is displayed, allowing you to update
         * it. For example, you could add extra buttons that perform tasks outside the scope of the normal keypad.
         *
         * The function receives the division to be displayed and the keypad instance object as parameters, while `this`
         * refers to the text field.
         *
         * ```javascript
         * $(selector).keypad({
         *   beforeShow: function(div, inst) {
         *     $("<button id=\"clickMe\" class=\"keypad-key\">Click me</button>")
         *       .appendTo(div)
         *       .click(function() {
         *         alert("Clicked");
         *       });
         *   }
         * });
         * ```
         * @since 1.2.0
         */
        beforeShow: BeforeShowListener | null;
        /**
         * A function that is called when a key is pressed on the keypad.
         *
         * The function receives the current key value ($.keypad.BS for the Back key, $.keypad.DEL for the Clear key,
         * and the empty string for other control keys), the full field value, and the keypad instance object as
         * parameters, while `this` refers to the text field.
         *
         * Of course, you can still have an `onchange` handler for the input field itself.
         *
         * ```javascript
         * $(selector).keypad({
         *   onKeypress: function(key, value, inst) {
         *     $("#keypress").text(key || " ");
         *     $("#current").text(value);
         *   }
         * });
         * ```
         * @since 1.2.0 - added current key parameter.
         * @since 1.2.1 - added `$.keypad.BS` and `$.keypad.DEL` characters for `Back` and `Clear` keys.
         */
        onKeypress: KeypressListener | null;
        /**
         * A function that is called when the keypad is closed.
         *
         * The function receives the current field value and the keypad instance object as parameters, while `this`
         * refers to the text field.
         *
         * ```javascript
         * $(selector).keypad({
         *   onClose: function(value, inst) {
         *     alert("Closed with value " + value);
         *   }
         * });
         * ```
         */
        onClose: CloseListener | null;
    }
    /**
     * Keypad related constants and utility method available on the `$.keypad` object.
     */
    export interface KeypadGlobals {
        /**
         * The button to close the keypad
         */
        CLOSE: string;
        /**
         * The button to clear the input field
         */
        CLEAR: string;
        /**
         * The button to erase the previous character
         */
        BACK: string;
        /**
         * The button to add a carriage return
         */
        ENTER: string;
        /**
         * Blank space equivalent to half a key
         */
        HALF_SPACE: string;
        /**
         * An extended space character
         */
        SPACE_BAR: string;
        /**
         * The button to toggle between upper/lower case characters
         */
        SHIFT: string;
        /**
         * Blank space equivalent to one key
         */
        SPACE: string;
        /**
         * The button to add a tab character
         */
        TAB: string;
        /**
         * The set of regional settings for the keypad fields. Entries are indexed by the country or region code with
         * the empty string providing the default (English) settings. Each entry is an object with the localized
         * messages. Language packages load new entries into this array and automatically apply them as global defaults.
         *
         * ```html
         * <script type="text/javascript"  src="jquery.keypad-fr.js"></script>
         * ```
         *
         * If necessary, you can then revert to the default language settings with
         *
         * ```javascript
         * $.keypad.setDefaults($.keypad.regionalOptions[""]);
         * ```
         *
         * and apply the language settings to individual fields with
         *
         * ```javascript
         * $("#frenchKeypad").keypad($.keypad.regionalOptions["fr"]);
         * ```
         * @since 2.0.0 - previously called regional.
         */
        regionalOptions: Record<string, Record<string, RegionalSettings>>;
        /**
         * The standard numeric button layout. This is the default value for the `layout` setting.
         * @since 2.0.0
         */
        numericLayout: string[];
        /**
         * The alphabetic character layout for the standard Qwerty keyboard. This is the default value for the
         * `alphabeticLayout` setting.
         */
        qwertyAlphabetic: string[];
        /**
         * The full keyboard layout for the standard Qwerty keyboard. This is the default value for the `fullLayout`
         * setting.
         * @since 1.2.0 - Reordered and space bar added.
         * @since 1.2.4 - Enter added
         * @since 1.4.0 - Tab added
         */
        qwertyLayout: string[];
        /**
         * Default test for English alphabetic characters - accepting "A" to "Z" and "a" to "z". See also the
         * `isAlphabetic` setting.
         * @param character A character to check.
         * @return `true` if the given character is contained in the set of alphabetic characters, or `false` otherwise.
         */
        isAlphabetic(character: string): boolean;
        /**
         * Default test for English numeric characters - accepting '0' to '9'. See also the `isNumeric` setting.
         * @param character A character to check.
         * @return `true` if the given character is contained in the set of numerical characters, or `false` otherwise.
         */
        isNumeric(character: string): boolean;
        /**
         * Convert a character into its upper case form - using the standard JavaScript `toUpperCase` function. See also
         * the `toUpper` setting.
         * @param character A character to convert to upper case.
         * @return The given character, converted to upper case, or the given character itself, if it cannot be
         * converted.
         * @since 1.5.0
         */
        toUpper(character: string): string;
        /**
         * Insert text into an input field at its current cursor position. This replaces text if a selection has been
         * made.
         * @param input An input element into which to insert the given value. May be either an element or a CSS
         * selector.
         * @param value The text to insert.
         */
        insertValue(input: HTMLElement | JQuery | string, value: string): void;
        /**
         * Update the default instance settings to use with all keypad instances.
         * @param settings The new default settings to apply.
         */
        setDefaults(settings: Partial<KeypadSettings>): void;
        /**
         * Define a new action key for use on the keypad. Up to 32 keys may be defined, including the built-in ones.
         *
         * ```javascript
         * $.keypad.addKeyDef("UPPER", "upper", function(inst) {
         *   this.val(this.val().toUpperCase()).focus();
         * });
         * $("#keypad").keypad({
         *   upperText: "U/C",
         *   upperStatus: "Convert to upper case",
         *   layout: [
         *     "abc" + $.keypad.CLOSE,
         *     "def" + $.keypad.CLEAR,
         *     "ghi" + $.keypad.BACK,
         *     "jkl" + $.keypad.UPPER
         *   ]
         * });
         * ```
         * @param id Name of the variable to use to include this key in a layout.
         * @param name Name used for a CSS styling class for the key (`keypad-<name>`) and for finding its display text
         * (using `<name>Text` and `<name>Status`).
         * @param action Code to be run when the key is pressed. it receives the keypad instance as a parameter, while this refers to the attached input field,
         * @param noHighlight `true` to suppress the highlight class when using `ThemeRoller` styling.
         *
         * @since 1.4.0
         */
        addKeyDef(id: string, name: string, action: (this: HTMLInputElement, instance: KeypadInstance) => void, noHighlight?: boolean): void;
    }
    /**
     * These settings comprise the regional settings that may be localised by a language package. They can be overridden
     * for individual instances:
     *
     * ```javascript
     * $(selector).keypad({backText: "BS"});
     * ```
     */
    export interface RegionalSettings {
        /**
         * The layout for alphabetic keyboard characters in this language.
         */
        alphabeticLayout: string[];
        /**
         * The description of the back button's purpose, used in a tool tip.
         */
        backStatus: string;
        /**
         * The text to display for the button to erase the previous character.
         */
        backText: string;
        /**
         * The description of the button's purpose, used in a tool tip.
         */
        buttonStatus: string;
        /**
         * The text to display on a trigger button for the keypad.
         */
        buttonText: string;
        /**
         * The description of the clear button's purpose, used in a tool tip.
         */
        clearStatus: string;
        /**
         * The text to display for the button to clear the text field.
         */
        clearText: string;
        /**
         * The description of the close button's purpose, used in a tool tip.
         */
        closeStatus: string;
        /**
         * The text to display for the button to close the keypad.
         */
        closeText: string;
        /**
         * The description of the enter button's purpose, used in a tool tip.
         * @since 1.2.4
         */
        enterStatus: string;
        /**
         * The text to display for the button to add a carriage return.
         * @since 1.2.4
         */
        enterText: string;
        /**
         * The layout for the full standard keyboard in this language.
         */
        fullLayout: string[];
        /**
         * A function to determine whether or not a character is alphabetic.
         *
         * The character to test is passed as the parameter and a boolean response is expected.
         *
         * The default accepts `A` to `Z` and `a` to `z`.
         * @param character A character to test.
         * @return `true` if the given character is contained in the set of alphabetical characters, or `false`
         * otherwise.
         */
        isAlphabetic(character: string): boolean;
        /**
         * A function to determine whether or not a character is numeric.
         *
         * The character to test is passed as the parameter and a boolean response is expected.
         *
         * The default accepts `0` to `9`.
         * @param character A character to test.
         * @return `true` if the given character is contained in the set of numerical characters, or `false`
         * otherwise.
         */
        isNumeric(character: string): boolean;
        /**
         * Set to `true` to indicate that the current language flows right-to-left.
         */
        isRTL: boolean;
        /**
         * The description of the shift button's purpose, used in a tool tip.
         */
        shiftStatus: string;
        /**
         * The text to display for the button to shift between upper and lower case characters.
         */
        shiftText: string;
        /**
         * The description of the space bar button's purpose, used in a tool tip.
         * @since 1.4.0.
         */
        spacebarStatus: string;
        /**
         * The text to display for the extended button to add a space.
         * @since 1.4.0.
         */
        spacebarText: string;
        /**
         * The description of the tab button's purpose, used in a tool tip.
         * @since 1.4.0.
         */
        tabStatus: string;
        /**
         * The text to display for the button to add a tab.
         * @since 1.4.0.
         */
        tabText: string;
        /**
         * A function to convert a character into its upper case form.
         *
         * It accepts one parameter being the current character and returns the corresponding upper case character.
         *
         * The default uses the standard JavaScript `toUpperCase` function.
         *
         * ```javascript
         * $(selector).keypad({
         *   layout: [
         *     "12345",
         *     $.keypad.SHIFT
         *   ],
         *   toUpper: function(ch) {
         *     return {
         *       "1": "!",
         *       "2": "@",
         *       "3": "#",
         *       "4": "$",
         *       "5": "%"
         *     }[ch] || ch;
         *   }
         * });
         * ```
         * @param character A character to convert to upper case.
         * @return The given character, converted to upper case; or the given character itself if it cannot be
         * converted.
         *
         * @since 1.5.0.
         */
        toUpper(character: string): string;
    }
    /**
     * The keypad instance that is used to control the keypad. It is passed to the callback function.
     */
    export interface KeypadInstance {
        /**
         * The name of this plugin, i.e. `keypad`.
         */
        name: "keypad";
        /**
         * The HTML input element for entering text.
         */
        elem: JQuery;
        /**
         * The current settings of this keypad instance.
         */
        options: KeypadSettings;
        /**
         * Whether the keypad is currently showing upper case characters (usually triggered by pressing the shift key).
         */
        ucase: boolean;
        /**
         * Whether the readonly attribute is set or removed from the input field.
         */
        saveReadonly: boolean;
    }
}
interface JQuery {
    /**
     * Initializes the keypad on the currently select elements.
     * @param settings Optional settings for customizing the keypad.
     * @return This jQuery instance for chaining.
     */
    keypad(settings?: Partial<JQueryKeypad.KeypadSettings>): this;
    /**
     * Determine whether the keypad functionality has been disabled for the first of the given field(s).
     * @param method The method to call on the existing keypad instance.
     * @return Whether the input field is disabled.
     */
    keypad(method: "isDisabled"): boolean;
    /**
     * Disable the keypad for the given field(s) as well as the field itself. Note that a field that is disabled when
     * the keypad is applied to it causes the keypad to become disabled as well.
     * @param method The method to call on the existing keypad instance.
     * @return this jQuery instance for chaining.
     */
    keypad(method: "disable"): this;
    /**
     * Enable the keypad for the given field(s) as well as the field itself.
     * @param method The method to call on the existing keypad instance.
     * @return this jQuery instance for chaining.
     */
    keypad(method: "enable"): this;
    /**
     * Hide the keypad for the given field.
     * @param method The method to call on the existing keypad instance.
     * @return this jQuery instance for chaining.
     */
    keypad(method: "hide"): this;
    /**
     * Pop up the keypad for the given field.
     * @param method The method to call on the existing keypad instance.
     * @return this jQuery instance for chaining.
     */
    keypad(method: "show"): this;
    /**
     * Remove the keypad functionality from the given field(s).
     * @param method The method to call on the existing keypad instance.
     * @return this jQuery instance for chaining.
     */
    keypad(method: "destroy"): this;
    /**
     * Retrieve all of the current settings for the first keypad instance attached to the given field(s).
     *
     * ```javascript
     * var settings = $(selector).keypad("option");
     * var prompt = $(selector).keypad("option").prompt;
     * ```
     * @param method The method to call on the existing keypad instance.
     * @return this jQuery instance for chaining.
     *
     * @since 1.5.0.
     */
    keypad(method: "option"): JQueryKeypad.KeypadSettings;
    /**
     * Retrieve one of the current settings for the first keypad instance attached to the given field(s).
     * @param optionName The name of the setting to retrieve.
     *
     * ```javascript
     * var settings = $(selector).keypad("option");
     * var prompt = $(selector).keypad("option", "prompt");
     * ```
     *
     * @typeparam K The key of the setting to retrieve.
     * @param method The method to call on the existing keypad instance.
     * @return this jQuery instance for chaining.
     *
     * @since 1.5.0.
     */
    keypad<K extends keyof JQueryKeypad.KeypadSettings>(method: "option", optionName: K): JQueryKeypad.KeypadSettings[K];
    /**
     * Update the settings for the keypad instance(s) attached to the given field(s).
     *
     * ```javascript
     * $(selector).keypad("option", {
     *   prompt: "Keypad",
     *   keypadOnly: false
     * });
     * ```
     * @param method The method to call on the existing keypad instance.
     * @param options The new options to set on the keypad instance.
     * @return this jQuery instance for chaining.
     *
     * @since 1.5.0 - previously you used the `change` command.
     */
    keypad(method: "option", options: Partial<JQueryKeypad.KeypadSettings>): this;
    /**
     * Update a particular setting for the keypad instance(s) attached to the given field(s).
     *
     * ```javascript
     * $(selector).keypad("option", "prompt", "Keypad");
     * ```
     * @typeparam K The key of the setting to update.
     * @param method The method to call on the existing keypad instance.
     * @param optionName Name of the option to update.
     * @param optionValue The new value for the option
     * @return this jQuery instance for chaining.
     *
     * @since 1.5.0 - previously you used the `change` command.
     */
    keypad<K extends keyof JQueryKeypad.KeypadSettings>(method: "option", optionName: K, optionValue: JQueryKeypad.KeypadSettings[K] | undefined): this;
}
interface JQueryStatic {
    /**
     * Keypad related constants and utility method available on the `$.keypad` object.
     */
    keypad: JQueryKeypad.KeypadGlobals;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Keyboard Widget__
     *
     * Keyboard is an input component that uses a virtual keyboard to provide the input. Notable features are the
     * customizable layouts and skinning capabilities.
     * @typeparam TCfg Defaults to `KeyboardCfg`. Type of the configuration object for this widget.
     */
    export class Keyboard<TCfg extends KeyboardCfg = KeyboardCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Unbind callback for the resize handler.
         */
        resizeHandler?: PrimeFaces.UnbindCallback;
        /**
         * Unbind callback for the scroll handler.
         */
        scrollHandler?: PrimeFaces.UnbindCallback;
        /**
         * Sets up all panel event listeners
         */
        private bindPanelEvents(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Unbind all panel event listeners
         */
        private unbindPanelEvents(): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Keyboard| Keyboard widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface KeyboardCfg extends JQueryKeypad.KeypadSettings, PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Callback that is invoked by the keyboard JQuery plugin before
         * the keyboard is brought up.
         */
        beforeShow: JQueryKeypad.BeforeShowListener;
        /**
         * The resolved and parsed keyboard layout to be used. Contains on item for each row, each
         * keyboard row contains the character codes of the keys shown on that row.
         */
        layout: string[];
        /**
         * The name of the built-in keyboard layout to use. Mutually exclusive with
         * `layoutTemplate`.
         */
        layoutName: string;
        /**
         * An optional custom keyboard layout template specified by the user. The keyboard rows
         * must be separated by a comma. Each row contains the keys that should be displayed on that row. To specify a control
         * button (space, back, shift etc.), separate the name of the control key with a dash.
         */
        layoutTemplate: string;
        /**
         * Callback that is invoked by the keyboard JQuery plugin before
         * the keyboard is closed.
         */
        onClose: JQueryKeypad.CloseListener;
    }
}
/**
 * Contains a few utilities for parsing and working with different keyboard layout. Used by the Keyboard widget.
 */
declare namespace PrimeFaces.widget.KeyboardUtils {
    /**
     * Contains the character codes for the available control characters on the keyboard, such as space and return.
     */
    export const controls: PrimeFaces.widget.KeyboardUtils.Controls;
    /**
     * This object contains the keyboard layouts that are available for the keyboard widget. Each keyboard layout is
     * a list with one entry for each keyboard row. Each row is a string that contains the characters available on that
     * row.
     */
    export const layouts: PrimeFaces.widget.KeyboardUtils.PresetLayouts;
    /**
     * Parses a custom template layout that was specified by the user. Rows are separated by commas, keys on a row may
     * be separated by dashes.
     * @param template A custom layout template specified by the user.
     * @return The parsed keyboard layout template. Contains one item for each row, each item contains the
     * characters on that keyboard row.
     */
    export function createLayoutFromTemplate(template: string): string[];
    /**
     * Finds and returns the keycode for the given control character. You can use this keycode in a custom keyboard
     * layout.
     * @param name string Name of the control keycode to get.
     * @return The keycode with the given name, if it exists.
     */
    export function getPresetControl(name: string): string | undefined;
    /**
     * Finds and returns a built-in layout with the given name. Currently available layout are `qwerty`, `qwertyBasic`,
     * and `alphabetic`.
     * @param name Name of a layout to get.
     * @return The layout with the given name, if it exists.
     */
    export function getPresetLayout(name: string): string | undefined;
    /**
     * Checks whether a built-in control with the given name exists. If it does, you can retrieve it via
     * `getPresetControl`.
     * @param key string Name of the control keycode to check.
     * @return `true` if a control for the given key is defined, `false` otherwise.
     */
    export function isDefinedControl(key: string): boolean;
}
declare namespace PrimeFaces.widget.KeyboardUtils {
    /**
     * Contains the character codes for the available control
     * characters on the keyboard, such as space and return.
     */
    export interface Controls {
        /**
         * The keyboard code for the back button that removes the character to the left of the cursor.
         */
        back: string;
        /**
         * The keyboard code for the button that clears the entered text.
         */
        clear: string;
        /**
         * The keyboard code for the button that closes (hides) the keyboard
         */
        close: string;
        /**
         * The keyboard code for the space button that inserts a half-width space.
         */
        halfspace: string;
        /**
         * The keyboard code for the modifying shift button.
         */
        shift: string;
        /**
         * The keyboard code for the space button that inserts a full-width space.
         */
        space: string;
        /**
         * The keyboard code for the space button that insert a whitespace.
         */
        spacebar: string;
    }
}
declare namespace PrimeFaces.widget.KeyboardUtils {
    /**
     * Contains the various different available keyboard layouts.
     */
    export interface PresetLayouts {
        /**
         * An alphabetical layout with the letter keys in alphabetical order. This is a list with one entry for each
         * keyboard row. Each row is a string that contains the characters available on that row.
         */
        alphabetic: string[];
        /**
         * A `qwerty` layout with some special characters. This is a list with one entry for each keyboard row. Each row
         * is a string that contains the characters available on that row.
         */
        qwerty: string[];
        /**
         * A basic `qwerty` layout without many special characters. This is a list with one entry for each keyboard row.
         * Each row is a string that contains the characters available on that row.
         */
        qwertyBasic: string[];
    }
}
/**
 * Namespace for the jQuery keyfilter plugin.
 *
 * KeyFilter is used to filter keyboard input on specified input components.
 */
declare namespace JQueryKeyfilter {
    /**
     * A test function that checks whether a newly entered characters is allowed or should be blocked. The this context
     * is set to the DOM element for the target input field.
     * @param character A newly entered character to check.
     * @return `true` to allow the entered character, or `false` to block it.
     */
    export type TestFunction = (this: JQuery, character: string) => boolean;
    /**
     * Default, global configuration for the key filter.
     */
    export interface Defaults {
        /**
         * A map between the built-in masks and the corresponding regular expressions.
         */
        masks: DefaultMasks;
    }
    /**
     * A map between the built-in masks and the corresponding regular expressions.
     */
    export interface DefaultMasks {
        /**
         * The mask that allows only letters.
         */
        alpha: RegExp;
        /**
         * The mask that allows only letters and numbers.
         */
        alphanum: RegExp;
        /**
         * The mask that allows only valid email characters (this does **NOT** check for well-formed emails)
         */
        email: RegExp;
        /**
         * The mask that allows only hexadecimal characters.
         */
        hex: RegExp;
        /**
         * The mask that allows only digits and dashes.
         */
        int: RegExp;
        /**
         * The mask that allows only digits, spaces, commas, and periods.
         */
        money: RegExp;
        /**
         * The mask that allows only digits, commas, and periods.
         */
        num: RegExp;
        /**
         * The mask that allows only digits.
         */
        pint: RegExp;
        /**
         * The mask that allows only digits and periods.
         */
        pnum: RegExp;
    }
    /**
     * The main object of the key filter plugin that contains both the JQuery plugin function as well as some global
     * configuration for the key filter.
     */
    export interface KeyfilterNamespace<TElement = HTMLElement> {
        /**
         * Enables the key filter on the current input filter. Only text that matches the given regular expression is
         * allowed in the input field.
         * @param regExp Regular expression the value of the input field needs to match.
         * @return this jQuery instance for chaining.
         */
        (regExp: RegExp): JQuery<TElement>;
        /**
         * Enables the key filter on the current input filter. An entered character needs to match the given test
         * function to be allowed. If it does not match, the character is blocked and the entered character is
         * discarded.
         * @param testFunction A function that tests whether an entered character is allowed.
         * @return this jQuery instance for chaining.
         */
        (testFunction: JQueryKeyfilter.TestFunction): JQuery<TElement>;
        /**
         * Default, global configuration for the key filter.
         */
        defaults: JQueryKeyfilter.Defaults;
        /**
         * The version of this plugin.
         */
        version: number;
    }
}
interface JQuery<TElement = HTMLElement> {
    /**
     * The main function of the key filter plugin that can be called on an input field to set up the key filter. Also
     * contains some default and global configuration for the key filter.
     */
    keyfilter: JQueryKeyfilter.KeyfilterNamespace<TElement>;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces KeyFilter Widget__
     *
     * KeyFilter is used to filter keyboard input on specified input components.
     * @typeparam TCfg Defaults to `KeyFilterCfg`. Type of the configuration object for this widget.
     */
    export class KeyFilter<TCfg extends KeyFilterCfg = KeyFilterCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * The DOM element for the target input element to which this key filter is applied.
         */
        target: JQuery;
        /**
         * Applies the key filter to the given input or textarea element.
         * @param input A jQuery selector object.
         * @param cfg The widget configuration.
         */
        private applyKeyFilter(input: JQuery, cfg: TCfg): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  KeyFilter| KeyFilter widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface KeyFilterCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Defines the regular expression which should be used to test the complete input text
         * content. The options `testFunction`, `regEx`, `inputRegEx`, and `mask` are mutually exclusive.
         */
        inputRegEx: RegExp;
        /**
         * Defines the predefined mask which should be used. The options
         * `testFunction`, `regEx`, `inputRegEx`, and `mask` are mutually exclusive.
         */
        mask: keyof JQueryKeyfilter.DefaultMasks;
        /**
         * Whether the component also should prevent paste.
         */
        preventPaste: boolean;
        /**
         * Defines the regular expression which should be used for filtering the input. The options
         * `testFunction`, `regEx`, `inputRegEx`, and `mask` are mutually exclusive.
         */
        regEx: RegExp;
        /**
         * The target input expression, defaults to the parent of this component.
         */
        target: string;
        /**
         * An optional function which should be used for filtering. The
         * options `testFunction`, `regEx`, `inputRegEx`, and `mask` are mutually exclusive.
         */
        testFunction: JQueryKeyfilter.TestFunction;
    }
}
/**
 * Namespace for the jQuery knob plugin.
 *
 * This is canvas based plugin that renders a knob the user can move with the mouse ; no png or jpg sprites required.
 *
 * See https://github.com/aterrien/jQuery-Knob
 */
declare namespace JQueryKnob {
    /**
     * Models the direction of a rotation, either clockwise or anti-clockwise.
     */
    export type RotationDirection = "clockwise" | "anticlockwise" | "cw" | "acw";
    /**
     * Settings for the jQuery knob plugin to configure the behavior and appearance.
     */
    export interface KnobSettings {
        /**
         * Arc size in degrees. Defaults to `360`.
         */
        angleArc: number;
        /**
         * Starting angle in degrees. Defaults to `0`.
         */
        angleOffset: number;
        /**
         * Maximum value of the knob. Defaults to `100`.
         */
        max: number;
        /**
         * Minimum value of the knob. Defaults to `0`.
         */
        min: number;
        /**
         * Disables input and events. Defaults to `false`.
         */
        readOnly: boolean;
        /**
         * Indicates in which direction the value of the knob increases. Defaults to `clockwise`.
         */
        rotation: RotationDirection;
        /**
         * Step size for incrementing and decrementing. Defaults to `1`.
         */
        step: number;
        /**
         * Stop at min & max on keydown and mousewheel. Defaults to `true`.
         */
        stopper: boolean;
        /**
         * Background color, i.e. the unfilled part of the knob.
         */
        bgColor: string;
        /**
         * Display mode "cursor", cursor size could be changed passing a numeric value to the option, default width is
         * used when passing boolean value `true`.
         */
        cursor: number | true;
        /**
         * Set to `false` to hide the input. Defaults to `true`.
         */
        displayInput: boolean;
        /**
         * `true` to display the previous value with transparency. Defaults to `false`.
         */
        displayPrevious: boolean;
        /**
         * Foreground color, i.e. the filled part of the knob.
         */
        fgColor: string;
        /**
         * Font family used for displaying the value.
         */
        font: string;
        /**
         * Font weight used for displaying the value.
         */
        fontWeight: string | number;
        /**
         * Dial height, as a CSS unit.
         */
        height: string;
        /**
         * Input value (number) color as a CSS unit.
         */
        inputColor: string;
        /**
         * Gauge stroke endings.
         */
        lineCap: "butt" | "round";
        /**
         * Gauge thickness, between 0 and 1.
         */
        thickness: number;
        /**
         * Dial width, as a CSS unit.
         */
        width: string;
        /**
         * Triggered when the Escape key is pressed.
         * @return `false` to abort the cancellation and apply the value.
         */
        cancel(this: KnobInstance): boolean;
        /**
         * Invoked at each change of the value.
         * @param currentValue The current value of this knob.
         * @return `false` to abort the change and revert the knob to its original value.
         */
        change(this: KnobInstance, currentValue: number): boolean;
        /**
         * Invoked when drawing the canvas.
         * @return `true` to continue with the default drawing of the knob widget, `false` to abort (such as when
         * performing custom drawing).
         */
        draw(this: KnobInstance): boolean;
        /**
         * Allows to format output (add unit such as `%` or `ms`).
         * @param currentValue Value to be formatted.
         * @return The formatted value.
         */
        format(this: KnobInstance, currentValue: number): string;
        /**
         * Invoked when the knob is released.
         * @param currentValue The current value of this knob.
         * @return `false` to abort the release and not change the value.
         */
        release(this: KnobInstance, currentValue: number): boolean;
    }
    /**
     * The knob instance controlling the knob. It is passed to the callback methods.
     */
    export interface KnobInstance {
        /**
         * The canvas on which the knob is drawn.
         */
        g: CanvasRenderingContext2D;
        /**
         * Element for the knob.
         */
        $: JQuery;
        /**
         * Current settings of the knob.
         */
        o: KnobSettings;
        /**
         * Input element of the knob.
         */
        i: JQuery;
    }
}
interface JQuery {
    /**
     * Initializes the knob on the current input element. Creates the canvas with the rendered knob.
     *
     * Options can also be  provided as data attributes, e.g.:
     *
     * ```html
     * <input type="text" class="dial" data-min="-50" data-max="50">
     * ```
     * @param settings Optional settings for the knob.
     * @return this jQuery instance for chaining.
     */
    knob(settings?: Partial<JQueryKnob.KnobSettings>): this;
}
declare namespace PrimeFaces.widget.Knob {
    /**
     * Client side callback to invoke when value changes. See also
     * {@link KnobCfg.onchange}.
     */
    export type OnChangeCallback =
    /**
     * @param currentValue Current numerical value of the knob.
     */
    (currentValue: number) => void;
}
declare namespace PrimeFaces.widget {
    /**
     * __PrimeFaces Knob Widget__
     *
     * Knob is an input component to insert numeric values in a range.
     * @typeparam TCfg Defaults to `KnobCfg`. Type of the configuration object for this widget.
     */
    export class Knob<TCfg extends KnobCfg = KnobCfg> extends PrimeFaces.widget.BaseWidget<TCfg> {
        /**
         * Name of the color theme to use. You can use on of the keys defined in
         * `PrimeFaces.widget.Knob.ColorThemes`.
         */
        colorTheme: string;
        /**
         * The DOM Element for the hidden input that stores the value of this widget.
         */
        input: JQuery;
        /**
         * The DOM element on which the JQuery knob plugin was initialized.
         */
        knob: JQuery;
        /**
         * Maximum allowed value for this knob.
         */
        max: number;
        /**
         * Minimum allowed value for this knob.
         */
        min: number;
        /**
         * Step size for incrementing or decrementing the value of this knob.
         */
        step: number;
        /**
         * Color theme data to be used.
         */
        themeObject: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * Creates the knob widget and sets up all event listeners.
         */
        private createKnob(): void;
        /**
         * Decrements the value of this knob by the current step size.
         */
        decrement(): void;
        /**
         * Disables this input so that the user cannot enter a value anymore.
         */
        disable(): void;
        /**
         * Enables this input so that the user can enter a value.
         */
        enable(): void;
        /**
         * Retrieves the current value of this knob, as a number.
         * @return The current numerical value of this knob.
         */
        getValue(): number;
        /**
         * Increments the value of this knob by the current step size.
         */
        increment(): void;
        /**
         * A widget class should not declare an explicit constructor, the default constructor provided by this base
         * widget should be used. Instead, override this initialize method which is called after the widget instance
         * was constructed. You can use this method to perform any initialization that is required. For widgets that
         * need to create custom HTML on the client-side this is also the place where you should call your render
         * method.
         *
         * Please make sure to call the super method first before adding your own custom logic to the init method:
         *
         * ```javascript
         * PrimeFaces.widget.MyWidget = PrimeFaces.widget.BaseWidget.extend({
         *   init: function(cfg) {
         *     this._super(cfg);
         *     // custom initialization
         *   }
         * });
         * ```
         * @override
         * @param cfg The widget configuration to be used for this widget instance.
         * This widget configuration is usually created on the server by the `javax.faces.render.Renderer` for this
         * component.
         */
        override init(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Used in ajax updates, reloads the widget configuration.
         *
         * When an AJAX call is made and this component is updated, the DOM element is replaced with the newly rendered
         * content. However, no new instance of the widget is created. Instead, after the DOM element was replaced, this
         * method is called with the new widget configuration from the server. This makes it possible to persist
         * client-side state during an update, such as the currently selected tab.
         *
         * Please note that instead of overriding this method, you should consider adding a refresh listener instead
         * via {@link addRefreshListener}. This has the advantage of letting you add multiple listeners, and makes it
         * possible to add additional listeners from code outside this widget.
         *
         * By default, this method calls all refresh listeners, then reinitializes the widget by calling the `init`
         * method.
         * @override
         * @param cfg The new widget configuration from the server.
         * @return The value as returned by the `init` method, which is often `undefined`.
         */
        override refresh(cfg: PrimeFaces.PartialWidgetCfg<TCfg>): void;
        /**
         * Sets the value of this knob widget to the given value.
         * @param value Value to set on this knob.
         */
        setValue(value: number): void;
    }
}
declare namespace PrimeFaces.widget {
    /**
     * The configuration for the {@link  Knob| Knob widget}.
     * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
     * configuration is usually meant to be read-only and should not be modified.
     */
    export interface KnobCfg extends PrimeFaces.widget.BaseWidgetCfg {
        /**
         * Foreground color of the component.
         */
        bgColor: string;
        /**
         * Theme of the knob.
         */
        colorTheme: string;
        /**
         * Background color of the component.
         */
        fgColor: string;
        /**
         * Template of the progress value e.g. `{value}%`.
         */
        labelTemplate: string;
        /**
         * Client side callback to invoke when value changes.
         */
        onchange: PrimeFaces.widget.Knob.OnChangeCallback;
        /**
         * Style class of the component.
         */
        styleClass: string;
    }
}
declare namespace PrimeFaces.widget.Knob {
    /**
     * A color theme for the knob, consisting of the color for the
     * filled and unfilled part of the knob.
     */
    export interface ColorTheme {
        /**
         * The background color, i.e. the color of the unfilled part of the knob. Must be a
         * CSS color, e.g., `#ff0000`.
         */
        bgColor: string;
        /**
         * The foreground color, i.e. the color of the filled part of the knob. Must be a CSS
         * color, e.g., `#ff0000`.
         */
        fgColor: string;
    }
}
declare namespace PrimeFaces.widget.Knob {
    /**
     * Interface for the the list of available builtin color themes for the {@link Knob} widget.
     */
    export interface ColorThemes {
        /**
         * The default afterdark theme.
         */
        afterdark: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default afternoon theme.
         */
        afternoon: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default afterwork theme.
         */
        afterwork: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default aristo theme.
         */
        aristo: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default black tie theme.
         */
        "black-tie": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default blitzer theme.
         */
        blitzer: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default bluesky theme.
         */
        bluesky: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default bootstrap theme.
         */
        bootstrap: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default casablanca theme.
         */
        casablanca: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default cruze theme.
         */
        cruze: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default cupertino theme.
         */
        cupertino: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default dark hive theme.
         */
        "dark-hive": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default delta theme.
         */
        delta: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default dot luv theme.
         */
        "dot-luv": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default eggplant theme.
         */
        eggplant: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default excite bike theme.
         */
        "excite-bike": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default flick theme.
         */
        flick: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default glass x theme.
         */
        "glass-x": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default home theme.
         */
        home: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default hot sneaks theme.
         */
        "hot-sneaks": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default humanity theme.
         */
        humanity: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default le frog theme.
         */
        "le-frog": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default midnight theme.
         */
        midnight: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default mint choc theme.
         */
        "mint-choc": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default overcast theme.
         */
        overcast: PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The default pepper grinder theme.
         */
        "pepper-grinder": PrimeFaces.widget.Knob.ColorTheme;
        /**
         * The def