import type { $ZodStringFormats } from "../core/checks.js";
import type * as errors from "../core/errors.js";
import * as util from "../core/util.js";

const Sizable: Record<string, { unit: string; verb: string }> = {
  string: { unit: "karakter", verb: "memiliki" },
  file: { unit: "byte", verb: "memiliki" },
  array: { unit: "item", verb: "memiliki" },
  set: { unit: "item", verb: "memiliki" },
};

function getSizing(origin: string): { unit: string; verb: string } | null {
  return Sizable[origin] ?? null;
}

export const parsedType = (data: any): string => {
  const t = typeof data;

  switch (t) {
    case "number": {
      return Number.isNaN(data) ? "NaN" : "number";
    }
    case "object": {
      if (Array.isArray(data)) {
        return "array";
      }
      if (data === null) {
        return "null";
      }

      if (Object.getPrototypeOf(data) !== Object.prototype && data.constructor) {
        return data.constructor.name;
      }
    }
  }
  return t;
};

const Nouns: {
  [k in $ZodStringFormats | (string & {})]?: string;
} = {
  regex: "input",
  email: "alamat email",
  url: "URL",
  emoji: "emoji",
  uuid: "UUID",
  uuidv4: "UUIDv4",
  uuidv6: "UUIDv6",
  nanoid: "nanoid",
  guid: "GUID",
  cuid: "cuid",
  cuid2: "cuid2",
  ulid: "ULID",
  xid: "XID",
  ksuid: "KSUID",
  datetime: "tanggal dan waktu format ISO",
  date: "tanggal format ISO",
  time: "jam format ISO",
  duration: "durasi format ISO",
  ipv4: "alamat IPv4",
  ipv6: "alamat IPv6",
  cidrv4: "rentang alamat IPv4",
  cidrv6: "rentang alamat IPv6",
  base64: "string dengan enkode base64",
  base64url: "string dengan enkode base64url",
  json_string: "string JSON",
  e164: "angka E.164",
  jwt: "JWT",
  template_literal: "input",
};

const error: errors.$ZodErrorMap = (issue) => {
  switch (issue.code) {
    case "invalid_type":
      return `Input tidak valid: diharapkan ${issue.expected}, diterima ${parsedType(issue.input)}`;
    case "invalid_value":
      if (issue.values.length === 1) return `Input tidak valid: diharapkan ${util.stringifyPrimitive(issue.values[0])}`;
      return `Pilihan tidak valid: diharapkan salah satu dari ${util.joinValues(issue.values, "|")}`;
    case "too_big": {
      const adj = issue.inclusive ? "<=" : "<";
      const sizing = getSizing(issue.origin);
      if (sizing)
        return `Terlalu besar: diharapkan ${issue.origin ?? "value"} memiliki ${adj}${issue.maximum.toString()} ${sizing.unit ?? "elemen"}`;
      return `Terlalu besar: diharapkan ${issue.origin ?? "value"} menjadi ${adj}${issue.maximum.toString()}`;
    }
    case "too_small": {
      const adj = issue.inclusive ? ">=" : ">";
      const sizing = getSizing(issue.origin);
      if (sizing) {
        return `Terlalu kecil: diharapkan ${issue.origin} memiliki ${adj}${issue.minimum.toString()} ${sizing.unit}`;
      }

      return `Terlalu kecil: diharapkan ${issue.origin} menjadi ${adj}${issue.minimum.toString()}`;
    }
    case "invalid_format": {
      const _issue = issue as errors.$ZodStringFormatIssues;
      if (_issue.format === "starts_with") return `String tidak valid: harus dimulai dengan "${_issue.prefix}"`;
      if (_issue.format === "ends_with") return `String tidak valid: harus berakhir dengan "${_issue.suffix}"`;
      if (_issue.format === "includes") return `String tidak valid: harus menyertakan "${_issue.includes}"`;
      if (_issue.format === "regex") return `String tidak valid: harus sesuai pola ${_issue.pattern}`;
      return `${Nouns[_issue.format] ?? issue.format} tidak valid`;
    }
    case "not_multiple_of":
      return `Angka tidak valid: harus kelipatan dari ${issue.divisor}`;
    case "unrecognized_keys":
      return `Kunci tidak dikenali ${issue.keys.length > 1 ? "s" : ""}: ${util.joinValues(issue.keys, ", ")}`;
    case "invalid_key":
      return `Kunci tidak valid di ${issue.origin}`;
    case "invalid_union":
      return "Input tidak valid";
    case "invalid_element":
      return `Nilai tidak valid di ${issue.origin}`;
    default:
      return `Input tidak valid`;
  }
};

export { error };

export default function (): { localeError: errors.$ZodErrorMap } {
  return {
    localeError: error,
  };
}
