import type { $ZodStringFormats } from "../core/checks.js";
import type * as errors from "../core/errors.js";
import * as util from "../core/util.js";

const Sizable: Record<string, { unit: string; verb: string }> = {
  string: { unit: "znakov", verb: "imeti" },
  file: { unit: "bajtov", verb: "imeti" },
  array: { unit: "elementov", verb: "imeti" },
  set: { unit: "elementov", verb: "imeti" },
};

function getSizing(origin: string): { unit: string; verb: string } | null {
  return Sizable[origin] ?? null;
}

export const parsedType = (data: any): string => {
  const t = typeof data;

  switch (t) {
    case "number": {
      return Number.isNaN(data) ? "NaN" : "število";
    }
    case "object": {
      if (Array.isArray(data)) {
        return "tabela";
      }
      if (data === null) {
        return "null";
      }

      if (Object.getPrototypeOf(data) !== Object.prototype && data.constructor) {
        return data.constructor.name;
      }
    }
  }
  return t;
};

const Nouns: {
  [k in $ZodStringFormats | (string & {})]?: string;
} = {
  regex: "vnos",
  email: "e-poštni naslov",
  url: "URL",
  emoji: "emoji",
  uuid: "UUID",
  uuidv4: "UUIDv4",
  uuidv6: "UUIDv6",
  nanoid: "nanoid",
  guid: "GUID",
  cuid: "cuid",
  cuid2: "cuid2",
  ulid: "ULID",
  xid: "XID",
  ksuid: "KSUID",
  datetime: "ISO datum in čas",
  date: "ISO datum",
  time: "ISO čas",
  duration: "ISO trajanje",
  ipv4: "IPv4 naslov",
  ipv6: "IPv6 naslov",
  cidrv4: "obseg IPv4",
  cidrv6: "obseg IPv6",
  base64: "base64 kodiran niz",
  base64url: "base64url kodiran niz",
  json_string: "JSON niz",
  e164: "E.164 številka",
  jwt: "JWT",
  template_literal: "vnos",
};

const error: errors.$ZodErrorMap = (issue) => {
  switch (issue.code) {
    case "invalid_type":
      return `Neveljaven vnos: pričakovano ${issue.expected}, prejeto ${parsedType(issue.input)}`;
    case "invalid_value":
      if (issue.values.length === 1) return `Neveljaven vnos: pričakovano ${util.stringifyPrimitive(issue.values[0])}`;
      return `Neveljavna možnost: pričakovano eno izmed ${util.joinValues(issue.values, "|")}`;
    case "too_big": {
      const adj = issue.inclusive ? "<=" : "<";
      const sizing = getSizing(issue.origin);
      if (sizing)
        return `Preveliko: pričakovano, da bo ${issue.origin ?? "vrednost"} imelo ${adj}${issue.maximum.toString()} ${sizing.unit ?? "elementov"}`;
      return `Preveliko: pričakovano, da bo ${issue.origin ?? "vrednost"} ${adj}${issue.maximum.toString()}`;
    }
    case "too_small": {
      const adj = issue.inclusive ? ">=" : ">";
      const sizing = getSizing(issue.origin);
      if (sizing) {
        return `Premajhno: pričakovano, da bo ${issue.origin} imelo ${adj}${issue.minimum.toString()} ${sizing.unit}`;
      }
      return `Premajhno: pričakovano, da bo ${issue.origin} ${adj}${issue.minimum.toString()}`;
    }
    case "invalid_format": {
      const _issue = issue as errors.$ZodStringFormatIssues;
      if (_issue.format === "starts_with") {
        return `Neveljaven niz: mora se začeti z "${_issue.prefix}"`;
      }
      if (_issue.format === "ends_with") return `Neveljaven niz: mora se končati z "${_issue.suffix}"`;
      if (_issue.format === "includes") return `Neveljaven niz: mora vsebovati "${_issue.includes}"`;
      if (_issue.format === "regex") return `Neveljaven niz: mora ustrezati vzorcu ${_issue.pattern}`;
      return `Neveljaven ${Nouns[_issue.format] ?? issue.format}`;
    }
    case "not_multiple_of":
      return `Neveljavno število: mora biti večkratnik ${issue.divisor}`;
    case "unrecognized_keys":
      return `Neprepoznan${issue.keys.length > 1 ? "i ključi" : " ključ"}: ${util.joinValues(issue.keys, ", ")}`;
    case "invalid_key":
      return `Neveljaven ključ v ${issue.origin}`;
    case "invalid_union":
      return "Neveljaven vnos";
    case "invalid_element":
      return `Neveljavna vrednost v ${issue.origin}`;
    default:
      return "Neveljaven vnos";
  }
};

export { error };

export default function (): { localeError: errors.$ZodErrorMap } {
  return {
    localeError: error,
  };
}
