package org.nakedobjects.applib;

import org.nakedobjects.applib.util.NameUtils;


public class Identifier implements Comparable<Identifier> {

    /**
     * What type of feature this identifies.
     */
    public static enum Type {
        CLASS, PROPERTY_OR_COLLECTION, ACTION
    }

    /**
     * @see #toIdentityString(int)
     */
    public static final int CLASS = 0;
    /**
     * @see #toIdentityString(int)
     */
    public static final int CLASS_NAME = 1;
    /**
     * @see #toIdentityString(int)
     */
    public static final int CLASS_NAME_PARMS = 2;
    /**
     * @see #toIdentityString(int)
     */
    public static final int NAME = 3;
    /**
     * @see #toIdentityString(int)
     */
    public static final int PARMS = 4;

    // ///////////////////////////////////////////////////////////////////////////
    // Factory methods
    // ///////////////////////////////////////////////////////////////////////////

    public static Identifier classIdentifier(final Class<?> clazz) {
        return classIdentifier(clazz.getName());
    }

    public static Identifier classIdentifier(final String className) {
        return new Identifier(className, "", new String[] {}, Type.CLASS);
    }

    public static Identifier propertyOrCollectionIdentifier(final Class<?> declaringClass, final String propertyOrCollectionName) {
        return propertyOrCollectionIdentifier(declaringClass.getCanonicalName(), propertyOrCollectionName);
    }

    public static Identifier propertyOrCollectionIdentifier(final String declaringClassName, final String propertyOrCollectionName) {
        return new Identifier(declaringClassName, propertyOrCollectionName, new String[] {}, Type.PROPERTY_OR_COLLECTION);
    }

    public static Identifier actionIdentifier(
            final Class<?> declaringClass,
            final String actionName,
            final Class<?>[] parameterClasses) {
        return actionIdentifier(declaringClass.getCanonicalName(), actionName, toParameterStringArray(parameterClasses));
    }

    public static Identifier actionIdentifier(
            final String declaringClassName,
            final String actionName,
            final Class<?>[] parameterClasses) {
        return actionIdentifier(declaringClassName, actionName, toParameterStringArray(parameterClasses));
    }

    public static Identifier actionIdentifier(
            final String declaringClassName,
            final String actionName,
            final String[] parameterClassNames) {
        return new Identifier(declaringClassName, actionName, parameterClassNames, Type.ACTION);
    }

    /**
     * Helper, used within contructor chaining
     */
    private static String[] toParameterStringArray(final Class<?>[] parameterClasses) {
        final String[] parameters = new String[parameterClasses == null ? 0 : parameterClasses.length];
        for (int i = 0; i < parameters.length; i++) {
            parameters[i] = parameterClasses[i].getName();
        }
        return parameters;
    }

    // ///////////////////////////////////////////////////////////////////////////
    // Instance variables
    // ///////////////////////////////////////////////////////////////////////////

    private final String className;
    private final String memberName;
    private final String[] parameterNames;
    private final Type type;
    private String identityString;

    /**
     * Caching of {@link #toString()}, for performance.
     */
    private String asString = null;

    // ///////////////////////////////////////////////////////////////////////////
    // Constructor
    // ///////////////////////////////////////////////////////////////////////////

    private Identifier(final String className, final String memberName, final String[] parameterNames, final Type type) {
        this.className = className;
        this.memberName = memberName;
        this.parameterNames = parameterNames;
        this.type = type;
    }

    public String getClassName() {
        return className;
    }

    public String getMemberName() {
        return memberName;
    }

    public String getMemberNaturalName() {
        return NameUtils.naturalName(memberName);
    }

    public String[] getMemberParameterNames() {
        return parameterNames;
    }

    public String[] getMemberParameterNaturalNames() {
        return NameUtils.naturalNames(parameterNames);
    }

    public Type getType() {
        return type;
    }

    /**
     * Convenience method.
     * 
     * @return
     */
    public boolean isPropertyOrCollection() {
        return type == Type.PROPERTY_OR_COLLECTION;
    }

    // ///////////////////////////////////////////////////////////////////////////
    // compareTo, equals
    // ///////////////////////////////////////////////////////////////////////////

    public int compareTo(final Identifier o2) {
        return toString().compareTo(o2.toString());
    }

    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        } else if (obj instanceof Identifier) {
            final Identifier other = (Identifier) obj;
            return equals(other.className, className) && equals(other.memberName, other.memberName)
                    && equals(other.parameterNames, parameterNames);
        }
        return false;
    }

    private boolean equals(final String[] a, final String[] b) {
        if (a == null && b == null) {
            return true;
        }
        if (a == null && b != null) {
            return false;
        }
        if (a != null && b == null) {
            return false;
        }
        if (a.length != b.length) {
            return false;
        }
        for (int i = 0; i < b.length; i++) {
            if (!a[i].equals(b[i])) {
                return false;
            }
        }
        return true;
    }

    private boolean equals(final String a, final String b) {
        if (a == b) {
            return true;
        }

        if (a != null) {
            return a.equals(b);
        }

        return false;
    }

    // ///////////////////////////////////////////////////////////////////////////
    // toXxxString
    // ///////////////////////////////////////////////////////////////////////////

    public String toIdentityString(final int depth) {
        // Assert.assertTrue(depth >= Identifier.CLASS && depth <= Identifier.PARMS);
        switch (depth) {
        case (Identifier.CLASS):
            return toClassIdentityString();
        case (Identifier.CLASS_NAME):
            return toClassAndNameIdentityString();
        case (Identifier.CLASS_NAME_PARMS):
            return toFullIdentityString();
        case (Identifier.NAME):
            return toNameIdentityString();
        case (Identifier.PARMS):
            return toParmsIdentityString();
        }
        return null;
    }

    public String toClassIdentityString() {
        return className;
    }

    public String toNameIdentityString() {
        return memberName;
    }

    public String toClassAndNameIdentityString() {
        return toClassIdentityString() + "#" + memberName;
    }

    public String toParmsIdentityString() {
        final StringBuilder str = new StringBuilder();
        if (type == Type.ACTION) {
            str.append('(');
            for (int i = 0; i < parameterNames.length; i++) {
                if (i > 0) {
                    str.append(",");
                }
                str.append(parameterNames[i]);
            }
            str.append(')');
        }
        return str.toString();
    }

    public String toNameParmsIdentityString() {
        return getMemberName() + toParmsIdentityString();
    }

    public String toFullIdentityString() {
        if (identityString == null) {
            if (memberName.length() == 0) {
                identityString = toClassIdentityString();
            } else {
                final StringBuilder str = new StringBuilder();
                str.append(toClassAndNameIdentityString());
                str.append(toParmsIdentityString());
                identityString = str.toString();
            }
        }
        return identityString;
    }

    @Override
    public String toString() {
        if (asString == null) {
            final StringBuilder str = new StringBuilder();
            str.append(className);
            str.append('#');
            str.append(memberName);
            str.append('(');
            for (int i = 0; i < parameterNames.length; i++) {
                if (i > 0) {
                    str.append(", ");
                }
                str.append(parameterNames[i]);
            }
            str.append(')');
            asString = str.toString();
        }
        return asString;
    }

}
// Copyright (c) Naked Objects Group Ltd.
