package org.nakedobjects.bytecode.javassist.persistence.objectfactory.internal;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatArg;

import java.lang.reflect.Method;

import javassist.util.proxy.MethodFilter;
import javassist.util.proxy.MethodHandler;
import javassist.util.proxy.ProxyFactory;
import javassist.util.proxy.ProxyObject;

import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.commons.lang.ArrayUtils;
import org.nakedobjects.metamodel.spec.JavaSpecification;
import org.nakedobjects.metamodel.spec.NakedObjectSpecification;
import org.nakedobjects.metamodel.spec.feature.NakedObjectMember;
import org.nakedobjects.metamodel.specloader.SpecificationLoader;
import org.nakedobjects.runtime.persistence.objectfactory.MethodUtils;
import org.nakedobjects.runtime.persistence.objectfactory.ObjectChanger;
import org.nakedobjects.runtime.persistence.objectfactory.ObjectResolver;

public class ObjectResolveAndObjectChangedEnhancer {
	
    private final ObjectResolver objectResolver;
    private final ObjectChanger objectChanger;
    private final SpecificationLoader specificationLoader;
	private MethodHandler methodHandler;

	public ObjectResolveAndObjectChangedEnhancer(
			final ObjectResolver objectResolver, 
			final ObjectChanger objectChanger, 
			final SpecificationLoader specificationLoader) {
	    ensureThatArg(objectResolver, is(notNullValue()));
        ensureThatArg(objectChanger, is(notNullValue()));
        ensureThatArg(specificationLoader, is(notNullValue()));
        
		this.objectResolver = objectResolver;
		this.objectChanger = objectChanger;
		this.specificationLoader = specificationLoader;
		
		this.methodHandler = new MethodHandler() {
		    public Object invoke(
		    		final Object proxied, Method proxyMethod, 
		    		Method proxiedMethod, Object[] args) throws Throwable {
		        //System.err.println("proxyMethod: " + proxyMethod + " proxiedMethod: " + proxiedMethod + " isAction: " + isAction);

		    	boolean ignore = proxyMethod.getDeclaringClass().equals(Object.class);
		    	
				final JavaSpecification targetObjSpec = getJavaSpecificationOfOwningClass(proxiedMethod); 

				final NakedObjectMember member = targetObjSpec.getMember(proxiedMethod);
				
				boolean isAssociation = member != null && member.isAssociation(); 
				boolean isAction = member != null && member.isAction(); 
				boolean isGetter = isAssociation && MethodUtils.isGetter(proxiedMethod);
				boolean isSetter = isAssociation && MethodUtils.isSetter(proxiedMethod);
				

				if (!ignore && (isGetter || isSetter)) {
					objectResolver.resolve(proxied, member.getName());
				}

		        Object proxiedReturn = proxiedMethod.invoke(proxied, args);  // execute the original method.
		        
				if (!ignore && (isSetter || isAction)) {
					objectChanger.objectChanged(proxied);
				}
			
				return proxiedReturn;
		    }
		};
		 
	}
	
	@SuppressWarnings("unchecked")
	public <T> T newInstance(Class<T> cls) {

		ProxyFactory proxyFactory = new ProxyFactory();
		proxyFactory.setSuperclass(cls);
		proxyFactory.setInterfaces(ArrayUtils.combine(
				cls.getInterfaces(), new Class<?>[]{JavassistEnhanced.class}));
		
		proxyFactory.setFilter(new MethodFilter() {
		     public boolean isHandled(Method m) {
		         // ignore finalize()
		         return !m.getName().equals("finalize");
		     }
		});
		
		Class<T> proxySubclass = proxyFactory.createClass();
		try {
			T newInstance = proxySubclass.newInstance();
			ProxyObject proxyObject = (ProxyObject)newInstance;
			proxyObject.setHandler(methodHandler);
			
			return newInstance;
		} catch (InstantiationException e) {
			throw new NakedObjectException(e);
		} catch (IllegalAccessException e) {
			throw new NakedObjectException(e);
		}
	}


    private JavaSpecification getJavaSpecificationOfOwningClass(final Method method) {
        return getJavaSpecification(method.getDeclaringClass());
    }

    private JavaSpecification getJavaSpecification(final Class<?> clazz) {
        final NakedObjectSpecification nos = getSpecification(clazz);
        if (!(nos instanceof JavaSpecification)) {
            throw new UnsupportedOperationException("Only Java is supported (specification is '"
                    + nos.getClass().getCanonicalName() + "')");
        }
        return (JavaSpecification) nos;
    }

    private NakedObjectSpecification getSpecification(final Class<?> type) {
        final NakedObjectSpecification nos = specificationLoader.loadSpecification(type);
        return nos;
    }


    // /////////////////////////////////////////////////////////////////
    // Dependencies (injected in constructor)
    // /////////////////////////////////////////////////////////////////


    public final ObjectResolver getObjectResolver() {
        return objectResolver;
    }
    
    public final ObjectChanger getObjectChanger() {
        return objectChanger;
    }
    
    public final SpecificationLoader getSpecificationLoader() {
        return specificationLoader;
    }

}
