package org.nakedobjects.metamodel.commons.encoding;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;

import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;


public class ByteEncoderBuffer implements ByteEncoder {

    static final String ENCODED_NULL_ARRAY = "null-array";
    static final String ENCODED_BOOLEAN_FALSE = "false";
    static final String ENCODED_BOOLEAN_TRUE = "true";
    static final String ENCODED_NULL_OBJECT = "null";

    private final OutputStream output;

    public ByteEncoderBuffer(final OutputStream output) {
        this.output = output;
    }

    public void add(final String entry) {
        final byte[] value = entry.getBytes();
        if (value.length > 255) {
            throw new NakedObjectException("Value too long to be encoded: " + entry);
        }
        add(value);
    }

    public void add(final byte[] value) {
        final int len = value.length;
        if (len > 255) {
            throw new NakedObjectException("byte array too long to be encoded");
        }
        add((byte) len);
        try {
            output.write(value);
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
    }

    private void add(final byte len) {
        try {
            output.write(len);
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
    }

    public void add(final String[] list) {
        add((byte) list.length);
        for (int i = 0; i < list.length; i++) {
            add(list[i]);
        }
    }

    public void add(final long value) {
        add(Long.toString(value).getBytes());
    }

    public void add(final int value) {
        add(Integer.toString(value).getBytes());
    }

    public void add(final boolean flag) {
        add(flag ? (byte) 1 : (byte) 0);
    }

    /**
     * Reciprocal of {@link ByteDecoderBuffer#getObject()}.
     */
    public void add(final Object object) {
        // handle null
        if (object == null) {
            add(ENCODED_NULL_OBJECT);
            return;
        }

        // handle boolean
        if (object instanceof Boolean) {
            final Boolean bool = (Boolean) object;
            add(bool.booleanValue() ? ENCODED_BOOLEAN_TRUE : ENCODED_BOOLEAN_FALSE);
            return;
        }

        final Class<? extends Object> clazz = object.getClass();
        final String className = clazz.getName();
        addClassName(className);

        // handle array
        if (clazz.isArray()) {
            final Object[] array = (Object[]) object;
            final int len = array.length;
            add(len);
            for (int i = 0; i < len; i++) {
                add(array[i]);
            }
            return;
        }

        // handle Encodable
        if (object instanceof Encodable) {
            final Encodable encodable = (Encodable) object;
            encodable.encode(this);
            return;
        }

        // last resort: handle Serializable
        if (object instanceof Serializable) {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ObjectOutputStream oos;
            try {
                oos = new ObjectOutputStream(baos);
                oos.writeObject(object);
            } catch (final IOException e) {
                throw new NakedObjectException("Object declared as Serializable but failed to write object: " + object, e);
            }
            final byte[] bytes = baos.toByteArray();
            add(bytes);
            return;
        }

        throw new NakedObjectException("Object is not Encodable or Serializable: " + object);
    }

    private void addClassName(final String className) {
        add(className);
    }

    public void add(final Object[] objects) {
        // handle nulls
        if (objects == null) {
            add(ENCODED_NULL_ARRAY);
            return;
        }

        add(objects.getClass().getComponentType().getName());
        add(objects.length);
        for (int i = 0; i < objects.length; i++) {
            add(objects[i]);
        }
    }

    public void end() {
        try {
            output.flush();
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
    }

}

// Copyright (c) Naked Objects Group Ltd.
