package org.nakedobjects.metamodel.value;

import org.nakedobjects.applib.adapters.EncoderDecoder;
import org.nakedobjects.applib.adapters.Parser;
import org.nakedobjects.applib.value.MultilineString;
import org.nakedobjects.metamodel.adapter.NakedObject;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.metamodel.facets.Facet;
import org.nakedobjects.metamodel.facets.FacetHolder;
import org.nakedobjects.metamodel.facets.value.MultilineStringValueFacet;
import org.nakedobjects.metamodel.runtimecontext.RuntimeContext;
import org.nakedobjects.metamodel.specloader.SpecificationLoader;


public class MultilineStringValueSemanticsProvider extends ValueSemanticsProviderAbstract implements MultilineStringValueFacet {

    public static Class<? extends Facet> type() {
        return MultilineStringValueFacet.class;
    }

    private static final int TYPICAL_LENGTH = 200;
    private static final boolean IMMUTABLE = true;
    private static final boolean EQUAL_BY_CONTENT = true;
    private static final Object DEFAULT_VALUE = null; // no default

    /**
     * Required because implementation of {@link Parser} and {@link EncoderDecoder}.
     */
    public MultilineStringValueSemanticsProvider() {
        this(null, null, null, null);
    }

    public MultilineStringValueSemanticsProvider(
    		final FacetHolder holder,
            final NakedObjectConfiguration configuration, 
            final SpecificationLoader specificationLoader, 
            final RuntimeContext runtimeContext) {
        super(type(), holder, MultilineString.class, TYPICAL_LENGTH, IMMUTABLE, EQUAL_BY_CONTENT, DEFAULT_VALUE, configuration, specificationLoader, runtimeContext);
    }

    // //////////////////////////////////////////////////////////////////
    // Parser
    // //////////////////////////////////////////////////////////////////

    @Override
    protected Object doParse(final Object original, final String text) {
        return new MultilineString(text);
    }

    @Override
    public String titleString(final Object value) {
        return value == null ? "" : stringValue(value);
    }

    // //////////////////////////////////////////////////////////////////
    // EncoderDecoder
    // //////////////////////////////////////////////////////////////////

    @Override
    protected String doEncode(final Object object) {
        final String text = stringValue(object);
        if (text.equals("NULL") || isEscaped(text)) {
            return escapeText(text);
        } else {
            return text;
        }
    }

    @Override
    protected Object doRestore(final String data) {
        if (isEscaped(data)) {
            return new MultilineString(data.substring(1));
        } else {
            return new MultilineString(data);
        }
    }

    private boolean isEscaped(final String text) {
        return text.startsWith("/");
    }

    private String escapeText(final String text) {
        return "/" + text;
    }

    // //////////////////////////////////////////////////////////////////
    // MultiLineStringValueFacet
    // //////////////////////////////////////////////////////////////////

    public NakedObject createValue(final String value) {
        return getRuntimeContext().adapterFor(new MultilineString(value));
    }

    public String stringValue(final NakedObject object) {
        return object == null ? null : stringValue(object.getObject());
    }

    private String stringValue(final Object object) {
        return ((MultilineString) object).getString();
    }


}
// Copyright (c) Naked Objects Group Ltd.
