package org.nakedobjects.metamodel.adapter.oid;

import java.io.Serializable;

import org.nakedobjects.metamodel.commons.encoding.ByteEncoder;
import org.nakedobjects.metamodel.commons.ensure.Assert;
import org.nakedobjects.metamodel.commons.exceptions.NotYetImplementedException;


/**
 * Used as the {@link Oid} for collections, values and <tt>@Aggregated</tt> types.
 * 
 * <p>
 * The Hibernate Object Store has custom handling for collections.
 */
public class AggregatedOid implements Oid, Serializable {

    private static final long serialVersionUID = 1L;

    private final Oid parentOid;
    private final String fieldName;

    public AggregatedOid(final Oid oid, final String id) {
        Assert.assertNotNull("Parent OID required", oid);
        Assert.assertNotNull("Field required", id);
        this.parentOid = oid;
        this.fieldName = id;
    }

    ///////////////////////////////////////////////////////////
    // Properties
    ///////////////////////////////////////////////////////////

    public Object getParentOid() {
        return parentOid;
    }

    public String getFieldName() {
        return fieldName;
    }


    ///////////////////////////////////////////////////////////
    // Previous
    ///////////////////////////////////////////////////////////

    public Oid getPrevious() {
        return null;
    }

    public boolean hasPrevious() {
        return false;
    }

    public void clearPrevious() {}


    ///////////////////////////////////////////////////////////
    // Encodeable
    ///////////////////////////////////////////////////////////

    public void encode(ByteEncoder encoder) {
        parentOid.encode(encoder);
        encoder.add(fieldName);
    }


    ///////////////////////////////////////////////////////////
    // Other OID stuff
    ///////////////////////////////////////////////////////////

    public void copyFrom(final Oid oid) {
        throw new NotYetImplementedException();
    }

    public boolean isTransient() {
        return parentOid.isTransient();
    }

    
    ///////////////////////////////////////////////////////////
    // Value semantics
    ///////////////////////////////////////////////////////////
    
    @Override
    public boolean equals(final Object other) {
        if (other == this) {
            return true;
        }
        if (other == null) {
            return false;
        }
        if (getClass() != other.getClass()) {
            return false;
        }
        return equals((AggregatedOid)other);
    }
    
    public boolean equals(final AggregatedOid other) {
        return other.parentOid.equals(parentOid) && 
               other.fieldName.equals(fieldName);
    }

    @Override
    public int hashCode() {
        int hashCode = 17;
        hashCode = 37 * hashCode + parentOid.hashCode();
        hashCode = 37 * hashCode + fieldName.hashCode();
        return hashCode;
    }

    
    ///////////////////////////////////////////////////////////
    // toString
    ///////////////////////////////////////////////////////////

    @Override
    public String toString() {
        return "AOID[" + parentOid + "," + fieldName + "]";
    }



}

// Copyright (c) Naked Objects Group Ltd.
