package org.nakedobjects.metamodel.facetdecorator;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.nakedobjects.metamodel.commons.component.ApplicationScopedComponent;
import org.nakedobjects.metamodel.commons.debug.DebugString;
import org.nakedobjects.metamodel.exceptions.ReflectionException;
import org.nakedobjects.metamodel.facets.Facet;
import org.nakedobjects.metamodel.facets.FacetHolder;



public class FacetDecoratorSet implements ApplicationScopedComponent {
	
    private final Map<Class<? extends Facet>, FacetDecorator> facetDecoratorByFacetType = new HashMap<Class<? extends Facet>, FacetDecorator>();
    private final Set<FacetDecorator> facetDecoratorSet = new LinkedHashSet<FacetDecorator>();

    //////////////////////////////////////////////////////////////
    // init, shutdown
    //////////////////////////////////////////////////////////////
    
    public void init() {}

    public void shutdown() {}

    //////////////////////////////////////////////////////////////
    // add, get, isEmpty
    //////////////////////////////////////////////////////////////

    public void add(final FacetDecorator decorator) {
        final Class<? extends Facet>[] forFacetTypes = decorator.forFacetTypes();
        for (int i = 0; i < forFacetTypes.length; i++) {
            facetDecoratorByFacetType.put(forFacetTypes[i], decorator);
            facetDecoratorSet.add(decorator);
        }
    }
    
    public void add(final List<FacetDecorator> decorators) {
        for(FacetDecorator decorator: decorators) {
            add(decorator);
        }
    }

	public Set<FacetDecorator> getFacetDecorators() {
		return Collections.unmodifiableSet(facetDecoratorSet);
	}

    public boolean isEmpty() {
        return facetDecoratorByFacetType.isEmpty();
    }

    //////////////////////////////////////////////////////////////
    // decorate
    //////////////////////////////////////////////////////////////

    public void decorateAllHoldersFacets(final FacetHolder holder) {
        if (isEmpty()) {
            return;
        }
        final Class<? extends Facet>[] facetTypes = holder.getFacetTypes();
        for (int i = 0; i < facetTypes.length; i++) {
            final Facet originalFacet = holder.getFacet(facetTypes[i]);
            decorateFacet(originalFacet, holder);
        }
    }

    private void decorateFacet(final Facet facet, final FacetHolder holder) {
        final Class<? extends Facet> cls = facet.facetType();
        final FacetDecorator decorator = (FacetDecorator) facetDecoratorByFacetType.get(cls);
        if (decorator == null) {
            return;
        }
        final Facet decoratedFacet = decorator.decorate(facet, holder);
        if (decoratedFacet != null) {
            ensureAtLeastOneOfTheDecoratingFacetsConsistentWithDecorated(decorator, decoratedFacet);
            facet.setFacetHolder(holder);
            holder.addFacet(decoratedFacet);
        }
    }

    private void ensureAtLeastOneOfTheDecoratingFacetsConsistentWithDecorated(final FacetDecorator decorator, final Facet decoratedFacet) {
        if (!isAtLeastOneOfTheDecoratingFacetsConsistentWithDecorated(decorator, decoratedFacet)) {
        	throw new ReflectionException(
        		"Inconsistent facet type " + decoratedFacet.getClass() + "; expectected one of " + decorator.getFacetTypeNames());
        }
    }

	private boolean isAtLeastOneOfTheDecoratingFacetsConsistentWithDecorated(final FacetDecorator decorator, final Facet decoratedFacet) {
		for (Class<? extends Facet> decoratedFacetType: decorator.forFacetTypes()) {
			if (decoratedFacetType.isAssignableFrom(decoratedFacet.getClass())) {
                return true;
            }
        }
		return false;
	}

    //////////////////////////////////////////////////////////////
    // debugging
    //////////////////////////////////////////////////////////////

    public void debugData(final DebugString str) {
        str.appendTitle("Facet decorators");
        str.indent();
        Set<Class<? extends Facet>> facetTypes = facetDecoratorByFacetType.keySet();
        if (facetTypes.size() == 0) {
        	str.append("none");
        } else {
        	for(final Class<? extends Facet> cls: facetTypes) {
                str.appendln(cls.getName(), facetDecoratorByFacetType.get(cls));
            }
        }
        str.unindent();
    }

}

// Copyright (c) Naked Objects Group Ltd.
