package org.nakedobjects.metamodel.commons.encoding;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;


public class ByteDecoderBuffer implements ByteDecoder {

    private final InputStream input;

    public ByteDecoderBuffer(final InputStream input) {
        this.input = input;
    }

    public String getString() {
        final int len = getByte();
        final byte[] val = new byte[len];
        try {
            final int read = input.read(val);
            if (read != len) {
                throw new NakedObjectException("failed to read all of string: " + new String(val) + ", "
                        + input.available() + " bytes still available");
            }
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
        return new String(val);
    }

    public int getByte() {
        try {
            return input.read();
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
    }

    public byte[] getBytes() {
        final int len = getByte();
        try {
            final byte[] array = new byte[len];
            input.read(array);
            return array;
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
    }

    public String[] getList() {
        final int len = getByte();
        final String[] list = new String[len];
        for (int i = 0; i < list.length; i++) {
            list[i] = getString();
        }
        return list;
    }

    public long getLong() {
        return Long.valueOf(getString()).longValue();
    }

    public boolean getBoolean() {
        return getByte() == 1;
    }

    public Object getObject() {
        final String className = getString();

        // handle nulls
        if (className.equals(ByteEncoderBuffer.ENCODED_NULL_OBJECT)) {
            return null;
        }

        // handle boolean true/false
        if (className.equals(ByteEncoderBuffer.ENCODED_BOOLEAN_TRUE)) {
            return Boolean.TRUE;
        }
        if (className.equals(ByteEncoderBuffer.ENCODED_BOOLEAN_FALSE)) {
            return Boolean.FALSE;
        }

        // before going any further, we need to get hold of the referenced class
        Class<?> cls;
        try {
            cls = Class.forName(className);
        } catch (final ClassNotFoundException e) {
            throw new NakedObjectException("failed to find class " + className, e);
        }

        // handle array, if present
        // TODO combine this with the getObjects method below
        if (className.startsWith("[")) {
            cls = cls.getComponentType();
            final int len = getInt();
            final Object[] array = (Object[]) Array.newInstance(cls, len);
            for (int i = 0; i < array.length; i++) {
                array[i] = getObject();
            }
            return array;
        }

        // Try to use Decoder (through object's constructor)
        Constructor<?> constructor = null;
        try {
            constructor = cls.getConstructor(new Class[] { ByteDecoder.class });
        } catch (final NoSuchMethodException e) {
            // ignore, may not be a Decoder
        }

        if (constructor != null) {
            try {
                return constructor.newInstance(new Object[] { this });
            } catch (final IllegalArgumentException e) {
                throw new NakedObjectException(e);
            } catch (final InstantiationException e) {
                throw new NakedObjectException(e);
            } catch (final IllegalAccessException e) {
                throw new NakedObjectException(e);
            } catch (final InvocationTargetException e) {
                throw new NakedObjectException(e);
            }
        }

        // handle serializable object, if possible
        if (Serializable.class.isAssignableFrom(cls)) {
            final byte[] bytes = getBytes();
            final ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
            ObjectInputStream ois;
            try {
                ois = new ObjectInputStream(bais);
                return ois.readObject();
            } catch (final IOException e) {
                throw new NakedObjectException(e);
            } catch (final ClassNotFoundException e) {
                throw new NakedObjectException(e);
            }
        }

        // can't deserialize
        throw new NakedObjectException("failed to decode object as either Encodable or Serializable");

    }

    public Object[] getObjects() {
        final String arrayClass = getString();
        if (arrayClass.equals(ByteEncoderBuffer.ENCODED_NULL_ARRAY)) {
            return null;
        }
        return createArray(arrayClass);
    }

    private Object[] createArray(final String arrayClass) {
        try {
            final int len = getInt();
            final Class<?> cls = Class.forName(arrayClass);
            final Object[] array = (Object[]) Array.newInstance(cls, len);
            for (int i = 0; i < array.length; i++) {
                array[i] = getObject();
            }
            return array;
        } catch (final ClassNotFoundException e) {
            throw new NakedObjectException(e);
        }
    }

    public int getInt() {
        return Integer.valueOf(getString()).intValue();
    }

    public void end() {
        try {
            final int available = input.available();
            if (available > 0) {
                final byte[] leftOver = new byte[available];
                input.read(leftOver);
                throw new NakedObjectException("more data available after completion of read: " + new String(leftOver));
            }
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
    }

}

// Copyright (c) Naked Objects Group Ltd.
