package org.nakedobjects.nof.core.adapter.value;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.TextEntryParseException;
import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;


public class BigDecimalAdapter extends AbstractValueAdapter {
    private final static NumberFormat DEFAULT_FORMAT = NumberFormat.getNumberInstance();
    private NumberFormat format;
    private BigDecimal value;

    public BigDecimalAdapter() {
        String formatRequired = NakedObjectsContext.getConfiguration().getString(Configuration.ROOT + "value.format.decimal");
        if (formatRequired == null) {
            format = DEFAULT_FORMAT;
        } else {
            setMask(formatRequired);
        }
    }

    public BigDecimalAdapter(final BigDecimal value) {
        this();
        this.value = value;
    }

    // ///// titleString, icon, valueClass ///////

    public String titleString() {
        return isEmpty() ? "" : format.format(value);
    }

    public String getIconName() {
        return "bigdecimal";
    }

    public Class getValueClass() {
        return BigDecimal.class;
    }

    // ///// Accessor ///////

    public BigDecimal bigDecimalValue() {
        return value;
    }

    public Object getObject() {
        return value;
    }

    // ///// Null Handling ///////

    public boolean canClear() {
        return true;
    }

    public void clear() {
        value = null;
    }

    public boolean isEmpty() {
        return value == null;
    }

    // ///// Mutator ///////

    public void setValue(final BigDecimal value) {
        this.value = value;
    }

    protected void doParse(String entry) {
        try {
            value = new BigDecimal(entry);
        } catch (NumberFormatException e) {
            throw new TextEntryParseException("Not an decimal " + entry, e);
        }
    }

    // ///// Encode, Decode ///////

    protected String doEncode() {
        return toPlainString();
    }

    protected void doRestore(String data) {
        value = new BigDecimal(data);
    }

    // ///// Semantics ///////

    public void setMask(String mask) {
        format = new DecimalFormat(mask);
    }

    public int defaultTypicalLength() {
        return 19;
    }

    // /////// toString ///////

    public String toString() {
        return "BigDecimalAdapter: " + (value != null ? toPlainString() : "NULL");
    }

    private String toPlainString() {
        // for dotnet compatibility - toString pre 1.3 was equivalent to toPlainString
        // later.
        try {
            final Class cls = value.getClass();
            try {
                return (String) cls.getMethod("toPlainString", (Class[]) null).invoke(value, (Object[]) null);
            } catch (NoSuchMethodException nsm) {
                return (String) cls.getMethod("toString", (Class[]) null).invoke(value, (Object[]) null);
            }
        } catch (Exception e) {
            throw new NakedObjectRuntimeException(e);
        }

    }

}
// Copyright (c) Naked Objects Group Ltd.
