package org.nakedobjects.nof.core.adapter.value;

import java.math.BigInteger;
import java.text.DecimalFormat;
import java.text.NumberFormat;

import org.nakedobjects.noa.adapter.TextEntryParseException;
import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;


public class BigIntegerAdapter extends AbstractValueAdapter {
    private final static NumberFormat DEFAULT_FORMAT = NumberFormat.getNumberInstance();
    private NumberFormat format;
    private BigInteger value;

    public BigIntegerAdapter() {
        String formatRequired = NakedObjectsContext.getConfiguration().getString(Configuration.ROOT + "value.format.int");
        if (formatRequired == null) {
            format = DEFAULT_FORMAT;
        } else {
            setMask(formatRequired);
        }
    }

    public BigIntegerAdapter(final BigInteger value) {
        this();
        this.value = value;
    }

    // ///// titleString, icon, valueClass ///////

    public String titleString() {
        return isEmpty() ? "" : format.format(value);
    }

    public String getIconName() {
        return "biginteger";
    }

    public Class getValueClass() {
        return BigInteger.class;
    }

    // ///// Accessor ///////

    /**
     * @deprecated use {@link #bigIntegerValue)
     */
    public BigInteger bigDecimalValue() {
        return value;
    }

    public BigInteger bigIntegerValue() {
        return value;
    }

    public Object getObject() {
        return value;
    }

    // ///// Null Handling ///////

    public boolean canClear() {
        return true;
    }

    public void clear() {
        value = null;
    }

    public boolean isEmpty() {
        return value == null;
    }

    // ///// Mutator ///////

    public void setValue(final BigInteger value) {
        this.value = value;
    }

    protected void doParse(String entry) {
        try {
            value = new BigInteger(entry);
        } catch (NumberFormatException e) {
            throw new TextEntryParseException("Not an decimal " + entry, e);
        }
    }

    // ///// Encode, Decode ///////

    protected String doEncode() {
        return value.toString();
    }

    protected void doRestore(String data) {
        value = new BigInteger(data);
    }

    // ///// Semantics ///////

    public void setMask(String mask) {
        format = new DecimalFormat(mask);
    }

    public int defaultTypicalLength() {
        return 19;
    }

    // /////// toString ///////

    public String toString() {
        return "BigIntegerAdapter: " + (value != null ? value.toString() : "NULL");
    }

}
// Copyright (c) Naked Objects Group Ltd.
