package org.nakedobjects.nof.core.adapter.value;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;

import org.nakedobjects.noa.adapter.TextEntryParseException;
import org.nakedobjects.noa.adapter.value.DoubleFloatingPointValue;
import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;


public class DoubleAdapter extends AbstractValueAdapter implements DoubleFloatingPointValue {
    private static final NumberFormat DEFAULT_FORMAT = NumberFormat.getNumberInstance();
    private NumberFormat format = DEFAULT_FORMAT;
    private Double value;
    
    public DoubleAdapter() {
        String formatRequired = NakedObjectsContext.getConfiguration().getString(Configuration.ROOT + "value.format.double");
        if (formatRequired == null) {
            format = DEFAULT_FORMAT;
        } else {
            setMask(formatRequired);
        }
    }

    public DoubleAdapter(final Double value) {
        this();
        this.value = value;
    }

    // ///// titleString, icon, valueClass ///////

    public String titleString() {
        return value == null ? "" : format.format(value);
    }

    public String getIconName() {
        return "double";
    }

    public Class getValueClass() {
        return Double.class;
    }

    // ///// Accessor ///////

    public Double doubleValue() {
        return value;
    }

    public Object getObject() {
        return value;
    }

    // ///// Null Handling ///////

    public boolean canClear() {
        return true;
    }

    public void clear() {
        value = null;
    }

    public boolean isEmpty() {
        return value == null;
    }

    // ///// Mutator ///////

    public void setValue(final Double value) {
        this.value = value;
    }

    protected void doParse(String entry) {
            try {
                value = new Double(format.parse(entry).doubleValue());
            } catch (ParseException e) {
                throw new TextEntryParseException("Not floating point number " + entry, e);
            }
    }

    // ///// Encode, Decode ///////

    protected String doEncode() {
        return value.toString();
    }

    protected void doRestore(String data) {
        value = new Double(data);
    }

    // ///// Semantics ///////

    public void setMask(String mask) {
        format = new DecimalFormat(mask);
    }

    public int defaultTypicalLength() {
        return 22;
    }

    // /////// toString ///////
    public String toString() {
        return "DoubleAdapter: " + value;
    }

}
// Copyright (c) Naked Objects Group Ltd.
