package org.nakedobjects.nof.core.adapter.value;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;

import org.nakedobjects.noa.adapter.TextEntryParseException;
import org.nakedobjects.noa.adapter.value.FloatingPointValue;
import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;


public class FloatAdapter extends AbstractValueAdapter implements FloatingPointValue {
    private static final NumberFormat DEFAULT_FORMAT = NumberFormat.getNumberInstance();
    private NumberFormat format = DEFAULT_FORMAT;
    private Float value;


    public FloatAdapter() {
        String formatRequired = NakedObjectsContext.getConfiguration().getString(Configuration.ROOT + "value.format.float");
        if (formatRequired == null) {
            format = DEFAULT_FORMAT;
        } else {
            setMask(formatRequired);
        }
    }

    public FloatAdapter(final Float value) {
        this();
        this.value = value;
    }

    // ///// titleString, icon, valueClass ///////
    public String titleString() {
        return value == null ? "" : format.format(value);
    }

    public String getIconName() {
        return "float";
    }

    public Class getValueClass() {
        return Float.class;
    }

    // ///// Accessor ///////
 
    public Float floatValue() {
        return value;
    }

    public Object getObject() {
        return value;
    }

    // ///// Null Handling ///////

    public boolean canClear() {
        return true;
    }

    public void clear() {
        value = null;
    }

    public boolean isEmpty() {
        return value == null;
    }

    // ///// Mutator ///////

    public void setValue(final Float value) {
        this.value = value;
    }

    protected void doParse(String entry) {
        try {
            value = new Float(format.parse(entry).floatValue());
        } catch (ParseException e) {
            throw new TextEntryParseException("Not a floating point number " + entry, e);
        }
    }

    // ///// Encode, Decode ///////

    protected String doEncode() {
        return value.toString();
    }

    protected void doRestore(String data) {
        value = new Float(data);
    }

    // ///// Semantics ///////

    public void setMask(String mask) {
        format = new DecimalFormat(mask);
    }

    public int defaultTypicalLength() {
        return 12;
    }

    // /////// toString ///////

    public String toString() {
        return "FloatAdapter: " + value;
    }

}
// Copyright (c) Naked Objects Group Ltd.
