package org.nakedobjects.nof.core.context;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.NakedObjectLoader;
import org.nakedobjects.noa.persist.NakedObjectPersistor;
import org.nakedobjects.noa.reflect.NakedObjectReflector;
import org.nakedobjects.noa.security.Session;
import org.nakedobjects.nof.core.image.TemplateImageLoader;
import org.nakedobjects.nof.core.util.Assert;
import org.nakedobjects.nof.core.util.DebugInfo;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nof.core.util.NakedObjectConfiguration;
import org.nakedobjects.object.MessageBroker;


/**
 * A repository of all the NOF components that are shared by a running system
 */
public abstract class NakedObjectsContext implements DebugInfo {
    private static final Logger LOG = Logger.getLogger(NakedObjectsContext.class);
    private static final boolean DEBUG = false;
    private static NakedObjectsContext singleton;
    private static NakedObjectConfiguration configuration;

    // Session

    /**
     * Terminate the session for the user in the current context.
     */
    public static void closeSession() {
        NakedObjectsContext instance = getInstance();
        //instance.terminateSession();
        instance.clearSession();
    }

    // Debug

    public static DebugInfo[] debug() {
        List l = new ArrayList();
        l.add(getInstance());
        
        l.add(getConfiguration());
        l.add(getReflector());
        l.add(getObjectLoader());
        l.add(getObjectPersistor());

        return (DebugInfo[]) l.toArray(new DebugInfo[l.size()]);
    }

    public void debugData(DebugString debug) {
        debug.appendln("context ", this);
    }

    // context properties
    /**
     * Returns the configuration component.
     */
    public static NakedObjectConfiguration getConfiguration() {
        return configuration;
    }

    /**
     * Returns the adapted Perspective for the user who is using this execution context.
     */
    public static NakedObject getPerspective() {
        String userName = getSession().getUserName();
        NakedObject perspective = getPerspective(userName);
        if (perspective == null) {
            return createPerspective(userName);
        } else {
            return perspective;
        }
    }

    private static NakedObject getPerspective(String userName) {
        NakedCollection instances = getObjectPersistor().findInstances(new PerspectiveCriteria(userName));
        if (instances.size() == 0) {
            return null;
        } else {
            return instances.firstElement();
        }
    }

    private static NakedObject createPerspective(String userName) {
        Perspective perspective = new Perspective();
        NakedObject template = getPerspective("_template");
        if (template == null) {
            perspective.setName("Naked Objects Exploration");
            perspective.setUserName(userName);
            Object[] services;
            services = NakedObjectsContext.getObjectLoader().getServices();
            if (services.length == 0) {
                throw new NakedObjectRuntimeException("No known services");
            }
            for (int i = 0; i < services.length; i++) {
                perspective.addToServices(services[i]);
            }
            LOG.info("creating exploration Perspective for " + userName);
        } else {
            String name = ((Perspective) template.getObject()).getName();
            perspective.setName(name);
            perspective.setUserName(userName);
            List services = ((Perspective) template.getObject()).getServices();
            perspective.getServices().addAll(services);
            LOG.info("creating Perspective, from template, for " + userName);
        }
        NakedObject adapter = NakedObjectsContext.getObjectLoader().createAdapterForTransient(perspective, false);
        NakedObjectsContext.getObjectPersistor().startTransaction();
        NakedObjectsContext.getObjectPersistor().makePersistent(adapter);
        NakedObjectsContext.getObjectPersistor().endTransaction();
        return adapter;
    }

    /**
     * Returns the session representing this user for this execution context.
     */
    public static Session getSession() {
        return getInstance().session();
    }

    /**
     * Returns an descriptive identifier for the execution context.
     */
    public static String getExecutionContextId() {
        return getInstance().executionContextId();
    }

    public static String[] getAllExecutionContextIds() {
        return getInstance().allExecutionContextIds();
    }

    public static ContextDebug getDebugContext(String executionContextId) {
        return getInstance().debugContext(executionContextId);
    }

    /**
     * Returns the singleton providing access to the set of execution contexts.
     */
    public static NakedObjectsContext getInstance() {
        return singleton;
    }

    private static void checkValidState() {
        if (DEBUG) {
            Assert.assertNotNull("no configuration set up", configuration);
            Assert.assertNotNull("no specification loader set up", singleton.reflector());
            Assert.assertNotNull("no object persistor set up", singleton.objectPersistor());
            Assert.assertNotNull("no object loader set up", singleton.objectLoader());
        }
    }

    /**
     * Returns the message broker for this execution context.
     */
    public static MessageBroker getMessageBroker() {
        return getInstance().messageBroker();
    }

    /**
     * Return the object loader for this execution context.
     */
    public static NakedObjectLoader getObjectLoader() {
        checkValidState();
        return getInstance().objectLoader();
    }

    /**
     * Returns the object persistor for this execution context.
     */
    public static NakedObjectPersistor getObjectPersistor() {
        checkValidState();
        return getInstance().objectPersistor();
    }

    /**
     * Return the specification loader for this execution context.
     */
    public static NakedObjectReflector getReflector() {
        checkValidState();
        return getInstance().reflector();
    }

    /**
     * Return the system wide image loader for this execution context.
     */
    public static TemplateImageLoader getTemplateImageLoader() {
        checkValidState();
        return getInstance().templateImageLoader();
    }

    /**
     * Returns the update notifier, which collects the changes made to objects.
     */
    public static UpdateNotifier getUpdateNotifer() {
        checkValidState();
        return getInstance().updateNotifier();
    }

    /**
     * Determine if the execution context is set up and ready to be used. Returns false if any of the
     * following are not set: configuration; specification loader; object loader; persistor; session; or
     * message broker.
     */
    public static boolean isReady() {
        return configuration != null && singleton.reflector() != null && singleton.objectPersistor() != null
                && singleton.objectLoader() != null && singleton.messageBroker() != null;
    }

    /**
     * Resets the singleton, so another can created.
     * 
     * @see #NakedObjects()
     */
    public static void reset() {
        singleton = null;
    }

    /**
     * Creates a new instance of the execution context holder. Will throw a StartupException if an instance
     * has already been created.
     */
    protected NakedObjectsContext() {
        // TODO remove next line once XAT test framework is updated
        reset();
        if (singleton != null) {
            throw new NakedObjectRuntimeException("Naked Objects Singleton already set up");
        }
        singleton = this;
    }

    /**
     * Terminate the session for the user in the current execution context.
     */
    protected abstract void terminateSession();

    /**
     * Find the session for the current execution context.
     */
    protected abstract Session session();

    /**
     * Find the update notifier for the current execution context.
     */
    protected abstract UpdateNotifier updateNotifier();

    /**
     * Generate the id for the current execution context.
     */
    protected abstract String executionContextId();

    protected abstract String[] allExecutionContextIds();

    protected abstract ContextDebug debugContext(String executionContextId);

    /**
     * Find the message broker for the current execution context.
     */
    protected abstract MessageBroker messageBroker();

    /**
     * Find the object loader for the current execution context.
     */
    protected abstract NakedObjectLoader objectLoader();

    /**
     * Find the object persistor for the current execution context.
     */
    protected abstract NakedObjectPersistor objectPersistor();

    public static void setConfiguration(NakedObjectConfiguration configuration) {
        NakedObjectsContext.configuration = configuration;
    }

    /**
     * Initialise the object loader for the current execution context.
     */
    public abstract void setObjectLoader(final NakedObjectLoader loader);

    /**
     * Initialise the object persistor for the current execution context.
     */
    public abstract void setObjectPersistor(final NakedObjectPersistor persistor);

    /**
     * Initialise the session for the current execution context.
     */
    public abstract void setSession(final Session session);

    /**
     * Terminate the session for the current execution context.
     */
    public abstract void clearSession();

    /**
     * Initialise the specification loader for the current execution context.
     */
    public abstract void setReflector(final NakedObjectReflector loader);

    /**
     * Initialise the template image loader for the current execution context.
     */
    public abstract void setTemplateImageLoader(final TemplateImageLoader loader);

    /**
     * Find the specification loader for the current execution context.
     */
    protected abstract NakedObjectReflector reflector();

    /**
     * Find the template image loader for the current execution context.
     */
    protected abstract TemplateImageLoader templateImageLoader();

    public static void shutdown() {
        LOG.info("shutting down");
        NakedObjectsContext instance = getInstance();
        instance.shutdownSession();
    }
    
    public abstract void shutdownSession();

}
// Copyright (c) Naked Objects Group Ltd.
