package org.nakedobjects.nof.core.image.java;

import java.awt.Canvas;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Toolkit;
import java.io.File;
import java.net.URL;
import java.util.Hashtable;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.image.TemplateImage;
import org.nakedobjects.nof.core.image.TemplateImageLoader;

/**
 * This class loads up file based images as resources (part of the classpath) or from the file system. Images
 * of type PNG, GIF and JPEG will be used.  The default directory is images.
 */
class AwtTemplateImageLoader implements TemplateImageLoader {
    private static final String[] EXTENSIONS = { "png", "gif", "jpg", "jpeg" };
    private final static Logger LOG = Logger.getLogger(AwtTemplateImageLoader.class);
    private final static String IMAGE_DIRECTORY = "images";
    private final static String IMAGE_DIRECTORY_PARAM = Properties.PROPERTY_BASE + "image-directory";
    private static final String SEPARATOR = "/";

    private Boolean alsoLoadAsFiles = null;
    protected final MediaTracker mt = new MediaTracker(new Canvas());
    /**
     * A keyed list of core images, one for each name, keyed by the image path.
     */
    private Hashtable loadedImages = new Hashtable();
    private Vector missingImages = new Vector();
    private String directory;

    /*
     * private Image createImage() { byte[] pixels = new byte[128 * 128]; for (int i = 0; i < pixels.length;
     * i++) { pixels[i] = (byte) (i % 128); }
     * 
     * byte[] r = new byte[] { 0, 127 }; byte[] g = new byte[] { 0, 127 }; byte[] b = new byte[] { 0, 127 };
     * IndexColorModel colorModel = new IndexColorModel(1, 2, r, g, b);
     * 
     * Image image = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(128, 128, colorModel,
     * pixels, 0, 128));
     * 
     * return image; }
     */

    /**
     * Returns an image template for the specified image (as specified by a path to a file or resource). If
     * the path has no extension (.gif, .png etc) then all valid extensions are searched for.
     * 
     * This method attempts to load the image from the jar/zip file this class was loaded from ie, your
     * application, and then from the file system as a file if can't be found as a resource. If neither method
     * works the default image is returned.
     * 
     * @return returns a TemplateImage for the specified image file, or null if none found.
     */
    public TemplateImage getTemplateImage(final String name) {
        String path = directory() + name;

        final int extensionAt = path.lastIndexOf('.');
        final String root = extensionAt == -1 ? path : path.substring(0, extensionAt);

        if (loadedImages.contains(root)) {
            return (AwtTemplateImage) loadedImages.get(root);

        } else if (missingImages.contains(root)) {
            return null;

        } else {
            return loadImage(path, extensionAt, root);
        }
    }

    private TemplateImage loadImage(String path, final int extensionAt, final String root) {
        LOG.debug("searching for image " + path);

        Image image = null;
        if (extensionAt >= 0) {
            image = load(path);
            return AwtTemplateImage.create(image);
        } else {
            for (int i = 0; i < EXTENSIONS.length; i++) {
                image = load(root + "." + EXTENSIONS[i]);
                if (image != null) {
                    return AwtTemplateImage.create(image);
                }
            }
        }
        LOG.debug("failed to find image for " + path);
        missingImages.addElement(root);
        return null;
    }

    private String directory() {
        if (directory == null) {
            directory = NakedObjectsContext.getConfiguration().getString(IMAGE_DIRECTORY_PARAM, IMAGE_DIRECTORY);
            if (!directory.endsWith(SEPARATOR)) {
                directory = directory.concat(SEPARATOR);
            }
        }
        return directory;
    }

    public void init() {
        LOG.info("images to be loaded from " + directory());
    }
    
    private Image load(final String path) {
        Image image = loadAsResource(path);
        if (image == null) {
        	if (alsoLoadAsFiles == null) {
                alsoLoadAsFiles = new Boolean(NakedObjectsContext.getConfiguration().getBoolean(Properties.PROPERTY_BASE + "load-images-from-files", true));
        	}
        	if (alsoLoadAsFiles.booleanValue()) {
        		image = loadAsFile(path);
        	}
        }
        return image;
    }

    /**
     * Get an Image object from the specified file path on the file system.
     */
    private Image loadAsFile(final String path) {
        final File file = new File(path);

        if (!file.exists()) {
            return null;
        } else {
            Toolkit t = Toolkit.getDefaultToolkit();
            Image image = t.getImage(file.getAbsolutePath());

            if (image != null) {
                mt.addImage(image, 0);

                try {
                    mt.waitForAll();
                } catch (Exception e) {
                    e.printStackTrace();
                }

                if (mt.isErrorAny()) {
                    LOG.error("found image file but failed to load it: " + file.getAbsolutePath());
                    mt.removeImage(image);
                    image = null;
                } else {
                    mt.removeImage(image);
                    LOG.info("image loaded from file: " + file);
                }
            }
            return image;
        }
    }

    /**
     * Get an Image object from the jar/zip file that this class was loaded from.
     */
    protected Image loadAsResource(final String path) {
        URL url = AwtTemplateImageLoader.class.getResource("/" + path);
        if (url == null) {
            LOG.debug("not found image in resources: " + url);
            return null;
        }

        Image image = Toolkit.getDefaultToolkit().getImage(url);
        if (image != null) {
            mt.addImage(image, 0);
            try {
                mt.waitForAll();
            } catch (Exception e) {
                e.printStackTrace();
            }

            if (mt.isErrorAny()) {
                LOG.error("found image but failed to load it from resources: " + url + " " + mt.getErrorsAny()[0]);
                mt.removeImage(image);
                image = null;
            } else {
                mt.removeImage(image);
                LOG.info("image loaded from resources: " + url);
            }
        }

        if (image == null || image.getWidth(null) == -1) {
            throw new RuntimeException(image.toString());
        }

        return image;
    }
    
    public void shutdown() {}
}
// Copyright (c) Naked Objects Group Ltd.
