package org.nakedobjects.nof.core.persist;

import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.persist.InstancesCriteria;
import org.nakedobjects.noa.persist.NakedObjectPersistor;
import org.nakedobjects.noa.persist.ObjectNotFoundException;
import org.nakedobjects.noa.persist.UnsupportedFindException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.util.DebugInfo;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nof.core.util.Logger;


public class ObjectPersistorLogger extends Logger implements NakedObjectPersistor, DebugInfo {
    private final NakedObjectPersistor decorated;

    public ObjectPersistorLogger(final NakedObjectPersistor decorated, final String logFileName) {
        super(logFileName, false);
        this.decorated = decorated;
    }

    public ObjectPersistorLogger(final NakedObjectPersistor decorated) {
        super(null, false);
        this.decorated = decorated;
    }

    public void abortTransaction() {
        log("abort transaction");
        decorated.abortTransaction();
    }

    public Oid createTransientOid(final Object object) {
        Oid oid = decorated.createTransientOid(object);
        log("create oid ", oid);
        return oid;
    }

    public void convertPersistentToTransientOid(final Oid oid) {
        decorated.convertPersistentToTransientOid(oid);
        log("convert persistent to transient Oid", oid);
    }

    public void convertTransientToPersistentOid(final Oid oid) {
        decorated.convertTransientToPersistentOid(oid);
        log("convert transient to persistent Oid", oid);
    }

    public void destroyObject(final NakedObject object) {
        log("destroy " + object.getObject());
        decorated.destroyObject(object);
    }

    public void endTransaction() {
        log("end transaction");
        decorated.endTransaction();
    }

    public NakedCollection findInstances(final InstancesCriteria criteria) throws UnsupportedFindException {
        log("find instances matching " + criteria);
        return decorated.findInstances(criteria);
    }

    public void debugData(final DebugString debug) {
        if (decorated instanceof DebugInfo) {
            ((DebugInfo) decorated).debugData(debug);
        }
    }

    public String debugTitle() {
        if (decorated instanceof DebugInfo) {
            return ((DebugInfo) decorated).debugTitle();
        } else {
            return "";
        }
    }

    protected Class getDecoratedClass() {
        return decorated.getClass();
    }

    public NakedObject getObject(final Oid oid, final NakedObjectSpecification hint) throws ObjectNotFoundException {
        NakedObject object = decorated.getObject(oid, hint);
        log("get object for " + oid + " (of type " + hint.getShortName() + ")", object.getObject());
        return object;
    }

    public Oid getOidForService(String name) {
        Oid oid = decorated.getOidForService(name);
        log("get oid for " + name + ": " + oid);
        return oid;
    }

    public boolean hasInstances(final NakedObjectSpecification specification, final boolean includeSubclasses) {
        boolean hasInstances = decorated.hasInstances(specification, false);
        log("has instances of " + specification.getShortName(), "" + hasInstances);
        return hasInstances;
    }

    public boolean isInitialized() {
        boolean isInitialized = decorated.isInitialized();
        log("is initialized: " + isInitialized);
        return isInitialized;
    }

    public void init() {
        log("initialising " + decorated);
        decorated.init();
    }

    public void makePersistent(final NakedObject object) {
        log("make object graph persistent: " + object);
        decorated.makePersistent(object);
    }

    public void objectChanged(final NakedObject object) {
        log("object changed " + object);
        decorated.objectChanged(object);
    }

    public void setServices(Object[] services) {
        decorated.setServices(services);
        StringBuffer servicesList = new StringBuffer();
        for (int i = 0; i < services.length; i++) {
            if (i > 0) {
                servicesList.append(',');
            }
            servicesList.append(services[i].getClass().getName());
        }
        log("registered services {" + servicesList+ "}");
    }

    public void reload(final NakedObject object) {
        decorated.reload(object);
        log("reload: " + object);
    }

    public void reset() {
        log("reset object manager");
        decorated.reset();
    }

    public void resolveImmediately(final NakedObject object) {
        decorated.resolveImmediately(object);
        log("Resolve immediately: " + object);
    }

    public void resolveField(final NakedObject object, final NakedObjectField field) {
        log("resolve eagerly object in field " + field + " of " + object);
        decorated.resolveField(object, field);
    }

    public void saveChanges() {
        log("saving changes");
        decorated.saveChanges();
    }

    public void shutdown() {
        log("shutting down " + decorated);
        decorated.shutdown();
        close();
    }

    public void startTransaction() {
        log("start transaction");
        decorated.startTransaction();
    }

    public boolean flushTransaction() {
        log("flush transaction");
        return decorated.flushTransaction();
    }
}
// Copyright (c) Naked Objects Group Ltd.
