package org.nakedobjects.nof.core.persist;

import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.util.ByteDecoder;
import org.nakedobjects.noa.util.ByteEncoder;
import org.nakedobjects.noa.util.Encodable;
import org.nakedobjects.nof.core.util.Assert;

import java.io.Serializable;


public final class SerialOid implements Oid, Encodable, Serializable {
    private static final long serialVersionUID = 1L;
	private static final boolean PERSISTENT = false;
    private static final boolean TRANSIENT = true;

    public static SerialOid createPersistent(final long serialNo) {
        return new SerialOid(serialNo, PERSISTENT);
    }

    public static SerialOid createTransient(final long serialNo) {
        return new SerialOid(serialNo, TRANSIENT);
    }

    private int hashCode;
    private boolean isTransient;
    private SerialOid previous;
    private long serialNo;
    private String toString;
    
    private SerialOid(final long serialNo, final boolean isTransient) {
        this.serialNo = serialNo;
        this.isTransient = isTransient;
        cacheState();
    }

    public SerialOid(ByteDecoder decoder) {
        serialNo = decoder.getLong();
        isTransient = decoder.getBoolean();
        boolean hasPrevious = decoder.getBoolean();
        if(hasPrevious) {
            long previousSerialNo = decoder.getLong();
            boolean previousIsTransient = decoder.getBoolean();
            previous = new SerialOid(previousSerialNo, previousIsTransient);
        }
        cacheState();
    }

    public void encode(ByteEncoder encoder) {
        encoder.add(serialNo);
        encoder.add(isTransient);
        encoder.add(previous != null);
        if (previous != null) {
            encoder.add(previous.serialNo);
            encoder.add(previous.isTransient);
        }
    }

    private void cacheState() {
        hashCode = 17;
        hashCode = 37 * hashCode + (int) (serialNo ^ (serialNo >>> 32));
        hashCode = 37 * hashCode + (isTransient ? 0 : 1);
        toString = (isTransient ? "T" : "") + "OID#" + Long.toHexString(serialNo).toUpperCase() + (previous == null ? "" : "+");
    }

    public void copyFrom(final Oid oid) {
        Assert.assertTrue(oid instanceof SerialOid);
        SerialOid from = (SerialOid) oid;
        this.serialNo = from.serialNo;
        this.isTransient = from.isTransient;
        cacheState();
    }

    void makePersistent(final long serialNo) {
        Assert.assertTrue(isTransient);
        previous = new SerialOid(this.serialNo, isTransient);
        this.serialNo = serialNo;
        this.isTransient = false;
        cacheState();
    }

    /*
     * public void setPrevious(SerialOid previous) { Assert.assertNull(previous); this.previous = previous; }
     */

    public boolean equals(final Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof SerialOid) {
            SerialOid otherOid = ((SerialOid) obj);
            return otherOid.serialNo == serialNo && otherOid.isTransient == isTransient;
        }
        return false;
    }

    /**
     * Overloaded to allow compiler to link directly if we know the compile-time type. (possible performance
     * improvement - called 166,000 times in normal ref data fixture.
     */
    public boolean equals(final SerialOid otherOid) {
        if (otherOid == this) {
            return true;
        }
        return otherOid.serialNo == serialNo && otherOid.isTransient == isTransient;
    }

    public Oid getPrevious() {
        return previous;
    }

    public long getSerialNo() {
        return serialNo;
    }

    public int hashCode() {
        return hashCode;
    }

    public boolean hasPrevious() {
        return previous != null;
    }

    public boolean isTransient() {
        return isTransient;
    }

    public String toString() {
        return toString;
    }
}
// Copyright (c) Naked Objects Group Ltd.
