package org.nakedobjects.nof.core.security;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.StringTokenizer;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.security.AuthenticationRequest;
import org.nakedobjects.noa.security.Authenticator;
import org.nakedobjects.nof.core.util.Assert;


public class PasswordFileAuthenticator implements Authenticator {
    private static final boolean FAILED_AUTHENTICATION = false;
    private static final boolean AUTHENTICATED = true;
    private final File passwordFile;

    public PasswordFileAuthenticator(File file) {
        passwordFile = file;
    }

    private final void setRoles(final String line, AuthenticationRequest request) {
        String[] roles;
        final StringTokenizer tokens = new StringTokenizer(line, "|", false);
        roles = new String[tokens.countTokens()];
        for (int i = 0; tokens.hasMoreTokens(); i++) {
            roles[i] = tokens.nextToken();
        }
        request.setRoles(roles);
    }

    public final boolean isValid(AuthenticationRequest request) {
        PasswordAuthenticationRequest passwordRequest = (PasswordAuthenticationRequest) request;
        String username = passwordRequest.getName();
        if (username == null || username.equals("")) {
            return FAILED_AUTHENTICATION;
        }
        String password = passwordRequest.getPassword();
        Assert.assertNotNull(password);

        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(new FileInputStream(passwordFile)));
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.startsWith("#") || line.trim().length() == 0) {
                    continue;
                }
                int posFirstColon = line.indexOf(':');
                if (posFirstColon == -1) {
                    throw new NakedObjectRuntimeException("Invalid entry in password file - no colon (:) found on line: "
                            + line);
                }
                String name = line.substring(0, posFirstColon);
                if (name.equals(username)) {
                    int posSecondColon = line.indexOf(':', posFirstColon + 1);
                    String expected = line.substring(posFirstColon + 1, posSecondColon == -1 ? line.length() : posSecondColon);
                    if (checkPassword(expected, password)) {
                        if (posSecondColon != -1) {
                            setRoles(line.substring(posSecondColon + 1), request);
                        }
                        return AUTHENTICATED;
                    } else {
                        return FAILED_AUTHENTICATION;
                    }
                }
            }
        } catch (IOException e) {
            throw new NakedObjectRuntimeException("Failed to open or read password file: " + passwordFile.getAbsolutePath());
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException ignore) {}
            }
        }

        return FAILED_AUTHENTICATION;
    }

    protected boolean checkPassword(String expected, String password) {
        return expected.equals(password);
    }

    public final boolean canAuthenticate(AuthenticationRequest request) {
        return request instanceof PasswordAuthenticationRequest;
    }

}

// Copyright (c) Naked Objects Group Ltd.
