package org.nakedobjects.nof.core.adapter;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.adapter.NakedReference;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.adapter.Persistable;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.adapter.Version;
import org.nakedobjects.noa.persist.ConcurrencyException;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.Assert;
import org.nakedobjects.nof.core.util.ToString;


public abstract class AbstractNakedReference implements NakedReference {
    private final static Logger LOG = Logger.getLogger(AbstractNakedReference.class);
    private String defaultTitle;
    private Oid oid;
    private transient ResolveState resolveState;
    private NakedObjectSpecification specification;
    private Version version;

    public AbstractNakedReference() {
        resolveState = ResolveState.NEW;
    }

    public void changeState(final ResolveState newState) {
        Assert.assertTrue("can't change to " + newState.name() + ": " + this, resolveState.isValidToChangeTo(newState));
        LOG.debug("recreate - change state " + this + " to " + newState.name());
        resolveState = newState;
    }

    public void checkLock(final Version version) {
        if (this.version != null && this.version.different(version)) {
            LOG.info("concurrency conflict on " + this + " (" + version + ")");
            throw new ConcurrencyException(this, version); 
        }
    }

    protected String getDefaultTitle() {
        return defaultTitle;
    }

    /**
     * Returns the name of the icon to use to represent this object.
     */
    public String getIconName() {
        return getSpecification().getIconName(this);
    }

    public Oid getOid() {
        return oid;
    }

    public ResolveState getResolveState() {
        return resolveState;
    }

    public NakedObjectSpecification getSpecification() {
        if (specification == null) {
            specification = NakedObjectsContext.getReflector().loadSpecification(getObject().getClass());
            defaultTitle = "A" + (" " + specification.getSingularName()).toLowerCase();
        }
        return specification;
    }

    public Version getVersion() {
        return version;
    }

    public Persistable persistable() {
        return getSpecification().persistable();
    }

    protected void setOid(final Oid oid) {
        if(oid == null) {
            throw new NullPointerException();
        }
        this.oid = oid;
    }

    public void setOptimisticLock(final Version version) {
        if (shouldSetVersion(version)) {
            this.version = version;
        }
    }

    private boolean shouldSetVersion(final Version version) {
        return this.version == null || version == null || version.different(this.version);
    }

    protected void toString(final ToString str) {
        str.append(resolveState.code());
        Oid oid = getOid();
        if (oid != null) {
            str.append(":");
            str.append(oid.toString().toUpperCase());
        } else {
            str.append(":-");
        }
        str.setAddComma();
        if (specification == null) {
            str.append("class", getObject().getClass().getName());
        } else {
            str.append("specification", specification.getShortName());
        }
        str.append("version", version == null ? null : version.sequence());
    }
}
// Copyright (c) Naked Objects Group Ltd.
