package org.nakedobjects.nof.core.adapter;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.reflect.NakedObjectAction;
import org.nakedobjects.noa.reflect.NakedObjectActionInstance;
import org.nakedobjects.noa.reflect.NakedObjectActionInstanceImpl;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.OneToManyAssociation;
import org.nakedobjects.noa.reflect.OneToManyAssociationInstance;
import org.nakedobjects.noa.reflect.OneToManyAssociationInstanceImpl;
import org.nakedobjects.noa.reflect.OneToOneAssociation;
import org.nakedobjects.noa.reflect.OneToOneAssociationInstance;
import org.nakedobjects.noa.reflect.OneToOneAssociationInstanceImpl;
import org.nakedobjects.noa.reflect.ValueAssociation;
import org.nakedobjects.noa.reflect.ValueAssociationInstance;
import org.nakedobjects.noa.reflect.ValueAssociationInstanceImpl;
import org.nakedobjects.noa.reflect.listeners.ObjectEvent;
import org.nakedobjects.noa.reflect.listeners.ObjectListener;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.ToString;


public class PojoAdapter extends AbstractNakedReference implements NakedObject {
    
    private final static Logger LOG = Logger.getLogger(PojoAdapter.class);
    private Object pojo;

    public PojoAdapter(final Object pojo, final Oid oid) {
        this.pojo = pojo;
        setOid(oid);
    }

    protected void finalize() throws Throwable {
        super.finalize();
        LOG.debug("finalizing pojo: " + pojo);
    }

    public Object getObject() {
        return pojo;
    }

    /**
     * Returns the title from the underlying business object. If the object has not yet been resolved the
     * specification will be asked for a unresolved title, which could of been persisted by the persistence
     * mechanism. If either of the above provides null as the title then this method will return a title
     * relating to the name of the object type, e.g. "A Customer", "A Product".
     */
    public String titleString() {
        if (getOid() == null) {
            throw new NakedObjectRuntimeException("OID not initialised");
        }

        NakedObjectSpecification specification = getSpecification();
        String title = specification.getTitle(this);
        if (title == null) {
            ResolveState resolveState = getResolveState();
            if (resolveState.isGhost()) {
                LOG.info("attempting to use unresolved object; resolving it immediately: " + this);
                NakedObjectsContext.getObjectPersistor().resolveImmediately(this);
            }
        }
        if (title == null) {
            title = getDefaultTitle();
        }
        return title;
    }

    public synchronized String toString() {
        ToString str = new ToString(this);
        toString(str);

        ResolveState resolveState = getResolveState();
        if (resolveState.isTransient() || resolveState.isResolved()) {
            // don't do title of unresolved objects as this may force the resolving of the object.
            str.append("title", titleString());
        }
        str.appendAsHex("pojo-hash", pojo.hashCode());
        return str.toString();
    }

    private Map actionInstancesByType = new HashMap();

    public NakedObjectActionInstance[] getActionInstances(NakedObjectAction.Type type) {
        NakedObjectActionInstance[] actionInstances = (NakedObjectActionInstance[]) actionInstancesByType.get(type);
        if (actionInstances == null) {
            Vector objectActionList = new Vector();
            appendToActionList(getSpecification().getObjectActions(type), objectActionList);
            actionInstances = toArray(objectActionList);
            actionInstancesByType.put(type, actionInstances);
        }
        return actionInstances;
    }

    private void appendToActionList(
            NakedObjectAction[] nakedObjectActions,
            Vector objectActionList) {
        for (int i = 0; i < nakedObjectActions.length; i++) {
            final NakedObjectAction noa = nakedObjectActions[i];
            final NakedObjectAction[] subActions = noa.getActions();
            if (subActions.length > 0) {
                appendToActionList(subActions, objectActionList);
            } else {
                objectActionList.add(noa);
            }
        }
    }

    private NakedObjectActionInstance[] toArray(
            Vector objectActionList) {
        NakedObjectActionInstance[] objectActions = new NakedObjectActionInstance[objectActionList.size()];
        Enumeration actionEnum = objectActionList.elements();
        int i=0;
        while(actionEnum.hasMoreElements()) {
            final NakedObjectAction noa = (NakedObjectAction) actionEnum.nextElement();
            objectActions[i++] = new NakedObjectActionInstanceImpl(this, noa);
        }
        return objectActions;
    }

    // TODO: use a Hashtable instead
    public NakedObjectActionInstance getActionInstance(NakedObjectAction action) {
        NakedObjectActionInstance[] actionInstances = getActionInstances(action.getType());
        for (int i = 0; i < actionInstances.length; i++) {
            if (actionInstances[i].getNakedObjectAction().getId().equals(action.getId())) {
                return actionInstances[i];
            }
        }
        return null;
    }

    private OneToManyAssociationInstance[] oneToManyAssociationInstances;

    public OneToManyAssociationInstance[] getOneToManyAssociationInstances() {
        if (oneToManyAssociationInstances == null) {
            NakedObjectField[] fields = getSpecification().getFields(NakedObjectField.COLLECTIONS);
            oneToManyAssociationInstances = new OneToManyAssociationInstance[fields.length];
            for (int i = 0; i < fields.length; i++) {
                oneToManyAssociationInstances[i] = new OneToManyAssociationInstanceImpl(this, (OneToManyAssociation) fields[i]);
            }
        }
        return oneToManyAssociationInstances;
    }

    // TODO: use a Hashtable instead
    public OneToManyAssociationInstance getOneToManyAssociation(OneToManyAssociation field) {
        OneToManyAssociationInstance[] oneToManyAssociationInstances = getOneToManyAssociationInstances();
        for (int i = 0; i < oneToManyAssociationInstances.length; i++) {
            if (oneToManyAssociationInstances[i].getOneToManyAssociation() == field) {
                return oneToManyAssociationInstances[i];
            }
        }
        return null;
    }

    private OneToOneAssociationInstance[] oneToOneAssociationInstances;

    public OneToOneAssociationInstance[] getOneToOneAssociationInstances() {
        if (oneToOneAssociationInstances == null) {
            NakedObjectField[] fields = getSpecification().getFields(NakedObjectField.REFERENCES);
            oneToOneAssociationInstances = new OneToOneAssociationInstance[fields.length];
            for (int i = 0; i < fields.length; i++) {
                oneToOneAssociationInstances[i] = new OneToOneAssociationInstanceImpl(this, (OneToOneAssociation) fields[i]);
            }
        }
        return oneToOneAssociationInstances;
    }

    // TODO: use a Hashtable instead
    public OneToOneAssociationInstance getOneToOneAssociation(OneToOneAssociation field) {
        OneToOneAssociationInstance[] oneToOneAssociationInstances = getOneToOneAssociationInstances();
        for (int i = 0; i < oneToOneAssociationInstances.length; i++) {
            if (oneToOneAssociationInstances[i].getOneToOneAssociation() == field) {
                return oneToOneAssociationInstances[i];
            }
        }
        return null;
    }

    private ValueAssociationInstance[] valueAssociationInstances;

    public ValueAssociationInstance[] getValueAssociationInstances() {
        if (valueAssociationInstances == null) {
            NakedObjectField[] fields = getSpecification().getFields(NakedObjectField.VALUES);
            valueAssociationInstances = new ValueAssociationInstance[fields.length];
            for (int i = 0; i < fields.length; i++) {
                valueAssociationInstances[i] = new ValueAssociationInstanceImpl(this, (ValueAssociation) fields[i]);
            }
        }
        return valueAssociationInstances;
    }

    // TODO: use a Hashtable instead
    public ValueAssociationInstance getValueAssociation(ValueAssociation field) {
        ValueAssociationInstance[] valueAssociationInstances = getValueAssociationInstances();
        for (int i = 0; i < valueAssociationInstances.length; i++) {
            if (valueAssociationInstances[i].getValueAssociation().getId().equals(field.getId())) {
                return valueAssociationInstances[i];
            }
        }
        return null;
    }

    private List listeners = new ArrayList();
    public void addObjectListener(
            ObjectListener listener) {
        if (listeners.contains(listener)) { return; }
        listeners.add(listener);
    }

    public void removeObjectListener(
            ObjectListener listener) {
        listeners.remove(listener);
    }
    
    public void fireChangedEvent() {
        ObjectEvent e = new ObjectEvent(this);
        List list = new ArrayList(listeners); // safeguard in case any listeners de-register themselves
        for(Iterator iter = list.iterator(); iter.hasNext(); ) {
            ObjectListener l = (ObjectListener) iter.next();
            l.objectChanged(e);
        }
    }

    private void fireInvalidEvent(
            String reason) {
        ObjectEvent e = new ObjectEvent(this, reason);
        List list = new ArrayList(listeners); // safeguard in case any listeners de-register themselves
        for(Iterator iter = list.iterator(); iter.hasNext(); ) {
            ObjectListener l = (ObjectListener) iter.next();
            l.objectInvalid(e);
        }
    }

    private void fireValidEvent() {
        ObjectEvent e = new ObjectEvent(this);
        List list = new ArrayList(listeners); // safeguard in case any listeners de-register themselves
        for(Iterator iter = list.iterator(); iter.hasNext(); ) {
            ObjectListener l = (ObjectListener) iter.next();
            l.objectValid(e);
        }
    }


}
// Copyright (c) Naked Objects Group Ltd.
