package org.nakedobjects.nof.core.adapter.value;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.TimeZone;

import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.adapter.TextEntryParseException;
import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;


public abstract class AbstractTemporalValueAdapter extends AbstractValueAdapter {
    protected static final String ISO_ENCODING_FORMAT = "iso_encoding";
    private static final TimeZone UTC_TIME_ZONE;
    private static DateFormat encodingFormat;

    static {
        TimeZone timeZone = TimeZone.getTimeZone("Etc/UTC");
        if (timeZone == null) {
            // for dotnet compatibility - Etc/UTC fails in dotnet
            timeZone = TimeZone.getTimeZone("UTC");
        }
        UTC_TIME_ZONE = timeZone;
    }

    protected static DateFormat createDateFormat(String mask) {
        return new SimpleDateFormat(mask);
    }

    protected DateFormat format;

    public AbstractTemporalValueAdapter(final String propertyName) {
        Hashtable formats = formats();
        Enumeration elements = formats.elements();
        while (elements.hasMoreElements()) {
            DateFormat format = (DateFormat) elements.nextElement();
            format.setLenient(false);
            if (ignoreTimeZone()) {
                format.setTimeZone(UTC_TIME_ZONE);
            }
        }
        String defaultFormat = NakedObjectsContext.getConfiguration().getString(
                Configuration.ROOT + "value.format." + propertyName, defaultFormat());
        String required = defaultFormat.toLowerCase().trim();
        format = (DateFormat) formats.get(required);
        if (format == null) {
            setMask(defaultFormat);
        }

        encodingFormat = (DateFormat) formats.get(ISO_ENCODING_FORMAT);
    }

    protected abstract void add(int years, int months, int days, int hours, int minutes);

    public boolean canClear() {
        return true;
    }

    protected void clearFields(Calendar cal) {}

    protected abstract Date dateValue();

    protected abstract String defaultFormat();

    public int defaultTypicalLength() {
        return 12;
    }

    protected abstract Hashtable formats();

    public String getIconName() {
        return "date";
    }

    public Oid getOid() {
        return null;
    }

    protected boolean ignoreTimeZone() {
        return false;
    }

    protected abstract void now();

    protected abstract void setDate(Date date);

    public void setMask(String mask) {
        format = new SimpleDateFormat(mask);
        format.setLenient(false);
        format.setTimeZone(UTC_TIME_ZONE);
    }

    public String titleString() {
        Date date = dateValue();
        return date == null ? "" : format.format(date);
    }

    // // Parsing

    private void parseDate(String dateString) {
        try {
            setDate(format.parse(dateString));
        } catch (ParseException e) {
            Hashtable formats = formats();
            Enumeration elements = formats.elements();
            setDate(parseDate(dateString, elements));
        }
    }

    private Date parseDate(final String dateString, final Enumeration elements) {
        DateFormat format = (DateFormat) elements.nextElement();
        try {
            return format.parse(dateString);
        } catch (ParseException e) {
            if (elements.hasMoreElements()) {
                return parseDate(dateString, elements);
            } else {
                throw new TextEntryParseException("Not recognised as a date: " + dateString);
            }
        }
    }

    protected void doParse(String entry) {
        String dateString = entry.trim();
        String str = dateString.toLowerCase();
        if (str.equals("today") || str.equals("now")) {
            now();
        } else if (dateString.startsWith("+")) {
            relativeDate(dateString, true);
        } else if (dateString.startsWith("-")) {
            relativeDate(dateString, false);
        } else {
            parseDate(dateString);
        }
    }

    private void relativeDate(String str, boolean add) {
        if (isEmpty()) {
            now();
        }

        StringTokenizer st = new StringTokenizer(str.substring(1), " ");
        while (st.hasMoreTokens()) {
            String token = st.nextToken();
            relativeDate2(token, add);
        }
    }

    private void relativeDate2(String str, boolean add) {
        int hours = 0;
        int minutes = 0;
        int days = 0;
        int months = 0;
        int years = 0;

        if (str.endsWith("H")) {
            str = str.substring(0, str.length() - 1);
            hours = Integer.valueOf(str).intValue();
        } else if (str.endsWith("M")) {
            str = str.substring(0, str.length() - 1);
            minutes = Integer.valueOf(str).intValue();
        } else if (str.endsWith("w")) {
            str = str.substring(0, str.length() - 1);
            days = 7 * Integer.valueOf(str).intValue();
        } else if (str.endsWith("y")) {
            str = str.substring(0, str.length() - 1);
            years = Integer.valueOf(str).intValue();
        } else if (str.endsWith("m")) {
            str = str.substring(0, str.length() - 1);
            months = Integer.valueOf(str).intValue();
        } else if (str.endsWith("d")) {
            str = str.substring(0, str.length() - 1);
            days = Integer.valueOf(str).intValue();
        } else {
            days = Integer.valueOf(str).intValue();
        }

        if (add) {
            add(years, months, days, hours, minutes);
        } else {
            add(-years, -months, -days, -hours, -minutes);
        }
    }

    // // Encoding

    protected String doEncode() {
        Date date = dateValue();
        return encodingFormat.format(date);
    }

    protected void doRestore(String data) {
        try {
            Calendar cal = Calendar.getInstance();
            cal.setTime(encodingFormat.parse(data));
            clearFields(cal);
            setDate(cal.getTime());
        } catch (ParseException e) {
            throw new EncodingException(e);
        }
    }

}
// Copyright (c) Naked Objects Group Ltd.
