package org.nakedobjects.nof.core.conf;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.nakedobjects.noa.NakedObjectRuntimeException;
/**
 * This class loads properties as resources from the ClassLoader.  By default the contextClassLoader is used to load
 * the properties file, and if that fails then this classes ClassLoader is used.  By initially attempting to
 * use the contextClassLoader it means that the jar containing this class can be deployed as a utility jar (eg in
 * the application server class path, or /WEB-INF/lib), but still be able to access the properties stored in 
 * another jar.
 */
class PropertiesResourceReader implements ConfigurationReader {
    private Properties properties = new Properties();

    public PropertiesResourceReader(final String pathname, final boolean ensureFileLoads) {
        InputStream in	= null;

        // Note: if this strategy for finding the properties file does not always work then
		// this class should be changed so that a classLoader can be passed in during the application
		// start up processing.  Since the properties file should be in the same jar as the application
		// it can just pass in its own ClassLoader.
		
		// try 1. load the properties from the ContextClassLoader. If this class is on the
		// system class path, then the class loader obtained from this.getClassLoader() 
		// won't be able to load resources from the application class path.
		ClassLoader classLoader = null;
		try {
			classLoader = Thread.currentThread().getContextClassLoader();
			in = classLoader.getResourceAsStream(pathname);
		} catch (SecurityException ignore) {}
		
		if (in == null)	{
			// try 2.  Load from this classes class loader.  This is useful if
			// a)  the contextClassLoader is not available
			// b)  the contextClassLoader does not load from this classes ClassLoader,
			//     for example when running under Ant while running unit tests as
			//     part of an automated build
			classLoader = getClass().getClassLoader();
			in = classLoader.getResourceAsStream(pathname);
		}
		if (in == null)	{
			if (ensureFileLoads) {
				throw new NakedObjectRuntimeException("Cannot find configuration parameters in ClassLoader : " +pathname);
			}
			return;
		}
		try	{
			properties.load(in);
		} catch (IOException e) {
			throw new NakedObjectRuntimeException("Error loading properties for "+pathname+" - " + e.getMessage());
		}
    }

    public Properties getProperties() {
        return properties;
    }
}
// Copyright (c) Naked Objects Group Ltd.