package org.nakedobjects.nof.core.context;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.security.Session;
import org.nakedobjects.nof.core.system.Monitor;
import org.nakedobjects.nof.core.util.DebugString;


/**
 * Basic multi-user implementation of NakedObjects that stores a set of components for each thread in use.
 */
public class ThreadContext extends MultiUserContext {
    private static final Logger LOG = Logger.getLogger(ThreadContext.class);

    public static NakedObjectsContext createInstance() {
        return new ThreadContext();
    }

    private Hashtable threads = new Hashtable();

    protected ThreadContext() {}
/*
    void clearSession(Session session) {
        NakedObjectsData local = getLocal();
        if (local.session == null) {
            throw new NakedObjectRuntimeException("No session exists on this thread");
        }
        if (local.session != session) {
            throw new NakedObjectRuntimeException("Wrong session exists on this thread: " + local.session);
        }
        terminateSession();
        local.session = session;

    }
*/
    /**
     * Removes the component set for the current thread.
     */
    private void clearThread() {
        Thread thread = Thread.currentThread();
        synchronized (threads) {
            if (threads.get(thread) == null) {
                throw new NakedObjectRuntimeException("Thread not currently registered");
            }
            threads.remove(thread);
        }
    }

    public void debugData(DebugString debug) {
        super.debugData(debug);
        debug.appendln();
        debug.appendTitle("Threads based Contexts");
        Enumeration e = threads.keys();
        while (e.hasMoreElements()) {
            Thread thread = (Thread) e.nextElement();
            Object data = threads.get(thread);
            debug.appendln(thread.toString(), data);
        }
    }

    public String debugTitle() {
        return "Naked Objects (by thread) " + Thread.currentThread().getName();
    }

    /**
     * Get the component set used the current thread. If no set exists then a new component set
     * (NakedObjectsData) is created.
     */
    protected NakedObjectsData getLocal() {
        Thread thread = Thread.currentThread();
        NakedObjectsData local;
        synchronized (threads) {
            local = (NakedObjectsData) threads.get(thread);

            if (local == null) {
                local = new NakedObjectsData();
                Monitor.addEvent("Context", "New thread using context (count " + threads.size() + ")");
                synchronized (threads) {
                    threads.put(thread, local);
                }
                LOG.info("  creating local " + local + "; now have " + threads.size() + " locals");
            }
        }
        return local;
    }

    protected String[] allExecutionContextIds() {
        String[] ids = new String[threads.size()];
        Enumeration e = threads.keys();
        int i = 0;
        while (e.hasMoreElements()) {
            Thread thread = (Thread) e.nextElement();
            NakedObjectsData data = (NakedObjectsData) threads.get(thread);
            ids[i++] = data.executionContextId();
        }
        return ids;
    }

    protected ContextDebug debugContext(String executionContextId) {
        Enumeration e = threads.keys();
        while (e.hasMoreElements()) {
            Thread thread = (Thread) e.nextElement();
            NakedObjectsData data = (NakedObjectsData) threads.get(thread);
            if (data.executionContextId().equals(executionContextId)) {
                return (ContextDebug) data;
            }
        }
        return null;
    }

    protected void terminateSession() {
        clearThread();
    }

    private void shutdownAllSessions() {
        synchronized (threads) {
            int i = 0;
            for (Iterator iter = threads.entrySet().iterator(); iter.hasNext();) {
                LOG.info("Shutting down thread: " + i++);
                final NakedObjectsData data = (NakedObjectsData) ((Map.Entry) iter.next()).getValue();
                if (data.objectPersistor != null) {
                    data.objectPersistor.shutdown();
                    data.objectPersistor = null;
                }
                if (data.objectLoader != null) {
                    data.objectLoader.shutdown();
                    data.objectLoader = null;
                }
            }
        }
    }

    public void shutdownSession() {
        shutdownAllSessions();
    }
}
// Copyright (c) Naked Objects Group Ltd.
