package org.nakedobjects.nof.core.service;

import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.system.InitialisationException;
import org.nakedobjects.nof.core.system.InstanceFactory;
import org.nakedobjects.nof.core.system.ServicesInstaller;
import org.nakedobjects.nof.core.util.NakedObjectConfiguration;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;


public class ServicesFromConfiguration implements ServicesInstaller {
    private static final char DELIMITER = '#';
    private static final Logger LOG = Logger.getLogger(ServicesFromConfiguration.class);
    private static final String SERVICES = "services";
    private static final String SERVICES_PREFIX = "services.prefix";

    public Object[] getServices(final boolean forServer) {
        LOG.info("installing " + this.getClass().getName());
        Object[] common = installServices(NakedObjectsContext.getConfiguration(), null);
        Object[] specific = installServices(NakedObjectsContext.getConfiguration(), forServer ? "server" : "client");
        Object[] combined = new Object[common.length + specific.length];
        System.arraycopy(common, 0, combined, 0, common.length);
        System.arraycopy(specific, 0, combined, common.length, specific.length);
        if (combined.length == 0) {
            throw new InitialisationException("No services specified");
        }
        return combined;
    }

    public String getName() {
        return "services-properties";
    }
    
    private Object installService(String className) {
        return InstanceFactory.createInstance(className);
    }

    private Object[] installServices(final NakedObjectConfiguration configuration, final String group) {
        String root = Configuration.ROOT + (group == null ? "" : group + ".");
        String servicePrefix = configuration.getString(root + SERVICES_PREFIX);
        if (group != null && servicePrefix == null) {
            servicePrefix = configuration.getString(Configuration.ROOT + SERVICES_PREFIX);
        }
        String prefix = servicePrefix(servicePrefix);
        String serviceList = configuration.getString(root + SERVICES);
        if (serviceList != null) {
            return installServices(prefix, serviceList);
        } else {
            return new Object[0];
        }
    }

    private Object[] installServices(String servicePrefix, String serviceList) {
        StringTokenizer services = new StringTokenizer(serviceList, Configuration.LIST_SEPARATOR);
        if (!services.hasMoreTokens()) {
            throw new InitialisationException("Services specified, but none loaded");
        }
        List list = new ArrayList();
        while (services.hasMoreTokens()) {
            String serviceName = services.nextToken().trim();
            if (serviceName.equals("")) {
                continue;
            }
            LOG.info("  creating service " + serviceName);
            Object service;
            if (serviceName.indexOf(DELIMITER) == -1) {
                service = installService(servicePrefix + serviceName);
            } else {
                service = installSimpleService(servicePrefix, serviceName);
            }
            list.add(service);
        }
        Object[] array = list.toArray(new Object[list.size()]);
        return array;
    }

    private Object installSimpleService(String prefix, String name) {
        int pos = name.indexOf(DELIMITER);
        String className = prefix + name.substring(pos + 1);
        Class cls = loadClass(className);
        String type = name.substring(0, pos);
        if ("repository".equals(type)) {
            return new SimpleRepository(cls);
        } else {
            throw new InitialisationException("Unknown service type " + type);
        }
    }

    private Class loadClass(String className) {
        try {
            return Class.forName(className);
        } catch (ClassNotFoundException e) {
            throw new InitialisationException("Cannot find service class " + className);
        }
    }

    private String servicePrefix(final String servicePrefix) {
        String prefix = servicePrefix == null ? "" : servicePrefix.trim();
        if (prefix.length() > 0 && !prefix.endsWith(Configuration.DELIMITER)) {
            prefix = prefix + Configuration.DELIMITER;
        }
        return prefix;
    }

}
// Copyright (c) Naked Objects Group Ltd.
