package org.nakedobjects.nof.persist;

import java.util.Enumeration;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Persistable;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.persist.ObjectPersistenceException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.OneToManyAssociation;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.ToString;


public class TopDownPersistAlgorithm implements PersistAlgorithm {
    private static final Logger LOG = Logger.getLogger(TopDownPersistAlgorithm.class);

    public void init() {}

    public void makePersistent(final NakedObject object, final PersistedObjectAdder persistor) {
        if (object.getResolveState().isPersistent() || object.persistable() == Persistable.TRANSIENT) {
            LOG.warn("can't make object persistent - either already persistent, or transient only: " + object);
            return;
        }

        LOG.info("persist " + object);
        object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.SAVING);
        NakedObjectsContext.getObjectLoader().madePersistent(object);
        persistor.addPersistedObject(object);
        object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.SAVED);

        NakedObjectField[] fields = object.getSpecification().getFields();
        for (int i = 0; i < fields.length; i++) {
            NakedObjectField field = fields[i];
            if (!field.isPersisted()) {
                continue;
            } else if (field.isValue()) {
                continue;
            } else if (field instanceof OneToManyAssociation) {
                NakedCollection collection = (NakedCollection) field.get(object);
                if (collection == null) {
                    throw new ObjectPersistenceException("Collection " + field.getName() + " does not exist in "
                            + object.getSpecification().getFullName());
                }
                makePersistent(collection, persistor);
            } else {
                Naked fieldValue = field.get(object);
                if (fieldValue == null) {
                    continue;
                }
                if (!(fieldValue instanceof NakedObject)) {
                    throw new NakedObjectRuntimeException(fieldValue + " is not a NakedObject");
                }
                makePersistent((NakedObject) fieldValue, persistor);
            }
        }
    }

    protected void makePersistent(final NakedCollection collection, final PersistedObjectAdder persistor) {
        LOG.info("persist " + collection);
        if (collection.getResolveState() == ResolveState.TRANSIENT) {
            collection.changeState(ResolveState.RESOLVED);
        }
        Enumeration elements = collection.elements();
        while (elements.hasMoreElements()) {
            makePersistent((NakedObject) elements.nextElement(), persistor);
        }
    }

    public String name() {
        return "Simple Top Down Persistence Walker";
    }

    public void shutdown() {}

    public String toString() {
        ToString toString = new ToString(this);
        return toString.toString();
    }

}
// Copyright (c) Naked Objects Group Ltd.
