package org.nakedobjects.nof.persist;

import java.util.Enumeration;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Persistable;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.persist.ObjectPersistenceException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.OneToManyAssociation;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.ToString;


public class TwoPassPersistAlgorithm implements PersistAlgorithm {
    private static final Logger LOG = Logger.getLogger(TwoPassPersistAlgorithm.class);

    public void init() {}

    public void makePersistent(final NakedObject object, final PersistedObjectAdder persistor) {
        if (object.getResolveState().isPersistent() || object.persistable() == Persistable.TRANSIENT) {
            return;
        }

        LOG.info("persist " + object);
        object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.SAVING);
        NakedObjectsContext.getObjectLoader().madePersistent(object);

        NakedObjectField[] fields = object.getSpecification().getFields();
        for (int i = 0; i < fields.length; i++) {
            NakedObjectField field = fields[i];
            if (!field.isPersisted()) {
                continue;
            } else if (field.isValue()) {
                continue;
            } else if (field.isCollection()) {
                // skip in first pass
                continue;
            } else {
                Naked fieldValue = field.get(object);
                if (fieldValue == null) {
                    continue;
                }
                if (!(fieldValue instanceof NakedObject)) {
                    throw new NakedObjectRuntimeException();
                }
                makePersistent((NakedObject) fieldValue, persistor);
            }
        }

        for (int i = 0; i < fields.length; i++) {
            NakedObjectField field = fields[i];
            if (field.isPersisted()) {
                continue;
            } else if (field.isValue()) {
                continue;
            } else if (field instanceof OneToManyAssociation) {
                NakedCollection collection = (NakedCollection) field.get(object);
                if (collection == null) {
                    throw new ObjectPersistenceException("Collection " + field.getName() + " does not exist in "
                            + object.getSpecification().getFullName());
                }
                makePersistent(collection, persistor);
                Enumeration elements = collection.elements();
                while (elements.hasMoreElements()) {
                    makePersistent((NakedObject) elements.nextElement(), persistor);
                }
            } else {
                // skip in second pass
                continue;
            }
        }

        persistor.addPersistedObject(object);
        object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.SAVED);
    }

    private void makePersistent(final NakedCollection collection, final PersistedObjectAdder persistor) {
        if (collection.getResolveState().isPersistent() || collection.persistable() == Persistable.TRANSIENT) {
            return;
        }
        LOG.info("persist " + collection);
        if (collection.getResolveState() == ResolveState.TRANSIENT) {
            collection.changeState(ResolveState.RESOLVED);
        }
        NakedObjectsContext.getObjectLoader().madePersistent(collection);
        Enumeration elements = collection.elements();
        while (elements.hasMoreElements()) {
            makePersistent((NakedObject) elements.nextElement(), persistor);
        }
    }

    public String name() {
        return "Two pass,  bottom up persistence walker";
    }

    public void shutdown() {}

    public String toString() {
        ToString toString = new ToString(this);
        return toString.toString();
    }

}
// Copyright (c) Naked Objects Group Ltd.
