package org.nakedobjects.nof.persist;

import junit.framework.TestCase;

import org.apache.log4j.BasicConfigurator;
import org.easymock.MockControl;
import org.nakedobjects.noa.adapter.Persistable;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.persist.NotPersistableException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.OneToOneAssociation;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.testing.DummyNakedObject;
import org.nakedobjects.testing.MockContext;
import org.nakedobjects.testing.TestSpecification;

public class DefaultPersistAlgorithmTest extends TestCase {
	private DefaultPersistAlgorithm algorithm;

	private MockContext context;

	private PersistedObjectAdder adder;

	private MockControl control;

	private DummyNakedObject object;

	private MockControl fieldControl;

	private OneToOneAssociation oneToOneAssociation;

	private DummyNakedObject fieldsObject;

	protected void setUp() throws Exception {
		BasicConfigurator.configure();

		algorithm = new DefaultPersistAlgorithm();

		context = new MockContext();
		context.initObjectLoader();

		control = MockControl.createControl(PersistedObjectAdder.class);
		adder = (PersistedObjectAdder) control.getMock();

		object = new DummyNakedObject(ResolveState.TRANSIENT);

		fieldControl = MockControl.createControl(OneToOneAssociation.class);
		oneToOneAssociation = (OneToOneAssociation) fieldControl.getMock();
		TestSpecification spec = new TestSpecification();
		NakedObjectField[] fields = new NakedObjectField[]{
				oneToOneAssociation
		};
		spec.setupFields(fields );
		object.setupSpecification(spec);

		fieldsObject = new DummyNakedObject(ResolveState.TRANSIENT);
		fieldsObject.setupSpecification(new TestSpecification());
	

	}

	public void testMakePersistentFailsIfObjectAlreadyPersistent() {
		object.setupResolveState(ResolveState.RESOLVED);
		try {
			algorithm.makePersistent(object, adder);
			fail();
		} catch (NotPersistableException expected) {
		}
	}

	public void testMakePersistentFailsIfObjectMustBeTransient() {
		try {
			object.setupPersistable(Persistable.TRANSIENT);
			algorithm.makePersistent(object, adder);
		} catch (NotPersistableException expected) {
		}
	}

	public void testMakePersistentRecursesThroughReferenceFields() {
		fieldControl.expectAndReturn(oneToOneAssociation.isPersisted(), true);
		fieldControl.expectAndReturn(oneToOneAssociation.isValue(), false);
		fieldControl.expectAndReturn(oneToOneAssociation.get(object), fieldsObject);
		
		NakedObjectsContext.getObjectLoader().madePersistent(object);
		NakedObjectsContext.getObjectLoader().madePersistent(fieldsObject);

		adder.addPersistedObject(object);
		adder.addPersistedObject(fieldsObject);

		replay();
		algorithm.makePersistent(object, adder);
		verify();
	}

	public void testMakePersistentRecursesThroughReferenceFieldsSkippingNullReferences() {
		fieldControl.expectAndReturn(oneToOneAssociation.isPersisted(), true);
		fieldControl.expectAndReturn(oneToOneAssociation.isValue(), false);
		fieldControl.expectAndReturn(oneToOneAssociation.get(object), null);
		
		NakedObjectsContext.getObjectLoader().madePersistent(object);

		adder.addPersistedObject(object);

		replay();
		algorithm.makePersistent(object, adder);
		verify();
	}

	public void testMakePersistentRecursesThroughReferenceFieldsSkippingNonPersistentFields() {
		fieldControl.expectAndReturn(oneToOneAssociation.isPersisted(), false);
		
		NakedObjectsContext.getObjectLoader().madePersistent(object);

		adder.addPersistedObject(object);

		replay();
		algorithm.makePersistent(object, adder);
		verify();
	}


	public void testMakePersistentRecursesThroughReferenceFieldsSkippingObjectsThatAreAlreadyPersistent() {
		fieldControl.expectAndReturn(oneToOneAssociation.isPersisted(), true);
		fieldControl.expectAndReturn(oneToOneAssociation.isValue(), false);
		fieldControl.expectAndReturn(oneToOneAssociation.get(object), fieldsObject);
		fieldsObject.setupResolveState(ResolveState.RESOLVED);
		
		NakedObjectsContext.getObjectLoader().madePersistent(object);

		adder.addPersistedObject(object);

		replay();
		algorithm.makePersistent(object, adder);
		verify();
	}


	private void verify() {
		fieldControl.verify();
		context.verify();
		control.verify();
	}

	private void replay() {
		fieldControl.replay();
		context.replay();
		control.replay();
	}

}

// Copyright (c) Naked Objects Group Ltd.
