package org.nakedobjects.nof.persist.objectstore;

import junit.framework.TestCase;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.persist.ObjectPersistenceException;
import org.nakedobjects.nof.core.persist.TransactionException;
import org.nakedobjects.nof.persist.transaction.CreateObjectCommand;
import org.nakedobjects.nof.persist.transaction.DestroyObjectCommand;
import org.nakedobjects.nof.persist.transaction.ExecutionContext;
import org.nakedobjects.nof.persist.transaction.SaveObjectCommand;
import org.nakedobjects.nof.persist.transaction.Transaction;
import org.nakedobjects.nof.testsystem.TestProxySystem;


public class TransactionTest extends TestCase {

    public static void main(final String[] args) {
        junit.textui.TestRunner.run(TransactionTest.class);
    }

    NakedObject object1;
    NakedObject object2;
    MockObjectStore os;
    private Transaction t;

    private CreateObjectCommand createCreateCommand(final NakedObject object, final String name) {
        return new CreateObjectCommand() {

            public void execute(final ExecutionContext context) throws ObjectPersistenceException {}

            public NakedObject onObject() {
                return object;
            }

            public String toString() {
                return name;
            }
        };
    }

    private DestroyObjectCommand createDestroyCommand(final NakedObject object, final String name) {
        return new DestroyObjectCommand() {

            public void execute(final ExecutionContext context) throws ObjectPersistenceException {}

            public NakedObject onObject() {
                return object;
            }

            public String toString() {
                return name;
            }
        };
    }

    private SaveObjectCommand createSaveCommand(final NakedObject object, final String name) {
        return new SaveObjectCommand() {
            public void execute(final ExecutionContext context) throws ObjectPersistenceException {}

            public NakedObject onObject() {
                return object;
            }

            public String toString() {
                return name;
            }
        };
    }

    private SaveObjectCommand createCommandThatAborts(final NakedObject object, final String name) {
        return new SaveObjectCommand() {
            public void execute(final ExecutionContext context) throws ObjectPersistenceException {
                throw new ObjectPersistenceException();
            }

            public NakedObject onObject() {
                return object;
            }

            public String toString() {
                return name;
            }
        };
    }

    protected void setUp() throws Exception {
        Logger.getRootLogger().setLevel(Level.OFF);

        TestProxySystem system = new TestProxySystem();
        system.init();

        os = new MockObjectStore();
        t = new ObjectStoreTransaction(os);

        object1 = system.createTransientTestObject();
        object2 = system.createTransientTestObject();
    }

    public void testAbort() throws Exception {
        t.addCommand(createSaveCommand(object1, "command 1"));
        t.addCommand(createSaveCommand(object2, "command 2"));
        t.abort();

        assertEquals(0, os.getActions().size());
    }

    public void testAbortBeforeCommand() throws Exception {
        t.abort();

        assertEquals(0, os.getActions().size());
    }

    public void testCommandThrowsAnExceptionCausingAbort() throws Exception {
        t.addCommand(createSaveCommand(object1, "command 1"));
        t.addCommand(createCommandThatAborts(object2, "command 2"));
        t.addCommand(createSaveCommand(object1, "command 3"));
        try {
            t.commit();
            fail();
        } catch (ObjectPersistenceException expected) {}
        os.assertAction(0, "endTransaction");
        os.assertAction(1, "run command 1");
        os.assertAction(2, "run command 2");
    }

    public void testAddCommands() throws Exception {
        t.addCommand(createSaveCommand(object1, "command 1"));
        t.addCommand(createSaveCommand(object2, "command 2"));
        t.commit();

        os.assertAction(0, "endTransaction");
        os.assertAction(1, "run command 1");
        os.assertAction(2, "run command 2");
        assertEquals(3, os.getActions().size());
    }

    public void testAddCreateCommandsButIgnoreSaveForSameObject() throws Exception {
        t.addCommand(createCreateCommand(object1, "create object 1"));
        /*
         * The next command should be ignored as the above create will have already saved the next object
         */
        t.addCommand(createSaveCommand(object1, "save object 1"));
        t.addCommand(createSaveCommand(object2, "save object 2"));
        t.commit();

        os.assertAction(0, "endTransaction");
        os.assertAction(1, "run create object 1");
        os.assertAction(2, "run save object 2");
        assertEquals(3, os.getActions().size());
    }

    public void testAddDestoryCommandsButRemovePreviousSaveForSameObject() throws Exception {
        t.addCommand(createSaveCommand(object1, "save object 1"));
        t.addCommand(createDestroyCommand(object1, "destroy object 1"));
        t.commit();

        os.assertAction(0, "endTransaction");
        os.assertAction(1, "run destroy object 1");
        assertEquals(2, os.getActions().size());
    }

    public void testIgnoreBothCreateAndDestroyCommandsWhenForSameObject() throws Exception {
        t.addCommand(createCreateCommand(object1, "create object 1"));
        t.addCommand(createDestroyCommand(object1, "destroy object 1"));
        t.addCommand(createDestroyCommand(object2, "destroy object 2"));
        t.commit();

        os.assertAction(0, "endTransaction");
        os.assertAction(1, "run destroy object 2");
        assertEquals(2, os.getActions().size());
    }

    public void testIgnoreSaveAfterDeleteForSameObject() throws Exception {
        t.addCommand(createDestroyCommand(object1, "destroy object 1"));
        t.addCommand(createSaveCommand(object1, "save object 1"));
        t.commit();

        os.assertAction(0, "endTransaction");
        os.assertAction(1, "run destroy object 1");
        assertEquals(2, os.getActions().size());
    }

    public void testNoCommands() throws Exception {
        t.commit();
        os.assertAction(0, "endTransaction");
        assertEquals(1, os.getActions().size());
    }

    public void testNoTransactionsWhenCommandCancelEachOtherOut() throws Exception {
        t.addCommand(createCreateCommand(object1, "create object 1"));
        t.addCommand(createDestroyCommand(object1, "destroy object 1"));
        t.commit();

        os.assertAction(0, "endTransaction");
        assertEquals(1, os.getActions().size());
    }

    public void testTransactionAlreadyCompleteAfterAbort() throws Exception {
        t.abort();

        try {
            t.abort();
            fail();
        } catch (TransactionException expected) {}

        try {
            t.commit();
            fail();
        } catch (TransactionException expected) {}
    }

    public void testTransactionAlreadyCompleteAfterCommit() throws Exception {
        t.commit();

        try {
            t.abort();
            fail();
        } catch (TransactionException expected) {}

        try {
            t.commit();
            fail();
        } catch (TransactionException expected) {}
    }

}
// Copyright (c) Naked Objects Group Ltd.
