package org.nakedobjects.nof.persist;

import java.util.Enumeration;

import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Persistable;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.persist.NotPersistableException;
import org.nakedobjects.noa.persist.ObjectPersistenceException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.OneToManyAssociation;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.ToString;
import org.nakedobjects.nof.core.util.UnknownTypeException;

import org.apache.log4j.Logger;


public class DefaultPersistAlgorithm implements PersistAlgorithm {
    private static final Logger LOG = Logger.getLogger(DefaultPersistAlgorithm.class);

    public void init() {}

    public void makePersistent(final NakedObject object, final PersistedObjectAdder persistor) {
        if (object.getResolveState().isPersistent() || object.persistable() == Persistable.TRANSIENT) {
            throw new NotPersistableException("can't make object persistent - either already persistent, or transient only: "
                    + object);
        }
        persist(object, persistor);
    }

    private void persist(final NakedObject object, final PersistedObjectAdder persistor) {
        if (object.getResolveState().isTransient() && object.persistable() != Persistable.TRANSIENT) {

            LOG.info("make persistent " + object);
            object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.SAVING);
            NakedObjectsContext.getObjectLoader().madePersistent(object);

            NakedObjectField[] fields = object.getSpecification().getFields();
            for (int i = 0; i < fields.length; i++) {
                NakedObjectField field = fields[i];
                if (!field.isPersisted()) {
                    continue;
                } else if (field.isValue()) {
                    continue;
                } else if (field instanceof OneToManyAssociation) {
                    NakedCollection collection = (NakedCollection) field.get(object);
                    if (collection == null) {
                        throw new ObjectPersistenceException("Collection " + field.getName() + " does not exist in "
                                + object.getSpecification().getFullName());
                    }
                    makePersistent(collection, persistor);
                } else {
                    Naked fieldValue = field.get(object);
                    if (fieldValue == null) {
                        continue;
                    }
                    if (!(fieldValue instanceof NakedObject)) {
                        throw new UnknownTypeException(fieldValue + " is not a NakedObject");
                    }
                    persist((NakedObject) fieldValue, persistor);
                }
            }
            persistor.addPersistedObject(object);
            object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.SAVED);
        }
    }

    protected void makePersistent(final NakedCollection collection, final PersistedObjectAdder persistor) {
        LOG.info("persist " + collection);
        if (collection.getResolveState() == ResolveState.GHOST) {
            collection.changeState(ResolveState.RESOLVING);
            collection.changeState(ResolveState.RESOLVED);
        } else if (collection.getResolveState() == ResolveState.TRANSIENT) {
            collection.changeState(ResolveState.RESOLVED);
        }
        Enumeration elements = collection.elements();
        while (elements.hasMoreElements()) {
            persist((NakedObject) elements.nextElement(), persistor);
        }
    }

    public String name() {
        return "Simple Bottom Up Persistence Walker";
    }

    public void shutdown() {}

    public String toString() {
        ToString toString = new ToString(this);
        return toString.toString();
    }

}
// Copyright (c) Naked Objects Group Ltd.
