package org.nakedobjects.nof.persist.objectstore;

import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.persist.InstancesCriteria;
import org.nakedobjects.noa.persist.NakedObjectPersistor;
import org.nakedobjects.noa.persist.ObjectNotFoundException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.util.DebugInfo;
import org.nakedobjects.nof.persist.transaction.CreateObjectCommand;
import org.nakedobjects.nof.persist.transaction.DestroyObjectCommand;
import org.nakedobjects.nof.persist.transaction.PersistenceCommand;
import org.nakedobjects.nof.persist.transaction.SaveObjectCommand;
import org.nakedobjects.object.RequiresSetup;


public interface NakedObjectStore extends RequiresSetup, DebugInfo {

    void abortTransaction();

    /**
     * Makes a naked object persistent. The specified object should be stored away via this object store's
     * persistence mechanism, and have an new and unique OID assigned to it (by calling the object's
     * <code>setOid</code> method). The object, should also be added to the cache as the object is
     * implicitly 'in use'.
     * 
     * <p>
     * If the object has any associations then each of these, where they aren't already persistent, should
     * also be made persistent by recursively calling this method.
     * </p>
     * 
     * <p>
     * If the object to be persisted is a collection, then each element of that collection, that is not
     * already persistent, should be made persistent by recursively calling this method.
     * </p>
     * 
     */
    CreateObjectCommand createCreateObjectCommand(NakedObject object);


    /**
     * Registers the specified service as having the specified OID.
     */
    void registerService(String name, Oid oid);

    /**
     * Removes the specified object from the object store. The specified object's data should be removed from
     * the persistence mechanism and, if it is cached (which it probably is), removed from the cache also.
     */
    DestroyObjectCommand createDestroyObjectCommand(NakedObject object);

    /**
     * Persists the specified object's state. Essentially the data held by the persistence mechanism should be
     * updated to reflect the state of the specified objects. Once updated, the object store should issue a
     * notification to all of the object's users via the <class>UpdateNotifier </class> object. This can be
     * achieved simply, if extending the <class> AbstractObjectStore </class> by calling its
     * <method>broadcastObjectUpdate </method> method.
     */
    SaveObjectCommand createSaveObjectCommand(NakedObject object);

    void endTransaction();

    NakedObject[] getInstances(InstancesCriteria criteria);

//    NakedObject[] getInstances(NakedObjectSpecification specification, boolean includeSubclasses);

    /**
     * Retrieves the object identified by the specified OID from the object store. The cache should be checked
     * first and, if the object is cached, the cached version should be returned. It is important that if this
     * method is called again, while the originally returned object is in working memory, then this method
     * must return that same Java object.
     * 
     * <para>Assuming that the object is not cached then the data for the object should be retreived from the
     * persistence mechanism and the object recreated (as describe previously). The specified OID should then
     * be assigned to the recreated object by calling its <method>setOID </method>. Before returning the
     * object its resolved flag should also be set by calling its <method>setResolved </method> method as
     * well. </para>
     * 
     * <para>If the persistence mechanism does not known of an object with the specified OID then a
     * <class>ObjectNotFoundException </class> should be thrown. </para>
     * 
     * <para>Note that the OID could be for an internal collection, and is therefore related to the parent
     * object (using a <class>CompositeOid </class>). The elements for an internal collection are commonly
     * stored as part of the parent object, so to get element the parent object needs to be retrieved first,
     * and the internal collection can be got from that. </para>
     * 
     * <para>Returns the stored NakedObject object that has the specified OID. </para>
     * 
     * @param oid
     *            of the object to be retrieved
     * 
     * @return the requested naked object
     * @throws ObjectNotFoundException
     *             when no object corresponding to the oid can be found
     */
    NakedObject getObject(Oid oid, NakedObjectSpecification hint);

    /**
     * Returns the OID for the adapted service. This allows a service object to be given the same OID that it
     * had when it was created in a different session.
     */
    Oid getOidForService(String name);

    /**
     * Checks whether there are any instances of the specified type. The object store should look for
     * instances of the type represented by <variable>type </variable> and return <code>true</code> if there
     * are, or <code>false</code> if there are not.
     * 
     * @param includeSubclasses
     */
    boolean hasInstances(NakedObjectSpecification specification, boolean includeSubclasses);

    /**
     * Determine if the object store has been initialized with its set of start up objects. This method is
     * called only once after the <code>init</code> has been called. If this flag returns <code>false</code>
     * the framework will run the fixtures to initialise the object store.
     */
    boolean isInitialized();

    /**
     * The name of this objects store (for logging/debugging purposes)
     */
    String name();

    /**
     * Called by the resolveEagerly method in NakedObjectManager.
     * 
     * @see NakedObjectPersistor#resolveField(NakedObject, NakedObjectField)
     */
    void resolveField(NakedObject object, NakedObjectField field);

    /**
     * Called by the resolveImmediately method in NakedObjectManager.
     * 
     * @see NakedObjectPersistor#resolveImmediately(NakedObject)
     */
    void resolveImmediately(NakedObject object);

    void execute(final PersistenceCommand[] commands);

    void startTransaction();
    
    // TODO this should be done by the execute method
    boolean flush(final PersistenceCommand[] commands);

    void reset();
}
// Copyright (c) Naked Objects Group Ltd.
