package org.nakedobjects.nof.persist.objectstore;

import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.persist.InstancesCriteria;
import org.nakedobjects.noa.persist.ObjectNotFoundException;
import org.nakedobjects.noa.persist.ObjectPersistenceException;
import org.nakedobjects.noa.persist.UnsupportedFindException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.conf.ConfigurationException;
import org.nakedobjects.nof.core.system.InstanceCreationException;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nof.core.util.Logger;
import org.nakedobjects.nof.persist.transaction.CreateObjectCommand;
import org.nakedobjects.nof.persist.transaction.DestroyObjectCommand;
import org.nakedobjects.nof.persist.transaction.PersistenceCommand;
import org.nakedobjects.nof.persist.transaction.SaveObjectCommand;


public class ObjectStoreLogger extends Logger implements NakedObjectStore {
    private final NakedObjectStore decorated;

    public ObjectStoreLogger(final NakedObjectStore decorated, final String logFileName) {
        super(logFileName, false);
        this.decorated = decorated;
    }

    public ObjectStoreLogger(final NakedObjectStore decorated) {
        super(null, true);
        this.decorated = decorated;
    }

    public void abortTransaction() throws ObjectPersistenceException {
        log("Abort transaction started");
        decorated.abortTransaction();
        log("Abort transaction complete");
    }

    public CreateObjectCommand createCreateObjectCommand(final NakedObject object) {
        log("Create object " + object);
        return decorated.createCreateObjectCommand(object);
    }

    public void registerService(String name, Oid oid) {
        log("register service " + name + " as " + oid);
        decorated.registerService(name, oid);
    }

    public DestroyObjectCommand createDestroyObjectCommand(final NakedObject object) {
        log("Destroy object " + object);
        return decorated.createDestroyObjectCommand(object);
    }

    public SaveObjectCommand createSaveObjectCommand(final NakedObject object) {
        log("Save object " + object);
        return decorated.createSaveObjectCommand(object);
    }

    public void endTransaction() throws ObjectPersistenceException {
        log("End transaction");
        decorated.endTransaction();
    }

    public void debugData(final DebugString debug) {
        decorated.debugData(debug);
    }

    public String debugTitle() {
        return decorated.debugTitle();
    }

    protected Class getDecoratedClass() {
        return decorated.getClass();
    }

    public NakedObject[] getInstances(final InstancesCriteria criteria) throws ObjectPersistenceException,
            UnsupportedFindException {
        log("Get instances matching " + criteria);
        return decorated.getInstances(criteria);
    }

    public NakedObject getObject(final Oid oid, final NakedObjectSpecification hint) throws ObjectNotFoundException,
            ObjectPersistenceException {
        NakedObject object = decorated.getObject(oid, hint);
        log("Get object for " + oid + " (of type " + hint.getShortName() + ")", object.getObject());
        return object;
    }

    public Oid getOidForService(String name) {
        Oid oid = decorated.getOidForService(name);
        log("Get OID for service " + name + ": " + oid);
        return oid;
    }
    
    public boolean hasInstances(final NakedObjectSpecification specification, final boolean includeSubclasses)
            throws ObjectPersistenceException {
        boolean hasInstances = decorated.hasInstances(specification, includeSubclasses);
        log("Has instances of " + specification.getShortName(), "" + hasInstances);
        return hasInstances;
    }
    
    public boolean isInitialized() {
        boolean isInitialized = decorated.isInitialized();
        log("is initialized: " + isInitialized);
        return isInitialized;
    }

    public void init() throws ConfigurationException, InstanceCreationException, ObjectPersistenceException {
        log("Initialising " + name());
        decorated.init();
    }

    public String name() {
        return decorated.name();
    }

    public void reset() {
        log("Reset");
        decorated.reset();
    }

    public void resolveField(final NakedObject object, final NakedObjectField field) throws ObjectPersistenceException {
        log("Resolve eagerly object in field " + field + " of " + object);
        decorated.resolveField(object, field);
    }

    public void resolveImmediately(final NakedObject object) throws ObjectPersistenceException {
        log("Resolve immediately: " + object);
        decorated.resolveImmediately(object);
    }

    public void execute(final PersistenceCommand[] commands) throws ObjectPersistenceException {
        log("Run transactions");
        for (int i = 0; i < commands.length; i++) {
            log("  " + i + " " + commands[i]);
        }
        decorated.execute(commands);
    }

    public boolean flush(final PersistenceCommand[] commands) throws ObjectPersistenceException {
        log("flush transactions");
        for (int i = 0; i < commands.length; i++) {
            log("  " + i + " " + commands[i]);
        }
        return decorated.flush(commands);
    }
    
    public void shutdown() throws ObjectPersistenceException {
        log("Shutting down " + decorated);
        decorated.shutdown();
        close();
    }

    public void startTransaction() throws ObjectPersistenceException {
        log("Start transaction");
        decorated.startTransaction();
    }
}
// Copyright (c) Naked Objects Group Ltd.
