package org.nakedobjects.nof.persist.objectstore;

import java.util.Enumeration;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.NakedObjectLoader;
import org.nakedobjects.noa.adapter.NakedReference;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.adapter.Persistable;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.persist.InstancesCriteria;
import org.nakedobjects.noa.persist.NotPersistableException;
import org.nakedobjects.noa.persist.ObjectPersistenceException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.NakedObjectReflector;
import org.nakedobjects.noa.spec.Features;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.persist.AbstractObjectPersistor;
import org.nakedobjects.nof.core.persist.TransactionException;
import org.nakedobjects.nof.core.service.ServiceUtil;
import org.nakedobjects.nof.core.util.Assert;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nof.core.util.ToString;
import org.nakedobjects.nof.persist.PersistAlgorithm;
import org.nakedobjects.nof.persist.PersistedObjectAdder;
import org.nakedobjects.nof.persist.transaction.DestroyObjectCommand;
import org.nakedobjects.nof.persist.transaction.Transaction;


public class ObjectStorePersistor extends AbstractObjectPersistor implements PersistedObjectAdder {
    private static final Logger LOG = Logger.getLogger(ObjectStorePersistor.class);
    private boolean checkObjectsForDirtyFlag;
    private NakedObjectStore objectStore;
    private Transaction transaction;
    private int transactionLevel;
    private PersistAlgorithm persistAlgorithm;
    private Object[] services;

    public ObjectStorePersistor() {
        LOG.debug("creating " + this);
    }

    public void abortTransaction() {
        if (transaction != null) {
            transaction.abort();
            transaction = null;
            transactionLevel = 0;
            objectStore.abortTransaction();
        }
    }

    public void addPersistedObject(final NakedObject object) {
        getTransaction().addCommand(objectStore.createCreateObjectCommand(object));
    }

    /**
     * Removes the specified object from the system. The specified object's data should be removed from the
     * persistence mechanism.
     */
    public void destroyObject(final NakedObject object) {
        LOG.info("destroyObject " + object);
        object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.DELETING);
        DestroyObjectCommand command = objectStore.createDestroyObjectCommand(object);
        getTransaction().addCommand(command);
        object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.DELETED);
    }

    private NakedObjectLoader loader() {
        return NakedObjectsContext.getObjectLoader();
    }

    public void endTransaction() {
        transactionLevel--;
        if (transactionLevel == 0) {
            // TODO collate changes before committing
            saveChanges();
            getTransaction().commit();
            transaction = null;
        } else if (transactionLevel < 0) {
            transactionLevel = 0;
            throw new TransactionException("No transaction running to end");
        }
    }

    protected void finalize() throws Throwable {
        super.finalize();
        LOG.info("finalizing object manager");
    }

    public void debugData(final DebugString debug) {
        super.debugData(debug);

        debug.appendTitle("Persistor");
        debug.appendln("Check dirty flag", checkObjectsForDirtyFlag);
        debug.appendln("Transaction", transaction);
        debug.appendln("Persist Algorithm", persistAlgorithm);
        debug.appendln("Object Store", objectStore);
        debug.appendln();

        objectStore.debugData(debug);
    }

    public String debugTitle() {
        return "Object Store Persistor";
    }

    protected NakedObject[] getInstances(final InstancesCriteria criteria) {
        LOG.info("getInstances matching " + criteria);
        NakedObject[] instances = objectStore.getInstances(criteria);
        clearChanges();
        return instances;
    }

    public NakedObject getObject(final Oid oid, final NakedObjectSpecification specification) {
        Assert.assertNotNull("needs an OID", oid);
        Assert.assertNotNull("needs a specification", specification);

        NakedObject object;
        if (NakedObjectsContext.getObjectLoader().isIdentityKnown(oid)) {
            object = NakedObjectsContext.getObjectLoader().getAdapterFor(oid);
        } else {
            object = objectStore.getObject(oid, specification);
        }
        return object;
    }

    public Oid getOidForService(String name) {
        return objectStore.getOidForService(name);
    }

    private Transaction getTransaction() {
        if (transaction == null) {
            return new ObjectStoreTransaction(objectStore);
        }
        return transaction;
    }

    /**
     * Checks whether there are any instances of the specified type. The object store should look for
     * instances of the type represented by <variable>type </variable> and return <code>true</code> if there
     * are, or <code>false</code> if there are not.
     */
    public boolean hasInstances(final NakedObjectSpecification specification, final boolean includeSubclasses) {
        LOG.info("hasInstances of " + specification.getShortName());
        return objectStore.hasInstances(specification, includeSubclasses);
    }

    public boolean isInitialized() {
        return objectStore.isInitialized();
    }

    /**
     * Initialize the object store so that calls to this object store access persisted objects and persist
     * changes to the object that are saved.
     */
    public void init() {
        LOG.debug("initialising " + this);
        Assert.assertNotNull("persist algorithm required", persistAlgorithm);
        Assert.assertNotNull("object store required", objectStore);
        objectStore.init();
        persistAlgorithm.init();
        super.init();
        setUpRegisterServices();
    }

    private void setUpRegisterServices() {
        NakedObjectReflector reflector = NakedObjectsContext.getReflector();
        NakedObjectLoader loader = NakedObjectsContext.getObjectLoader();
        startTransaction();
        for (int i = 0; i < services.length; i++) {
            reflector.installServiceSpecification(services[i].getClass());
            Oid oid = getOidForService(ServiceUtil.id(services[i]));
            if (oid == null) {
                NakedObject adapter = loader.createAdapterForTransient(services[i], false);
                loader.madePersistent(adapter);
                oid = adapter.getOid();
                registerService(ServiceUtil.id(services[i]), oid);
            }
        }
        endTransaction();
    }

    private boolean isPersistent(final NakedReference object) {
        return object.getResolveState().isPersistent();
    }

    /**
     * Makes a naked object persistent. The specified object should be stored away via this object store's
     * persistence mechanism, and have an new and unique OID assigned to it (by calling the object's
     * <code>setOid</code> method). The object, should also be added to the cache as the object is
     * implicitly 'in use'.
     * 
     * <p>
     * If the object has any associations then each of these, where they aren't already persistent, should
     * also be made persistent by recursively calling this method.
     * </p>
     * 
     * <p>
     * If the object to be persisted is a collection, then each element of that collection, that is not
     * already persistent, should be made persistent by recursively calling this method.
     * </p>
     * 
     */
    public void makePersistent(final NakedObject object) {
        if (isPersistent(object)) {
            throw new NotPersistableException("Object already persistent: " + object);
        }
        if (object.persistable() == Persistable.TRANSIENT) {
            throw new NotPersistableException("Object must be kept transient: " + object);
        }
        NakedObjectSpecification specification = object.getSpecification();
        if (Features.isService(specification)) {
            throw new NotPersistableException("Cannot persist services: " + object);
        }

        persistAlgorithm.makePersistent(object, this);
    }

    private void registerService(String name, Oid oid) {
        objectStore.registerService(name, oid);
    }

    public void objectChanged(final NakedObject object) {
        ResolveState resolveState = object.getResolveState();
        if (resolveState.respondToChangesInPersistentObjects()) {
            NakedObjectSpecification specification = object.getSpecification();
            if (Features.isAlwaysImmutable(specification)
                    || (Features.isImmutableOncePersisted(specification) && resolveState.isPersistent())) {
                throw new ObjectPersistenceException("cannot change immutable object");
            }
            object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.UPDATING);
            getTransaction().addCommand(objectStore.createSaveObjectCommand(object));
            object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.UPDATED);
            NakedObjectsContext.getUpdateNotifer().addChangedObject(object);
        }
        if (resolveState.respondToChangesInPersistentObjects() || resolveState.isTransient()) {
            object.fireChangedEvent();
            NakedObjectsContext.getUpdateNotifer().addChangedObject(object);
        }
    }

    public void reset() {
        objectStore.reset();
    }

    public void resolveField(final NakedObject object, final NakedObjectField field) {
        if (field.isValue()) {
            return;
        }
        NakedReference reference = (NakedReference) field.get(object);
        if (reference == null || reference.getResolveState().isResolved()) {
            return;
        }
        if (!reference.getResolveState().isPersistent()) {
            return;
        }
        if (LOG.isInfoEnabled()) {
            // don't log object - it's toString() may use the unresolved field or unresolved collection
            LOG.info("resolve field " + object.getSpecification().getShortName() + "." + field.getId() + ": "
                    + reference.getSpecification().getShortName() + " " + reference.getResolveState().code() + " "
                    + reference.getOid());
        }
        objectStore.resolveField(object, field);
    }

    public void reload(final NakedObject object) {}

    public void resolveImmediately(final NakedObject object) {
        ResolveState resolveState = object.getResolveState();
        if (resolveState.isResolvable(ResolveState.RESOLVING)) {
            Assert.assertFalse("only resolve object that is not yet resolved", object, object.getResolveState().isResolved());
            Assert.assertTrue("only resolve object that is persistent", object, object.getResolveState().isPersistent());
            if (LOG.isInfoEnabled()) {
                // don't log object - it's toString() may use the unresolved field, or unresolved collection
                LOG.info("resolve immediately: " + object.getSpecification().getShortName() + " "+ object.getResolveState().code() + " " + object.getOid());
            }
            object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.LOADING);
            objectStore.resolveImmediately(object);
            object.getSpecification().lifecycleEvent(object, NakedObjectSpecification.LOADED);
        }
    }

    public void saveChanges() {
        collateChanges();
    }

    /**
     * Somewhat misnamed; just invokes the <tt>isDirty</tt> method on every known
     * object and if returns <tt>true</tt> then calls {@link #objectChanged(NakedObject)}
     * for it.
     * 
     * <p>
     * What was this for, again?
     */
    private synchronized void collateChanges() {
        if (checkObjectsForDirtyFlag) {
            LOG.debug("collating changed objects");
            Enumeration e = loader().getIdentifiedObjects();
            while (e.hasMoreElements()) {
                Object o = e.nextElement();
                if (o instanceof NakedObject) {
                    NakedObject object = (NakedObject) o;
                    if (object.getSpecification().isDirty(object)) {
                        LOG.debug("  found dirty object " + object);
                        objectChanged(object);
                        object.getSpecification().clearDirty(object);
                    }
                }
            }
        }
    }

    private synchronized void clearChanges() {
        if (checkObjectsForDirtyFlag) {
            LOG.debug("clearing changed objects");
            Enumeration e = loader().getIdentifiedObjects();
            while (e.hasMoreElements()) {
                Object o = e.nextElement();
                if (o instanceof NakedObject) {
                    NakedObject object = (NakedObject) o;
                    if (object.getSpecification().isDirty(object)) {
                        LOG.debug("  found dirty object " + object);
                        object.getSpecification().clearDirty(object);
                    }
                }
            }
        }
    }

    /**
     * Expose as a .NET property
     * 
     * @property
     */
    public void set_CheckObjectsForDirtyFlag(final boolean checkObjectsForDirtyFlag) {
        this.checkObjectsForDirtyFlag = checkObjectsForDirtyFlag;
    }

    /**
     * Expose as a .NET property
     * 
     * @property
     */
    public void set_ObjectStore(final NakedObjectStore objectStore) {
        setObjectStore(objectStore);
    }

    public void setCheckObjectsForDirtyFlag(final boolean checkObjectsForDirtyFlag) {
        this.checkObjectsForDirtyFlag = checkObjectsForDirtyFlag;
    }

    public void setObjectStore(final NakedObjectStore objectStore) {
        this.objectStore = objectStore;
    }

    public void shutdown() {
        LOG.info("shutting down " + this);
        super.shutdown();
        if (transaction != null) {
            try {
                abortTransaction();
            } catch (Exception e2) {
                LOG.error("failure during abort", e2);
            }
        }
        persistAlgorithm.shutdown();
        objectStore.shutdown();
        objectStore = null;
    }

    public void startTransaction() {
        if (transaction == null) {
            transaction = new ObjectStoreTransaction(objectStore);
            transactionLevel = 0;
            objectStore.startTransaction();
        }
        transactionLevel++;
    }

    public boolean flushTransaction() {
        if (transaction != null) {
            saveChanges();
            return transaction.flush();
        }
        return false; 
    }

    public String toString() {
        ToString toString = new ToString(this);
        if (objectStore != null) {
            toString.append("objectStore", objectStore.name());
        }
        if (persistAlgorithm != null) {
            toString.append("persistAlgorithm", persistAlgorithm.name());
        }
        return toString.toString();
    }

    /**
     * Expose as a .NET property
     * 
     * @property
     */
    public void set_PersistAlgorithm(final PersistAlgorithm persistAlgorithm) {
        this.persistAlgorithm = persistAlgorithm;
    }

    public void setPersistAlgorithm(final PersistAlgorithm persistAlgorithm) {
        this.persistAlgorithm = persistAlgorithm;
    }
    
    public void setServices(Object[] services) {
        this.services = services;
    }
}
// Copyright (c) Naked Objects Group Ltd.
