package org.nakedobjects.nof.persist.objectstore.inmemory;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.adapter.Version;
import org.nakedobjects.noa.persist.InstancesCriteria;
import org.nakedobjects.noa.persist.ObjectNotFoundException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.adapter.SerialNumberVersion;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nof.core.util.Dump;
import org.nakedobjects.nof.core.util.UnexpectedCallException;
import org.nakedobjects.nof.persist.objectstore.NakedObjectStore;
import org.nakedobjects.nof.persist.transaction.CreateObjectCommand;
import org.nakedobjects.nof.persist.transaction.DestroyObjectCommand;
import org.nakedobjects.nof.persist.transaction.ExecutionContext;
import org.nakedobjects.nof.persist.transaction.PersistenceCommand;
import org.nakedobjects.nof.persist.transaction.SaveObjectCommand;

import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import org.apache.log4j.Logger;


/**
 * This object store keep all objects in memory and simply provides a index of instances for each particular
 * type. This store does not exhibit the same behaviour as real object stores that persist the objects' data.
 * For a more realistic store use the Memory Object Store
 * 
 * @see org.nakedobjects.nof.persist.objectstore.inmemory.MemoryObjectStore
 */
public class TransientObjectStore implements NakedObjectStore {
    private final static Logger LOG = Logger.getLogger(TransientObjectStore.class);
    private final Hashtable oidObjectMap;
    private final Hashtable instances;
    private final Hashtable oidServiceMap;

    public TransientObjectStore() {
        LOG.debug("creating " + this);
        instances = new Hashtable();
        oidObjectMap = new Hashtable();
        oidServiceMap = new Hashtable();
    }

    public void abortTransaction() {
        LOG.debug("transaction aborted");
    }

    public CreateObjectCommand createCreateObjectCommand(final NakedObject object) {
        return new CreateObjectCommand() {
            public void execute(final ExecutionContext context) {
                LOG.info("  create object " + object);
                NakedObjectSpecification specification = object.getSpecification();
                LOG.debug("   saving object " + object + " as instance of " + specification.getFullName());
                TransientObjectStoreInstances ins = instancesFor(specification);
                ins.add(object);
                oidObjectMap.put(object.getOid(), object);
                object.setOptimisticLock(new SerialNumberVersion(1, "user", new Date()));
            }

            public NakedObject onObject() {
                return object;
            }

            public String toString() {
                return "CreateObjectCommand [object=" + object + "]";
            }
        };
    }

    public DestroyObjectCommand createDestroyObjectCommand(final NakedObject object) {
        return new DestroyObjectCommand() {
            public void execute(final ExecutionContext context) {
                LOG.info("  delete object '" + object + "'");
                oidObjectMap.remove(object.getOid());

                NakedObjectSpecification specification = object.getSpecification();
                LOG.debug("   destroy object " + object + " as instance of " + specification.getFullName());
                TransientObjectStoreInstances ins = instancesFor(specification);
                ins.remove(object);

                // TODO need to do garbage collection instead
                // NakedObjects.getObjectLoader().unloaded(object);
            }

            public NakedObject onObject() {
                return object;
            }

            public String toString() {
                return "DestroyObjectCommand [object=" + object + "]";
            }
        };
    }

    public SaveObjectCommand createSaveObjectCommand(final NakedObject object) {
        return new SaveObjectCommand() {
            public void execute(final ExecutionContext context) {
                NakedObjectSpecification specification = object.getSpecification();
                LOG.info("   saving object " + object + " as instance of " + specification.getFullName());

                Version version = object.getVersion();
                SerialNumberVersion serialNumberVersion = (SerialNumberVersion) version;
                Version next = serialNumberVersion.next("user", new Date());
                object.setOptimisticLock(next);
            }

            public NakedObject onObject() {
                return object;
            }

            public String toString() {
                return "SaveObjectCommand [object=" + object + "]";
            }
        };
    }

    public void endTransaction() {
        LOG.debug("end transaction");
    }

    protected void finalize() throws Throwable {
        super.finalize();
        LOG.info("finalizing object store");
    }

    public void debugData(final DebugString debug) {
        debug.appendTitle("Business Objects");
        Enumeration e = instances.keys();
        while (e.hasMoreElements()) {
            NakedObjectSpecification spec = (NakedObjectSpecification) e.nextElement();
            debug.appendln(spec.getFullName());
            TransientObjectStoreInstances instances = instancesFor(spec);
            Vector v = new Vector();
            instances.instances(v);
            Enumeration f = v.elements();
            debug.indent();
            if (!f.hasMoreElements()) {
                debug.appendln("no instances");
            } else {
                while (f.hasMoreElements()) {
                    debug.appendln("" + f.nextElement());
                }
            }
            debug.appendln();
            debug.unindent();
        }
        debug.appendln();

        debug.appendTitle("Service Objects");
        e = oidServiceMap.keys();
        while (e.hasMoreElements()) {
            String name = (String) e.nextElement();
            debug.appendln(oidServiceMap.get(name) + " -> " + name);
        }
        debug.appendln();
        
        
        debug.appendTitle("Object graphs");
        Vector dump = new Vector();
        e = instances.keys();
        while (e.hasMoreElements()) {
            NakedObjectSpecification spec = (NakedObjectSpecification) e.nextElement();
            TransientObjectStoreInstances instances = instancesFor(spec);
            Vector v = new Vector();
            instances.instances(v);
            Enumeration f = v.elements();
            while (f.hasMoreElements()) {
                debug.append(spec.getFullName());
                debug.append(": ");
                NakedObject object = (NakedObject) f.nextElement();
                debug.appendln(Dump.graph(object, dump));
            }
        }
    }

    public String debugTitle() {
        return name();
    }

    public NakedObject[] getInstances(final InstancesCriteria criteria) {
        Vector allInstances = new Vector();
        getInstances(criteria, allInstances);
        NakedObject[] matchedInstances = new NakedObject[allInstances.size()];
        int matches = 0;
        for (int i = 0; i < allInstances.size(); i++) {
            NakedObject object = (NakedObject) allInstances.elementAt(i);
            if (criteria.matches(object)) {
                matchedInstances[matches++] = object;
            }
        }

        NakedObject[] ins = new NakedObject[matches];
        System.arraycopy(matchedInstances, 0, ins, 0, matches);
        return ins;
    }

    private void getInstances(final InstancesCriteria criteria, final Vector instances) {
        NakedObjectSpecification spec = criteria.getSpecification();
        instancesFor(spec).instances(instances);
        if (criteria.includeSubclasses()) {
            NakedObjectSpecification[] subclasses = spec.subclasses();
            for (int i = 0; i < subclasses.length; i++) {
                getInstances(subclasses[i], instances, criteria.includeSubclasses());
            }
        }
    }

    private void getInstances(final NakedObjectSpecification spec, final Vector instances, final boolean includeSubclasses) {
        instancesFor(spec).instances(instances);
        if (includeSubclasses) {
            NakedObjectSpecification[] subclasses = spec.subclasses();
            for (int i = 0; i < subclasses.length; i++) {
                getInstances(subclasses[i], instances, includeSubclasses);
            }
        }
    }

    public NakedObject getObject(final Oid oid, final NakedObjectSpecification hint) {
        LOG.debug("getObject " + oid);
        NakedObject nakedObject = (NakedObject) oidObjectMap.get(oid);
        if (nakedObject == null) {
            throw new ObjectNotFoundException(oid);
        }
        return nakedObject;
    }

    public Oid getOidForService(String name) {
        return (Oid) oidServiceMap.get(name);
    }

    public boolean hasInstances(final NakedObjectSpecification spec, final boolean includeSubclasses) {
        if (instancesFor(spec).hasInstances()) {
            return true;
        }
        if (includeSubclasses) {
            NakedObjectSpecification[] subclasses = spec.subclasses();
            for (int i = 0; i < subclasses.length; i++) {
                if (hasInstances(subclasses[i], includeSubclasses)) {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean isInitialized() {
        return false;
    }

    public void init() {
        LOG.debug("initialising " + this);
    }

    private TransientObjectStoreInstances instancesFor(final NakedObjectSpecification spec) {
        TransientObjectStoreInstances ins = (TransientObjectStoreInstances) instances.get(spec);
        if (ins == null) {
            ins = new TransientObjectStoreInstances();
            instances.put(spec, ins);
        }
        return ins;
    }

    public String name() {
        return "Transient Object Store";
    }

    public void resolveField(final NakedObject object, final NakedObjectField field) {
        if (field.isCollection()) {
            NakedCollection collection = (NakedCollection) field.get(object);
            NakedObjectsContext.getObjectLoader().start(collection, ResolveState.RESOLVING);
            NakedObjectsContext.getObjectLoader().end(collection);
        } else {
            throw new UnexpectedCallException();
        }
    }

    public void registerService(String name, Oid oid) {
        if (oidServiceMap.containsKey(name)) {
            throw new NakedObjectRuntimeException("Service already exists: " + name);
        }
        oidServiceMap.put(name, oid);
    }

    public void reset() {}

    public void execute(final PersistenceCommand[] commands) {
        LOG.info("start execution of transaction ");
        for (int i = 0; i < commands.length; i++) {
            commands[i].execute(null);
        }
        LOG.info("end execution");
    }

    public boolean flush(final PersistenceCommand[] commands) {
        LOG.info("start flush of transaction ");
        for (int i = 0; i < commands.length; i++) {
            commands[i].execute(null);
        }
        LOG.info("end flush");
        return commands.length > 0;
    }

    public void resolveImmediately(final NakedObject object) {
        LOG.debug("resolve " + object);
    }

    public void shutdown() {
        LOG.info("shutting down " + this);
        oidObjectMap.clear();
        for (Enumeration e = instances.elements(); e.hasMoreElements();) {
            TransientObjectStoreInstances inst = (TransientObjectStoreInstances) e.nextElement();
            inst.shutdown();
        }
        instances.clear();
    }

    public void startTransaction() {
        LOG.debug("start transaction");
    }

}
// Copyright (c) Naked Objects Group Ltd.
