package org.nakedobjects.nof.persist.objectstore.inmemory;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Oid;
import org.nakedobjects.noa.persist.ObjectNotFoundException;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.persist.AllInstances;
import org.nakedobjects.nof.core.persist.TitleCriteria;
import org.nakedobjects.nof.persist.transaction.PersistenceCommand;
import org.nakedobjects.nof.testsystem.ProxyTestCase;
import org.nakedobjects.nof.testsystem.TestProxyNakedObject;
import org.nakedobjects.nof.testsystem.TestProxyOid;


public class MemoryObjectStoreTest extends ProxyTestCase {
    private MemoryObjectStore store;

    protected void setUp() throws Exception {
        super.setUp();
        store = new MemoryObjectStore();
        store.init();
    }

    protected void tearDown() throws Exception {
        store.shutdown();
        super.tearDown();
    }

    public void testObjectNotPersistedWhenCreated() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();

        NakedObjectSpecification specification = object.getSpecification();
        assertEquals(false, store.hasInstances(specification, false));
        assertEquals(0, store.getInstances(new AllInstances(specification, false)).length);
        try {
            store.getObject(object.getOid(), specification);
            fail();
        } catch (ObjectNotFoundException expected) {}
    }

    public void testRetrievedInstanceAdapterIsIsolatedFromOriginal() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();
        addObjectToStore(object);

        NakedObjectSpecification specification = object.getSpecification();
        NakedObject[] retrievedInstance = store.getInstances(new AllInstances(specification, false));
        assertEquals(1, retrievedInstance.length);
        assertNotSame(object, retrievedInstance[0]);
        assertSame(object.getObject(), retrievedInstance[0].getObject());
    }

    public void testRetrievedObjectAdapterIsIsolatedFromOriginal() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();

        addObjectToStore(object);

        NakedObjectSpecification specification = object.getSpecification();
        NakedObject retrievedObject = store.getObject(object.getOid(), specification);
        assertNotSame(object, retrievedObject);
        assertEquals(object.getObject(), retrievedObject.getObject());
    }

    public void testResetClearsAdapterFromLoader() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();
        addObjectToStore(object);

        assertNull(NakedObjectsContext.getObjectLoader().getAdapterFor(object.getObject()));
    }

    public void testHasInstances() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();
        addObjectToStore(object);

        NakedObjectSpecification specification = object.getSpecification();
        assertEquals(true, store.hasInstances(specification, false));
    }

    public void testRetrievedInstancesByTitle() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();
        object.setupTitleString("title string");
        addObjectToStore(object);

        NakedObjectSpecification specification = object.getSpecification();
        NakedObject[] retrievedInstance = store.getInstances(new TitleCriteria(specification, "le STR", false));
        assertEquals(1, retrievedInstance.length);
        assertNotSame(object, retrievedInstance[0]);
        assertSame(object.getObject(), retrievedInstance[0].getObject());
    }

    public void testOidForService() throws Exception {
        TestProxyOid oid = new TestProxyOid(14);
        store.registerService("service name", oid);
        reset();
        Oid oidForService = store.getOidForService("service name");
        assertEquals(oid, oidForService);
    }

    private void reset() {
        store.reset();
        NakedObjectsContext.getObjectLoader().reset();
    }

    public void testCantRegisterServiceMoreThanOnce() throws Exception {
        TestProxyOid oid = new TestProxyOid(14);
        store.registerService("service name", oid);
        oid = new TestProxyOid(15);
        try {
            store.registerService("service name", oid);
            fail();
        } catch (NakedObjectRuntimeException expected) {}
    }
    
    public void testRemoveInstance() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();
        addObjectToStore(object);
        
        PersistenceCommand command = store.createDestroyObjectCommand(object);
        assertEquals(object, command.onObject());
        store.execute(new PersistenceCommand[] { command });
        
        reset();
        
        NakedObjectSpecification specification = object.getSpecification();
        assertEquals(false, store.hasInstances(specification, false));
    }

    
    public void testSaveInstance() throws Exception {
        TestProxyNakedObject object = system.createPersistentTestObject();
        addObjectToStore(object);
        
        NakedObjectSpecification specification = object.getSpecification();
        NakedObject[] retrievedInstance = store.getInstances(new TitleCriteria(specification, "changed", false));
        assertEquals(0, retrievedInstance.length);

        object.setupTitleString("changed title");
        PersistenceCommand command = store.createSaveObjectCommand(object);
        assertEquals(object, command.onObject());
        store.execute(new PersistenceCommand[] { command });
        
        reset();
        
        retrievedInstance = store.getInstances(new TitleCriteria(specification, "changed", false));
        assertEquals(1, retrievedInstance.length);
        assertNotSame(object, retrievedInstance[0]);
    }

    private void addObjectToStore(TestProxyNakedObject object) {
        PersistenceCommand command = store.createCreateObjectCommand(object);
        assertEquals(object, command.onObject());
        store.execute(new PersistenceCommand[] { command });
        reset();
    }

}

// Copyright (c) Naked Objects Group Ltd.
