package org.nakedobjects.nof.persist.objectstore.inmemory;

import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.Version;
import org.nakedobjects.noa.persist.InstancesCriteria;
import org.nakedobjects.noa.persist.ObjectNotFoundException;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.adapter.SerialNumberVersion;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nof.persist.transaction.PersistenceCommand;
import org.nakedobjects.nof.persist.transaction.SaveObjectCommand;
import org.nakedobjects.nof.testsystem.TestPojo;
import org.nakedobjects.nof.testsystem.TestProxyNakedObject;
import org.nakedobjects.nof.testsystem.TestProxyOid;
import org.nakedobjects.nof.testsystem.TestProxySpecification;
import org.nakedobjects.nof.testsystem.TestProxySystem;
import org.nakedobjects.testing.TestSpecification;

import java.util.Vector;

import junit.framework.TestCase;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;


class TestCriteria implements InstancesCriteria {
    private boolean includeSubclasses;
    private final Vector matches = new Vector();
    private final NakedObjectSpecification spec;

    public TestCriteria(final NakedObjectSpecification spec, final boolean includeSubclasses) {
        this.spec = spec;
        this.includeSubclasses = includeSubclasses;
    }

    public void addMatch(final NakedObject match) {
        matches.addElement(match);
    }

    public NakedObjectSpecification getSpecification() {
        return spec;
    }

    public boolean includeSubclasses() {
        return includeSubclasses;
    }

    public boolean matches(final NakedObject object) {
        return matches.contains(object);
    }
}

class AllCriteria implements InstancesCriteria {
    private boolean includeSubclasses;
    private final NakedObjectSpecification spec;

    public AllCriteria(final NakedObjectSpecification spec, final boolean includeSubclasses) {
        this.spec = spec;
        this.includeSubclasses = includeSubclasses;
    }

    public NakedObjectSpecification getSpecification() {
        return spec;
    }

    public boolean includeSubclasses() {
        return includeSubclasses;
    }

    public boolean matches(final NakedObject object) {
        return true;
    }
}

public class TransientObjectStoreTest extends TestCase {
    private int nextId;
    private TestProxySpecification objectSpec;
    private TransientObjectStore objectStore;
    private TestProxySpecification superClassObjectSpec;
    private NakedObject object1;
    private NakedObject object2;
    private NakedObject object3;
    private NakedObject object4;
    private NakedObject object5;
    private TestProxySpecification superSuperClassObjectSpec;
    private TestProxySystem system;

    private void assertEquals(final NakedObject object, final NakedObject v) {
        assertEquals(object.getObject(), v.getObject());
        assertEquals(object.getOid(), v.getOid());
    }

    private NakedObject createTestObject() {
        TestProxyNakedObject nakedObject = new TestProxyNakedObject();
        TestProxyOid oid = new TestProxyOid(nextId++, true);
        nakedObject.setupOid(oid);
        nakedObject.setupSpecification(objectSpec);
        nakedObject.setupObject(new TestPojo());
        return nakedObject;
    }

    private NakedObject createTestSuperClassObject() {
        TestProxyNakedObject nakedObject = new TestProxyNakedObject();
        TestProxyOid oid = new TestProxyOid(nextId++, true);
        nakedObject.setupOid(oid);
        nakedObject.setupSpecification(superClassObjectSpec);
        nakedObject.setupObject(new TestPojo());
        return nakedObject;
    }

    protected void setUp() throws Exception {
        BasicConfigurator.configure();
        Logger.getRootLogger().setLevel(Level.OFF);
        
        system = new TestProxySystem();
        system.init();

        superSuperClassObjectSpec = new TestProxySpecification(TestPojo.class);
        superClassObjectSpec = new TestProxySpecification(TestPojo.class);
        objectSpec = new TestProxySpecification(TestPojo.class);
        superClassObjectSpec.setupSubclasses(new NakedObjectSpecification[] { objectSpec });
        superSuperClassObjectSpec.setupSubclasses(new NakedObjectSpecification[] { superClassObjectSpec });

        objectStore = new TransientObjectStore();
        objectStore.init();

        object1 = createTestObject();
        object2 = createTestObject();
        object3 = createTestSuperClassObject();
        object4 = createTestSuperClassObject();
        object5 = createTestObject();
        PersistenceCommand[] commands = new PersistenceCommand[] { objectStore.createCreateObjectCommand(object1),
                objectStore.createCreateObjectCommand(object2), objectStore.createCreateObjectCommand(object3),
                objectStore.createCreateObjectCommand(object4), objectStore.createCreateObjectCommand(object5) };
        objectStore.startTransaction();
        objectStore.execute(commands);
        objectStore.endTransaction();
        
        system.resetLoader();
    }

    protected void tearDown() throws Exception {
        objectStore.shutdown();
    }

    public void testCreateInstances() throws Exception {
        assertEquals(3, objectStore.getInstances(new AllCriteria(objectSpec, false)).length);
    }

    public void testDestroyObject() throws Exception {
        PersistenceCommand[] commands = new PersistenceCommand[] { objectStore.createDestroyObjectCommand(object1) };
        objectStore.execute(commands);

        NakedObject[] instances = objectStore.getInstances(new AllCriteria(objectSpec, false));
        assertEquals(2, instances.length);
        assertEquals(object2, instances[0]);
        assertEquals(object5, instances[1]);
    }

    public void testGetInstancesByCriteria() throws Exception {
        TestCriteria criteria = new TestCriteria(superClassObjectSpec, false);
        criteria.addMatch(object1);
        criteria.addMatch(object3);

        NakedObject[] instances = objectStore.getInstances(criteria);
        assertEquals(1, instances.length);
        assertEquals(object3, instances[0]);
    }

    public void testGetInstancesByCriteriaIncludingSubclasses() throws Exception {
        TestCriteria criteria = new TestCriteria(superClassObjectSpec, true);
        criteria.addMatch(object1);
        criteria.addMatch(object3);
        NakedObject[] instances = objectStore.getInstances(criteria);

        assertEquals(2, instances.length);
        assertEquals(object3, instances[0]);
        assertEquals(object1, instances[1]);
    }

    public void testGetInstancesBySpecification() throws Exception {
        NakedObject[] instances = objectStore.getInstances(new AllCriteria(superClassObjectSpec, false));
        assertEquals(2, instances.length);
        assertEquals(object3, instances[0]);
        assertEquals(object4, instances[1]);
    }

    public void testGetInstancesBySpecificationIncludingSubclasses() throws Exception {
        NakedObject[] instances = objectStore.getInstances(new AllCriteria(superClassObjectSpec, true));
        assertEquals(5, instances.length);
        assertEquals(object3, instances[0]);
        assertEquals(object4, instances[1]);
        assertEquals(object1, instances[2]);
        assertEquals(object2, instances[3]);
        assertEquals(object5, instances[4]);
    }

    public void testGetObject() throws Exception {
        assertEquals(object1, objectStore.getObject(object1.getOid(), objectSpec));
        assertEquals(object3, objectStore.getObject(object3.getOid(), superClassObjectSpec));
    }

    public void testGetObjectCantFindObject() throws Exception {
        try {
            objectStore.getObject(new TestProxyOid(-3, true), objectSpec);
            fail();
        } catch (ObjectNotFoundException expected) {}
    }

    public void testHasInstances() throws Exception {
        assertTrue(objectStore.hasInstances(superClassObjectSpec, false));
    }

    public void testHasInstancesIncludingSubclasses() throws Exception {
        assertTrue(objectStore.hasInstances(superSuperClassObjectSpec, true));
    }

    public void testHasNoInstances() throws Exception {
        TestSpecification spec = new TestSpecification();
        assertFalse(objectStore.hasInstances(spec, false));
    }

    public void testHasNoInstancesIncludingSubclasses() throws Exception {
        TestSpecification spec = new TestSpecification();
        TestSpecification superSpec = new TestSpecification();
        superSpec.setupSubclasses(new NakedObjectSpecification[] { spec });
        assertFalse(objectStore.hasInstances(superSpec, true));
    }

    public void testNumberOfInstancesIncludingSubclasses() throws Exception {
        assertEquals(5, objectStore.getInstances(new AllCriteria(superClassObjectSpec, true)).length);
    }

    public void testShutdown() throws Exception {
        objectStore.shutdown();
        assertEquals(0, objectStore.getInstances(new AllCriteria(superClassObjectSpec, false)).length);
        assertEquals(0, objectStore.getInstances(new AllCriteria(objectSpec, false)).length);
    }

    public void testVersionOfRetrievedObject() throws Exception {
        NakedObject obj = objectStore.getObject(object1.getOid(), null);
        assertSame(object1.getObject(), obj.getObject());
        assertEquals(1L, ((SerialNumberVersion) obj.getVersion()).getSequence());
        assertEquals("user", obj.getVersion().getUser());
    }

    public void testVersionOfNewObject() throws Exception {
        assertEquals("user", object1.getVersion().getUser());
        assertEquals(1L, ((SerialNumberVersion) object1.getVersion()).getSequence());
    }

    public void testVersionOfSavedObject() throws Exception {
        SaveObjectCommand saveCommand = objectStore.createSaveObjectCommand(object1);
        objectStore.execute(new PersistenceCommand[] {saveCommand});
        assertEquals("user", object1.getVersion().getUser());
        assertEquals(2L, ((SerialNumberVersion) object1.getVersion()).getSequence());
    }
    
    public void testDebug() {
        DebugString dtr = new DebugString();
        objectStore.debugData(dtr);
        assertTrue(dtr.toString().length() > 0);
    }

    public void testSave() {
        ((TestProxyNakedObject) object3).setupTitleString("title");

        Version version = object3.getVersion();
        PersistenceCommand[] commands = new PersistenceCommand[] { objectStore.createSaveObjectCommand(object3) };
        objectStore.startTransaction();
        objectStore.execute(commands);
        objectStore.endTransaction();

        assertNotSame(version, object3.getVersion());

    }
}
// Copyright (c) Naked Objects Group Ltd.
